#!/usr/bin/env wish
#  projfiles.tcl
#  Copyright (C) 2021-2024 Kostas Michalopoulos
#
#  This software is provided 'as-is', without any express or implied
#  warranty.  In no event will the authors be held liable for any damages
#  arising from the use of this software.
#
#  Permission is granted to anyone to use this software for any purpose,
#  including commercial applications, and to alter it and redistribute it
#  freely, subject to the following restrictions:
#
#  1. The origin of this software must not be misrepresented; you must not
#     claim that you wrote the original software. If you use this software
#     in a product, an acknowledgment in the product documentation would be
#     appreciated but is not required.
#  2. Altered source versions must be plainly marked as such, and must not be
#     misrepresented as being the original software.
#  3. This notice may not be removed or altered from any source distribution.
#
#  Kostas Michalopoulos <badsector@runtimeterror.com>

set Title {Project Files}
set FilePatterns {*.c *.h}
set MakeCommand {make all ; read -p "Press Enter to continue... "}
set CleanCommand {make clean ; read -p "Press Enter to continue... "}
set EditCommand {vi %%PATH%%}

# Example for Configurations (pattern is name1 code1 name2 code2, etc):
# set Configurations {
#     "Normal" { set FilePatterns { *.cpp *.hpp *.c *.h } MakeCommand { make normal } }
#     "Extra" { set FilePatterns { *.cpp *.hpp *.c *.h *.x } MakeCommand { make extra } }
# }
set Configurations {}
# Like Configurations but adds buttons. Two dashes (e.g. Buttons { - - }) make a new row.
set Buttons {}

# The .custom frame can be used to add custom GUI if needed, for example:
#
# button .custom.r -text Hello -command { puts Hello }
# pack .custom.r

# Sizes
set DirCount 12
set FileCount 24
set ViewWidth 40

# The following functions can be used and/or redefined by scripts:

# InputPrompt {prompt} - shows a dialog box with an input line
# RunTerminalCommand {cmd} - runs the command $cmd in the terminal
# RunTerminalAt {path} - opens a terminal window in the directory at $path
# RunMakeCommand {cmd} - run the command $cmd for 'making' the project
# RunCleanCommand {cmd} - run the command $cmd for 'cleaning' the project
# RunEditCommand {cmd} - run the command $cmd for launching the editor

################################################################################
# Internal stuff -- do not use any of the stuff below this line as they may
# change in future versions of projfiles.tcl and your scripts will stop working!
################################################################################
set guidone 0
set usepats 1
set usefilter 1
set showsubdirs 1
set lastcurdir {.}
set filtertxt {}

if {$argc >= 1} { cd [lindex $argv 0] }

if { [lindex $tcl_platform(os) 0] == "Windows" } {
    set IsWindows 1
    set EditCommand {nano %%PATH%%}
    set minttyfound 0
    catch {
        if {[string first mintty [exec mintty.exe --version]] != -1} {
            set minttyfound 1
        }
    }
} else {
    set IsWindows 0
}

proc InputPrompt {prompt} {
    global IsWindows InputPrompt_value InputPrompt_visible
    if {[info exists InputPrompt_visible] && $InputPrompt_visible} return
    set InputPrompt_visible 1
    set InputPrompt_value {}
    toplevel .t
    if {!$IsWindows} {
        wm attributes .t -type dialog
    }
    label .t.prompt -text $prompt
    entry .t.value -width 40 -textvariable InputPrompt_value
    button .t.ok -text "OK" -command { destroy .t }
    button .t.cancel -text "Cancel" -command { 
        set InputPrompt_value {}
        destroy .t
    }
    grid .t.prompt .t.value -sticky ew
    grid .t.ok .t.cancel
    grid configure .t.prompt -column 0 -row 0 -columnspan 2
    grid configure .t.value -column 0 -row 1 -columnspan 2
    grid configure .t.ok -column 0 -row 2
    grid configure .t.cancel -column 1 -row 2
    grid columnconfigure .t .t.value -weight 1
    bind .t <Return> { .t.ok invoke }
    bind .t <Escape> { .t.cancel invoke }
    focus .t.value
    wm title .t Question
    tkwait window .t
    set InputPrompt_visible 0
    return $InputPrompt_value
}

proc RunTerminalCommand {cmd} {
    global IsWindows
    if $IsWindows {
        exec mintty.exe -e bash -c $cmd &
    } else {
        exec x-terminal-emulator --command $cmd &
    }
}

proc RunTerminalAt {path} {
    global IsWindows
    if $IsWindows {
        exec mintty.exe -e bash -c "cd \"$path\"; exec bash" &
    } else {
        exec x-terminal-emulator "--working-directory=\"$path\"" &
    }
}

proc RunMakeCommand {cmd} {
    RunTerminalCommand $cmd
}

proc RunCleanCommand {cmd} {
    RunTerminalCommand $cmd
}

proc RunEditCommand {cmd} {
    RunTerminalCommand $cmd
}

if [file exists "$env(HOME)/.projfilesrc.tcl"] {
    source "$env(HOME)/.projfilesrc.tcl"
}

if [file exists pwconfig.tcl] {
    source pwconfig.tcl
}

set configs {}
foreach {name code} $Configurations {
    set "CfgCode_${name}" $code
    lappend configs $name
}
if {$configs == ""} {
    set configs {None}
    set CfgCode_None {}
}

frame .buttons
if {$Buttons != ""} {
    set i 0
    set r 0
    frame .buttons.r0
    foreach {name code} $Buttons {
        if {$name == "-" && $code == "-"} {
            pack .buttons.r$r -fill x
            incr r
            frame .buttons.r$r
            set i 0
            continue
        }
        button .buttons.r${r}.btn$i -text $name -command $code
        grid .buttons.r${r}.btn$i -column $i -row 0 -sticky ew
        grid columnconfigure .buttons.r${r} .buttons.r${r}.btn$i -weight 1
        incr i
    }
    pack .buttons.r$r -fill x
}

proc scanonedir {dir updatedirs issub} {
    global alldirs allfiles curdir usepats usefilter filtertxt showsubdirs FilePatterns
    if $usepats {
        set patterns $FilePatterns
        set foundany 0
    } else {
        set patterns {*}
        set foundany 1
    }
    foreach pat $patterns {
        if [string equal {-} [string range $pat 0 0]] {
            continue
        }
        foreach file [glob -nocomplain -directory $dir -type f $pat] {
            set skipthis 0
            foreach nopat $patterns {
                if [string equal - [string range $nopat 0 0]] {
                    if [string match [string range $nopat 1 end] $file] {
                        set skipthis 1
                        break
                    }
                }
            }
            if $skipthis continue
            if [expr ( $issub && $showsubdirs ) || !$issub] {
                if [expr $usefilter && ![string equal $filtertxt {}]] {
                    set skipthis 1
                    foreach flt $filtertxt {
                        if [string match $flt $file] {
                            set skipthis 0
                            break
                        }
                    }
                }
                if [expr !$skipthis] {
                    lappend allfiles [string range $file [expr [string length $curdir] + 1] end]
                }
            }
            set foundany 1
        }
    }
    foreach subdir [glob -nocomplain -directory $dir -type d *] {
        scanonedir $subdir $updatedirs 1
    }
    if {$updatedirs && ($foundany || [string equal . $dir])} {
        lappend alldirs $dir
    }
}

proc scanprojdir {} {
    global alldirs allfiles curdir lastcurdir
    set alldirs {}
    set allfiles {}
    set curdir .
    set lastcurdir .
    scanonedir . 1 0
    set alldirs [lsort $alldirs]
    set allfiles [lsort $allfiles]
}

proc scanspecdir {dir} {
    global allfiles curdir
    set allfiles {}
    set curdir $dir
    scanonedir $curdir 0 0
    set allfiles [lsort $allfiles]
}

proc scancurdir {} {
    global lastcurdir
    scanspecdir $lastcurdir
}

proc selectfile {file} {
    global allfiles
    set index [lsearch -exact $allfiles $file]
    if {$index != -1} {
        focus .filesframe.files
        .filesframe.files see $index
        .filesframe.files activate $index
    }
}

proc domake {} {
    global MakeCommand
    RunMakeCommand $MakeCommand
}

proc doclean {} {
    global CleanCommand
    RunCleanCommand $CleanCommand
}

proc donew {} {
    global curdir
    set filename [string trim [InputPrompt "Create new file under $curdir"]]
    if {[string length $filename] == 0} return
    set filename "${curdir}/$filename"
    exec touch $filename
    doeditfile $filename
    scanspecdir $curdir
    selectfile [string range $filename [expr [string length $curdir] + 1] end]
}

proc doeditfile {path} {
    global EditCommand
    set mapping {}
    lappend mapping {%%PATH%%}
    lappend mapping $path
    set cmd [string map $mapping $EditCommand]
    RunEditCommand $cmd
}

proc doedit {} {
    global curdir
    if {[string length [.filesframe.files curselection]] == 0} return
    doeditfile "${curdir}/[.filesframe.files get [.filesframe.files curselection]]"
}

proc doterm {} {
    global curdir
    RunTerminalAt [file normalize $curdir]
}

scanprojdir

proc curconfigtracer {var args} {
    global curconfig
    global CfgCode_${curconfig}
    uplevel #0 [set CfgCode_${curconfig}]
    scancurdir
}

trace add variable curconfig write "curconfigtracer v"

wm title . $Title
button .refresh -text Refresh -command { scanprojdir }
button .make -text Make -command { domake }
button .clean -text Clean -command { doclean }
button .new -text New -command { donew }
button .term -text Term -command { doterm }
tk_optionMenu .configs curconfig {*}$configs
frame .custom
frame .patline
checkbutton .patline.usepats -variable usepats -command { scanprojdir }
entry .patline.pats -textvariable FilePatterns
frame .dirsframe
listbox .dirsframe.dirs -height $DirCount -listvariable alldirs -width $ViewWidth -yscrollcommand {.dirsframe.scroll set}
scrollbar .dirsframe.scroll -command {.dirsframe.dirs yview}
frame .filesframe
listbox .filesframe.files -height $FileCount -listvariable allfiles -width $ViewWidth -yscrollcommand {.filesframe.scroll set}
scrollbar .filesframe.scroll -command {.filesframe.files yview}
frame .filesframe.cfg
checkbutton .filesframe.cfg.subdirs -text Subdirs -variable showsubdirs -command { scancurdir }
checkbutton .filesframe.cfg.usefilter -text Filter: -variable usefilter -command { scancurdir }
entry .filesframe.cfg.filter -textvariable filtertxt
bind .filesframe.cfg.filter <KeyPress> {
    if [string equal %K Return] {
        scancurdir
    }
}
pack .patline.usepats -side left
pack .patline.pats -fill x
pack .dirsframe.scroll -side right -fill y
pack .dirsframe.dirs -fill both
pack .filesframe.cfg.subdirs -side left
pack .filesframe.cfg.usefilter -side left
pack .filesframe.cfg.filter -fill x
pack .filesframe.cfg -side top -fill x
pack .filesframe.scroll -side right -fill y
pack .filesframe.files -fill both -expand 1
grid .refresh .make .clean .new .term .buttons .configs .patline .dirsframe .filesframe -sticky nsew
grid configure .refresh -column 0 -row 0
grid configure .make -column 1 -row 0
grid configure .clean -column 2 -row 0
grid configure .new -column 3 -row 0
grid configure .term -column 4 -row 0
grid configure .buttons -column 0 -row 1 -columnspan 5
grid configure .configs -column 0 -row 2 -columnspan 5
grid configure .custom -column 0 -row 3 -columnspan 5
grid configure .patline -column 0 -row 4 -columnspan 5
grid configure .dirsframe -column 0 -row 5 -columnspan 5
grid configure .filesframe -column 0 -row 6 -columnspan 5
grid rowconfigure . .filesframe -weight 1
grid columnconfigure . .filesframe -weight 1

set guidone 1

if {$Configurations == ""} {
    grid forget .configs
}

if {$Buttons == ""} {
    grid forget .buttons
}

bind .dirsframe.dirs <<ListboxSelect>> {
    if {[string length [.dirsframe.dirs curselection]] > 0} {
        set lastcurdir [.dirsframe.dirs get [.dirsframe.dirs curselection]]
        scancurdir
    }
}

bind .filesframe.files <Double-Button-1> { doedit }

if {$IsWindows && !$minttyfound} {
    tk_messageBox -type ok -icon warning -message "Couldn't locate mintty.exe, make sure Wish is running under MSYS2" -title "mintty not found"
}

if {[info exists InputPrompt_visible] && $InputPrompt_visible} return
