unit UMain;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Forms, Controls, Graphics, Dialogs, Menus, ExtCtrls,
  ComCtrls, StdCtrls, Spin, Buttons, Viewports, FirstTimeInit, Maths,
  UPieces;

type

  { TMain }

  TMain = class(TForm)
    MainMenu1: TMainMenu;
    mFile: TMenuItem;
    mFileNew: TMenuItem;
    mFileOpen: TMenuItem;
    mFileSave: TMenuItem;
    mFileSaveAs: TMenuItem;
    mFileExit: TMenuItem;
    MenuItem7: TMenuItem;
    plSidebar: TPanel;
    Splitter1: TSplitter;
    tvPieces: TTreeView;
    Splitter2: TSplitter;
    Panel1: TPanel;
    Viewport1: TViewport;
    ViewportManager1: TViewportManager;
    FirstTimeInit1: TFirstTimeInit;
    mView: TMenuItem;
    mViewWireframe: TMenuItem;
    Label1: TLabel;
    fseRefX: TFloatSpinEdit;
    fseRefY: TFloatSpinEdit;
    fseRefZ: TFloatSpinEdit;
    sbZeroRef: TSpeedButton;
    sbToCenter: TSpeedButton;
    sbToTop: TSpeedButton;
    sbToBottom: TSpeedButton;
    Bevel1: TBevel;
    sbSetOrigin: TSpeedButton;
    sbGetOrigin: TSpeedButton;
    Label2: TLabel;
    fsePosX: TFloatSpinEdit;
    fsePosY: TFloatSpinEdit;
    fsePosZ: TFloatSpinEdit;
    Label3: TLabel;
    fseRotX: TFloatSpinEdit;
    fseRotY: TFloatSpinEdit;
    fseRotZ: TFloatSpinEdit;
    sbResetTransform: TSpeedButton;
    sbCopyTransform: TSpeedButton;
    sbPastePosition: TSpeedButton;
    sbPasteRotation: TSpeedButton;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    Panel2: TPanel;
    Label4: TLabel;
    cbAnimations: TComboBox;
    pbTimeline: TPaintBox;
    sbSetKey: TSpeedButton;
    sbDelKey: TSpeedButton;
    SpeedButton1: TSpeedButton;
    MenuItem1: TMenuItem;
    mAnimationAdd: TMenuItem;
    mAnimationEdit: TMenuItem;
    mAnimationRemove: TMenuItem;
    mAnimationPreviousFrame: TMenuItem;
    MenuItem6: TMenuItem;
    cbAnimationNextFrame: TMenuItem;
    MenuItem2: TMenuItem;
    mAnimationSetFrames: TMenuItem;
    mAnimationInsertFrame: TMenuItem;
    mAnimationDeleteFrame: TMenuItem;
    sbSaveRootPosition: TSpeedButton;
    sbCopyPose: TSpeedButton;
    sbPastePose: TSpeedButton;
    sbClearPose: TSpeedButton;
    sbRestoreRootPosition: TSpeedButton;
    sbPasteFromPose: TSpeedButton;
    mAnimationLooping: TMenuItem;
    MenuItem4: TMenuItem;
    tmrAnimationPlayback: TTimer;
    mAnimationFirstFrame: TMenuItem;
    mAnimationLastFrame: TMenuItem;
    mAnimationPlay: TMenuItem;
    MenuItem5: TMenuItem;
    mAnimationSetFPS: TMenuItem;
    mAnimationSmooth: TMenuItem;
    pmPieces: TPopupMenu;
    pmPiecesAddPiece: TMenuItem;
    pmPiecesLoadMesh: TMenuItem;
    MenuItem3: TMenuItem;
    pmPiecesRename: TMenuItem;
    MenuItem8: TMenuItem;
    pmPiecesDelete: TMenuItem;
    odMesh: TOpenDialog;
    mHelp: TMenuItem;
    pmHelpAbout: TMenuItem;
    procedure FirstTimeInit1Initialize(Sender: TObject);
    procedure Viewport1Render(Sender: TObject);
    procedure tvPiecesSelectionChanged(Sender: TObject);
    procedure Viewport1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Viewport1RenderOverlay(Sender: TObject);
    procedure sbZeroRefClick(Sender: TObject);
    procedure sbToCenterClick(Sender: TObject);
    procedure sbToTopClick(Sender: TObject);
    procedure sbToBottomClick(Sender: TObject);
    procedure fseRefXChange(Sender: TObject);
    procedure fseRefYChange(Sender: TObject);
    procedure fseRefZChange(Sender: TObject);
    procedure sbSetOriginClick(Sender: TObject);
    procedure sbGetOriginClick(Sender: TObject);
    procedure fsePosXChange(Sender: TObject);
    procedure fsePosYChange(Sender: TObject);
    procedure fsePosZChange(Sender: TObject);
    procedure fseRotXChange(Sender: TObject);
    procedure fseRotYChange(Sender: TObject);
    procedure fseRotZChange(Sender: TObject);
    procedure sbResetTransformClick(Sender: TObject);
    procedure sbCopyTransformClick(Sender: TObject);
    procedure sbPastePositionClick(Sender: TObject);
    procedure sbPasteRotationClick(Sender: TObject);
    procedure mFileNewClick(Sender: TObject);
    procedure mFileOpenClick(Sender: TObject);
    procedure mFileSaveClick(Sender: TObject);
    procedure mFileSaveAsClick(Sender: TObject);
    procedure pbTimelinePaint(Sender: TObject);
    procedure cbAnimationsSelect(Sender: TObject);
    procedure mAnimationPreviousFrameClick(Sender: TObject);
    procedure cbAnimationNextFrameClick(Sender: TObject);
    procedure sbSetKeyClick(Sender: TObject);
    procedure sbDelKeyClick(Sender: TObject);
    procedure mAnimationSetFramesClick(Sender: TObject);
    procedure mAnimationInsertFrameClick(Sender: TObject);
    procedure mAnimationDeleteFrameClick(Sender: TObject);
    procedure mAnimationAddClick(Sender: TObject);
    procedure cbAnimationsChange(Sender: TObject);
    procedure mAnimationEditClick(Sender: TObject);
    procedure mAnimationRemoveClick(Sender: TObject);
    procedure pbTimelineMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure sbSaveRootPositionClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure sbCopyPoseClick(Sender: TObject);
    procedure sbPastePoseClick(Sender: TObject);
    procedure sbClearPoseClick(Sender: TObject);
    procedure pbTimelineMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure pbTimelineMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure sbRestoreRootPositionClick(Sender: TObject);
    procedure sbPasteFromPoseClick(Sender: TObject);
    procedure pbTimelineDblClick(Sender: TObject);
    procedure mAnimationLoopingClick(Sender: TObject);
    procedure mAnimationFirstFrameClick(Sender: TObject);
    procedure mAnimationLastFrameClick(Sender: TObject);
    procedure mAnimationPlayClick(Sender: TObject);
    procedure SpeedButton1Click(Sender: TObject);
    procedure tmrAnimationPlaybackTimer(Sender: TObject);
    procedure mAnimationSetFPSClick(Sender: TObject);
    procedure mAnimationSmoothClick(Sender: TObject);
    procedure pmPiecesAddPieceClick(Sender: TObject);
    procedure pmPiecesPopup(Sender: TObject);
    procedure pmPiecesRenameClick(Sender: TObject);
    procedure pmPiecesDeleteClick(Sender: TObject);
    procedure pmPiecesLoadMeshClick(Sender: TObject);
    procedure pmHelpAboutClick(Sender: TObject);
  private
    FSelectedPiece: TPiece;
    procedure SetSelectedPiece(AValue: TPiece);
  private
    Model: TPieceModel;
    Animation: TAnimation;
    CurFrame: Integer;
    Ref: TVector;
    ScratchTransform: TTransform;
    ScratchFrame: TFrame;
    FileName: string;
    BaseTime: QWord;
    procedure UpdateCaption;
    procedure UpdatePiecesTree;
    procedure UpdateAnimationsList;
    procedure UpdateTimeline;
    procedure UpdateAnimation;
    procedure NewModel;
    procedure FreeModel;
    procedure RenderModel;
    procedure UpdateRefUI;
    procedure UpdateTransformUI;
    property SelectedPiece: TPiece read FSelectedPiece write SetSelectedPiece;
  public
  end;

var
  Main: TMain;

implementation

uses
  GL, MeshLoader;

{$R *.lfm}

{ TMain }

procedure TMain.FirstTimeInit1Initialize(Sender: TObject);
begin
  NewModel;
end;

procedure TMain.Viewport1Render(Sender: TObject);
begin
  RenderModel;
end;

procedure TMain.tvPiecesSelectionChanged(Sender: TObject);
begin
  if Assigned(tvPieces.Selected) and Assigned(tvPieces.Selected.Data) then
    SelectedPiece:=TPiece(tvPieces.Selected.Data);
end;

procedure TMain.Viewport1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  Ray: TRay;
  RayPiece: TPiece;
begin
  Ray:=Viewport1.RayAt(X, Y);
  RayPiece:=Model.RayPick(Ray);
  if Assigned(RayPiece) then SelectedPiece:=RayPiece;
end;

procedure TMain.Viewport1RenderOverlay(Sender: TObject);
var
  V: TVector;
begin
  glDisable(GL_DEPTH_TEST);
  glPointSize(6);
  glColor3f(0, 1, 1);
  glBegin(GL_POINTS);
  glVertex3d(Ref.X, Ref.Y, Ref.Z);
  glEnd();
  glPointSize(4);
  glColor3f(1, 1, 0);
  glBegin(GL_POINTS);
  glVertex3d(Ref.X, Ref.Y, Ref.Z);
  if Assigned(SelectedPiece) then begin
    glColor3f(1, 0, 0);
    V:=SelectedPiece.WorldMatrix.Transformed(Vector(0, 0, 0));
    glVertex3d(V.x,
               V.y,
               V.z);
  end;
  glEnd();
  glEnable(GL_DEPTH_TEST);
  glPointSize(1);
end;

procedure TMain.sbZeroRefClick(Sender: TObject);
begin
  Ref.Zero;
  UpdateRefUI;
end;

procedure TMain.sbToCenterClick(Sender: TObject);
begin
  if not Assigned(SelectedPiece) then Exit;
  if Assigned(SelectedPiece.Mesh) then
    Ref:=SelectedPiece.Mesh.CalcBoundingBox.Center
  else
    Ref.Zero;
  SelectedPiece.WorldMatrix.Transform(Ref);
  UpdateRefUI;
end;

procedure TMain.sbToTopClick(Sender: TObject);
begin
  if not Assigned(SelectedPiece) then Exit;
  if Assigned(SelectedPiece.Mesh) then begin
    Ref:=SelectedPiece.Mesh.CalcBoundingBox.Center;
    Ref.Y:=SelectedPiece.Mesh.CalcBoundingBox.b.y;
  end else
    Ref.Zero;
  SelectedPiece.WorldMatrix.Transform(Ref);
  UpdateRefUI;
end;

procedure TMain.sbToBottomClick(Sender: TObject);
begin
  if not Assigned(SelectedPiece) then Exit;
  if Assigned(SelectedPiece.Mesh) then begin
    Ref:=SelectedPiece.Mesh.CalcBoundingBox.Center;
    Ref.Y:=SelectedPiece.Mesh.CalcBoundingBox.a.y;
  end else
    Ref.Zero;
  SelectedPiece.WorldMatrix.Transform(Ref);
  UpdateRefUI;
end;

procedure TMain.fseRefXChange(Sender: TObject);
begin
  Ref.x:=fseRefX.Value;
end;

procedure TMain.fseRefYChange(Sender: TObject);
begin
  Ref.y:=fseRefY.Value;
end;

procedure TMain.fseRefZChange(Sender: TObject);
begin
  Ref.z:=fseRefZ.Value;
end;

procedure TMain.sbSetOriginClick(Sender: TObject);
var
  T: TTransform;
  LRef, OldTranslation: TVector;
  I, J: Integer;
begin
  if not Assigned(SelectedPiece) then Exit;
  SelectedPiece.UpdateWorldMatrix;
  LRef:=Ref;
  if SelectedPiece.Parent is TPiece then
    TPiece(SelectedPiece.Parent).WorldMatrix.Inverted.Transform(LRef);
  T:=SelectedPiece.Transform;
  OldTranslation:=T.Translation;
  T.Translation:=LRef;
  SelectedPiece.Transform:=T;
  if Assigned(SelectedPiece.Mesh) then SelectedPiece.Mesh.SetOrigin(LRef.Subbed(OldTranslation));
  for I:=0 to SelectedPiece.ChildCount - 1 do with TPiece(SelectedPiece.Children[I]) do begin
    T:=Transform;
    T.Translation.Sub(LRef.Subbed(OldTranslation));
    Transform:=T;
  end;
  if SelectedPiece=Model.Root then begin
    for I:=0 to Model.AnimationCount - 1 do
      with Model.Animations[I] do
        for J:=0 to Frames - 1 do
          Frame[J].RootPos.Add(Model.Root.Transform.Translation.Subbed(OldTranslation));
  end;
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.sbGetOriginClick(Sender: TObject);
begin
  if Assigned(SelectedPiece) then begin
    Ref:=SelectedPiece.WorldMatrix.Transformed(Vector(0, 0, 0));
    UpdateRefUI;
  end;
end;

procedure TMain.fsePosXChange(Sender: TObject);
var
  T: TTransform;
begin
  if not Assigned(SelectedPiece) then Exit;
  T:=SelectedPiece.Transform;
  T.Translation.x:=fsePosX.Value;
  SelectedPiece.Transform:=T;
end;

procedure TMain.fsePosYChange(Sender: TObject);
var
  T: TTransform;
begin
  if not Assigned(SelectedPiece) then Exit;
  T:=SelectedPiece.Transform;
  T.Translation.y:=fsePosY.Value;
  SelectedPiece.Transform:=T;
end;

procedure TMain.fsePosZChange(Sender: TObject);
var
  T: TTransform;
begin
  if not Assigned(SelectedPiece) then Exit;
  T:=SelectedPiece.Transform;
  T.Translation.z:=fsePosZ.Value;
  SelectedPiece.Transform:=T;
end;

procedure TMain.fseRotXChange(Sender: TObject);
var
  T: TTransform;
begin
  if not Assigned(SelectedPiece) then Exit;
  T:=SelectedPiece.Transform;
  T.Rotation.x:=fseRotX.Value;
  SelectedPiece.Transform:=T;
end;

procedure TMain.fseRotYChange(Sender: TObject);
var
  T: TTransform;
begin
  if not Assigned(SelectedPiece) then Exit;
  T:=SelectedPiece.Transform;
  T.Rotation.y:=fseRotY.Value;
  SelectedPiece.Transform:=T;
end;

procedure TMain.fseRotZChange(Sender: TObject);
var
  T: TTransform;
begin
  if not Assigned(SelectedPiece) then Exit;
  T:=SelectedPiece.Transform;
  T.Rotation.z:=fseRotZ.Value;
  SelectedPiece.Transform:=T;
end;

procedure TMain.sbResetTransformClick(Sender: TObject);
var
  T: TTransform;
begin
  if Assigned(SelectedPiece) then begin
    T.Reset;
    SelectedPiece.Transform:=T;
    UpdateTransformUI;
  end;
end;

procedure TMain.sbCopyTransformClick(Sender: TObject);
begin
  if Assigned(SelectedPiece) then
    ScratchTransform:=SelectedPiece.Transform;
end;

procedure TMain.sbPastePositionClick(Sender: TObject);
var
  T: TTransform;
begin
  if Assigned(SelectedPiece) then begin
    T:=SelectedPiece.Transform;
    T.Translation:=ScratchTransform.Translation;
    SelectedPiece.Transform:=T;
    UpdateTransformUI;
  end;
end;

procedure TMain.sbPasteRotationClick(Sender: TObject);
var
  T: TTransform;
begin
  if Assigned(SelectedPiece) then begin
    T:=SelectedPiece.Transform;
    T.Rotation:=ScratchTransform.Rotation;
    SelectedPiece.Transform:=T;
    UpdateTransformUI;
  end;
end;

procedure TMain.mFileNewClick(Sender: TObject);
begin
  if MessageDlg('New Model', 'Are you sure that you want to proceed? Any changes will be lost.', mtConfirmation, mbYesNo, 0) <> mrYes then Exit;
  NewModel;
end;

procedure TMain.mFileOpenClick(Sender: TObject);
begin
  if MessageDlg('New Model', 'Are you sure that you want to proceed? Any changes will be lost.', mtConfirmation, mbYesNo, 0) <> mrYes then Exit;
  OpenDialog1.FileName:=FileName;
  if OpenDialog1.Execute then begin
    NewModel;
    Model.LoadFromFile(OpenDialog1.FileName);
    Animation:=Model.Animations[0];
    mAnimationSmooth.Checked:=Animation.Smooth;
    mAnimationLooping.Checked:=Animation.Looping;
    FileName:=OpenDialog1.FileName;
    UpdatePiecesTree;
    UpdateAnimation;
    UpdateAnimationsList;
    UpdateTimeline;
    UpdateCaption;
  end;
end;

procedure TMain.mFileSaveClick(Sender: TObject);
begin
  if FileName='' then begin
    mFileSaveAs.Click;
    Exit;
  end;
  Model.SaveToFile(FileName);
end;

procedure TMain.mFileSaveAsClick(Sender: TObject);
begin
  SaveDialog1.FileName:=FileName;
  if SaveDialog1.Execute then begin
    Model.SaveToFile(SaveDialog1.FileName);
    FileName:=SaveDialog1.FileName;
    UpdateCaption;
  end;
end;

procedure TMain.pbTimelinePaint(Sender: TObject);
var
  R: TRect;
  I: Integer;
begin
  with pbTimeline.Canvas do begin
    R:=pbTimeline.ClientRect;
    Brush.Color:=$404040;
    Pen.Color:=clBlack;
    Rectangle(R);
    if Assigned(Animation) then begin
      CurFrame:=CurFrame mod Animation.Frames;
      for I:=0 to Animation.Frames - 1 do begin
        if Animation.Frame[I].IsKey then Brush.Color:=clRed else Brush.Color:=clSilver;
        Rectangle(I*6, 0, 7 + I*6, r.Bottom);
        if I=CurFrame then begin
          Brush.Color:=clGreen;
          Rectangle(I*6 + 1, 3, 6 + I*6, r.Bottom - 3);
        end;
        if Animation.Frame[I].IsEvent then begin
          Brush.Color:=clYellow;
          Rectangle(I*6 + 1, r.Bottom - 6, 6 + I*6, r.bottom - 2);
        end;
      end;
    end;
    Frame3D(R, clBtnShadow, clBtnHighlight, 1);
  end;
end;

procedure TMain.cbAnimationsSelect(Sender: TObject);
begin
  if cbAnimations.ItemIndex=-1 then Exit;
  Animation:=TAnimation(cbAnimations.Items.Objects[cbAnimations.ItemIndex]);
  mAnimationSmooth.Checked:=Animation.Smooth;
  mAnimationLooping.Checked:=Animation.Looping;
  UpdateTimeline;
end;

procedure TMain.mAnimationPreviousFrameClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  if CurFrame=0 then CurFrame:=Animation.Frames - 1 else Dec(CurFrame);
  UpdateTimeline;
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.cbAnimationNextFrameClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  CurFrame:=(CurFrame + 1) mod Animation.Frames;
  UpdateTimeline;
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.sbSetKeyClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.SetKeyFrame(CurFrame);
  UpdateTimeline;
  UpdateAnimation;
end;

procedure TMain.sbDelKeyClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.DeleteKeyFrame(CurFrame);
  UpdateTimeline;
  UpdateAnimation;
end;

procedure TMain.mAnimationSetFramesClick(Sender: TObject);
var
  Frames: Integer;
begin
  Frames:=StrToIntDef(InputBox('Set animation frames',
    'Frame count (current is ' + IntToStr(Animation.Frames) + ', current index is ' + IntToStr(CurFrame) + '):',
    IntToStr(Animation.Frames)), Animation.Frames);
  if Frames < 0 then Frames:=0;
  Animation.Frames:=Frames;
  if Frames=0 then Animation.Frames:=1;
  UpdateTimeline;
  UpdateAnimation;
end;

procedure TMain.mAnimationInsertFrameClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.InsertFrame(CurFrame);
  UpdateTimeline;
end;

procedure TMain.mAnimationDeleteFrameClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.DeleteFrame(CurFrame);
  UpdateTimeline;
end;

procedure TMain.mAnimationAddClick(Sender: TObject);
var
  AnimName: string;
  NewAnim: TAnimation;
begin
  if not Assigned(Model) then Exit;
  AnimName:=Trim(InputBox('New Animation', 'Enter name (must be unique):', ''));
  if AnimName='' then Exit;
  NewAnim:=Model.AddAnimation(AnimName);
  if not Assigned(NewAnim) then begin
    ShowMessage('The name already exists');
    Exit;
  end;
  Animation:=NewAnim;
  Animation.ResetFramePose(0);
  Animation.Frame[0].IsKey:=True;
  mAnimationSmooth.Checked:=Animation.Smooth;
  mAnimationLooping.Checked:=Animation.Looping;
  CurFrame:=0;
  UpdateAnimationsList;
  UpdateTimeline;
  UpdateAnimation;
end;

procedure TMain.cbAnimationsChange(Sender: TObject);
begin
  if not Assigned(Model) then Exit;
  if cbAnimations.ItemIndex=-1 then Exit;
  Animation:=TAnimation(cbAnimations.Items.Objects[cbAnimations.ItemIndex]);
  mAnimationSmooth.Checked:=Animation.Smooth;
  mAnimationLooping.Checked:=Animation.Looping;
  CurFrame:=0;
  UpdateTimeline;
  UpdateAnimation;
end;

procedure TMain.mAnimationEditClick(Sender: TObject);
var
  NewName: String;
  Existing: TAnimation;
begin
  if not Assigned(Animation) then Exit;
  NewName:=Trim(InputBox('Edit Animation', 'Enter new name:', Animation.Name));
  Existing:=Model.FindAnimation(NewName);
  if Assigned(Existing) and (Existing <> Animation) then begin
    ShowMessage('An animation with the name "' + NewNAme + '" already exists.');
    Exit;
  end;
  Animation.Name:=NewName;
  UpdateAnimationsList;
end;

procedure TMain.mAnimationRemoveClick(Sender: TObject);
var
  Index: Integer;
begin
  if not Assigned(Model) then Exit;
  if MessageDlg('Confirm', 'Do you want to remove the animation "' + Animation.Name + '"? This cannot be undone!', mtConfirmation, mbYesNo, 0) <> mrYes then Exit;
  Index:=Model.IndexOfAnimation(Animation);
  Model.RemoveAnimation(Animation);
  if Index >= Model.AnimationCount then Index:=Model.AnimationCount - 1;
  if Index=-1 then begin
    Animation:=Model.AddAnimation('idle');
    Animation.ResetFramePose(0);
    Animation.Frame[0].IsKey:=True;
  end else
    Animation:=Model.Animations[Index];
  mAnimationSmooth.Checked:=Animation.Smooth;
  mAnimationLooping.Checked:=Animation.Looping;
  CurFrame:=0;
  UpdateAnimation;
  UpdateAnimationsList;
  UpdateTimeline;
end;

procedure TMain.pbTimelineMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if not Assigned(Animation) then Exit;
  if Button=mbLeft then begin
    CurFrame:=X div 6;
    if CurFrame >= Animation.Frames then CurFrame:=Animation.Frames - 1;
    UpdateTimeline;
    UpdateAnimation;
    UpdateTransformUI;
    pbTimeline.Tag:=1;
  end;
end;

procedure TMain.sbSaveRootPositionClick(Sender: TObject);
begin
  if Assigned(Model) then Model.SaveRootPosition;
end;

procedure TMain.FormCreate(Sender: TObject);
begin
  ScratchFrame:=TFrame.Create;
end;

procedure TMain.FormDestroy(Sender: TObject);
begin
  ScratchFrame.Free;
  FreeModel;
end;

procedure TMain.sbCopyPoseClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  ScratchFrame.CopyPoseFrom(Animation.Frame[CurFrame]);
end;

procedure TMain.sbPastePoseClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.Frame[CurFrame].CopyPoseFrom(ScratchFrame);
  Animation.CreatePieceKeyFrames;
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.sbClearPoseClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.ResetFramePose(CurFrame);
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.pbTimelineMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  if not Assigned(Animation) then Exit;
  if pbTimeline.Tag=1 then begin
    CurFrame:=X div 6;
    if CurFrame < 0 then CurFrame:=0;
    if CurFrame >= Animation.Frames then CurFrame:=Animation.Frames - 1;
    UpdateTimeline;
    UpdateAnimation;
    UpdateTransformUI;
  end;
end;

procedure TMain.pbTimelineMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Button=mbLeft then pbTimeline.Tag:=0;
end;

procedure TMain.sbRestoreRootPositionClick(Sender: TObject);
begin
  if Assigned(Model) then Model.RestoreRootPosition;
end;

procedure TMain.sbPasteFromPoseClick(Sender: TObject);
var
  I: Integer;
  T: TTransform;
begin
  for I:=0 to High(ScratchFrame.KeyFrames) do
    if ScratchFrame.KeyFrames[I].Piece=SelectedPiece then begin
      T:=SelectedPiece.Transform;
      T.Rotation:=ScratchFrame.KeyFrames[I].Rotation;
      SelectedPiece.Transform:=T;
      UpdateTransformUI;
      Break;
    end;
end;

procedure TMain.pbTimelineDblClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.Frame[CurFrame].IsEvent:=not Animation.Frame[CurFrame].IsEvent;
  UpdateTimeline;
end;

procedure TMain.mAnimationLoopingClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.Looping:=not Animation.Looping;
  mAnimationLooping.Checked:=Animation.Looping;
  UpdateAnimation;
end;

procedure TMain.mAnimationFirstFrameClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  if tmrAnimationPlayback.Enabled then BaseTime:=GetTickCount64;
  CurFrame:=0;
  UpdateTimeline;
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.mAnimationLastFrameClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  CurFrame:=Animation.Frames - 1;
  UpdateTimeline;
  UpdateAnimation;
  UpdateTransformUI;
end;

procedure TMain.mAnimationPlayClick(Sender: TObject);
begin
  tmrAnimationPlayback.Enabled:=not tmrAnimationPlayback.Enabled;
  if tmrAnimationPlayback.Enabled then BaseTime:=GetTickCount64;
  mAnimationPlay.Checked:=tmrAnimationPlayback.Enabled;
  SpeedButton1.Down:=mAnimationPlay.Checked;
end;

procedure TMain.SpeedButton1Click(Sender: TObject);
begin
  mAnimationPlay.Click;
end;

procedure TMain.tmrAnimationPlaybackTimer(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  CurFrame:=Animation.ApplyTime((GetTickCount64 - BaseTime)/1000);
  UpdateTimeline;
  UpdateTransformUI;
  if (not Animation.Looping) and (CurFrame=Animation.Frames-1) then mAnimationPlay.Click;
end;

procedure TMain.mAnimationSetFPSClick(Sender: TObject);
var
  NewValue: string;
  NewFPS: Integer;
begin
  if not Assigned(Animation) then Exit;
  NewValue:=Trim(InputBox('Animation FPS', 'Enter new FPS (current FPS is ' + IntToStr(Animation.FPS) + '):', IntToStr(Animation.FPS)));
  if NewValue='' then Exit;
  NewFPS:=StrToIntDef(NewValue, 0);
  if NewFPS < 1 then begin
    ShowMessage('Invalid value');
    Exit;
  end;
  Animation.FPS:=NewFPS;
  UpdateAnimation;
end;

procedure TMain.mAnimationSmoothClick(Sender: TObject);
begin
  if not Assigned(Animation) then Exit;
  Animation.Smooth:=not Animation.Smooth;
  mAnimationSmooth.Checked:=Animation.Smooth;
  UpdateAnimation;
end;

procedure TMain.pmPiecesAddPieceClick(Sender: TObject);
begin
  SelectedPiece.AddChildPiece('');
  UpdatePiecesTree;
end;

procedure TMain.pmPiecesPopup(Sender: TObject);
begin
  pmPiecesAddPiece.Enabled:=Assigned(SelectedPiece);
  pmPiecesLoadMesh.Enabled:=Assigned(SelectedPiece);
  pmPiecesRename.Enabled:=Assigned(SelectedPiece);
end;

procedure TMain.pmPiecesRenameClick(Sender: TObject);
var
  NewName: String;
begin
  NewName:=InputBox('Rename Piece ' + SelectedPiece.Name, 'Enter new name:', Trim(SelectedPiece.Name));
  if NewName='' then Exit;
  if NewName=Trim(SelectedPiece.Name) then Exit;
  if Assigned(Model.Root.FindChild(NewName)) then begin
    if MessageDlg('There is already a piece named "' + NewName + '", are you sure that you want to use the same name?', mtConfirmation, mbYesNo, 0) <> mrYes then
      Exit;
  end;
  SelectedPiece.Name:=NewName;
  UpdatePiecesTree;
end;

procedure TMain.pmPiecesDeleteClick(Sender: TObject);
begin
  if SelectedPiece=Model.Root then begin
    ShowMessage('Cannot delete the root piece');
    Exit;
  end;
  if MessageDlg('Are you sure that you want to delete the piece ' + SelectedPiece.Name + '?', mtConfirmation, mbYesNo, 0) <> mrYes then
    Exit;
  SelectedPiece.Parent.Delete(SelectedPiece);
  FSelectedPiece:=nil;
  UpdatePiecesTree;
  UpdateAnimation;
  UpdateTimeline;
end;

procedure TMain.pmPiecesLoadMeshClick(Sender: TObject);
begin
  odMesh.Filter:=GetSupportedMeshFilesFilter;
  odMesh.FileName:='';
  if not odMesh.Execute then Exit;
  try
    SelectedPiece.LoadMesh(odMesh.FileName);
  except
    ShowMessage('Error loading mesh');
  end;
  UpdatePiecesTree;
  UpdateAnimation;
end;

procedure TMain.pmHelpAboutClick(Sender: TObject);
begin
  ShowMessage('Piece Model Editor version 1.0 by Kostas "Bad Sector" Michalopoulos' + LineEnding + 'Made with Lazarus and Free Pascal, released under the zlib license');
end;

procedure TMain.SetSelectedPiece(AValue: TPiece);
begin
  if FSelectedPiece=AValue then Exit;
  FSelectedPiece:=AValue;
  tvPieces.Selected:=tvPieces.Items.FindNodeWithData(AValue);
  UpdateTransformUI;
end;

procedure TMain.UpdateCaption;
begin
  if FileName='' then Caption:='Piece Model Editor'
  else Caption:=ExtractFileName(FileName) + ' - Piece Model Editor';
end;

procedure TMain.UpdatePiecesTree;

  procedure AddPiece(Piece: TPiece; TreeParent: TTreeNode);
  var
    TreeItem: TTreeNode;
    I: Integer;
    VisibleName: string;
  begin
    if Piece.Name='' then begin
      if Assigned(Piece.Parent) then
        VisibleName:='(unnamed piece)'
      else
        VisibleName:='(root piece)';
    end else
      VisibleName:=Piece.Name;
    TreeItem:=tvPieces.Items.AddChild(TreeParent, VisibleName);
    TreeItem.Data:=Piece;
    for I:=0 to Piece.ChildCount - 1 do
      AddPiece(TPiece(Piece.Children[I]), TreeItem);
  end;

begin
  tvPieces.BeginUpdate;
  tvPieces.Items.Clear;
  if Assigned(Model) then AddPiece(Model.Root, nil);
  tvPieces.FullExpand;
  tvPieces.EndUpdate;
end;

procedure TMain.UpdateAnimationsList;
var
  I: Integer;
begin
  if not Assigned(Model) then begin
    cbAnimations.Clear;
    Exit;
  end;
  cbAnimations.Items.BeginUpdate;
  cbAnimations.Clear;
  for I:=0 to Model.AnimationCount - 1 do
    cbAnimations.Items.AddObject(Model.Animations[I].Name + ' (' + IntToStr(I) + ')', Model.Animations[I]);
  cbAnimations.Items.EndUpdate;
  cbAnimations.ItemIndex:=Model.IndexOfAnimation(Animation);
  UpdateTimeline;
end;

procedure TMain.UpdateTimeline;
begin
  pbTimeline.Invalidate;
end;

procedure TMain.UpdateAnimation;
begin
  if not Assigned(Animation) then Exit;
  CurFrame:=CurFrame mod Animation.Frames;
  Animation.ApplyTransformations(CurFrame);
end;

procedure TMain.NewModel;
begin
  FreeModel;
  Model:=TPieceModel.Create(Self);
  Animation:=Model.Animations[0];
  mAnimationSmooth.Checked:=Animation.Smooth;
  mAnimationLooping.Checked:=Animation.Looping;
  UpdatePiecesTree;
  UpdateAnimationsList;
  UpdateTimeline;
  ScratchTransform.Reset;
  Ref.Zero;
  FileName:='';
  UpdateCaption;
end;

procedure TMain.FreeModel;
begin
  FreeAndNil(Model);
  tmrAnimationPlayback.Enabled:=False;
  Animation:=nil;
  mAnimationSmooth.Checked:=False;
  mAnimationLooping.Checked:=False;
  UpdatePiecesTree;
  UpdateAnimationsList;
  UpdateTimeline;
  FileName:='';
  UpdateCaption;
end;

procedure TMain.RenderModel;

  procedure RenderPiece(Piece: TPiece);
  var
    I: Integer;
  begin
    glPushMatrix();
    glMultMatrixd(Piece.WorldMatrix.ToArray);
    if Assigned(Piece.Mesh) then begin
      if Piece=SelectedPiece then glColor3f(1, 0.5, 0.5) else glColor3f(1, 1, 1);
      glCallList(Piece.Mesh.DisplayList);
    end;
    glPopMatrix();
    for I:=0 to Piece.ChildCount - 1 do
      RenderPiece(TPiece(Piece.Children[I]));
  end;

begin
  if not Assigned(Model) then Exit;
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_COLOR_MATERIAL);
  if mViewWireframe.Checked then
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE)
  else
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  RenderPiece(Model.Root);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_LIGHTING);
end;

procedure TMain.UpdateRefUI;
begin
  fseRefX.Value:=Ref.x;
  fseRefY.Value:=Ref.y;
  fseRefZ.Value:=Ref.z;
end;

procedure TMain.UpdateTransformUI;
begin
  if not Assigned(SelectedPiece) then Exit;
  fsePosX.Value:=SelectedPiece.Transform.Translation.x;
  fsePosY.Value:=SelectedPiece.Transform.Translation.y;
  fsePosZ.Value:=SelectedPiece.Transform.Translation.z;
  fseRotX.Value:=SelectedPiece.Transform.Rotation.x;
  fseRotY.Value:=SelectedPiece.Transform.Rotation.y;
  fseRotZ.Value:=SelectedPiece.Transform.Rotation.z;
end;

end.

