unit UMain;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Forms, Controls, Graphics, Dialogs, Menus, Viewports,
  Textures, Maths, Meshes, MeshLoader, UtilMeshData;

type

  TViewMode = (vmWall, vmFloor, vmMesh);

  { TMain }

  TMain = class(TForm)
    pmPopupAbout: TMenuItem;
    pmPopupFullScreen: TMenuItem;
    odOpenMesh: TOpenDialog;
    pmPopupReloadMesh: TMenuItem;
    pmPopupOpenMesh: TMenuItem;
    pmPopupWall: TMenuItem;
    pmPopupFloor: TMenuItem;
    pmPopupMesh: TMenuItem;
    pmPopupMoreTiles: TMenuItem;
    pmPopupLessTiles: TMenuItem;
    pmPopupFiltered: TMenuItem;
    pmPopupPaste: TMenuItem;
    pmPopupExit: TMenuItem;
    pmPopup: TPopupMenu;
    Separator1: TMenuItem;
    Separator2: TMenuItem;
    Separator3: TMenuItem;
    Separator4: TMenuItem;
    Separator5: TMenuItem;
    Separator6: TMenuItem;
    Viewport1: TViewport;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure pmPopupAboutClick(Sender: TObject);
    procedure pmPopupFullScreenClick(Sender: TObject);
    procedure pmPopupFilteredClick(Sender: TObject);
    procedure pmPopupExitClick(Sender: TObject);
    procedure pmPopupFloorClick(Sender: TObject);
    procedure pmPopupLessTilesClick(Sender: TObject);
    procedure pmPopupMeshClick(Sender: TObject);
    procedure pmPopupMoreTilesClick(Sender: TObject);
    procedure pmPopupOpenMeshClick(Sender: TObject);
    procedure pmPopupPasteClick(Sender: TObject);
    procedure pmPopupPopup(Sender: TObject);
    procedure pmPopupReloadMeshClick(Sender: TObject);
    procedure pmPopupWallClick(Sender: TObject);
    procedure Viewport1DblClick(Sender: TObject);
    procedure Viewport1PreMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer; var Veto: Boolean);
    procedure Viewport1PreMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer; var Veto: Boolean);
    procedure Viewport1Render(Sender: TObject);
  private
    Texture: TTexture;
    Mesh: TMesh;
    Tiles: Integer;
    MouseDownPos: TVector;
    ViewMode: TViewMode;
    LastMeshFile: TFileName;
    FullScreenMode: Boolean;
    procedure OpenMeshFile(AFileName: TFileName);
  public
  end;

var
  Main: TMain;

implementation

uses
  GL, LCLType, Clipbrd;

{$R *.lfm}

{ TMain }

procedure TMain.Viewport1Render(Sender: TObject);
var
  X, Y, Base: Integer;
  Asp: Double;
begin
  glColor3f(1, 1, 1);
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, Texture.GLName);
  if pmPopupFiltered.Checked then begin
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  end else begin
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  end;
  Base:=Tiles*64 - 32;
  Asp:=Texture.Width/Texture.Height;
  case ViewMode of
    vmWall: begin
      glBegin(GL_QUADS);
      for Y:=-Tiles to Tiles do
        for X:=-Tiles to Tiles do begin
          glTexCoord2f(0, 0); glVertex2f((-32 + X*64)*Asp, Base + 32 + Y*64 + 64);
          glTexCoord2f(0, 1); glVertex2f((-32 + X*64)*Asp, Base + 32 + Y*64 + 0);
          glTexCoord2f(1, 1); glVertex2f((-32 + X*64 + 64)*Asp, Base + 32 + Y*64 + 0);
          glTexCoord2f(1, 0); glVertex2f((-32 + X*64 + 64)*Asp, Base + 32 + Y*64 + 64);
        end;
      glEnd();
    end;
    vmFloor: begin
      glBegin(GL_QUADS);
      for Y:=-Tiles to Tiles do
        for X:=-Tiles to Tiles do begin
          glTexCoord2f(0, 0); glVertex3f((-32 + X*64)*Asp, 2, -32 + Y*64 + 64);
          glTexCoord2f(0, -1);glVertex3f((-32 + X*64)*Asp, 2, -32 + Y*64 + 0);
          glTexCoord2f(1, -1);glVertex3f((-32 + X*64 + 64)*Asp, 2, -32 + Y*64 + 0);
          glTexCoord2f(1, 0); glVertex3f((-32 + X*64 + 64)*Asp, 2, -32 + Y*64 + 64);
        end;
      glEnd();
    end;
    vmMesh: begin
      glCallList(Mesh.DisplayList);
    end;
  end;
  glBindTexture(GL_TEXTURE_2D, 0);
end;

procedure TMain.OpenMeshFile(AFileName: TFileName);
var
  NewMesh: TMesh = nil;
begin
  try
    pmPopupReloadMesh.Enabled:=False;
    NewMesh:=LoadMeshFromFile(AFileName, False);
    if Assigned(NewMesh) then begin
      NewMesh.ScaleToHeight(128);
      NewMesh.Translate(Vector(0, NewMesh.CalcBoundingBox.A.Y, 0));
      Mesh.Free;
      Mesh:=NewMesh;
    end;
    LastMeshFile:=AFileName;
    pmPopupReloadMesh.Enabled:=True;
  except
    NewMesh.Free;
    ShowMessage('Error: ' + Exception(ExceptObject).Message);
  end;
end;

procedure TMain.FormCreate(Sender: TObject);
var
  X, Y, C: Integer;
  RGB: array of Byte = nil;
begin
  // Default texture
  SetLength(RGB, 64*64*4);
  for Y:=0 to 63 do
    for X:=0 to 63 do begin
      C:=255;
      if (X=0) then C:=128;
      if (Y=0) then C:=192;
      if (X=63) then C:=96;
      if (Y=63) then C:=64;
      RGB[(Y*64 + X)*4]:=C;
      RGB[(Y*64 + X)*4 + 1]:=C;
      RGB[(Y*64 + X)*4 + 2]:=C;
      RGB[(Y*64 + X)*4 + 3]:=255;
    end;
  Texture:=TTexture.Create('');
  Texture.SetFromPixels(64, 64, RGB, False);
  // Default mesh
  Mesh:=CreateMeshFor(CubeVertices);
  Mesh.ScaleToHeight(64);
  Mesh.CalcTexCoords(1/64, 0.5, 0.5);
  Mesh.Translate(Vector(0, 32, 0));
  // Mesh formats
  odOpenMesh.Filter:=GetSupportedMeshFilesFilter(False);
  // Initial state
  Tiles:=1;
end;

procedure TMain.FormDestroy(Sender: TObject);
begin
  Texture.Free;
  Mesh.Free;
end;

procedure TMain.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if (Key=VK_F4) and (ssAlt in Shift) then pmPopupExit.Click;
  if (Key=VK_V) and (ssCtrl in Shift) then pmPopupPaste.Click;
  if Key=VK_F then pmPopupFiltered.Click;
  if Key=VK_F11 then pmPopupFullScreen.Click;
  if Key=VK_1 then pmPopupWall.Click;
  if Key=VK_2 then pmPopupFloor.Click;
  if Key=VK_3 then pmPopupMesh.Click;
  if (Key=VK_O) and (ssCtrl in Shift) then pmPopupOpenMesh.Click;
  if (Key=VK_R) and (ssCtrl in Shift) and pmPopupReloadMesh.Enabled then pmPopupReloadMesh.Click;
  if Key=VK_OEM_4 then pmPopupLessTiles.Click;
  if Key=VK_OEM_6 then pmPopupMoreTiles.Click;
end;

procedure TMain.pmPopupAboutClick(Sender: TObject);
begin
  ShowMessage('Texture Paste Preview version 1.2' + LineEnding + 'Copyright (C) 2020-2022 Kostas Michalopoulos.');
end;

procedure TMain.pmPopupFullScreenClick(Sender: TObject);
begin
  FullScreenMode:=not FullScreenMode;
  // This works around some LCL bugs
  if FullScreenMode then begin
    WindowState:=wsNormal;
    WindowState:=wsMaximized;
    WindowState:=wsFullScreen;
  end else begin
    WindowState:=wsNormal;
    WindowState:=wsMaximized;
    WindowState:=wsNormal;
  end;
end;

procedure TMain.pmPopupFilteredClick(Sender: TObject);
begin
  pmPopupFiltered.Checked:=not pmPopupFiltered.Checked;
end;

procedure TMain.pmPopupExitClick(Sender: TObject);
begin
  Close;
end;

procedure TMain.pmPopupFloorClick(Sender: TObject);
begin
  ViewMode:=vmFloor;
end;

procedure TMain.pmPopupLessTilesClick(Sender: TObject);
begin
  if Tiles > 0 then Dec(Tiles);
end;

procedure TMain.pmPopupMeshClick(Sender: TObject);
begin
  ViewMode:=vmMesh;
end;

procedure TMain.pmPopupMoreTilesClick(Sender: TObject);
begin
  Inc(Tiles);
end;

procedure TMain.pmPopupOpenMeshClick(Sender: TObject);
begin
  odOpenMesh.FileName:=LastMeshFile;
  if odOpenMesh.Execute then begin
    OpenMeshFile(odOpenMesh.FileName);
    if LastMeshFile <> '' then ViewMode:=vmMesh;
  end;
end;

procedure TMain.pmPopupPasteClick(Sender: TObject);
var
  Pic: TPicture;
  Bmp: TBitmap;
  RGB: array of Byte = nil;
  Y: Integer;
begin
  if Clipboard.HasPictureFormat then begin
    Clipboard.HasPictureFormat;
    Pic:=TPicture.Create;
    try
      Pic.LoadFromClipboardFormatID(Clipboard.ClipboardType, Clipboard.FindPictureFormatID);
    except
      Pic.Free;
      ShowMessage('Failed to paste the image: ' + Exception(ExceptObject).Message);
      Exit;
    end;
    Bmp:=TBitmap.Create;
    Bmp.Width:=Pic.Width;
    Bmp.Height:=Pic.Height;
    Bmp.PixelFormat:=pf32bit;
    Bmp.Canvas.Draw(0, 0, Pic.Graphic);
    Pic.Free;
    Bmp.BeginUpdate();
    SetLength(RGB, Bmp.Width*Bmp.Height*4);
    for Y:=0 to Bmp.Height - 1 do
      Move(Bmp.RawImage.GetLineStart(Y)^, RGB[Y*Bmp.Width*4], Bmp.Width*4);
    Bmp.EndUpdate();
    Texture.SetFromPixels(Bmp.Width, Bmp.Height, RGB, False);
    Bmp.Free;
  end else
    ShowMessage('No picture in clipboard');
end;

procedure TMain.pmPopupPopup(Sender: TObject);
begin
  pmPopupFullScreen.Checked:=FullScreenMode;
  pmPopupWall.Checked:=ViewMode=vmWall;
  pmPopupFloor.Checked:=ViewMode=vmFloor;
  pmPopupMesh.Checked:=ViewMode=vmMesh;
end;

procedure TMain.pmPopupReloadMeshClick(Sender: TObject);
begin
  if LastMeshFile <> '' then OpenMeshFile(LastMeshFile);
end;

procedure TMain.pmPopupWallClick(Sender: TObject);
begin
  ViewMode:=vmWall;
end;

procedure TMain.Viewport1DblClick(Sender: TObject);
begin
  pmPopupFullScreen.Click;
end;

procedure TMain.Viewport1PreMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer; var Veto: Boolean);
begin
  if Button in [mbRight, mbMiddle] then begin
    MouseDownPos:=Vector(X, Y, 0);
  end;
end;

procedure TMain.Viewport1PreMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer; var Veto: Boolean);
begin
  if (Button=mbRight) and (Distance(Vector(X, Y, 0), MouseDownPos) < 2) then begin
    pmPopup.PopUp;
  end;
  if (Button=mbMiddle) and (Distance(Vector(X, Y, 0), MouseDownPos) < 2) then begin
    pmPopupPaste.Click;
  end;
end;

end.

