{*
 * Outliner Lighto
 * Copyright (C) 2011-2013 Kostas Michalopoulos
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * Kostas Michalopoulos <badsector@runtimeterror.com>
 *}
unit Scripts;
{$MODE OBJFPC}{$H+}{$UNITPATH ../lil/fplil}
interface
uses SysUtils, UI, Nodes, Tree, Defines, Video, State, FPLIL;

var
  LIL: TLIL;

function NodeToID(Node: TNode): Integer;
function IDToNode(ID: Integer): TNode;
procedure UnregisterScriptID(ID: Integer);
procedure RunScript(FileName: string);
function Eval(Code: string): string;

implementation
{$PUSH}
{$HINTS-}
uses Process;

var
  NodeList: array of TNode;

function NewScriptID: Integer;
begin
  for Result:=1 to Length(NodeList) do
    if NodeList[Result - 1]=nil then Exit;
  Result:=Length(NodeList) + 1;
  SetLength(NodeList, Result);
end;

function NodeToID(Node: TNode): Integer;
begin
  if Node.ScriptID=0 then begin
    Node.ScriptID:=NewScriptID;
    NodeList[Node.ScriptID - 1]:=Node;
  end;
  Result:=Node.ScriptID;
end;

function IDToNode(ID: Integer): TNode;
begin
  Dec(ID);
  if (ID >= 0) and (ID < Length(NodeList)) then Result:=NodeList[ID] else Result:=nil;
end;

procedure UnregisterScriptID(ID: Integer);
begin
  Dec(ID);
  if (ID >= 0) and (ID < Length(NodeList)) then NodeList[ID]:=nil;
end;

function FncRoot(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  Result:=TLIL.AllocInteger(NodeToID(Root));
end;

function FncCurrent(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  Result:=TLIL.AllocInteger(NodeToID(CNode));
end;

function FncParent(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Node: TNode;
begin
  if Length(Args) < 1 then Exit(nil);
  Node:=IDToNode(Args[0].IntegerValue);
  if not Assigned(Node) then Exit(nil);
  Result:=TLIL.AllocInteger(NodeToID(Node.Parent));
end;

function FncChildren(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Node: TNode;
  List: TLILList;
  i: Integer;
begin
  if Length(Args) < 1 then Exit(nil);
  Node:=IDToNode(Args[0].IntegerValue);
  if not Assigned(Node) then Exit(nil);
  List:=TLILList.Create;
  for i:=0 to High(Node.Children) do
    List.AddInteger(NodeToID(Node.Children[i]));
  Result:=List.ToValue;
  List.Free;
end;

function FncTarget(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Node: TNode;
begin
  if Length(Args) < 1 then Exit(nil);
  Node:=IDToNode(Args[0].IntegerValue);
  if not Assigned(Node) then Exit(nil);
  if Node.NodeType=ntPointer then Node:=Node.Target;
  Result:=TLIL.AllocInteger(NodeToID(Node));
end;

function FncText(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Node: TNode;
begin
  if Length(Args) < 1 then Exit(nil);
  Node:=IDToNode(Args[0].IntegerValue);
  if not Assigned(Node) then Exit(nil);
  if Length(Args) > 1 then Node.Text:=Args[1].StringValue;
  Result:=TLIL.AllocString(Node.Text);
end;

function FncType(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Node: TNode;
begin
  if Length(Args) < 1 then Exit(nil);
  Node:=IDToNode(Args[0].IntegerValue);
  if not Assigned(Node) then Exit(nil);
  case Node.NodeType of
    ntNormal: Result:=TLIL.AllocString('normal');
    ntTickable: Result:=TLIL.AllocString('tickable');
    ntPointer: Result:=TLIL.AllocString('pointer');
    else Result:=nil;
  end;
end;

function FncDone(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Node: TNode;
begin
  if Length(Args) < 1 then Exit(nil);
  Node:=IDToNode(Args[0].IntegerValue);
  if not Assigned(Node) then Exit(nil);
  Result:=TLIL.AllocInteger(Node.TickPercent);
end;

function FncAdd(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Text: string;
  Node, Parent: TNode;
begin
  if Length(Args)=0 then Exit(nil);
  if Length(Args)=1 then begin
    Text:=Args[0].StringValue;
    Parent:=CNode;
  end else begin
    Text:=Args[1].StringValue;
    Parent:=IDToNode(Args[0].IntegerValue);
    if not Assigned(Parent) then Exit(nil);
  end;
  Parent.Fresh:=False;
  Node:=TNode.Create;
  Node.Text:=Text;
  Parent.Insert(Node, Length(Parent.Children));
  Result:=ALIL.AllocInteger(NodeToID(Node));
end;

function FncAsk(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Msg: string = '';
  Value: string = '';
begin
  if Length(Args) > 0 then Msg:=Args[0].StringValue;
  GotoXY(0, ScreenHeight - 1);
  Color(15, 0);
  WriteStr(Msg);
  ClearEOL;
  Color(7, 0);
  UpdateScreen(True);
  Value:='';
  if not Input(Length(Msg), ScreenHeight - 1, ScreenWidth - Length(Msg) - 1, Value) then Value:='';
  Result:=TLIL.AllocString(Value);
end;

function FncSystem(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Command: string;
  Output: string = '';
  i: Integer;
begin
  for i:=0 to High(Args) do begin
    if i > 0 then Command:=Command + ' ';
    Command:=Command + Args[i].StringValue;
  end;
  RunCommand(Command, Output);
  Result:=TLIL.AllocString(Output);
end;

function FncSelect(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Index: Integer;
  ChoiceList: TLILList;
  Choices: array of string;
  Title: string;
  I: Integer;
begin
  if Length(Args)=0 then Exit(nil);
  if Length(Args)=1 then begin
    Title:='';
    ChoiceList:=ALIL.SubstituteToList(Args[0]);
    Index:=0;
  end else begin
    Title:=Args[0].StringValue;
    ChoiceList:=ALIL.SubstituteToList(Args[1]);
    if Length(Args) > 2 then
      Index:=Args[2].IntegerValue
    else
      Index:=0;
    if Index < 0 then Index:=0;
    if Index >= ChoiceList.Count then Index:=ChoiceList.Count - 1;
  end;
  if ChoiceList.Count=0 then begin
    ChoiceList.Free;
    Exit(nil);
  end;
  SetLength(Choices, ChoiceList.Count);
  for I:=0 to High(Choices) do Choices[I]:=ChoiceList[I].StringValue;
  ChoiceList.Free;
  Index:=Select(Title, Choices, Index);
  if Index=-1 then Result:=nil else Result:=ALIL.AllocInteger(Index);
end;
              
function FncDisplay(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Title, Text: string;
  QuitChar: Char;
begin
  if Length(Args)=0 then Exit(nil);
  if Length(Args)=1 then begin
    Title:='';
    Text:=Args[0].StringValue;
    QuitChar:='Q';
  end;
  if Length(Args)=2 then begin
    Title:=Args[0].StringValue;
    Text:=Args[1].StringValue;
    QuitChar:='Q';
  end;
  if Length(Args) > 2 then begin
    Title:=Args[0].StringValue;
    Text:=Args[2].StringValue;
    if Args[0].StringValue <> '' then
      QuitChar:=Args[0].StringValue[1]
    else
      QuitChar:='Q';
  end;
  FullScreenText(Title, Text, QuitChar);
  Result:=nil;
end;

function FncBrief(ALIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args) > 0 then BriefMode:=Args[0].BooleanValue;
  Result:=ALIL.AllocInteger(Integer(BriefMode));
end;

procedure InitProcs;
begin
  with LIL do begin
    Register('root', @FncRoot);
    Register('current', @FncCurrent);
    Register('parent', @FncParent);
    Register('children', @FncChildren);
    Register('target', @FncTarget);
    Register('text', @FncText);
    Register('type', @FncType);
    Register('done', @FncDone);
    Register('add', @FncAdd);
    Register('ask', @FncAsk);
    Register('system', @FncSystem);
    Register('select', @FncSelect);
    Register('display', @FncDisplay);
    Register('brief', @FncBrief);
  end;
end;

procedure RunScript(FileName: string);
var
  Code: string = '';
  Line: string;
  f: Text;
begin
  Assign(f, FileName);
  {$I-}
  Reset(f);
  {$I+}
  if IOResult <> 0 then Exit;
  while not Eof(f) do begin
    Readln(f, Line);
    Code:=Code + Line + LineEnding;
  end;
  Close(f);
  Eval(Code);
end;

function Eval(Code: string): string;
var
  ResVal: TLILValue;
begin
  ResVal:=LIL.Parse(Code);
  Result:=ResVal.StringValue;
  ResVal.Free;
end;
{$POP}

initialization
  LIL:=TLIL.Create(nil);
  InitProcs;
  Eval('foreach [preinit init loaded saved key shutdown] func $i {} {}');
finalization
  FreeAndNil(LIL);
end.
