{*
 * Outliner Lighto
 * Copyright (C) 2011-2018 Kostas Michalopoulos
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * Kostas Michalopoulos <badsector@runtimeterror.com>
 *}
program OutlinerLighto;
{$MODE OBJFPC}{$H+}
uses SysUtils, Crt, Keyboard, Video, UI, Nodes, Tree, Scripts,
  State
  {$IFNDEF WINDOWS}, baseunix{$ENDIF}
  {$IFDEF OLPTC}, ptckvm{$ENDIF};

{$IFDEF WINDOWS}
{$R OLICON.RC}
{$ENDIF}

{$I keyshelp.inc}

const
  LevelColor: array [0..3] of Byte = (3, 2, 6, 5);
var
  Scroll: Integer;
  CNodeY: Integer;
  UpParent: array of TNode; // note: contains navigation parents (for pointers)
  Editing, Moving, Running: Boolean;
  Cx, Cy: Integer;
  EdPos: Integer;
  CutNode: TNode;
  FileName: string;
  SearchTerm: string;
  i: Integer;

procedure Save;
begin
  Eval('saved {' + FileName + '}');
  SaveTree(FileName, Root);
end;

procedure Load;
begin
  Eval('loaded {' + FileName + '}');
  LoadTree(FileName, Root);
end;

procedure DrawTree;
var
  lev, x, y: Integer;

  procedure WriteText(Str, Cmt: string; Editing: Boolean);
  var
    w: string;
    Pos, lx, i, CmtAttr: Integer;

    function WriteW: Boolean;
    var
      i: Integer;
    begin
      if lx > x then begin
        if (y > 0) and (y < ScreenHeight - 1) then WriteChar(' ');
        if Editing and (Pos < EdPos) then Inc(Cx);
        Inc(Pos);
      end;
      if (lx > x) and (lx + Length(w) >= ScreenWidth) then begin
        if BriefMode and not Editing then begin
          ClrTo(lx, ScreenWidth - 1, y);
          CmtAttr:=CAttr;
          CAttr:=4;
          GotoXY(lx, y);
          while lx < ScreenWidth - 1 do begin
            WriteChar('-');
            Inc(lx);
          end;
          CAttr:=12;
          WriteChar('>');
          CAttr:=CmtAttr;
          Exit(False);
        end;
        lx:=x;
        Inc(y);
        CAttr:=0;
        if (y > 0) and (y < ScreenHeight - 1) then ClrTo(0, x, y);
        if Editing then begin
          CAttr:=8;
          GotoXY(x - 2, y);
          if (y > 0) and (y < ScreenHeight - 1) then WriteChar('|');
        end;
        Color(FColor, BColor);
      end;
      GotoXY(lx, y);
      if Editing and (Pos <= EdPos) and (lx=x) then begin
        Cx:=x;
        Cy:=y;
      end;
      if (y > 0) and (y < ScreenHeight - 1) then WriteStr(w);
      if Editing then begin
        for i:=1 to Length(w) do begin
          if Pos < EdPos then begin
            Inc(Cx);
            Cy:=y;
          end;
          Inc(Pos);
        end;
      end;
      Inc(lx, Length(w) + 1);
      Result:=True;
    end;

  begin
    w:='';
    lx:=x;
    Pos:=1;
    if Editing then begin
      Cx:=x;
      Cy:=y;
    end;
    for i:=1 to Length(Str) do
      if Str[i] in [#9, ' '] then begin
        WriteW;
        w:='';
      end else w:=w + Str[i];
    if not WriteW then Exit;
    if Cmt <> '' then begin
      CmtAttr:=CAttr;
      Color(4, 0);
      w:='';
      for i:=1 to Length(Cmt) do
        if Cmt[i] in [#9, ' '] then begin
          if not WriteW then Exit;
          w:='';
        end else w:=w + Cmt[i];
      WriteW;
      Color(CmtAttr and $0F, CmtAttr shr 4);
    end;
  end;

  procedure DrawNode(ANode: TNode);
  var
    i, Per: Integer;
    CNY1: Integer;
    Child: TNode;
    NodeType: TNodeType;
  begin
    Inc(x, 4);
    if x > ScreenWidth then Exit;
    Inc(lev);
    for i:=0 to Length(ANode.Children) - 1 do begin
      Child:=ANode.Children[i];
      if (((ANode=Root) and (Length(UpParent)=0)) or ((Length(UpParent) > 0) and (UpParent[Length(UpParent) - 1]=ANode))) and (CNIdx=i) then begin
        if Editing then
          Color(11, 0)
        else if Moving then
          Color(0, 6)
        else
          Color(0, 7);
        CNY1:=y;
      end else
        if Child.HasChildren then begin
          Color(LevelColor[lev and 3], 0);
        end else
          if (Child.Parent.NodeType=ntTickable) and (Child.NodeType=ntNormal) then
            Color(14, 0)
          else
            Color(7, 0);
      if (Child.NodeType=ntPointer) and (y > 0) and (y < ScreenHeight - 1) then begin
        GotoXY(x - 7, y);
        CAttr:=9;
        WriteStr('<-');
      end;
      GotoXY(x - 4, y);
      if Editing and (CParent=ANode) and (CNIdx=i) then begin
        CAttr:=8;
        WriteStr('  | ');
      end else if (y > 0) and (y < ScreenHeight - 1) then begin
        NodeType:=Child.TargetNodeType;
        if Child.HasChildren then begin
          if NodeType=ntNormal then begin
            CAttr:=7;
            if Child.Open then
              WriteStr('  - ')
            else
              WriteStr('  + ');
          end else if NodeType=ntTickable then begin
            if (y > 0) and (y < ScreenHeight - 1) then begin
              Per:=Child.TickPercent;
              if Per=100 then begin
                CAttr:=15;
                WriteStr('[X] ')
              end else begin
                if Per in [0..24] then CAttr:=8
                else if Per in [25..49] then CAttr:=7
                else if Per in [50..74] then CAttr:=13
                else if Per in [75..99] then CAttr:=14;
                WriteStr(Chr(Ord('0')+(Per div 10)) + Chr(Ord('0')+(Per mod 10)) + '%');
              end;
            end;
          end;
        end else begin
          if NodeType=ntNormal then begin
            if (Child.Parent.NodeType=ntTickable) and (Child.NodeType=ntNormal) then begin
              CAttr:=15;
              WriteStr(' -- ');
            end else begin
              CAttr:=1;
              WriteStr('  o ')
            end;
          end else if NodeType=ntTickable then begin
            CAttr:=2;
            if Child.Tick then
              WriteStr('[X] ')
            else
              WriteStr('[ ] ');
          end;
        end;
      end;
      Color(FColor, BColor);
      GotoXY(x, y);
      if Child.Text='' then begin
        if (y > 0) and (y < ScreenHeight - 1) then WriteChar(' ');
        if Editing and (CParent=ANode) and (CNIdx=i) then begin
          Cx:=x;
          Cy:=y;
        end;
      end else begin
        if (Child.NodeType=ntPointer) and Assigned(Child.Target) and Assigned(Child.Target.Parent) then
          WriteText(Child.Text, '[' + Child.Target.Parent.Text + ']', Editing and (CParent=ANode) and (CNIdx=i))
        else
          WriteText(Child.Text, '', Editing and (CParent=ANode) and (CNIdx=i));
      end;
      Color(7, 0);
      if (CParent=ANode) and (CNIdx=i) then begin
        CNodeY:=(CNY1 + y) div 2;
      end;
      Inc(y);
      if Child.Open and Child.HasChildren then begin
        DrawNode(Child);
      end;
    end;
    Dec(lev);
    Dec(x, 4);
  end;

begin
  if ScreenHeight < 6 then Exit;
  DoWrites:=False;
  while True do begin
    y:=-Scroll;
    x:=0;
    lev:=-1;
    DrawNode(Root);
    if CNodeY < ScreenHeight div 4 then
      Dec(Scroll)
    else if CNodeY > ScreenHeight div 4 * 3 then
      Inc(Scroll)
    else begin
      y:=-Scroll;
      x:=0;
      lev:=-1;
      DoWrites:=True;
      DrawNode(Root);
      break;
    end;
  end;
end;

procedure GoToPrev;
begin
  if CNIdx > 0 then begin
    if CNode.Fresh and (CNode.Text='') and (CNIdx=Length(CParent.Children)-1) then begin
      CParent.Remove(CNode);
      CNode.Free;
    end;
    Dec(CNIdx);
    CNode:=CParent.Children[CNIdx];
  end;
end;

procedure GoToNext;
begin
  if CNIdx < Length(CParent.Children) - 1 then begin
    Inc(CNIdx);
    CNode:=CParent.Children[CNIdx];
  end else if not CNode.Fresh then begin
    CNode:=CParent.AddStr('');
    CNode.Fresh:=True;
    CNode.NodeType:=CParent.TargetNodeType;
    CNIdx:=Length(CParent.Children) - 1;
  end;
end;

function Dive: TNode;
var
  Node: TNode;
begin
  Result:=nil;
  if CNode.Fresh then Exit;
  if not CNode.HasChildren then begin
    Node:=TNode.Create;
    Node.NodeType:=CNode.TargetNodeType;
    Node.Text:='';
    Node.Fresh:=True;
    CNode.Add(Node);
    Result:=Node;
  end;
  SetLength(UpParent, Length(UpParent) + 1);
  UpParent[Length(UpParent) - 1]:=CNode;
  CNIdx:=0;
  CNode.WasOpen:=CNode.Open;
  CNode.Open:=True;
  CNode:=CNode.Children[0];
  CParent:=CNode.Parent;
end;

procedure Rise;
begin
  if Length(UpParent) > 0 then begin
    if CNode.Fresh and (CNode.Text='') then begin
      CParent.Remove(CNode);
      CNode.Free;
    end;
    CNode:=UpParent[Length(UpParent) - 1];
    SetLength(UpParent, Length(UpParent) - 1);
    CParent:=CNode.Parent;
    CNIdx:=CParent.IndexOf(CNode);
    CNode.Open:=CNode.WasOpen;
    if (Length(CNode.Children)=1) and (CNode.Children[0].Fresh) then begin
      CNode.Remove(CNode.Children[0]);
      CNode.Children[0].Free;
    end;
  end;
end;

procedure GoToHome;
begin
  if CParent=Root then begin
    CNIdx:=0;
    CNode:=Root.Children[0];
  end else while CParent <> Root do Rise;
end;

procedure OpenClose;
begin
  if CParent.Children[CNIdx].HasChildren then begin
    CParent.Children[CNIdx].Open:=not CParent.Children[CNIdx].Open;
    CParent.Children[CNIdx].WasOpen:=CParent.Children[CNIdx].Open;
  end;
end;

function EditMode: Boolean;
var
  Key: TKeyEvent;
  SaveText: string;
  SaveFresh: Boolean;
  Sx, Sy: Integer;
  Ch: Char;
  ToDeleteOnCancel: array of TNode;

  procedure MoveLeft;
  begin
    if EdPos > 1 then Dec(EdPos);
  end;

  procedure MoveRight;
  begin
    if EdPos <= Length(CNode.Text) then Inc(EdPos);
  end;

  function CursorUnderWordSeparator: Boolean;
  begin
    if not (CNode.Text[EdPos] in ['a'..'z', 'A'..'Z', '0'..'9']) then Exit(True);
    Result:=False;
  end;

  procedure MoveWordLeft;
  var
    Start: Boolean;
  begin
    MoveLeft;
    Start:=CursorUnderWordSeparator;
    while (EdPos > 1) and (Start=CursorUnderWordSeparator) do
      MoveLeft;
  end;

  procedure MoveWordRight;
  var
    Start: Boolean;
  begin
    MoveRight;
    Start:=CursorUnderWordSeparator;
    while (EdPos <= Length(CNode.Text)) and (Start=CursorUnderWordSeparator) do
      MoveRight;
  end;

  procedure SplitNode;
  var
    NNode: TNode;
  begin
    NNode:=TNode.Create;
    NNode.Fresh:=False;
    NNode.Text:=Copy(CNode.Text, EdPos, Length(CNode.Text));
    NNode.NodeType:=CNode.NodeType;
    CParent.Insert(NNode, CNIdx + 1);
    CNode.Text:=Copy(CNode.Text, 1, EdPos - 1);
    CNode.Fresh:=False;
    SetLength(ToDeleteOnCancel, Length(ToDeleteOnCancel) + 1);
    ToDeleteOnCancel[High(ToDeleteOnCancel)]:=NNode;
  end;

begin
  Result:=True;
  Editing:=True;
  SaveText:=CNode.Text;
  SaveFresh:=CNode.Fresh;
  EdPos:=Length(SaveText) + 1;
  ToDeleteOnCancel:=nil;
  while Editing do begin
    ClearBackScreen;
    Status(' Editing Node: ~Enter~ Done ~Arrows~ Move ~ESC~ Cancel');
    DrawTree;
    UpdateScreen(False);
    SetCursorPos(Cx, Cy);
    repeat
      {$IFDEF OLPTC}InvertCharAt(Cx, Cy);{$ENDIF}
      Key:=TranslateKeyEvent(GetKeyEvent);
      {$IFDEF OLPTC}InvertCharAt(Cx, Cy);{$ENDIF}
      case GetKeyEventFlags(Key) of
        kbASCII: begin
          Ch:=GetKeyEventChar(Key);
          case Ch of
            #8: if EdPos > 1 then begin
              CNode.Text:=Copy(CNode.Text, 1, EdPos - 2) + Copy(CNode.Text, EdPos, Length(CNode.Text));
              Dec(EdPos);
            end;
            #27: begin
              Editing:=False;
              CNode.Text:=SaveText;
              CNode.Fresh:=SaveFresh;
              for I:=0 to High(ToDeleteOnCancel) do begin
                ToDeleteOnCancel[I].Parent.Remove(ToDeleteOnCancel[I]);
                ToDeleteOnCancel[I].Free;
              end;
              Result:=False;
            end;
            #13: begin
              Editing:=False;
              if CNode.Text <> '' then CNode.Fresh:=False;
            end;
            ^S: if EdPos <= Length(CNode.Text) then SplitNode;
            else if Ch in [#32..#127] then begin
              CNode.Text:=Copy(CNode.Text, 1, EdPos - 1) + Ch + Copy(CNode.Text, EdPos, Length(CNode.Text));
              Inc(EdPos);
            end;
          end;
        end;
        else case GetKeyEventCode(Key) of
          kbdDelete: if EdPos <= Length(CNode.Text) then begin
            CNode.Text:=Copy(CNode.Text, 1, EdPos - 1) + Copy(CNode.Text, EdPos + 1, Length(CNode.Text));
          end;
          kbdHome: EdPos:=1;
          kbdEnd: EdPos:=Length(CNode.Text) + 1;
          kbdLeft: MoveLeft;
          kbdRight: MoveRight;
          29440: MoveWordLeft; // is this portable?
          29696: MoveWordRight; // is this portable?
          kbdUp: begin
            Sx:=Cx;
            Sy:=Cy - 1;
            while (EdPos > 1) and (not ((Cx=Sx) and (Cy=Sy))) do begin
              Dec(EdPos);
              DrawTree;
            end;
          end;
          kbdDown: begin
            Sx:=Cx;
            Sy:=Cy + 1;
            while (EdPos <= Length(CNode.Text)) and (not ((Cx=Sx) and (Cy=Sy))) do begin
              Inc(EdPos);
              DrawTree;
            end;
          end;
        end;
      end;
    until not KeyPressed;
  end;
end;

procedure DeleteNode;
var
  UParent: TNode;
begin
  FreeAndNil(CutNode);
  if not CNode.Fresh then CutNode:=CNode;
  CParent.Remove(CNode);
  if Length(UpParent)=0 then UParent:=Root else UParent:=UpParent[Length(UpParent) - 1];
  if UParent.HasChildren then begin
    if Length(UParent.Children) <= CNIdx then begin
      CNIdx:=Length(UParent.Children) - 1;
      CNode:=UParent.Children[CNIdx];
    end else begin
      CNode:=UParent.Children[CNIdx];
    end;
  end else begin
    if UParent=Root then begin
      CNode:=Root.AddStr('');
      CNode.Fresh:=True;
    end else begin
      //CNode:=UParent;
      //CNIdx:=CNode.Parent.IndexOf(CNode);
      //CParent:=CNode.Parent;
      Rise;
    end;
  end;
end;

procedure MultiEditMode;
var
  Node: TNode;
  Res, AddedAny: Boolean;
begin
  AddedAny:=False;
  if CNode.Fresh and (CNode.Text='') then begin
    while True do begin
      Res:=EditMode;
      if (not Res) or (CNode.Fresh and (CNode.Text='')) then begin
        if (CNIdx=Length(CParent.Children)-1) and (not AddedAny) then break;
        Node:=CutNode;
        CutNode:=nil;
        DeleteNode;
        if CutNode <> nil then CutNode.Free;
        CutNode:=Node;
        if CNIdx < Length(CParent.Children)-1 then GoToPrev;
        break;
      end;
      AddedAny:=True;
      Node:=TNode.Create;
      Node.Fresh:=True;
      Node.NodeType:=CParent.TargetNodeType;
      CParent.Insert(Node, CNIdx + 1);
      CNode:=Node;
      Inc(CNIdx);
    end;
  end else EditMode;
end;

procedure MergeNodes;
var
  NNode: TNode;
begin
  NNode:=CParent.Children[CNIdx + 1];
  CNode.Text:=CNode.Text + NNode.Text;
  while NNode.HasChildren do CNode.Add(NNode.Children[0]);
  CParent.Remove(NNode);
  NNode.Free;
end;

procedure MergeNodesRequest;
var
  Key: TKeyEvent;
begin
  if CNIdx >= Length(CParent.Children) - 1 then begin
    Status(' ~There must be at least one node below to merge with. Press a key to continue.');
    UpdateScreen(False);
    GetKeyEvent;
    Exit;
  end;
  if CNode.HasChildren or CParent.Children[CNIdx + 1].HasChildren then
    Status(' Press ~M~ again to merge the node texts and their children')
  else
    Status(' Press ~M~ again to merge the node texts');
  UpdateScreen(False);
  Key:=TranslateKeyEvent(GetKeyEvent);
  if (GetKeyEventFlags(Key)=kbASCII) and (UpCase(GetKeyEventChar(Key))='M') then
    MergeNodes;
end;

procedure DeleteRequest(KeyName: string);
var
  Key: TKeyEvent;
begin
  if CNode.HasChildren then
    Status(' Press ~' + KeyName + '~ again to delete the node and it''s children')
  else
    Status(' Press ~' + KeyName + '~ again to delete the node');
  DrawTree;
  UpdateScreen(False);
  Key:=TranslateKeyEvent(GetKeyEvent);
  if (GetKeyEventFlags(Key)=kbASCII) and (UpCase(GetKeyEventChar(Key))='D') then begin
    DeleteNode;
  end else if GetKeyEventCode(Key)=kbdDelete then begin
    DeleteNode;
  end;
end;

procedure PasteNode;
begin
  if CutNode=nil then exit;
  if CNode.Fresh and (CNode.Text='') then begin
    CParent.Remove(CNode);
    CNode.Free;
    CNode:=CutNode.Clone;
  end;
  CNode:=CutNode.Clone;
  CParent.Insert(CNode, CNIdx);
end;

procedure ChangeType;
var
  i: Integer;
begin
  if CNode.NodeType=ntNormal then begin
    CNode.NodeType:=ntTickable;
    CParent.NodeType:=ntTickable;
  end else begin
    CNode.NodeType:=ntNormal;
    CParent.NodeType:=ntNormal;
    for i:=0 to Length(CParent.Children)-1 do
      if CParent.Children[i].NodeType=ntTickable then begin
        Cparent.NodeType:=ntTickable;
        break;
      end;
  end;
end;

procedure GrabMode;
var
  Key: TKeyEvent;
  Ch: Char;
  OParent, Node, MNode: TNode;
  OIndex: Integer;

  procedure Restore;
  var
    Nodes: array of TNode;
    Node: TNode;
    i: Integer;
  begin
    CParent.Remove(MNode);
    GoToHome;
    SetLength(Nodes, 0);
    Node:=OParent;
    while Node <> Root do begin
      SetLength(Nodes, Length(Nodes) + 1);
      Nodes[Length(Nodes) - 1]:=Node;
      Node:=Node.Parent;
    end;
    for i:=Length(Nodes) - 1 downto 0 do begin
      CNode:=Nodes[i];
      CNIdx:=CParent.IndexOf(CNode);
      Node:=Dive;
      if Node <> nil then begin
        CParent.Remove(Node);
        Node.Free;
      end;
    end;
    OParent.Insert(MNode, OIndex);
    CNode:=MNode;
    CParent:=OParent;
    CNIdx:=OIndex;
  end;

begin
  Moving:=True;
  MNode:=CNode;
  OParent:=CParent;
  OIndex:=CNIdx;
  while Moving do begin
    ClearBackScreen;
    DefaultTop;
    Status(' Grabbed Node: ~G or Enter~ Drop here ~Arrows~ Move ~Q or ESC~ Cancel');
    DrawTree;
    UpdateScreen(False);
    Key:=TranslateKeyEvent(GetKeyEvent);
    case GetKeyEventFlags(Key) of
      kbASCII: begin
        Ch:=GetKeyEventChar(Key);
        case Ch of
          #27, 'q', 'Q': begin
            Restore;
            Moving:=False;
          end;
          #13, 'g', 'G': begin
            Moving:=False;
          end;
        end;
      end;
      else case GetKeyEventCode(Key) of
        kbdLeft: if CParent <> Root then begin
          CParent.Remove(MNode);
          Node:=CParent;
          Rise;
          CParent.Insert(MNode, CParent.IndexOf(Node) + 1);
          CNode:=MNode;
          CNIdx:=CParent.IndexOf(CNode);
        end;
        kbdRight: if CNIdx > 0 then begin
          GoToPrev;
          CParent.Remove(MNode);
          CNode.Add(MNode);
          Dive;
          CNode:=MNode;
          CNIdx:=CParent.IndexOf(CNode);
        end;
        kbdUp: if CNIdx > 0 then begin
          CParent.Remove(MNode);
          CParent.Insert(MNode, CNIdx - 1);
          Dec(CNIdx);
        end;
        kbdDown: if CNIdx < Length(CParent.Children) - 1 then begin
          CParent.Remove(MNode);
          CParent.Insert(MNode, CNIdx + 1);
          Inc(CNIdx);
        end;
      end;
    end;
  end;
end;

procedure MakePointer;
var
  Node: TNode;
begin
  if CNode.Fresh then Exit;
  Node:=TNode.Create;
  Node.NodeType:=ntPointer;
  Node.Target:=CNode;
  CParent.Insert(Node, CNIdx + 1);
  GoToNext;
  GrabMode;
end;

procedure OrderNodes;
var
  Key: TKeyEvent;
begin
  Status(' Press ~O~ again to order the nodes alphabetically');
  DrawTree;
  UpdateScreen(False);
  Key:=TranslateKeyEvent(GetKeyEvent);
  if (GetKeyEventFlags(Key)=kbASCII) and (UpCase(GetKeyEventChar(Key))='O') then begin
    CParent.OrderChildren;
    CNIdx:=CParent.IndexOf(CNode);
  end;
end;

procedure EvalCode;
var
  Code: string;
begin
  GotoXY(0, ScreenHeight - 1);
  Color(15, 4);
  WriteStr('Evaluate Code: ');
  ClearEOL;
  Color(14, 4);
  UpdateScreen(False);
  Code:='';
  if Input(15, ScreenHeight - 1, ScreenWidth - 18, Code) then begin
    Eval(Code);
  end;
end;

procedure FlowNode;
var
  Text: string;

  procedure ScanNode(N: TNode; Depth: Integer);
  var
    I: Integer;
  begin
    if Text <> '' then Text += #10#10;
    if Length(N.Children)=0 then Text += #7 else Text += Char(Depth mod 4 + 2);
    Text += N.Text;
    for I:=0 to High(N.Children) do ScanNode(N.Children[I], Depth + 1);
  end;

begin
  Text:='';
  ScanNode(CNode, 0);
  FullScreenText('Flow contents of ' + CNode.Text, Text, 'F');
end;

procedure SearchNext;

  function StepForward: Boolean;
  begin
    if not CNode.Fresh and CNode.HasChildren then begin
      Dive;
      Exit(True);
    end;
    if (CNIdx < Length(CParent.Children) - 1) and not CParent.Children[CNIdx + 1].Fresh then begin
      GoToNext;
      Exit(True);
    end;
    while CParent <> Root do begin
      Rise;
      if (CNIdx < Length(CParent.Children) - 1) and not CParent.Children[CNIdx + 1].Fresh then begin
        GoToNext;
        Exit(True);
      end;
    end;
    Result:=False;
  end;

begin
  while StepForward do begin
    if Pos(UpperCase(SearchTerm), UpperCase(CNode.Text)) <> 0 then Break;
  end;
end;

procedure Search;
var
  Term: string;
begin
  GotoXY(0, ScreenHeight - 1);
  Color(4, 7);
  WriteStr('Search: ');
  ClearEol;
  Color(0, 7);
  UpdateScreen(False);
  Term:=SearchTerm;
  if Input(8, ScreenHeight - 1, ScreenWidth - 9, Term) then begin
    Term:=Trim(Term);
    if Term <> '' then begin
      SearchTerm:=Term;
      SearchNext;
    end;
  end;
end;

procedure MainMode;
var
  Key: TKeyEvent;
  Node: TNode;
begin
  while Running do begin
    ClearBackScreen;
    DefaultTop;
    Status(' ~Space~ Toggle ~Arrows~ Browse ~Enter~ Edit ~I~ Insert ~A~ Append ~D~ Delete ~Q~ Quit ~?~ Help ');
    DrawTree;
    UpdateScreen(False);
    SetCursorPos(ScreenWidth - 1, ScreenHeight - 1);
    Key:=TranslateKeyEvent(GetKeyEvent);
    case GetKeyEventFlags(Key) of
      kbASCII: case GetKeyEventChar(Key) of
        #27, 'q', 'Q': Running:=False;
        'r', 'R': begin
          DoneVideo;
          InitVideo;
        end;
        's', 'S': Save;
        #32: OpenClose;
        'l', 'L': Dive;
        'h', 'H': Rise;
        'k', 'K': GoToPrev;
        'j', 'J': GoToNext;
        #13, 'c', 'C': MultiEditMode;
        'm', 'M': MergeNodesRequest;
        'i', 'I': begin
          if not CNode.Fresh then begin
            Node:=TNode.Create;
            Node.Fresh:=True;
            Node.NodeType:=CParent.TargetNodeType;
            CParent.Insert(Node, CNIdx);
            CNode:=Node;
          end;
          MultiEditMode;
        end;
        'a', 'A': begin
          if not CNode.Fresh then begin
            Node:=TNode.Create;
            Node.Fresh:=True;
            Node.NodeType:=CParent.TargetNodeType;
            CParent.Insert(Node, CNIdx + 1);
            GoToNext;
          end;
          MultiEditMode;
        end;
        't', 'T': ChangeType;
        'x', 'X': begin
          if CNode.TargetNodeType=ntTickable then
            CNode.Tick:=not CNode.Tick;
        end;
        'd', 'D': DeleteRequest('D');
        'p', 'P': PasteNode;
        'u', 'U': GoToHome;
        'g', 'G': if not CNode.Fresh then GrabMode;
        'o', 'O': OrderNodes;
        'e', 'E': EvalCode;
        'b', 'B': BriefMode:=not BriefMode;
        'f', 'F': FlowNode;
        '.': if not CNode.Fresh then MakePointer;
        '/': Search;
        '?': FullScreenText('Help', KeysHelp);
        else Eval('key ' + UpCase(GetKeyEventChar(Key)));
      end;
      else case GetKeyEventCode(Key) of
        kbdF1..kbdF4, kbdF6..kbdF12: Eval('key F' + IntToStr(Ord(GetKeyEventCode(Key)) - Ord(kbdF1) + 1));
        kbdF5: begin
          DoneVideo;
          InitVideo;
        end;
        kbdUp: GoToPrev;
        kbdDown: GoToNext;
        kbdRight: Dive;
        kbdLeft: Rise;
        kbdDelete: DeleteRequest('Del');
        kbdHome: GoToHome;
      end;
    end;
  end;
end;

{$IFNDEF WINDOWS}
procedure HandleSignal(Signal: CInt); cdecl;
begin
  Save;
  DestroyTree;
  ClearScreen;
  DoneKeyboard;
  DoneVideo;
  Crt.GotoXY(1, ScreenHeight);
  FreeAndNil(CutNode);
  Halt;
end;

procedure InstallSignalHandlers;
var
  oa, na: PSigActionRec;
begin
  New(oa);
  New(na);
  na^.sa_Handler:=SigActionHandler(@HandleSignal);
  FillChar(na^.sa_Mask, SizeOf(na^.sa_Mask), #0);
  na^.sa_Flags:=0;
  {$IFDEF LINUX}
  na^.sa_Restorer:=nil;
  {$ENDIF}
  if fpSigAction(SigQuit, na, oa) <> 0 then Exit;
  if fpSigAction(SigTerm, na, oa) <> 0 then Exit;
  if fpSigAction(SigHup, na, oa) <> 0 then Exit;
  Dispose(oa);
  Dispose(na);
end;
{$ENDIF}

{$IFDEF OLPTC}
var
  PTCRows : Integer = 100;
  PTCCols : Integer = 40;
procedure SetPTCVideoMode;
var
  VideoMode: TVideoMode;
begin
  VideoMode.Col:=PTCRows;
  VideoMode.Row:=PTCCols;
  VideoMode.Color:=True;
  KVMSetWindowTitle('Outliner Lighto (PasPTC/PTCKVM version)');
  SetVideoMode(VideoMode);
end;
{$ENDIF}

begin
  RunScript(GetUserDir + '.ol.lil');
  FileName:=GetUserDir + '.ol.olol';
  Eval('preinit');
  for i:=1 to ParamCount do begin
    if ParamStr(i)='--help' then begin
      WriteLn('Usage: ol [options] [path-to-olol-file]');
      WriteLn('Options:');
      WriteLn('  --help           These help instructions');
      {$IFDEF OLPTC}
      WriteLn('  --ptcsize=c,r    Set the site for the PasPTC/PTCKVM window to');
      WriteLn('                   <c> columns and <r> rows');
      {$ENDIF}
      exit;
    end else
    {$IFDEF OLPTC}
    if Copy(ParamStr(i), 1, 10)='--ptcsize=' then begin
      try
        Tmp:=Copy(ParamStr(i), 11, Length(ParamStr(i)));
        PTCRows:=StrToInt(Copy(Tmp, 1, Pos(',', Tmp) - 1));
        PTCCols:=StrToInt(Copy(Tmp, Pos(',', Tmp) + 1, Length(Tmp)));
      except
        PTCRows:=40;
        PTCCols:=100;
      end;
    end else
    {$ENDIF}
      FileName:=ParamStr(i);
  end;
  InitVideo;
  InitKeyboard;
  {$IFDEF OLPTC}
  SetPTCVideoMode;
  {$ENDIF}
  Eval('init');
  CreateTree;
  Load;
  CParent:=Root;
  CNode:=Root.Children[0];
  CNIdx:=0;
  Color(7, 0);
  ClearScreen;
  Scroll:=0;
  CNodeY:=0;
  Running:=True;
  {$IFNDEF WINDOWS}
  InstallSignalHandlers;
  {$ENDIF}
  MainMode;
  Save;
  Eval('shutdown');
  DestroyTree;
  ClearScreen;
  DoneKeyboard;
  DoneVideo;
  Crt.GotoXY(1, ScreenHeight);
  FreeAndNil(CutNode);
end.
