{*
 * Outliner Lighto
 * Copyright (C) 2011 Kostas Michalopoulos
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * Kostas Michalopoulos <badsector@runtimeterror.com>
 *}
unit Nodes;
interface
{$MODE OBJFPC}{$H+}
uses SysUtils;
type
  TNode = class;
  TNodeType = (ntNormal, ntTickable, ntPointer);

  TNodeChildren = array of TNode;

  { TNode }

  TNode = class
  private
    // Do not forget Clone when adding new stuff
    FTarget: TNode;
    FTargetAddress: string;
    FText: string;
    FNodeType: TNodeType;
    FTick: Boolean;
    FParent: TNode;
    FChildren: TNodeChildren;
    FOpen: Boolean;
    FWasOpen: Boolean;
    FFresh: Boolean;
    FScriptID: Integer;
    function GetChildren: TNodeChildren;
    function GetTargetNodeType: TNodeType;
    function GetText: string;
    function GetTick: Boolean;
    procedure SetFresh(const AValue: Boolean);
    procedure SetNodeType(const AValue: TNodeType);
    procedure SetOpen(const AValue: Boolean);
    procedure SetParent(const AValue: TNode);
    procedure SetTarget(const AValue: TNode);
    procedure SetText(const AValue: string);
    procedure SetTick(const AValue: Boolean);
    procedure SetWasOpen(const AValue: Boolean);
  public
    constructor Create;
    destructor Destroy; override;

    procedure Add(AChild: TNode);
    function AddStr(Str: string): TNode;
    procedure Remove(AChild: TNode);
    procedure Insert(AChild: TNode; AIndex: Integer);
    function IndexOf(AChild: TNode): Integer;
    procedure OrderChildren;
    function HasChildren: Boolean;
    function TickPercent: Integer;
    function Address: string;
    procedure Save(var f: TextFile);
    procedure Load(var f: TextFile);
    function Clone: TNode;

    // Do not forget Clone when adding new stuff
    property Text: string read GetText write SetText;
    property NodeType: TNodeType read FNodeType write SetNodeType;
    property TargetNodeType: TNodeType read GetTargetNodeType;
    property Tick: Boolean read GetTick write SetTick;
    property Parent: TNode read FParent;
    property Children: TNodeChildren read GetChildren;
    property Open: Boolean read FOpen write SetOpen;
    property WasOpen: Boolean read FWasOpen write SetWasOpen;
    property Fresh: Boolean read FFresh write SetFresh;
    property Target: TNode read FTarget write SetTarget;
    property TargetAddress: string read FTargetAddress write FTargetAddress; // note: used for loading
    property ScriptID: Integer read FScriptID write FScriptID;
  end;

implementation

uses
  Scripts;

procedure TNode.SetText(const AValue: string);
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then FTarget.Text:=AValue;
  end else begin
    if FText=AValue then Exit;
    FText:=AValue;
  end;
end;

function TNode.GetChildren: TNodeChildren;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Result:=FTarget.Children else Result:=nil;
  end else Result:=FChildren;
end;

function TNode.GetTargetNodeType: TNodeType;
begin
  if FNodeType=ntPointer then begin
    if Assigned(FTarget) then Exit(FTarget.TargetNodeType) else Exit(ntNormal);
  end else Exit(FNodeType);
end;

function TNode.GetText: string;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Result:=FTarget.Text else Result:='(no target)';
  end else Result:=FText;
end;

function TNode.GetTick: Boolean;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Result:=FTarget.Tick else Result:=False;
  end else Result:=FTick;
end;

procedure TNode.SetFresh(const AValue: Boolean);
begin
  if FFresh=AValue then Exit;
  FFresh:=AValue;
end;

procedure TNode.SetNodeType(const AValue: TNodeType);
begin
  if FNodeType=AValue then Exit;
  FNodeType:=AValue;
end;

procedure TNode.SetOpen(const AValue: Boolean);
begin
  if FOpen=AValue then Exit;
  FOpen:=AValue;
end;

procedure TNode.SetParent(const AValue: TNode);
begin
  if FParent=AValue then Exit;
  FParent:=AValue;
end;

procedure TNode.SetTarget(const AValue: TNode);
begin
  if FTarget=AValue then Exit;
  FTarget:=AValue;
end;

procedure TNode.SetTick(const AValue: Boolean);
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then FTarget.Tick:=AValue;
  end else begin
    if FTick=AValue then Exit;
    FTick:=AValue;
  end;
end;

procedure TNode.SetWasOpen(const AValue: Boolean);
begin
  if FWasOpen=AValue then Exit;
  FWasOpen:=AValue;
end;

constructor TNode.Create;
begin
end;

destructor TNode.Destroy;
var
  i: Integer;
begin
  if ScriptID <> 0 then UnregisterScriptID(ScriptID);
  for i:=0 to Length(FChildren) - 1 do FreeAndNil(FChildren[i]);
end;

procedure TNode.Add(AChild: TNode);
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then FTarget.Add(AChild);
    Exit;
  end;
  if AChild.Parent <> nil then AChild.Parent.Remove(AChild);
  AChild.FParent:=self;
  SetLength(FChildren, Length(FChildren) + 1);
  FChildren[Length(FChildren) - 1]:=AChild;
end;

function TNode.AddStr(Str: string): TNode;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Result:=FTarget.AddStr(Str) else Result:=nil;
    Exit;
  end;
  Result:=TNode.Create;
  Result.Text:=Str;
  Add(Result);
end;

procedure TNode.Remove(AChild: TNode);
var
  Idx, i: Integer;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then FTarget.Remove(AChild);
    Exit;
  end;
  Idx:=IndexOf(AChild);
  if Idx=-1 then Exit;
  for i:=Idx to Length(FChildren) - 2 do FChildren[i]:=FChildren[i + 1];
  SetLength(FChildren, Length(FChildren) - 1);
  AChild.FParent:=nil;
end;

procedure TNode.Insert(AChild: TNode; AIndex: Integer);
var
  i: Integer;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then FTarget.Insert(AChild, AIndex);
    Exit;
  end;
  if AChild.FParent <> nil then AChild.FParent.Remove(AChild);
  SetLength(FChildren, Length(FChildren) + 1);
  for i:=Length(FChildren) - 1 downto AIndex + 1 do FChildren[i]:=FChildren[i - 1];
  FChildren[AIndex]:=AChild;
  AChild.FParent:=Self;
end;

function TNode.IndexOf(AChild: TNode): Integer;
var
  i: Integer;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Result:=FTarget.IndexOf(AChild) else Result:=-1;
    Exit;
  end;
  for i:=0 to Length(FChildren) - 1 do if FChildren[i]=AChild then Exit(i);
  Result:=-1;
end;

procedure TNode.OrderChildren;
var
  i, j: Integer;
  Tmp: TNode;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then FTarget.OrderChildren;
    Exit;
  end;
  for i:=1 to Length(FChildren) - 1 do begin
    Tmp:=FChildren[i];
    j:=i - 1;
    while True do begin
      if AnsiCompareStr(FChildren[j].Text, Tmp.Text) > 0 then begin
        Children[j + 1]:=FChildren[j];
        j:=j - 1;
        if j < 0 then break;
      end else break;
    end;
    FChildren[j + 1]:=Tmp;
  end;
end;

function TNode.HasChildren: Boolean;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Result:=FTarget.HasChildren else Result:=False;
    Exit;
  end;
  Result:=Length(FChildren) > 0;
end;

function TNode.TickPercent: Integer;
var
  i, Per: Integer;
  Done, From: Int64;
begin
  if NodeType=ntPointer then begin
    if Assigned(FTarget) then Exit(FTarget.TickPercent) else Exit(100);
  end;
  if NodeType <> ntTickable then Exit(100);
  if not HasChildren then
    if Tick then Exit(100) else Exit(0);
  Done:=0;
  From:=0;
  for i:=0 to Length(FChildren) - 1 do begin
    if not (FChildren[i].NodeType in [ntTickable, ntPointer]) then Continue;
    Per:=FChildren[i].TickPercent;
    Done:=Done + Per;
    Inc(From);
  end;
  if From=0 then Exit(0);
  Result:=Done div From;
end;

function TNode.Address: string;
begin
  if Assigned(FParent) then Address:=Trim(FParent.Address + ' ' + IntToStr(FParent.IndexOf(Self)));
end;

procedure TNode.Save(var f: TextFile);
var
  i: Integer;
begin
  WriteLn(f, 'NODE ', Text);
  case NodeType of
    ntNormal: WriteLn(f, 'TYPE NORMAL');
    ntTickable: WriteLn(f, 'TYPE TICKABLE');
    ntPointer: begin
      WriteLn(f, 'TYPE POINTER');
      if Assigned(FTarget) then WriteLn(f, 'LINK ', FTarget.Address);
    end;
  end;
  if Tick then WriteLn(f, 'TICK');
  if Open and WasOpen then WriteLn(f, 'OPEN');
  for i:=0 to Length(FChildren) - 1 do begin
    if FChildren[i].Fresh and (FChildren[i].Text='') then Continue;
    FChildren[i].Save(f);
  end;
  WriteLn(f, 'DONE');
end;

procedure TNode.Load(var f: TextFile);
var
  s, Cmd, Arg: string;
begin
  while not Eof(f) do begin
    Readln(f, s);
    if Length(s) < 4 then Continue;
    Cmd:=Copy(s, 1, 4);
    Arg:=Copy(s, 6, Length(s));
    if Cmd='DONE' then break
    else if Cmd='TYPE' then begin
      if Arg='NORMAL' then NodeType:=ntNormal
      else if Arg='TICKABLE' then NodeType:=ntTickable
      else if Arg='POINTER' then NodeType:=ntPointer;
    end else if Cmd='TICK' then
      Tick:=True
    else if Cmd='OPEN' then begin
      Open:=True;
      WasOpen:=True;
    end else if Cmd='NODE' then begin
      AddStr(Arg).Load(f);
    end else if Cmd='LINK' then begin
      TargetAddress:=Arg;
    end;
  end;
end;

function TNode.Clone: TNode;
var
  I: Integer;
  ChildClone: TNode;
begin
  Result:=TNode.Create;
  Result.FTarget:=FTarget;
  Result.FText:=FText;
  Result.FNodeType:=FNodeType;
  Result.FTick:=FTick;
  Result.FOpen:=FOpen;
  Result.FWasOpen:=FWasOpen;
  for I:=0 to High(FChildren) do begin
    ChildClone:=FChildren[I].Clone;
    Result.Add(ChildClone);
  end;
end;

end.
