unit UMain;
{$mode objfpc}{$H+}
{$modeswitch advancedrecords}
{$ifdef WIN64}{$codealign localmin=16}{$ENDIF}
interface

uses
  Classes, SysUtils, Forms, Controls, Graphics, Dialogs, StdCtrls, Menus,
  ExtCtrls, Spin, ComCtrls, FGL, UDWARFScanner;

const
  Version = '1.1';
  MaxCallStackDepth = 256;

type

  { TDebugInfoFormat }
  TDebugInfoFormat = (difNone, difSTABS, difDWARF);

  { TSample }
  TSample = record
    Thread: UIntPtr;
    IP: UIntPtr;
    CallStack: array of UIntPtr;
  end;

  { TFuncInfo }
  TFuncInfo = class
    Name: string;
    Address, Size: UIntPtr;
    Hits: IntPtr;
    Fake, Approximate: Boolean;
  end;

  TFuncInfoList = specialize TFPGObjectList<TFuncInfo>;

  { TModuleInfo }
  TModuleInfo = record
    Name: string;
    StartAddr: UIntPtr;
    EndAddr: UIntPtr;
  end;

  { TCallTrace }
  TCallTrace = class
    Calls: array of TFuncInfo;
    Hits: IntPtr;
  end;
  TCallTraceList = specialize TFPGObjectList<TCallTrace>;

  { TCallInfo }
  TCallInfo = class
    Func: TFuncInfo;
    Hits: IntPtr;
  end;
  TCallInfoList = specialize TFPGObjectList<TCallInfo>;

  { TMain }
  TMain = class(TForm)
    cbClosest: TCheckBox;
    cbCaptureStack: TCheckBox;
    lbDetails: TListBox;
    MainMenu1: TMainMenu;
    MenuItem1: TMenuItem;
    MenuItem3: TMenuItem;
    pmResultsFind: TMenuItem;
    pmDetailsCopyName: TMenuItem;
    pmResultsCopyName: TMenuItem;
    pmDetailsShowDirectCallersCallees: TMenuItem;
    pmResultsShowDirectCallersCallees: TMenuItem;
    pmDetailsShowCallTracesAnywhere: TMenuItem;
    pmDetailsShowCallTracesFromFunction: TMenuItem;
    pmResultsShowCallTracesFromFunction: TMenuItem;
    pmResultsShowCallTracesAnywhere: TMenuItem;
    MenuItem2: TMenuItem;
    mHelpAbout: TMenuItem;
    mHelp: TMenuItem;
    mFile: TMenuItem;
    mFileExit: TMenuItem;
    mView: TMenuItem;
    mViewRefreshProcesses: TMenuItem;
    lbProcesses: TListBox;
    pmDetails: TPopupMenu;
    Splitter1: TSplitter;
    plMainPanel: TPanel;
    Panel1: TPanel;
    btStartSampling: TButton;
    btStopSampling: TButton;
    lbResults: TListBox;
    ApplicationProperties1: TApplicationProperties;
    spDetails: TSplitter;
    Timer1: TTimer;
    cbThreads: TComboBox;
    cbAutoStart: TCheckBox;
    edEXE: TEdit;
    cbMaxSamples: TCheckBox;
    seMaxSamples: TSpinEdit;
    Timer2: TTimer;
    Label1: TLabel;
    cbStartShortcut: TComboBox;
    Label2: TLabel;
    cbStopShortcut: TComboBox;
    Label3: TLabel;
    cbOnlyKnown: TCheckBox;
    pbProgress: TProgressBar;
    sbStatus: TStatusBar;
    pmResults: TPopupMenu;
    pmResultsCopyAll: TMenuItem;
    MenuItem4: TMenuItem;
    pmDetailsHide: TMenuItem;
    procedure cbClosestChange(Sender: TObject);
    procedure mFileExitClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure btStartSamplingClick(Sender: TObject);
    procedure btStopSamplingClick(Sender: TObject);
    procedure ApplicationProperties1Idle(Sender: TObject; var Done: Boolean);
    procedure mHelpAboutClick(Sender: TObject);
    procedure pmDetailsCopyNameClick(Sender: TObject);
    procedure pmDetailsPopup(Sender: TObject);
    procedure pmDetailsShowCallTracesAnywhereClick(Sender: TObject);
    procedure pmDetailsShowCallTracesFromFunctionClick(Sender: TObject);
    procedure pmDetailsShowDirectCallersCalleesClick(Sender: TObject);
    procedure pmResultsCopyNameClick(Sender: TObject);
    procedure pmResultsFindClick(Sender: TObject);
    procedure pmResultsPopup(Sender: TObject);
    procedure pmResultsShowCallTracesAnywhereClick(Sender: TObject);
    procedure pmResultsShowCallTracesFromFunctionClick(Sender: TObject);
    procedure pmResultsShowDirectCallersCalleesClick(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure mViewRefreshProcessesClick(Sender: TObject);
    procedure lbProcessesClick(Sender: TObject);
    procedure cbThreadsChange(Sender: TObject);
    procedure Timer2Timer(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure pmResultsCopyAllClick(Sender: TObject);
    procedure pmDetailsHideClick(Sender: TObject);
  private
    SampledProcessID: UIntPtr;
    SampledProcessHandle: THandle;
    Samples: array of TSample;
    StackCaptureEnabled: Boolean;
    FullSampleCount: UIntPtr;
    FuncInfos: TFuncInfoList;
    UsedThreads: array of UIntPtr;
    Modules: array of TModuleInfo;
    ClickedProcessPath: string;
    ClickedDebugInfoFormat: TDebugInfoFormat;
    function GetDebugInfoFormat(AFileName: string): TDebugInfoFormat;
    procedure LoadSTABSDebugInfo(AFileName: string);
    procedure LoadDWARFDebugInfo(AFileName: string);
    procedure StartSampling(AProcessID: UIntPtr);
    procedure StopSampling;
    procedure CollectSamples;
    procedure FindUsedThreads;
    procedure CreateProfile;
    procedure RefreshProcessList;
    procedure SetDetailsVisible(AVisible: Boolean);
    procedure ShowCallTracesFor(AFuncInfo: TFuncInfo; FromFunction: Boolean);
    procedure ShowDirectCallersCallees(AFuncInfo: TFuncInfo);
  public

  end;

var
  Main: TMain;

implementation

uses
  Windows, JwaTlHelp32, JwaWinBase, ExeInfo, mmsystem, Clipbrd, imagehlp;

{$R *.lfm}

{ Functions }
function FuncInfoHitSort(const A, B: TFuncInfo): Integer;
begin
  Result:=B.Hits - A.Hits;
end;

function CallTraceHitSort(const A, B: TCallTrace): Integer;
begin
  Result:=B.Hits - A.Hits;
end;

function CallInfoHitSort(const A, B: TCallInfo): Integer;
begin
  Result:=B.Hits - A.Hits;
end;

{ TMain }
                           
procedure TMain.StartSampling(AProcessID: UIntPtr);
var
  Snap: THandle;
  ME: TModuleEntry32;
begin
  Assert(SampledProcessHandle=INVALID_HANDLE_VALUE, 'Already sampling a process');
  SetDetailsVisible(False);
  SampledProcessID:=0;
  SampledProcessHandle:=OpenProcess(PROCESS_ALL_ACCESS, False, AProcessID);
  StackCaptureEnabled:=cbCaptureStack.Checked;
  if StackCaptureEnabled then SymInitialize(SampledProcessHandle, nil, True);
  sbStatus.Panels[0].Text:='';
  sbStatus.Panels[1].Text:='';
  sbStatus.Panels[2].Text:='';
  if SampledProcessHandle=0 then begin
    SampledProcessHandle:=INVALID_HANDLE_VALUE;
    sbStatus.Panels[2].Text:='Failed to open process (you may want to refresh, press Ctrl+R)';
    Exit;
  end;
  Modules:=nil;
  if not cbOnlyKnown.Checked then begin
    Snap:=CreateToolhelp32Snapshot(TH32CS_SNAPMODULE, AProcessID);
    if Snap=INVALID_HANDLE_VALUE then begin
      CloseHandle(SampledProcessHandle);
      SampledProcessHandle:=INVALID_HANDLE_VALUE;
      sbStatus.Panels[2].Text:='Failed to create system snapshot (you may want to refresh, press Ctrl+R)';
      Exit;
    end;
    ME.dwSize:=SizeOf(ME);
    if Module32First(Snap, ME) then repeat
      SetLength(Modules, Length(Modules) + 1);
      Modules[High(Modules)].Name:=ME.szModule;
      Move(ME.modBaseAddr, Modules[High(Modules)].StartAddr, SizeOf(UIntPtr));
      Modules[High(Modules)].EndAddr:=Modules[High(Modules)].StartAddr + ME.modBaseSize;
    until not Module32Next(Snap, ME);
    CloseHandle(Snap);
  end;
  SampledProcessID:=AProcessID;
  Samples:=nil;
  FullSampleCount:=0;
  lbResults.Clear;
  sbStatus.Panels[1].Text:='Sampling...';
end;

procedure TMain.StopSampling;
begin
  Assert(SampledProcessHandle <> INVALID_HANDLE_VALUE, 'Not sampling a process');
  if StackCaptureEnabled then SymCleanup(SampledProcessHandle);
  CloseHandle(SampledProcessHandle);
  SampledProcessHandle:=INVALID_HANDLE_VALUE;
  SampledProcessID:=0;
  sbStatus.Panels[1].Text:='Stopped';
end;

procedure TMain.CollectSamples;
var
  Snap: THandle;
  Threads: array of DWORD;
  TE: TThreadEntry32;
  I, CSLevel: Integer;
  ThreadHandle: THandle;
  Ctx: Windows.TContext;
  CallStack: array [0..MaxCallStackDepth - 1] of UIntPtr;
  StackFrame: TSTACKFRAME64;
begin
  Assert(SampledProcessHandle <> INVALID_HANDLE_VALUE, 'Not sampling a process');
  Snap:=CreateToolhelp32Snapshot(TH32CS_SNAPTHREAD, 0);
  if Snap=INVALID_HANDLE_VALUE then Exit;
  ZeroMemory(@TE, SizeOf(TE));
  TE.dwSize:=SizeOf(TE);
  Threads:=nil;
  if Thread32First(Snap, TE) then repeat
    if TE.th32OwnerProcessID=SampledProcessID then begin
      SetLength(Threads, Length(Threads) + 1);
      Threads[High(Threads)]:=TE.th32ThreadID;
    end;
  until not Thread32Next(Snap, TE);
  CloseHandle(Snap);
  if Length(Threads)=0 then begin
    btStopSampling.Click;
    sbStatus.Panels[2].Text:='Program exited.';
    Exit;
  end;
  ZeroMemory(@CallStack, SizeOf(CallStack));
  for I:=0 to High(Threads) do begin
    ThreadHandle:=OpenThread(THREAD_GET_CONTEXT or THREAD_SET_CONTEXT or THREAD_QUERY_INFORMATION or THREAD_SUSPEND_RESUME, False, Threads[I]);
    if ThreadHandle=INVALID_HANDLE_VALUE then Continue;
    if SuspendThread(ThreadHandle) <> DWORD(-1) then begin
      Ctx.ContextFlags:=CONTEXT_CONTROL or CONTEXT_INTEGER;
      if Windows.GetThreadContext(ThreadHandle, Ctx) then begin
        SetLength(Samples, Length(Samples) + 1);
        Samples[High(Samples)].Thread:=Threads[I];
        Samples[High(Samples)].IP:=Ctx.{$IFDEF CPUAMD64}Rip{$ELSE}Eip{$ENDIF};
        if StackCaptureEnabled then begin
          CSLevel:=0;
          ZeroMemory(@StackFrame, SizeOf(StackFrame));
          StackFrame.AddrPC.Offset:=Ctx.{$IFDEF CPUAMD64}Rip{$ELSE}Eip{$ENDIF};
          StackFrame.AddrPC.Mode:=AddrModeFlat;
          StackFrame.AddrStack.Offset:=Ctx.{$IFDEF CPUAMD64}Rsp{$ELSE}Esp{$ENDIF};
          StackFrame.AddrStack.Mode:=AddrModeFlat;
          StackFrame.AddrFrame.Offset:=Ctx.{$IFDEF CPUAMD64}Rbp{$ELSE}Ebp{$ENDIF};
          StackFrame.AddrFrame.Mode:=AddrModeFlat;
          while StackWalk64({$IFDEF CPUAMD64}IMAGE_FILE_MACHINE_AMD64{$ELSE}IMAGE_FILE_MACHINE_I386{$ENDIF}, SampledProcessHandle, ThreadHandle, @StackFrame, @Ctx, nil, @SymFunctionTableAccess64, @SymGetModuleBase64, nil) do begin
            if (StackFrame.AddrPC.Offset=0) or (CSLevel=MaxCallStackDepth) then Break;
            CallStack[CSLevel]:=StackFrame.AddrPC.Offset;
            Inc(CSLevel);
          end;
          SetLength(Samples[High(Samples)].CallStack, CSLevel);
          if CSLevel <> 0 then Move(CallStack[0], Samples[High(Samples)].CallStack[0], CSLevel*SizeOf(UIntPtr));
        end;
      end else begin
        ShowMessage(IntToStr(GetLastError));
        Halt;
      end;
      ResumeThread(ThreadHandle);
    end;
    CloseHandle(ThreadHandle);
  end;
  Inc(FullSampleCount);
end;

procedure TMain.FindUsedThreads;
var
  I, J: Integer;
  Found: Boolean;
begin
  UsedThreads:=nil;
  for I:=0 to High(Samples) do begin
    Found:=False;
    for J:=0 to High(UsedThreads) do
      if UsedThreads[J]=Samples[I].Thread then begin
        Found:=True;
        Break;
      end;
    if not Found then begin
      SetLength(UsedThreads, Length(UsedThreads) + 1);
      UsedThreads[High(UsedThreads)]:=Samples[I].Thread;
    end;
  end;
  cbThreads.Clear;
  cbThreads.Items.BeginUpdate;
  for I:=0 to High(UsedThreads) do
    cbThreads.Items.Add('#' + IntToStr(I) + ' ' + HexStr(UsedThreads[I], 8));
  cbThreads.Items.EndUpdate;
  if Length(UsedThreads) > 0 then cbThreads.ItemIndex:=0;
end;

procedure TMain.CreateProfile;
var
  I, J, K: Integer;
  Found: Boolean;
  ThreadFilter, TotalHits, UnknownHits: UIntPtr;
  FuncInfo: TFuncInfo;
  Entry: string;
begin
  sbStatus.Panels[1].Text:='Stopped';
  lbResults.Clear;
  SetDetailsVisible(False);
  if Length(Samples)=0 then Exit;
  sbStatus.Panels[1].Text:='Profiling...';
  sbStatus.Repaint;
  Application.ProcessMessages;
  if cbThreads.ItemIndex=-1 then
    ThreadFilter:=Samples[0].Thread
  else
    ThreadFilter:=UsedThreads[cbThreads.ItemIndex];
  for I:=FuncInfos.Count - 1 downto 0 do begin
    FuncInfos[I].Hits:=0;
    FuncInfos[I].Approximate:=False;
    if FuncInfos[I].Fake then FuncInfos.Delete(I);
  end;
  TotalHits:=0;
  UnknownHits:=0;
  for I:=0 to High(Samples) do if ThreadFilter=Samples[I].Thread then begin
    Found:=False;
    for J:=0 to FuncInfos.Count - 1 do
      if (Samples[I].IP >= FuncInfos[J].Address) and
         (Samples[I].IP < FuncInfos[J].Address + FuncInfos[J].Size) then begin
        Inc(FuncInfos[J].Hits);
        Inc(TotalHits);
        Found:=True;
        Break;
      end;
    if StackCaptureEnabled and (not Found) and cbClosest.Checked then begin
      for K:=0 to High(Samples[I].CallStack) do begin
        for J:=0 to FuncInfos.Count - 1 do if not FuncInfos[J].Fake then begin
          if (Samples[I].CallStack[K] >= FuncInfos[J].Address) and
             (Samples[I].CallStack[K] < FuncInfos[J].Address + FuncInfos[J].Size) then begin
            Inc(FuncInfos[J].Hits);
            Inc(TotalHits);
            FuncInfos[J].Approximate:=True;
            Found:=True;
            Break;
          end;
        end;
        if Found then Break;
      end;
    end;
    if not Found then begin
      for J:=0 to High(Modules) do if (Samples[I].IP >= Modules[J].StartAddr) and (Samples[I].IP < Modules[J].EndAddr) then begin
        FuncInfo:=TFuncInfo.Create;
        FuncInfo.Name:='Somewhere in ' + Modules[J].Name;
        FuncInfo.Address:=Modules[J].StartAddr;
        FuncInfo.Size:=Modules[J].EndAddr - Modules[J].StartAddr;
        FuncInfo.Hits:=1;
        FuncInfo.Fake:=True;
        FuncInfos.Add(FuncInfo);
        Inc(TotalHits);
        Found:=True;
        Break;
      end;
    end;
    if not Found then Inc(UnknownHits);
  end;
  FuncInfos.Sort(@FuncInfoHitSort);
  if TotalHits=0 then begin
    lbResults.Items.Add('No samples from known functions in this thread');
    sbStatus.Panels[1].Text:='Stopped';
    Exit;
  end;
  lbResults.Items.BeginUpdate;
  for I:=0 to FuncInfos.Count - 1 do begin
    Entry:=FuncInfos[I].Name;
    if FuncInfos[I].Approximate then Entry += '*';
    Entry += ' (' + IntToStr(Round(FuncInfos[I].Hits/TotalHits*100)) + '%, ' + IntToStr(FuncInfos[I].Hits) + ' samples)';
    lbResults.Items.AddObject(Entry, FuncInfos[I]);
  end;
  if UnknownHits > 0 then lbResults.Items.Add('Samples from unknown functions (not counted in percentages above): ' + IntToStr(UnknownHits));
  lbResults.Items.EndUpdate;
  sbStatus.Panels[0].Text:='Total samples: ' + IntToStr(FullSampleCount);
  sbStatus.Panels[1].Text:='Stopped';
end;

procedure TMain.RefreshProcessList;
var
  Snap: THandle;
  PE: TProcessEntry32;
begin
  lbProcesses.Items.BeginUpdate;
  lbProcesses.Items.Clear;
  lbProcesses.Sorted:=False;
  Snap:=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
  if Snap=INVALID_HANDLE_VALUE then begin
    lbProcesses.Items.EndUpdate;
    Exit;
  end;
  ZeroMemory(@PE, SizeOf(PE));
  PE.dwSize:=SizeOf(PE);
  if Process32First(Snap, PE) then repeat
    lbProcesses.Items.AddObject(LowerCase(PE.szExeFile), TObject(UIntPtr(PE.th32ProcessID)));
  until not Process32Next(Snap, PE);
  CloseHandle(Snap);
  lbProcesses.Sorted:=True;
  lbProcesses.Items.EndUpdate;
end;

procedure TMain.SetDetailsVisible(AVisible: Boolean);
begin
  lbDetails.Visible:=AVisible;
  spDetails.Visible:=AVisible;
  if AVisible then begin
    spDetails.Left:=lbResults.Left + lbResults.Width + 1;
    lbDetails.Left:=spDetails.Left + spDetails.Width + 1;
  end;
end;

procedure TMain.ShowCallTracesFor(AFuncInfo: TFuncInfo; FromFunction: Boolean);
var
  Trace: TCallTrace;
  Traces: TCallTraceList;
  I, J, K, StartIdx: Integer;
  Found: Boolean;
  TotalHits: IntPtr = 0;
begin
  SetDetailsVisible(True);
  lbDetails.Clear;
  Traces:=TCallTraceList.Create(True);
  for I:=0 to High(Samples) do begin
    StartIdx:=-1;
    for J:=0 to High(Samples[I].CallStack) do
      if (Samples[I].CallStack[J] >= AFuncInfo.Address) and
         (Samples[I].CallStack[J] < AFuncInfo.Address + AFuncInfo.Size) then begin
        StartIdx:=J;
        Break;
      end;
    if (not FromFunction) and (StartIdx <> -1) then StartIdx:=0;
    if (StartIdx <> -1) and (Length(Samples[I].CallStack) - StartIdx > 0) then begin
      Inc(TotalHits);
      Trace:=TCallTrace.Create;
      Trace.Hits:=1;
      Trace.Calls:=nil;
      SetLength(Trace.Calls, Length(Samples[I].CallStack) - StartIdx);
      for J:=StartIdx to High(Samples[I].CallStack) do begin
        Found:=False;
        for K:=0 to FuncInfos.Count - 1 do if not FuncInfos[K].Fake then
          if (Samples[I].CallStack[J] >= FuncInfos[K].Address) and
             (Samples[I].CallStack[J] < FuncInfos[K].Address + FuncInfos[K].Size) then begin
            Found:=True;
            Trace.Calls[J - StartIdx]:=FuncInfos[K];
            Break;
          end;
        if not Found then begin
          for K:=0 to FuncInfos.Count - 1 do if FuncInfos[K].Fake then
            if (Samples[I].CallStack[J] >= FuncInfos[K].Address) and
               (Samples[I].CallStack[J] < FuncInfos[K].Address + FuncInfos[K].Size) then begin
              Found:=True;
              Trace.Calls[J - StartIdx]:=FuncInfos[K];
              Break;
            end;
        end;
        if not Found then Trace.Calls[J - StartIdx]:=nil;
      end;
      while (Length(Trace.Calls) > 0) and not Assigned(Trace.Calls[High(Trace.Calls)]) do
        SetLength(Trace.Calls, Length(Trace.Calls) - 1);
      if Length(Trace.Calls) > 0 then begin
        Found:=False;
        K:=-1;
        for J:=0 to Traces.Count - 1 do
          if (Length(Traces[J].Calls)=Length(Trace.Calls)) and
             (CompareByte(Traces[J].Calls[0], Trace.Calls[0], Length(Trace.Calls)*SizeOf(TFuncInfo))=0) then begin
            Found:=True;
            K:=J;
          end;
        if Found then begin
          Inc(Traces[K].Hits);
          Trace.Free;
        end else begin
          Traces.Add(Trace);
        end;
      end else Trace.Free;
    end;
  end;
  Traces.Sort(@CallTraceHitSort);
  lbDetails.Items.BeginUpdate;
  lbDetails.Items.AddObject(AFuncInfo.Name + ' is in ' + IntToStr(Traces.Count) + ' traces:', AFuncInfo);
  lbDetails.Items.Add('');
  for I:=0 to Traces.Count - 1 do with Traces[I] do begin
    lbDetails.Items.Add('Trace #' + IntToStr(I) + ', ' + IntToStr(Hits) + ' hits (' + IntToStr(Round(Hits/TotalHits*100)) + '%):');
    for J:=0 to High(Calls) do
      if Assigned(Calls[J]) then
        lbDetails.Items.AddObject('  #' + IntToStr(J) + ' ' + Calls[J].Name + ' (' + IntToStr(Calls[J].Hits) + ' samples)', Calls[J])
      else
        lbDetails.Items.Add('  #' + IntToStr(J) + ' ??');
    lbDetails.Items.Add('');
  end;
  lbDetails.Items.EndUpdate;
  Traces.Free;
end;

procedure TMain.ShowDirectCallersCallees(AFuncInfo: TFuncInfo);
var
  Callers, Callees: TCallInfoList;
  Caller, Callee: TFuncInfo;
  I, J, K: Integer;
  TotalCallerHits, TotalCalleeHits: IntPtr;
  Found: Boolean;
begin
  SetDetailsVisible(True);
  lbDetails.Clear;
  lbDetails.Items.AddObject(AFuncInfo.Name + ' callers and callees:', AFuncInfo);
  lbDetails.Items.Add('');
  Callers:=TCallInfoList.Create(True);
  Callees:=TCallInfoList.Create(True);
  TotalCallerHits:=0;
  TotalCalleeHits:=0;
  for I:=0 to High(Samples) do with Samples[I] do begin
    Caller:=nil;
    Callee:=nil;
    for J:=0 to High(CallStack) do
      if (CallStack[J] >= AFuncInfo.Address) and
         (CallStack[J] < AFuncInfo.Address + AFuncInfo.Size) then begin
        if J > 0 then begin
          for K:=0 to FuncInfos.Count - 1 do
            if (CallStack[J - 1] >= FuncInfos[K].Address) and
               (CallStack[J - 1] < FuncInfos[K].Address + FuncInfos[K].Size) then begin
              Callee:=FuncInfos[K];
              Break;
            end;
        end;
        if J < High(CallStack) then begin
          for K:=0 to FuncInfos.Count - 1 do
            if (CallStack[J + 1] >= FuncInfos[K].Address) and
               (CallStack[J + 1] < FuncInfos[K].Address + FuncInfos[K].Size) then begin
              Caller:=FuncInfos[K];
              Break;
            end;
        end;
        Break;
    end;
    if Assigned(Caller) then begin
      Found:=False;
      for J:=0 to Callers.Count - 1 do
        if Callers[J].Func=Caller then begin
          Inc(Callers[J].Hits);
          Found:=True;
          Break;
        end;
      if not Found then begin
        Callers.Add(TCallInfo.Create);
        Callers.Last.Func:=Caller;
        Callers.Last.Hits:=1;
      end;
      Inc(TotalCallerHits);
    end;
    if Assigned(Callee) then begin
      Found:=False;
      for J:=0 to Callees.Count - 1 do
        if Callees[J].Func=Callee then begin
          Inc(Callees[J].Hits);
          Found:=True;
          Break;
        end;
      if not Found then begin
        Callees.Add(TCallInfo.Create);
        Callees.Last.Func:=Callee;
        Callees.Last.Hits:=1;
      end;
      Inc(TotalCalleeHits);
    end;
  end;
  Callers.Sort(@CallInfoHitSort);
  Callees.Sort(@CallInfoHitSort);
  lbDetails.Items.BeginUpdate;
  if Callers.Count > 0 then begin
    lbDetails.Items.Add(IntToStr(Callers.Count) + ' callers:');
    for I:=0 to Callers.Count - 1 do begin
      lbDetails.Items.AddObject('  #' + IntToStr(I) + ' ' + Callers[I].Func.Name + ' (' +
        IntToStr(Round(Callers[I].Hits/TotalCallerHits*100)) + '%, ' + IntToStr(Callers[I].Hits) + ' calls)', Callers[I].Func);
    end;
    lbDetails.Items.Add('');
  end;
  if Callees.Count > 0 then begin
    lbDetails.Items.Add(IntToStr(Callees.Count) + ' callees:');
    for I:=0 to Callees.Count - 1 do begin
      lbDetails.Items.AddObject('  #' + IntToStr(I) + ' ' + Callees[I].Func.Name + ' (' +
        IntToStr(Round(Callees[I].Hits/TotalCalleeHits*100)) + '%, ' + IntToStr(Callees[I].Hits) + ' calls)', Callees[I].Func);
    end;
    lbDetails.Items.Add('');
  end;
  lbDetails.Items.EndUpdate;
  Callees.Free;
  Callers.Free;
end;

procedure TMain.mFileExitClick(Sender: TObject);
begin
  Close;
end;

procedure TMain.cbClosestChange(Sender: TObject);
begin
  CreateProfile;
end;

procedure TMain.FormShow(Sender: TObject);
begin
  {$IFDEF WIN64}
  Caption:='Free Pascal Windows Profiler version ' + Version + ' (for 64bit programs only)';
  {$ELSE}
  Caption:='Free Pascal Windows Profiler version ' + Version + ' (for 32bit programs only)';
  {$ENDIF}
  FuncInfos:=TFuncInfoList.Create;
  SampledProcessHandle:=INVALID_HANDLE_VALUE;
  RefreshProcessList;
end;

procedure TMain.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  if SampledProcessHandle <> INVALID_HANDLE_VALUE then StopSampling;
  FuncInfos.Free;
end;

procedure TMain.btStartSamplingClick(Sender: TObject);
var
  ID: UIntPtr;
begin
  if SampledProcessHandle <> INVALID_HANDLE_VALUE then StopSampling;
  btStartSampling.Enabled:=False;
  if lbProcesses.ItemIndex=-1 then Exit;
  ID:=UIntPtr(lbProcesses.Items.Objects[lbProcesses.ItemIndex]);
  if ID <> 0 then StartSampling(ID);
end;

procedure TMain.btStopSamplingClick(Sender: TObject);
begin
  if SampledProcessHandle <> INVALID_HANDLE_VALUE then begin
    btStopSampling.Enabled:=False;
    StopSampling;
    FindUsedThreads;
    case ClickedDebugInfoFormat of
      difDWARF: LoadDWARFDebugInfo(ClickedProcessPath);
      difSTABS: LoadSTABSDebugInfo(ClickedProcessPath);
    end;
    CreateProfile;
  end;
end;

procedure TMain.ApplicationProperties1Idle(Sender: TObject; var Done: Boolean);
begin
  btStartSampling.Enabled:=(SampledProcessHandle=INVALID_HANDLE_VALUE) and (lbProcesses.ItemIndex <> -1);
  btStopSampling.Enabled:=SampledProcessHandle <> INVALID_HANDLE_VALUE;
  cbCaptureStack.Enabled:=SampledProcessHandle=INVALID_HANDLE_VALUE;
  cbClosest.Enabled:=(SampledProcessHandle=INVALID_HANDLE_VALUE) and cbCaptureStack.Checked;
  cbOnlyKnown.Enabled:=SampledProcessHandle=INVALID_HANDLE_VALUE;
  cbThreads.Enabled:=SampledProcessHandle=INVALID_HANDLE_VALUE;
  Done:=True;
end;

procedure TMain.mHelpAboutClick(Sender: TObject);
begin
  ShowMessage('Free Pascal Windows Profiler version ' + Version + LineEnding +
              'Copyright (C) 2021 Kostas Michalopoulos' + LineEnding + LineEnding +
              'Released under the zlib license.' + LineEnding +
              'Find more at http://runtimeterror.com/tools/fpwprof');
end;

procedure TMain.pmDetailsCopyNameClick(Sender: TObject);
begin
  Clipboard.AsText:=TFuncInfo(lbDetails.Items.Objects[lbDetails.ItemIndex]).Name;
end;

procedure TMain.pmDetailsPopup(Sender: TObject);
begin
  pmDetailsShowCallTracesAnywhere.Enabled:=(lbDetails.ItemIndex <> -1) and Assigned(lbDetails.Items.Objects[lbDetails.ItemIndex]);
  pmDetailsShowCallTracesFromFunction.Enabled:=pmDetailsShowCallTracesAnywhere.Enabled;
  pmDetailsShowDirectCallersCallees.Enabled:=pmDetailsShowCallTracesAnywhere.Enabled;
  pmDetailsCopyName.Enabled:=pmDetailsShowCallTracesAnywhere.Enabled;
end;

procedure TMain.pmDetailsShowCallTracesAnywhereClick(Sender: TObject);
begin
  ShowCallTracesFor(TFuncInfo(lbDetails.Items.Objects[lbDetails.ItemIndex]), False);
end;

procedure TMain.pmDetailsShowCallTracesFromFunctionClick(Sender: TObject);
begin
  ShowCallTracesFor(TFuncInfo(lbDetails.Items.Objects[lbDetails.ItemIndex]), True);
end;

procedure TMain.pmDetailsShowDirectCallersCalleesClick(Sender: TObject);
begin
  ShowDirectCallersCallees(TFuncInfo(lbDetails.Items.Objects[lbDetails.ItemIndex]));
end;

procedure TMain.pmResultsCopyNameClick(Sender: TObject);
begin
  Clipboard.AsText:=TFuncInfo(lbResults.Items.Objects[lbResults.ItemIndex]).Name;
end;

procedure TMain.pmResultsFindClick(Sender: TObject);
var
  S: string;
  StartIdx, I: Integer;
begin
  S:='';
  if not InputQuery('Find', 'Text to search for:', S) then Exit;
  S:=LowerCase(Trim(S));
  if S='' then Exit;
  StartIdx:=lbResults.ItemIndex + 1;
  if (StartIdx=-1) or (StartIdx >= lbResults.Items.Count) then StartIdx:=0;
  for I:=StartIdx to lbResults.Items.Count - 1 do
    if Pos(S, LowerCase(lbResults.Items[I])) <> 0 then begin
      lbResults.ItemIndex:=I;
      Exit;
    end;
  for I:=0 to StartIdx - 1 do
    if Pos(S, LowerCase(lbResults.Items[I])) <> 0 then begin
      lbResults.ItemIndex:=I;
      Exit;
    end;
  ShowMessage('Text not found');
end;

procedure TMain.pmResultsPopup(Sender: TObject);
begin
  pmResultsShowCallTracesAnywhere.Enabled:=(lbResults.ItemIndex <> -1) and Assigned(lbResults.Items.Objects[lbResults.ItemIndex]);
  pmResultsShowCallTracesFromFunction.Enabled:=pmResultsShowCallTracesAnywhere.Enabled;
  pmResultsShowDirectCallersCallees.Enabled:=pmResultsShowCallTracesAnywhere.Enabled;
  pmResultsCopyName.Enabled:=pmResultsShowCallTracesAnywhere.Enabled;
end;

procedure TMain.pmResultsShowCallTracesAnywhereClick(Sender: TObject);
begin
  ShowCallTracesFor(TFuncInfo(lbResults.Items.Objects[lbResults.ItemIndex]), False);
end;

procedure TMain.pmResultsShowCallTracesFromFunctionClick(Sender: TObject);
begin
  ShowCallTracesFor(TFuncInfo(lbResults.Items.Objects[lbResults.ItemIndex]), True);
end;

procedure TMain.pmResultsShowDirectCallersCalleesClick(Sender: TObject);
begin
  ShowDirectCallersCallees(TFuncInfo(lbResults.Items.Objects[lbResults.ItemIndex]));
end;

procedure TMain.Timer1Timer(Sender: TObject);
begin
  if (cbStartShortcut.ItemIndex > 0) and btStartSampling.Enabled and ((GetAsyncKeyState(VK_F1 + cbStartShortcut.ItemIndex - 1) and $8000)=$8000) then begin
    btStartSampling.Click;
    SysUtils.Beep;
  end;
  if (cbStopShortcut.ItemIndex > 0) and cbStopShortcut.Enabled and ((GetAsyncKeyState(VK_F1 + cbStopShortcut.ItemIndex - 1) and $8000)=$8000) then begin
    btStopSampling.Click;
    SysUtils.Beep;
  end;
  if SampledProcessHandle=INVALID_HANDLE_VALUE then Exit;
  CollectSamples;
  sbStatus.Panels[0].Text:=IntToStr(FullSampleCount) + ' samples';
  if cbMaxSamples.Checked and (FullSampleCount >= UIntPtr(seMaxSamples.Value)) then begin
    btStopSampling.Click;
    SysUtils.Beep;
  end;
end;

procedure TMain.mViewRefreshProcessesClick(Sender: TObject);
begin
  RefreshProcessList;
end;

procedure TMain.lbProcessesClick(Sender: TObject);
var
  ID: UIntPtr;
  Snap: THandle;
  ME: TModuleEntry32;
begin
  if SampledProcessHandle <> INVALID_HANDLE_VALUE then Exit;
  sbStatus.Panels[2].Text:='';
  if lbProcesses.ItemIndex=-1 then Exit;
  ID:=UIntPtr(lbProcesses.Items.Objects[lbProcesses.ItemIndex]);
  if ID=0 then Exit;
  Snap:=CreateToolhelp32Snapshot(TH32CS_SNAPMODULE, ID);
  ZeroMemory(@ME, SizeOf(ME));
  ME.dwSize:=SizeOf(ME);
  if not Module32First(Snap, ME) then begin
    sbStatus.Panels[2].Text:='Cannot obtain first module for ' + lbProcesses.Items[lbProcesses.ItemIndex];
    CloseHandle(Snap);
    Exit;
  end;
  CloseHandle(Snap);
  if not FileExists(ME.szExePath) then begin
    sbStatus.Panels[2].Text:='Cannot find EXE file at ' + ME.szExePath;
    Exit;
  end;
  ClickedProcessPath:=ME.szExePath;
  ClickedDebugInfoFormat:=GetDebugInfoFormat(ClickedProcessPath);
  case ClickedDebugInfoFormat of
    difNone: sbStatus.Panels[2].Text:='No debug data in ' + ClickedProcessPath;
    difDWARF: sbStatus.Panels[2].Text:='DWARF debug data found ' + ClickedProcessPath;
    difSTABS: sbStatus.Panels[2].Text:='STABS debug data found ' + ClickedProcessPath;
  end;
end;

procedure TMain.cbThreadsChange(Sender: TObject);
begin
  CreateProfile;
end;

procedure TMain.Timer2Timer(Sender: TObject);
var
  Idx: Integer;
begin
  if SampledProcessHandle <> INVALID_HANDLE_VALUE then Exit;
  if cbAutoStart.Checked and (edEXE.Text <> '') then begin
    RefreshProcessList;
    Idx:=lbProcesses.Items.IndexOf(LowerCase(edEXE.Text));
    if Idx <> -1 then begin
      cbAutoStart.Checked:=False;
      lbProcesses.ItemIndex:=Idx;
      lbProcessesClick(lbProcesses);
      if cbStartShortcut.ItemIndex < 1 then btStartSampling.Click;
    end;
  end;
end;

procedure TMain.FormCreate(Sender: TObject);
begin
  timeBeginPeriod(1);
end;

procedure TMain.FormDestroy(Sender: TObject);
begin
  timeEndPeriod(1);
end;

procedure TMain.pmResultsCopyAllClick(Sender: TObject);
begin
  Clipboard.AsText:=lbResults.Items.Text;
end;

procedure TMain.pmDetailsHideClick(Sender: TObject);
begin
  SetDetailsVisible(False);
end;

function TMain.GetDebugInfoFormat(AFileName: string): TDebugInfoFormat;
var
  E: TExeFile;
  Ofs: LongInt = 0;
  Len: LongInt = 0;
begin
  ZeroMemory(@E, SizeOf(E));
  if not OpenExeFile(E, AFileName) then Exit(difNone);
  if FindExeSection(E, '.debug_info', Ofs, Len) and FindExeSection(E, '.debug_abbrev', Ofs, Len) then begin
    CloseExeFile(E);
    Exit(difDWARF);
  end;
  if FindExeSection(E, '.gnu_debuglink', Ofs, Len) then begin
    CloseExeFile(E);
    Exit(difDWARF);
  end;
  if FindExeSection(E, '.stab', Ofs, Len) and FindExeSection(E, '.stabstr', Ofs, Len) then begin
    CloseExeFile(E);
    Exit(difSTABS);
  end;
  CloseExeFile(E);
  Result:=difNone;
end;

procedure TMain.LoadSTABSDebugInfo(AFileName: string);
const
  N_Function = $24;
  N_RBrack = $E0;
type
  TStabEntry = packed record
    StrPos: LongInt;
    NType: Byte;
    NOther: Byte;
    NDesc: Word;
    NValue: DWord;
  end;
var
  E: TExeFile;
  StabOfs, StabLen, StabStrOfs, StabStrLen: LongInt;
  StabStrData: array of Char;
  Stabs: array of TStabEntry;
  StabStr: array of string;
  I, J: Integer;
  FuncName: string;
  FuncSize: UIntPtr;
  ColonPos: SizeInt;
  FuncInfo: TFuncInfo;
begin
  sbStatus.Panels[1].Text:='';
  FuncInfos.Clear;
  ZeroMemory(@E, SizeOf(E));
  if not OpenExeFile(E, AFileName) then begin
    sbStatus.Panels[2].Text:='Failed to open EXE';
    Exit;
  end;
  StabOfs:=0;
  StabLen:=0;
  StabStrOfs:=0;
  StabStrLen:=0;
  if not (FindExeSection(E, '.stab', StabOfs, StabLen) and FindExeSection(E, '.stabstr', StabStrOfs, StabStrLen)) then begin
    CloseExeFile(E);
    sbStatus.Panels[2].Text:='No STABS debug info';
    Exit;
  end;
  SetLength(Stabs, StabLen div Sizeof(TStabEntry));
  SetLength(StabStrData, StabStrLen);
  SetLength(StabStr, Length(Stabs));
  IOResult;
  {$I-}
  Seek(E.f, StabOfs);
  BlockRead(E.f, Stabs[0], StabLen);
  Seek(E.f, StabStrOfs);
  BlockRead(E.f, StabStrData[0], StabStrLen);
  {$I+}
  if IOResult <> 0 then begin
    sbStatus.Panels[2].Text:='Error reading STABS data';
    CloseExeFile(E);
    Exit;
  end;
  CloseExeFile(E);
  for I:=0 to High(Stabs) do StabStr[I]:=PChar(@StabStrData[Stabs[I].StrPos]);
  pbProgress.Visible:=True;
  pbProgress.Position:=0;
  pbProgress.Repaint;
  sbStatus.Panels[1].Text:='Reading STABS...';
  sbStatus.Repaint;
  Application.ProcessMessages;
  for I:=0 to High(StabStr) do begin
    pbProgress.Position:=I*1000 div High(StabStr);
    if (I and $FF)=0 then Application.ProcessMessages;
    if Stabs[I].NType=N_Function then begin
      FuncName:=StabStr[I];
      ColonPos:=Pos(':F', FuncName);
      if ColonPos=0 then Continue;
      FuncName:=Copy(FuncName, 1, ColonPos - 1);
      FuncSize:=0;
      for J:=I + 1 to High(Stabs) do
        if Stabs[J].NType=N_RBrack then
          FuncSize:=Stabs[J].NValue
        else if Stabs[J].NType=N_Function then
          Break;
      if FuncSize=0 then Continue;
      FuncInfo:=TFuncInfo.Create;
      FuncInfo.Name:=FuncName;
      FuncInfo.Address:=Stabs[I].NValue;
      FuncInfo.Size:=FuncSize;
      FuncInfos.Add(FuncInfo);
    end;
  end;
  sbStatus.Panels[1].Text:='Stopped';
  pbProgress.Visible:=False;
end;

procedure TMain.LoadDWARFDebugInfo(AFileName: string);
var
  Scanner: TDWARFScanner;
  Header: TDWARFCompilationUnitHeader;
  Entry: TDWARFEntry;

  // This skips over entries
  procedure ScanEntries;
  var
    Entry: TDWARFEntry;
  begin
    while Scanner.ReadEntry(Entry)=dserEntryRead do begin
      try
        if Entry.HasChildren then ScanEntries;
      finally
        Entry.Free;
      end;
    end;
  end;

  // Scan subprogram (functions, methods, etc)
  procedure ScanSub(SubEntry: TDWARFEntry; Prefix: string);
  var
    LoAddr: Int64 = 0;
    HiAddr: Int64 = 0;
    I: Integer;
    Name: string = '';
    FuncInfo: TFuncInfo;
  begin
    for I:=0 to SubEntry.AttributeCount - 1 do
      case SubEntry.Attributes[I].Name of
        DW_AT_name: Name:=Prefix + TDWARFEntryStringAttribute(SubEntry.Attributes[I]).Value;
        DW_AT_low_pc: LoAddr:=TDWARFEntryAddressAttribute(SubEntry.Attributes[I]).Address;
        DW_AT_high_pc: HiAddr:=TDWARFEntryAddressAttribute(SubEntry.Attributes[I]).Address;
      end;
    if (Name <> '') and (LoAddr <> 0) and (HiAddr <> 0) then begin
      FuncInfo:=TFuncInfo.Create;
      FuncInfo.Name:=Name;
      FuncInfo.Address:=LoAddr;
      FuncInfo.Size:=HiAddr - LoAddr;
      FuncInfos.Add(FuncInfo);
    end;
    if SubEntry.HasChildren then ScanEntries;
  end;

  // Scan a custom data type that may contain subs (classes, structures)
  procedure ScanType(TypeEntry: TDWARFEntry; Prefix: string);
  var
    I: Integer;
    Entry: TDWARFEntry;
  begin
    for I:=0 to TypeEntry.AttributeCount - 1 do
      if TypeEntry.Attributes[I].Name=DW_AT_name then
        Prefix:=Prefix + TDWARFEntryStringAttribute(TypeEntry.Attributes[I]).Value + '.';
    while Scanner.ReadEntry(Entry)=dserEntryRead do begin
      try
        if Entry.Tag=DW_TAG_subprogram then
          ScanSub(Entry, Prefix)
        else if Entry.HasChildren and (Entry.Tag in [DW_TAG_class_type, DW_TAG_structure_type]) then
          ScanType(Entry, Prefix)
        else if Entry.HasChildren then
          ScanEntries;
      finally
        Entry.Free;
      end;
    end;
  end;

  // Scan a compilation unit
  procedure ScanUnit(UnitEntry: TDWARFEntry);
  var
    Entry: TDWARFEntry;
  begin
    while Scanner.ReadEntry(Entry)=dserEntryRead do begin
      try
        if Entry.Tag=DW_TAG_subprogram then
          ScanSub(Entry, '')
        else if Entry.HasChildren and (Entry.Tag in [DW_TAG_class_type, DW_TAG_structure_type]) then
          ScanType(Entry, '')
        else if Entry.HasChildren then
          ScanEntries;
      finally
        Entry.Free;
      end;
    end;
  end;

begin
  FuncInfos.Clear;
  pbProgress.Visible:=True;
  pbProgress.Position:=0;
  pbProgress.Repaint;
  sbStatus.Panels[1].Text:='Reading DWARF...';
  sbStatus.Repaint;
  Application.ProcessMessages;
  Scanner:=TDWARFScanner.Create;
  Enabled:=False;
  try
    Scanner.Open(ClickedProcessPath);
    while Scanner.HasMore do begin
      if Scanner.ReadCompilationUnitHeader(Header) then begin
        while Scanner.ReadEntry(Entry)=dserEntryRead do begin
          pbProgress.Position:=Round(1000*Scanner.Progress);
          Application.ProcessMessages;
          try
            if Entry.Tag=DW_TAG_compile_unit then
              ScanUnit(Entry)
            else if Entry.HasChildren then
              ScanEntries;
          finally
            Entry.Free;
          end;
        end;
      end else begin
        sbStatus.Panels[2].Text:='Some compilation units have unsupported DWARF data (this is normal for MinGW)';
      end;
    end;
  except
    sbStatus.Panels[2].Text:='DWARF parse error';
    ShowMessage('DWARF parse error: ' + Exception(ExceptObject).ToString +
      LineEnding + LineEnding + 'If this program was compiled with an old version of Free Pascal consider using STABS instead');
  end;
  Enabled:=True;
  Scanner.Free;
  pbProgress.Visible:=False;
  sbStatus.Panels[1].Text:='Stopped';
end;

end.

