{ Provids a DWARF debug data scanner }
unit UDWARFScanner;
{$mode objfpc}{$H+}
interface
uses
  Classes, SysUtils, ExeInfo;


const
  { Tags }
  DW_TAG_array_type = $01;
  DW_TAG_class_type = $02;
  DW_TAG_entry_point = $03;
  DW_TAG_enumeration_type = $04;
  DW_TAG_formal_parameter = $05;
  DW_TAG_imported_declaration = $08;
  DW_TAG_label = $0A;
  DW_TAG_lexical_block = $0B;
  DW_TAG_member = $0D;
  DW_TAG_pointer_type = $0F;
  DW_TAG_reference_type = $10;
  DW_TAG_compile_unit = $11;
  DW_TAG_string_type = $12;
  DW_TAG_structure_type = $13;
  DW_TAG_subroutine_type = $15;
  DW_TAG_typedef = $16;
  DW_TAG_union_type = $17;
  DW_TAG_unspecified_parameters = $18;
  DW_TAG_variant = $19;
  DW_TAG_common_block = $1A;
  DW_TAG_common_inclusion = $1B;
  DW_TAG_inheritance = $1C;
  DW_TAG_inlined_subroutine = $1D;
  DW_TAG_module = $1E;
  DW_TAG_ptr_to_member_type = $1F;
  DW_TAG_set_type = $20;
  DW_TAG_subrange_type = $21;
  DW_TAG_with_stmt = $22;
  DW_TAG_access_declaration = $23;
  DW_TAG_base_type = $24;
  DW_TAG_catch_block = $25;
  DW_TAG_const_type = $26;
  DW_TAG_constant = $27;
  DW_TAG_enumerator = $28;
  DW_TAG_file_type = $29;
  DW_TAG_friend = $2A;
  DW_TAG_namelist = $2B;
  DW_TAG_namelist_item = $2C;
  DW_TAG_packed_type = $2D;
  DW_TAG_subprogram = $2E;
  DW_TAG_template_type_param = $2F;
  DW_TAG_template_value_param = $30;
  DW_TAG_thrown_type = $31;
  DW_TAG_try_block = $32;
  DW_TAG_variant_part = $33;
  DW_TAG_variable = $34;
  DW_TAG_volatile_type = $35;
  DW_TAG_lo_user = $4080;
  DW_TAG_hi_user = $FFFF;

  { Child determination name }
  DW_CHILDREN_no = 0;
  DW_CHILDREN_yes = 1;

  { Attributes }
  DW_AT_sibling = $01;
  DW_AT_location = $02;
  DW_AT_name = $03;
  DW_AT_ordering = $09;
  DW_AT_byte_size = $0B;
  DW_AT_bit_offset = $0C;
  DW_AT_bit_size = $0D;
  DW_AT_stmt_list = $10;
  DW_AT_low_pc = $11;
  DW_AT_high_pc = $12;
  DW_AT_language = $13;
  DW_AT_discr = $15;
  DW_AT_discr_value = $16;
  DW_AT_visibility = $17;
  DW_AT_import = $18;
  DW_AT_string_length = $19;
  DW_AT_common_reference = $1A;
  DW_AT_comp_dir = $1B;
  DW_AT_const_value = $1C;
  DW_AT_containing_type = $1D;
  DW_AT_default_value = $1E;
  DW_AT_inline = $20;
  DW_AT_is_optional = $21;
  DW_AT_lower_bound = $22;
  DW_AT_producer = $25;
  DW_AT_prototyped = $27;
  DW_AT_return_addr = $2A;
  DW_AT_start_scope = $2C;
  DW_AT_stride_size = $2E;
  DW_AT_upper_bound = $2F;
  DW_AT_abstract_origin = $31;
  DW_AT_accessibility = $32;
  DW_AT_address_class = $33;
  DW_AT_artificial = $34;
  DW_AT_base_types = $35;
  DW_AT_calling_convention = $36;
  DW_AT_count = $37;
  DW_AT_data_member_location = $38;
  DW_AT_decl_column = $39;
  DW_AT_decl_file = $3A;
  DW_AT_decl_line = $3B;
  DW_AT_declaration = $3C;
  DW_AT_discr_list = $3D;
  DW_AT_encoding = $3E;
  DW_AT_external = $3F;
  DW_AT_frame_base = $40;
  DW_AT_friend = $41;
  DW_AT_identifier_case = $42;
  DW_AT_macro_info = $43;
  DW_AT_namelist_item = $44;
  DW_AT_priority = $45;
  DW_AT_segment = $46;
  DW_AT_specification = $47;
  DW_AT_static_link = $48;
  DW_AT_type = $49;
  DW_AT_use_location = $4A;
  DW_AT_variable_parameter = $4B;
  DW_AT_virtuality = $4C;
  DW_AT_vtable_elem_location = $4D;
  DW_AT_lo_user = $2000;
  DW_AT_hi_user = $3FFF;

  { Attribute forms }
  DW_FORM_addr = $01;
  DW_FORM_block2 = $03;
  DW_FORM_block4 = $04;
  DW_FORM_data2 = $05;
  DW_FORM_data4 = $06;
  DW_FORM_data8 = $07;
  DW_FORM_string = $08;
  DW_FORM_block = $09;
  DW_FORM_block1 = $0A;
  DW_FORM_data1 = $0B;
  DW_FORM_flag = $0C;
  DW_FORM_sdata = $0D;
  DW_FORM_strp = $0E;
  DW_FORM_udata = $0F;
  DW_FORM_ref_addr = $10;
  DW_FORM_ref1 = $11;
  DW_FORM_ref2 = $12;
  DW_FORM_ref4 = $13;
  DW_FORM_ref8 = $14;
  DW_FORM_ref_udata = $15;
  DW_FORM_indirect = $16;

  { Location opcodes }
  DW_OP_addr = $03;
  DW_OP_deref = $06;
  DW_OP_const1u = $08;
  DW_OP_const1s = $09;
  DW_OP_const2u = $0A;
  DW_OP_const2s = $0B;
  DW_OP_const4u = $0C;
  DW_OP_const4s = $0D;
  DW_OP_const8u = $0E;
  DW_OP_const8s = $0F;
  DW_OP_constu = $10;
  DW_OP_consts = $11;
  DW_OP_dup = $12;
  DW_OP_drop = $13;
  DW_OP_over = $14;
  DW_OP_pick = $15;
  DW_OP_swap = $16;
  DW_OP_rot = $17;
  DW_OP_xderef = $18;
  DW_OP_abs = $19;
  DW_OP_and = $1A;
  DW_OP_div = $1B;
  DW_OP_minus = $1C;
  DW_OP_mod = $1D;
  DW_OP_mul = $1E;
  DW_OP_neg = $1F;
  DW_OP_not = $20;
  DW_OP_or = $21;
  DW_OP_plus = $22;
  DW_OP_plus_uconst = $23;
  DW_OP_shl = $24;
  DW_OP_shr = $25;
  DW_OP_shra = $26;
  DW_OP_xor = $27;
  DW_OP_skip = $2F;
  DW_OP_bra = $28;
  DW_OP_eq = $29;
  DW_OP_ge = $2A;
  DW_OP_gt = $2B;
  DW_OP_le = $2C;
  DW_OP_lt = $2D;
  DW_OP_ne = $2E;
  DW_OP_lit0 = $30;  // 31 more literals follow
  DW_OP_reg0 = $50;  // 31 more regs follow
  DW_OP_breg0 = $70; // 31 more bregs follow
  DW_OP_regx = $90;
  DW_OP_fbreg = $91;
  DW_OP_bregx = $92;
  DW_OP_piece = $93;
  DW_OP_deref_size = $94;
  DW_OP_xderef_size = $95;
  DW_OP_nop = $96;
  DW_OP_lo_user = $E0;
  DW_OP_hi_user = $FF;

  { Base types }
  DW_ATE_address = $1;
  DW_ATE_boolean = $2;
  DW_ATE_complex_float = $3;
  DW_ATE_float = $4;
  DW_ATE_signed = $5;
  DW_ATE_signed_char = $6;
  DW_ATE_unsigned = $7;
  DW_ATE_unsigned_char = $8;
  DW_ATE_lo_user = $80;
  DW_ATE_hi_user = $FF;

  { Accessibilities }
  DW_ACCESS_public = 1;
  DW_ACCESS_protected = 2;
  DW_ACCESS_private = 3;

  { Visibilities }
  DW_VIS_local = 1;
  DW_VIS_exported = 2;
  DW_VIS_qualified = 3;

  { Virtualities }
  DW_VIRTUALITY_none = 0;
  DW_VIRTUALITY_virtual = 1;
  DW_VIRTUALITY_pure_virtual = 2;

  { Languages }
  DW_LANG_C89 = $0001;
  DW_LANG_C = $0002;
  DW_LANG_Ada83 = $0003;
  DW_LANG_C_plus_plus = $0004;
  DW_LANG_Cobol74 = $0005;
  DW_LANG_Cobol85 = $0006;
  DW_LANG_Fortran77 = $0007;
  DW_LANG_Fortran90 = $0008;
  DW_LANG_Pascal83 = $0009;
  DW_LANG_Modula2 = $000A;
  DW_LANG_lo_user = $8000;
  DW_LANG_hi_user = $FFFF;

  { Identifier cases }
  DW_ID_case_sensitive = 0;
  DW_ID_up_case = 1;
  DW_ID_down_case = 2;
  DW_ID_case_insensitive = 3;

  { Calling conventions }
  DW_CC_normal = $1;
  DW_CC_program = $2;
  DW_CC_nocall = $3;
  DW_CC_lo_user = $40;
  DW_CC_hi_user = $FF;

  { Inlining }
  DW_INL_not_inlined = 0;
  DW_INL_inlined = 1;
  DW_INL_declared_not_inlined = 2;
  DW_INL_declared_inlined = 3;

  { Array ordering }
  DW_ORD_row_major = 0;
  DW_ORD_col_major = 1;

  { Discriminant descriptors }
  DW_DSC_label = 0;
  DW_DSC_range = 1;

  { Line number standard opcodes }
  DW_LNS_copy = 1;
  DW_LNS_advance_pc = 2;
  DW_LNS_advance_line = 3;
  DW_LNS_set_file = 4;
  DW_LNS_set_column = 5;
  DW_LNS_negate_stmt = 6;
  DW_LNS_set_basic_block = 7;
  DW_LNS_const_add_pc = 8;
  DW_LNS_fixed_advance_pc = 9;

  { Line number extended opcodes }
  DW_LNE_end_sequence = 1;
  DW_LNE_set_address = 2;
  DW_LNE_define_file = 3;

  { Macinfo types }
  DW_MACINFO_define = 1;
  DW_MACINFO_undef = 2;
  DW_MACINFO_start_file = 3;
  DW_MACINFO_end_file = 4;
  DW_MACINFO_vendor_ext = 255;

  { Callframe instructions }
  DW_CFA_nop = $00;
  DW_CFA_set_loc = $01;
  DW_CFA_advance_loc1 = $02;
  DW_CFA_advance_loc2 = $03;
  DW_CFA_advance_loc4 = $04;
  DW_CFA_offset_extended = $05;
  DW_CFA_restore_extended = $06;
  DW_CFA_undefined = $07;
  DW_CFA_same_value = $08;
  DW_CFA_register = $09;
  DW_CFA_remember_state = $0A;
  DW_CFA_restore_state = $0B;
  DW_CFA_def_cfa = $0C;
  DW_CFA_def_cfa_register = $0D;
  DW_CFA_def_cfa_offset = $0E;
  DW_CFA_lo_user = $1C;
  DW_CFA_hi_user = $3F;
  DW_CFA_advance_loc = $40;
  DW_CFA_offset = $80;
  DW_CFA_restore = $C0;

type
  { Exceptions }
  EDWARFScannerError = class(Exception);

  { TDWARFScanEntryResult - Result after scanning a single entry }
  TDWARFScanEntryResult = (
    // An entry was read
    dserEntryRead,
    // Reached the end of an entry chain
    dserEndOfChain,
    // Reached the end of the compilation unit
    dserEndOfUnit
  );

  { TDWARFCompilationUnitHeader - Header for a compilation unit }
  TDWARFCompilationUnitHeader = packed record
    // The compilation unit's size
    Size: UInt32;
    // The DWARF version (2)
    Version: UInt16;
    // Offset into the abbreviation data
    AbbrevOffset: UInt32;
    // The address size (4 or 8)
    AddressSize: UInt8;
  end;

  { TDWARFEntryAttribute - Base class for DWARF entry attributes }
  TDWARFEntryAttribute = class
  private
    // Property storage
    FName: Cardinal;
    FForm: Cardinal;
  public
    // Construct the attribute
    constructor Create(AName, AForm: Cardinal);
  public
    // The attribute's name
    property Name: Cardinal read FName;
    // The attribute's form
    property Form: Cardinal read FForm;
  end;

  { TDWARFEntryAddressAttribute - Contains an address attribute }
  TDWARFEntryAddressAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FAddress: Int64;
  public
    // Construct the attribute
    constructor Create(AName: Cardinal; AAddress: Int64);
  public
    // The address stored in this attribute
    property Address: Int64 read FAddress;
  end;

  { TDWARFEntryArbitraryDataAttribute - Base class for attributes with arbitrary
    data stored as a pair of Data+Size fields }
  TDWARFEntryArbitraryDataAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FData: Pointer;
    FSize: UIntPtr;
  public
    // Construct the attribute
    constructor Create(AName, AForm: Cardinal; AData: Pointer; ASize: UIntPtr);
    // Destroy the attribute
    destructor Destroy; override;
  public
    // The data stored in this attribute
    property Data: Pointer read FData;
    // The size of the stored data
    property Size: UIntPtr read FSize;
  end;

  { TDWARFEntryBlockAttribute - Contains a block attribute of arbitrary data }
  TDWARFEntryBlockAttribute = class(TDWARFEntryArbitraryDataAttribute);

  { TDWARFEntryDataAttribute - Contains constant data as a numeric value }
  TDWARFEntryDataAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FValue: UInt64;
  public
    // Construct the attribute
    constructor Create(AName, AForm: Cardinal; AValue: UInt64);
  public
    // The stored value
    property Value: UInt64 read FValue;
  end;

  { TDWARFEntrySignedDataAttribute - Contains constant data as a signed numeric value }
  TDWARFEntrySignedDataAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FValue: Int64;
  public
    // Construct the attribute
    constructor Create(AName, AForm: Cardinal; AValue: Int64);
  public
    // The stored value
    property Value: Int64 read FValue;
  end;

  { TDWARFEntryStringAttribute - Contains a string }
  TDWARFEntryStringAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FValue: string;
  public
    // Construct the attribute
    constructor Create(AName, AForm: Cardinal; AValue: string);
  public
    // The string value
    property Value: string read FValue;
  end;

  { TDWARFEntryFlagAttribute - Contains a boolean flag }
  TDWARFEntryFlagAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FValue: Boolean;
  public
    // Construct the attribute
    constructor Create(AName: Cardinal; AValue: Boolean);
  public
    // The flag value
    property Value: Boolean read FValue;
  end;

  { TDWARFEntryReferenceAttribute - Contains a reference to another entry,
    this hasn't been implemented properly yet }
  TDWARFEntryReferenceAttribute = class(TDWARFEntryAttribute)
  private
  private
    // The reference offset
    Offset: UInt64;
  public
  end;

  { TDWARFEntryAddressReferenceAttribute - Contains a reference to an address }
  TDWARFEntryAddressReferenceAttribute = class(TDWARFEntryAttribute)
  private
    // Property storage
    FAddress: Int64;
  public
    // Construct the attribute
    constructor Create(AName: Cardinal; AAddress: Int64);
  public
    // The address stored in this attribute
    property Address: Int64 read FAddress;
  end;

  { TDWARFEntry - A single entry in the DWARF debug info }
  TDWARFEntry = class
  private
    // Property storage
    FTag: Cardinal;
    FAttributes: array of TDWARFEntryAttribute;
    FHasChildren: Boolean;
    function GetAttributeCount: Integer;
    function GetAttributes(AIndex: Integer): TDWARFEntryAttribute;
  public
    // Construct the entry
    constructor Create(ATag: Cardinal);
    // Destroy the entry
    destructor Destroy; override;
    // Find the attribute with the given name
    function FindAttribute(AName: Cardinal): TDWARFEntryAttribute;
  public
    // The entry's tag
    property Tag: Cardinal read FTag;
    // The number of attributes in the entry
    property AttributeCount: Integer read GetAttributeCount;
    // The entry attributes
    property Attributes[AIndex: Integer]: TDWARFEntryAttribute read GetAttributes;
    // True if this entry has children
    property HasChildren: Boolean read FHasChildren;
  end;

  { TDWARFScanner - Scans DWARF data }
  TDWARFScanner = class
  private type
    { Pointers }
    PAbbrevEntry = ^TAbbrevEntry;
    { TSectionInfo - Information about a single section }
    TSectionInfo = record
      Position, Size: LongInt;
    end;
    { TAbbrevEntry - Abbreviation table entry }
    TAbbrevEntry = record
      Code: Cardinal;
      Tag: Cardinal;
      Children: Boolean;
      AttrName: array of Cardinal;
      AttrForm: array of Cardinal;
    end;
  private
    // Property storage
    function GetProgress: Single;
  private
    // The executable being scanned
    Exe: TExeFile;
    // The abbreviation table data
    AbbrevData: TMemoryStream;
    // The debug data
    Data: TMemoryStream;
    // The debug string data (optional)
    StrData: TMemoryStream;
    // The current abbreviation table
    AbbrevTable: array of TAbbrevEntry;
    // The current compilation unit header
    CurrentHeader: TDWARFCompilationUnitHeader;
    // The base position for the current compilation unit
    BasePosition: Int64;
  private
    // Find the given executable section
    function FindSection(AName: string; out SectionInfo: TSectionInfo): Boolean;
    // Seek to the given section, returns the section's size
    function SeekSection(AName: string): Cardinal;
    // Load the entire section into a memory stream
    function LoadSection(AName: string): TMemoryStream;
    // Read an unsigned LEB128 integer from the given stream
    function ReadUnsignedLEB128(Stream: TStream): UInt64;
    // Read a signed LEB128 integer from the given stream
    function ReadSignedLEB128(Stream: TStream): Int64;
    // Read the abbreviations table
    procedure ReadAbbrevTable(Offset: Int64);
    // Find the abbreviation table entry for the given code
    function FindAbbrevEntry(ACode: Cardinal): PAbbrevEntry;
  public
    // Create the scanner, if the given filename is not empty it will be opened
    constructor Create(AFileName: string='');
    // Destroy the scanner, will close it if it is open
    destructor Destroy; override;
    // Open the reader for the given executable
    procedure Open(AFileName: string);
    // Close the reader
    procedure Close;
    // Returns true if there is more data to be scanned
    function HasMore: Boolean;
    // Rewind the scan to the first compilation unit
    procedure Rewind;
    // Read a compilation unit header, returns false if the header cannot be
    // handled by this scanner (unsupported version, address size, etc) in
    // which case the scanner jumps right after the data (so another call to
    // ReadCompilationUnitHeader can be made for the next unit)
    function ReadCompilationUnitHeader(out Header: TDWARFCompilationUnitHeader): Boolean;
    // Read an entry, this allocates a new object which must be freed
    function ReadEntry(out Entry: TDWARFEntry): TDWARFScanEntryResult;
  public
    // Is the reader open?
    property IsOpen: Boolean read Exe.IsOpen;
    // The current progress
    property Progress: Single read GetProgress;
  end;

function DWARFTagToString(Tag: Cardinal): string;
function DWARFAttributeToString(Attr: Cardinal): string;
function DWARFFormToString(Form: Cardinal): string;

implementation

{ Functions }
function DWARFTagToString(Tag: Cardinal): string;
begin
  case Tag of
    DW_TAG_array_type: Result:='DW_TAG_array_type';
    DW_TAG_class_type: Result:='DW_TAG_class_type';
    DW_TAG_entry_point: Result:='DW_TAG_entry_point';
    DW_TAG_enumeration_type: Result:='DW_TAG_enumeration_type';
    DW_TAG_formal_parameter: Result:='DW_TAG_formal_parameter';
    DW_TAG_imported_declaration: Result:='DW_TAG_imported_declaration';
    DW_TAG_label: Result:='DW_TAG_label';
    DW_TAG_lexical_block: Result:='DW_TAG_lexical_block';
    DW_TAG_member: Result:='DW_TAG_member';
    DW_TAG_pointer_type: Result:='DW_TAG_pointer_type';
    DW_TAG_reference_type: Result:='DW_TAG_reference_type';
    DW_TAG_compile_unit: Result:='DW_TAG_compile_unit';
    DW_TAG_string_type: Result:='DW_TAG_string_type';
    DW_TAG_structure_type: Result:='DW_TAG_structure_type';
    DW_TAG_subroutine_type: Result:='DW_TAG_subroutine_type';
    DW_TAG_typedef: Result:='DW_TAG_typedef';
    DW_TAG_union_type: Result:='DW_TAG_union_type';
    DW_TAG_unspecified_parameters: Result:='DW_TAG_unspecified_parameters';
    DW_TAG_variant: Result:='DW_TAG_variant';
    DW_TAG_common_block: Result:='DW_TAG_common_block';
    DW_TAG_common_inclusion: Result:='DW_TAG_common_inclusion';
    DW_TAG_inheritance: Result:='DW_TAG_inheritance';
    DW_TAG_inlined_subroutine: Result:='DW_TAG_inlined_subroutine';
    DW_TAG_module: Result:='DW_TAG_module';
    DW_TAG_ptr_to_member_type: Result:='DW_TAG_ptr_to_member_type';
    DW_TAG_set_type: Result:='DW_TAG_set_type';
    DW_TAG_subrange_type: Result:='DW_TAG_subrange_type';
    DW_TAG_with_stmt: Result:='DW_TAG_with_stmt';
    DW_TAG_access_declaration: Result:='DW_TAG_access_declaration';
    DW_TAG_base_type: Result:='DW_TAG_base_type';
    DW_TAG_catch_block: Result:='DW_TAG_catch_block';
    DW_TAG_const_type: Result:='DW_TAG_const_type';
    DW_TAG_constant: Result:='DW_TAG_constant';
    DW_TAG_enumerator: Result:='DW_TAG_enumerator';
    DW_TAG_file_type: Result:='DW_TAG_file_type';
    DW_TAG_friend: Result:='DW_TAG_friend';
    DW_TAG_namelist: Result:='DW_TAG_namelist';
    DW_TAG_namelist_item: Result:='DW_TAG_namelist_item';
    DW_TAG_packed_type: Result:='DW_TAG_packed_type';
    DW_TAG_subprogram: Result:='DW_TAG_subprogram';
    DW_TAG_template_type_param: Result:='DW_TAG_template_type_param';
    DW_TAG_template_value_param: Result:='DW_TAG_template_value_param';
    DW_TAG_thrown_type: Result:='DW_TAG_thrown_type';
    DW_TAG_try_block: Result:='DW_TAG_try_block';
    DW_TAG_variant_part: Result:='DW_TAG_variant_part';
    DW_TAG_variable: Result:='DW_TAG_variable';
    DW_TAG_volatile_type: Result:='DW_TAG_volatile_type';
    DW_TAG_lo_user: Result:='DW_TAG_lo_user';
    DW_TAG_hi_user: Result:='DW_TAG_hi_user';
    else Result:='(' + HexStr(Tag, 8) + ')';
  end;
end;

function DWARFAttributeToString(Attr: Cardinal): string;
begin
  case Attr of
    DW_AT_sibling: Result:='DW_AT_sibling';
    DW_AT_location: Result:='DW_AT_location';
    DW_AT_name: Result:='DW_AT_name';
    DW_AT_ordering: Result:='DW_AT_ordering';
    DW_AT_byte_size: Result:='DW_AT_byte_size';
    DW_AT_bit_offset: Result:='DW_AT_bit_offset';
    DW_AT_bit_size: Result:='DW_AT_bit_size';
    DW_AT_stmt_list: Result:='DW_AT_stmt_list';
    DW_AT_low_pc: Result:='DW_AT_low_pc';
    DW_AT_high_pc: Result:='DW_AT_high_pc';
    DW_AT_language: Result:='DW_AT_language';
    DW_AT_discr: Result:='DW_AT_discr';
    DW_AT_discr_value: Result:='DW_AT_discr_value';
    DW_AT_visibility: Result:='DW_AT_visibility';
    DW_AT_import: Result:='DW_AT_import';
    DW_AT_string_length: Result:='DW_AT_string_length';
    DW_AT_common_reference: Result:='DW_AT_common_reference';
    DW_AT_comp_dir: Result:='DW_AT_comp_dir';
    DW_AT_const_value: Result:='DW_AT_const_value';
    DW_AT_containing_type: Result:='DW_AT_containing_type';
    DW_AT_default_value: Result:='DW_AT_default_value';
    DW_AT_inline: Result:='DW_AT_inline';
    DW_AT_is_optional: Result:='DW_AT_is_optional';
    DW_AT_lower_bound: Result:='DW_AT_lower_bound';
    DW_AT_producer: Result:='DW_AT_producer';
    DW_AT_prototyped: Result:='DW_AT_prototyped';
    DW_AT_return_addr: Result:='DW_AT_return_addr';
    DW_AT_start_scope: Result:='DW_AT_start_scope';
    DW_AT_stride_size: Result:='DW_AT_stride_size';
    DW_AT_upper_bound: Result:='DW_AT_upper_bound';
    DW_AT_abstract_origin: Result:='DW_AT_abstract_origin';
    DW_AT_accessibility: Result:='DW_AT_accessibility';
    DW_AT_address_class: Result:='DW_AT_address_class';
    DW_AT_artificial: Result:='DW_AT_artificial';
    DW_AT_base_types: Result:='DW_AT_base_types';
    DW_AT_calling_convention: Result:='DW_AT_calling_convention';
    DW_AT_count: Result:='DW_AT_count';
    DW_AT_data_member_location: Result:='DW_AT_data_member_location';
    DW_AT_decl_column: Result:='DW_AT_decl_column';
    DW_AT_decl_file: Result:='DW_AT_decl_file';
    DW_AT_decl_line: Result:='DW_AT_decl_line';
    DW_AT_declaration: Result:='DW_AT_declaration';
    DW_AT_discr_list: Result:='DW_AT_discr_list';
    DW_AT_encoding: Result:='DW_AT_encoding';
    DW_AT_external: Result:='DW_AT_external';
    DW_AT_frame_base: Result:='DW_AT_frame_base';
    DW_AT_friend: Result:='DW_AT_friend';
    DW_AT_identifier_case: Result:='DW_AT_identifier_case';
    DW_AT_macro_info: Result:='DW_AT_macro_info';
    DW_AT_namelist_item: Result:='DW_AT_namelist_item';
    DW_AT_priority: Result:='DW_AT_priority';
    DW_AT_segment: Result:='DW_AT_segment';
    DW_AT_specification: Result:='DW_AT_specification';
    DW_AT_static_link: Result:='DW_AT_static_link';
    DW_AT_type: Result:='DW_AT_type';
    DW_AT_use_location: Result:='DW_AT_use_location';
    DW_AT_variable_parameter: Result:='DW_AT_variable_parameter';
    DW_AT_virtuality: Result:='DW_AT_virtuality';
    DW_AT_vtable_elem_location: Result:='DW_AT_vtable_elem_location';
    DW_AT_lo_user: Result:='DW_AT_lo_user';
    DW_AT_hi_user: Result:='DW_AT_hi_user';
    else Result:='(' + HexStr(Attr, 8) + ')';
  end;
end;

function DWARFFormToString(Form: Cardinal): string;
begin
  case Form of
    DW_FORM_addr: Result:='DW_FORM_addr';
    DW_FORM_block2: Result:='DW_FORM_block2';
    DW_FORM_block4: Result:='DW_FORM_block4';
    DW_FORM_data2: Result:='DW_FORM_data2';
    DW_FORM_data4: Result:='DW_FORM_data4';
    DW_FORM_data8: Result:='DW_FORM_data8';
    DW_FORM_string: Result:='DW_FORM_string';
    DW_FORM_block: Result:='DW_FORM_block';
    DW_FORM_block1: Result:='DW_FORM_block1';
    DW_FORM_data1: Result:='DW_FORM_data1';
    DW_FORM_flag: Result:='DW_FORM_flag';
    DW_FORM_sdata: Result:='DW_FORM_sdata';
    DW_FORM_strp: Result:='DW_FORM_strp';
    DW_FORM_udata: Result:='DW_FORM_udata';
    DW_FORM_ref_addr: Result:='DW_FORM_ref_addr';
    DW_FORM_ref1: Result:='DW_FORM_ref1';
    DW_FORM_ref2: Result:='DW_FORM_ref2';
    DW_FORM_ref4: Result:='DW_FORM_ref4';
    DW_FORM_ref8: Result:='DW_FORM_ref8';
    DW_FORM_ref_udata: Result:='DW_FORM_ref_udata';
    DW_FORM_indirect: Result:='DW_FORM_indirect';
    else Result:='(' + HexStr(Form, 8) + ')';
  end;
end;

{ TDWARFEntrySignedDataAttribute }
constructor TDWARFEntrySignedDataAttribute.Create(AName, AForm: Cardinal; AValue: Int64);
begin
  inherited Create(AName, AForm);
  FValue:=AValue;
end;

{ TDWARFEntryAttribute }
constructor TDWARFEntryAttribute.Create(AName, AForm: Cardinal);
begin
  inherited Create;
  FName:=AName;
  FForm:=AForm;
end;

{ TDWARFEntryAddressAttribute }
constructor TDWARFEntryAddressAttribute.Create(AName: Cardinal; AAddress: Int64);
begin
  inherited Create(AName, DW_FORM_addr);
  FAddress:=AAddress;
end;

{ TDWARFEntryArbitraryDataAttribute }
constructor TDWARFEntryArbitraryDataAttribute.Create(AName, AForm: Cardinal; AData: Pointer; ASize: UIntPtr);
begin
  inherited Create(AName, AForm);
  FData:=AData;
  FSize:=ASize;
end;

destructor TDWARFEntryArbitraryDataAttribute.Destroy;
begin
  FreeMem(FData, FSize);
  inherited Destroy;
end;

{ TDWARFEntryDataAttribute }
constructor TDWARFEntryDataAttribute.Create(AName, AForm: Cardinal; AValue: UInt64);
begin
  inherited Create(AName, AForm);
  FValue:=AValue;
end;

{ TDWARFEntryStringAttribute }
constructor TDWARFEntryStringAttribute.Create(AName, AForm: Cardinal; AValue: string);
begin
  inherited Create(AName, AForm);
  FValue:=AValue;
end;

{ TDWARFEntryFlagAttribute }
constructor TDWARFEntryFlagAttribute.Create(AName: Cardinal; AValue: Boolean);
begin
  inherited Create(AName, DW_FORM_flag);
  FValue:=AValue;
end;

{ TDWARFEntryAddressReferenceAttribute }
constructor TDWARFEntryAddressReferenceAttribute.Create(AName: Cardinal; AAddress: Int64);
begin
  inherited Create(AName, DW_FORM_ref_addr);
  FAddress:=AAddress;
end;

{ TDWARFEntry }
constructor TDWARFEntry.Create(ATag: Cardinal);
begin
  inherited Create;
  FTag:=ATag;
end;

destructor TDWARFEntry.Destroy;
var
  I: Integer;
begin
  for I:=0 to AttributeCount - 1 do Attributes[I].Free;
  FAttributes:=nil;
  inherited Destroy;
end;

function TDWARFEntry.FindAttribute(AName: Cardinal): TDWARFEntryAttribute;
var
  I: Integer;
begin
  for I:=0 to AttributeCount - 1 do
    if Attributes[I].Name=AName then Exit(Attributes[I]);
  Result:=nil;
end;

function TDWARFEntry.GetAttributeCount: Integer;
begin
  Result:=Length(FAttributes);
end;

function TDWARFEntry.GetAttributes(AIndex: Integer): TDWARFEntryAttribute;
begin
  Result:=FAttributes[AIndex];
end;

{ TDWARFScanner }
constructor TDWARFScanner.Create(AFileName: string);
begin
  inherited Create;
  if AFileName <> '' then Open(AFileName);
end;

destructor TDWARFScanner.Destroy;
begin
  if IsOpen then Close;
  inherited Destroy;
end;

function TDWARFScanner.GetProgress: Single;
begin
  if Assigned(Data) and (Data.Size > 1) then Result:=Data.Position/(Data.Size - 1) else Result:=0;
end;

function TDWARFScanner.FindSection(AName: string; out SectionInfo: TSectionInfo): Boolean;
begin
  Assert(IsOpen, 'Called FindSection with a closed reader');
  SectionInfo.Position:=0;
  SectionInfo.Size:=0;
  Result:=FindExeSection(Exe, AName, SectionInfo.Position, SectionInfo.Size);
end;

function TDWARFScanner.SeekSection(AName: string): Cardinal;
var
  SectionInfo: TSectionInfo;
begin
  if not FindSection(AName, SectionInfo) then raise EDWARFScannerError.Create('Failed to find section ' + AName);
  IOResult;
  {$I-}Seek(Exe.F, SectionInfo.Position);{$I+}
  if IOResult <> 0 then raise EDWARFScannerError.Create('Failed to seek to ' + IntToStr(SectionInfo.Position));
  Result:=SectionInfo.Size;
end;

function TDWARFScanner.LoadSection(AName: string): TMemoryStream;
var
  Size: Cardinal;
  SaveExePos: Int64;
begin
  SaveExePos:=FilePos(Exe.F);
  Size:=SeekSection(AName);
  Result:=TMemoryStream.Create;
  Result.Size:=Size;
  IOResult;
  {$I-}BlockRead(Exe.F, Result.Memory^, Size);{$I+}
  if IOResult <> 0 then begin
    Result.Free;
    raise EDWARFScannerError.Create('Failed to load ' + AName);
  end;
  Seek(Exe.F, SaveExePos);
end;

function TDWARFScanner.ReadUnsignedLEB128(Stream: TStream): UInt64;
var
  B: Byte;
  Shift: UInt64 = 0;
begin
  Result:=0;
  while True do begin
    B:=Stream.ReadByte;
    Result:=Result or (UInt64(B and $7F) shl Shift);
    if (B and $80)=0 then Break;
    Inc(Shift, 7);
  end;
end;

function TDWARFScanner.ReadSignedLEB128(Stream: TStream): Int64;
var
  B: Byte;
  Shift: Int64 = 0;
begin
  Result:=0;
  while True do begin
    B:=Stream.ReadByte;
    Result:=Result or (Int64(B and $7F) shl Shift);
    if (B and $80)=0 then Break;
    Inc(Shift, 7);
  end;
  Result:=(not ((Result and (Int64(1) shl (Shift - 1))) - 1)) or Result;
end;

procedure TDWARFScanner.ReadAbbrevTable(Offset: Int64);
var
  CodeValue, AttrNameValue, AttrFormValue: Cardinal;
  B: Byte = 0;
  AbbrevEntryCount: Integer;
begin
  // Clear any existing entries
  AbbrevTable:=nil;
  // Count the entries in the table
  AbbrevData.Position:=Offset;
  AbbrevEntryCount:=0;
  while AbbrevData.Position < AbbrevData.Size do begin
    CodeValue:=ReadUnsignedLEB128(AbbrevData);
    if CodeValue=0 then Break; // Code
    ReadUnsignedLEB128(AbbrevData); // Tag
    AbbrevData.Position:=AbbrevData.Position + 1;
    while AbbrevData.Position < AbbrevData.Size do begin
      AttrNameValue:=ReadUnsignedLEB128(AbbrevData); // Attribute
      AttrFormValue:=ReadUnsignedLEB128(AbbrevData); // Form
      if (AttrNameValue=0) and (AttrFormValue=0) then Break;
    end;
    Inc(AbbrevEntryCount);
  end;
  // Allocate entries and re-read the table
  SetLength(AbbrevTable, AbbrevEntryCount);
  AbbrevData.Position:=Offset;
  AbbrevEntryCount:=0;
  while AbbrevData.Position < AbbrevData.Size do begin
    CodeValue:=ReadUnsignedLEB128(AbbrevData);
    if CodeValue=0 then Break;
    with AbbrevTable[AbbrevEntryCount] do begin
      Code:=CodeValue;
      Tag:=ReadUnsignedLEB128(AbbrevData);
      B:=AbbrevData.ReadByte;
      Children:=B <> DW_CHILDREN_no;
      AttrName:=nil;
      AttrForm:=nil;
      while AbbrevData.Position < AbbrevData.Size do begin
        AttrNameValue:=ReadUnsignedLEB128(AbbrevData);
        AttrFormValue:=ReadUnsignedLEB128(AbbrevData);
        if (AttrNameValue=0) and (AttrFormValue=0) then Break;
        SetLength(AttrName, Length(AttrName) + 1);
        SetLength(AttrForm, Length(AttrName));
        AttrName[High(AttrName)]:=AttrNameValue;
        AttrForm[High(AttrForm)]:=AttrFormValue;
      end;
    end;
    Inc(AbbrevEntryCount);
  end;
end;

function TDWARFScanner.FindAbbrevEntry(ACode: Cardinal): PAbbrevEntry;
var
  I: Integer;
begin
  // Often codes are stored sequentially so check that case to avoid looping
  if (ACode >= 1) and (ACode <= Cardinal(Length(AbbrevTable))) and (AbbrevTable[ACode - 1].Code=ACode) then
    Exit(@AbbrevTable[ACode - 1]);
  // Check normally
  for I:=0 to High(AbbrevTable) do
    if AbbrevTable[I].Code=ACode then Exit(@AbbrevTable[I]);
  // Not found
  Result:=nil;
end;

procedure TDWARFScanner.Open(AFileName: string);
var
  DebugFn: ShortString;
  SectionInfo: TSectionInfo;
begin
  // Quick error check
  if IsOpen then raise EDWARFScannerError.Create('The reader is already open');
  if AFileName='' then raise EDWARFScannerError.Create('An empty filename was given');
  // Check if the file exists
  AFileName:=ExpandFileName(AFileName);
  if not FileExists(AFileName) then raise EDWARFScannerError.Create('File not found ' + AFileName);
  // Open the executable
  if not OpenExeFile(Exe, AFileName) then raise EDWARFScannerError.Create('Failed to open executable file ' + AFileName);
  // Check for debug link
  DebugFn:='';
  if ReadDebugLink(Exe, DebugFn) then begin
    // Close the executable and switch to the debug link file
    CloseExeFile(Exe);
    if not OpenExeFile(Exe, DebugFn) then begin
      FillChar(Exe, SizeOf(Exe), 0);
      raise EDWARFScannerError.Create('Failed to open debug linked file ' + DebugFn);
    end;
  end;
  // Check for debug information
  if not (FindSection('.debug_info', SectionInfo) and FindSection('.debug_abbrev', SectionInfo)) then begin
    CloseExeFile(Exe);
    FillChar(Exe, SizeOf(Exe), 0);
    raise EDWARFScannerError.Create('Debug information not found in ' + AFileName);
  end;
  // Read the abbreviation data
  try
    AbbrevData:=LoadSection('.debug_abbrev');
  except
    CloseExeFile(Exe);
    FillChar(Exe, SizeOf(Exe), 0);
    raise;
  end;
  // Read the debug data
  try
    Data:=LoadSection('.debug_info');
  except
    FreeAndNil(AbbrevData);
    CloseExeFile(Exe);
    FillChar(Exe, SizeOf(Exe), 0);
    raise;
  end;
  // Read the debug string data, if any
  if FindSection('.debug_str', SectionInfo) then begin
    try
      StrData:=LoadSection('.debug_str');
    except
      FreeAndNil(Data);
      FreeAndNil(AbbrevData);
      CloseExeFile(Exe);
      FillChar(Exe, SizeOf(Exe), 0);
      raise;
    end;
  end else StrData:=nil;
end;

procedure TDWARFScanner.Close;
begin
  if not IsOpen then EDWARFScannerError.Create('The reader is not open');
  // Release section data
  FreeAndNil(StrData);
  FreeAndNil(Data);
  FreeAndNil(AbbrevData);
  // Close executable
  CloseExeFile(Exe);
  FillChar(Exe, SizeOf(Exe), 0);
end;

function TDWARFScanner.HasMore: Boolean;
begin
  if IsOpen then
    Result:=Data.Position < Data.Size
  else
    Result:=False;
end;

procedure TDWARFScanner.Rewind;
begin
  if not IsOpen then raise EDWARFScannerError.Create('The scanner is not open');
  Data.Position:=0;
  FillChar(CurrentHeader, SizeOf(CurrentHeader), 0);
end;

function TDWARFScanner.ReadCompilationUnitHeader(out Header: TDWARFCompilationUnitHeader): Boolean;
begin
  if not IsOpen then raise EDWARFScannerError.Create('The scanner is not open');
  // Store the position for later use
  BasePosition:=Data.Position;
  // Read the header
  FillChar((@Header)^, SizeOf(Header), 0);
  Data.Read(Header, SizeOf(Header));
  if (Header.Version <> 2) or not (Header.AddressSize in [4, 8]) then begin
    // Skip the header
    Data.Position:=BasePosition + Header.Size + 4;
    BasePosition:=Data.Position;
    FillChar((@CurrentHeader)^, SizeOf(CurrentHeader), 0);
    FillChar((@Header)^, SizeOf(Header), 0);
    Exit(False);
  end;
  CurrentHeader:=Header;
  // Read the abbrevations table
  ReadAbbrevTable(Header.AbbrevOffset);
  Result:=True;
end;

function TDWARFScanner.ReadEntry(out Entry: TDWARFEntry): TDWARFScanEntryResult;
var
  AbbrevEntryCode: UInt64;
  AbbrevEntry: PAbbrevEntry;

  // Read the data for the current entry
  procedure ReadEntryData;
  var
    I: Integer;
    Attr: TDWARFEntryAttribute;
    AttrForm: Cardinal;
    B: Byte = 0;
    U8: UInt64 = 0;
    S: String;
    S8: Int64;

    procedure ReadBlockAttr(Size: UInt64; Name, Form: Cardinal);
    var
      BlockData: PChar;
      ToRead, ActuallyRead: UInt64;
    begin
      BlockData:=GetMem(Size);
      ActuallyRead:=0;
      while ActuallyRead < Size do begin
        if Size - ActuallyRead > $FFFFFFFF then
          ToRead:=$FFFFFFFF
        else
          ToRead:=Size - ActuallyRead;
        ActuallyRead += Data.Read(BlockData[ActuallyRead], ToRead);
      end;
      Attr:=TDWARFEntryBlockAttribute.Create(Name, Form, BlockData, Size);
    end;

  begin
    // Allocate entry attributes
    SetLength(Entry.FAttributes, Length(AbbrevEntry^.AttrName));
    for I:=0 to High(AbbrevEntry^.AttrName) do Entry.FAttributes[I]:=nil;
    // Load data
    for I:=0 to High(AbbrevEntry^.AttrName) do begin
      // Resolve attribute form
      AttrForm:=AbbrevEntry^.AttrForm[I];
      if AttrForm=DW_FORM_indirect then AttrForm:=ReadUnsignedLEB128(Data);
      // Create appropriate attribute object
      Attr:=nil;
      IOResult;
      {$I-}
      case AttrForm of
        DW_FORM_addr: begin
          if CurrentHeader.AddressSize=4 then
            Attr:=TDWARFEntryAddressAttribute.Create(AbbrevEntry^.AttrName[I], Data.ReadDWord)
          else if CurrentHeader.AddressSize=8 then
            Attr:=TDWARFEntryAddressAttribute.Create(AbbrevEntry^.AttrName[I], Data.ReadQWord);
        end;
        DW_FORM_block2: ReadBlockAttr(Data.ReadWord, AbbrevEntry^.AttrName[I], AttrForm);
        DW_FORM_block4: ReadBlockAttr(Data.ReadDWord, AbbrevEntry^.AttrName[I], AttrForm);
        DW_FORM_data2: Attr:=TDWARFEntryDataAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, Data.ReadWord);
        DW_FORM_data4: Attr:=TDWARFEntryDataAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, Data.ReadDWord);
        DW_FORM_data8: Attr:=TDWARFEntryDataAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, Data.ReadQWord);
        DW_FORM_string: begin
          S:='';
          while Data.Position < Data.Size do begin
            B:=Data.ReadByte;
            if B=0 then Break;
            S += Chr(B);
          end;
          Attr:=TDWARFEntryStringAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, S);
        end;
        DW_FORM_block: begin
          U8:=ReadUnsignedLEB128(Data);
          ReadBlockAttr(U8, AbbrevEntry^.AttrName[I], AttrForm);
        end;
        DW_FORM_block1: ReadBlockAttr(Data.ReadByte, AbbrevEntry^.AttrName[I], AttrForm);
        DW_FORM_data1: Attr:=TDWARFEntryDataAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, Data.ReadByte);
        DW_FORM_flag: Attr:=TDWARFEntryFlagAttribute.Create(AbbrevEntry^.AttrName[I], Data.ReadByte <> 0);
        DW_FORM_sdata: begin
          S8:=ReadSignedLEB128(Data);
          Attr:=TDWARFEntrySignedDataAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, S8);
        end;
        DW_FORM_strp: begin
          if Assigned(StrData) then begin
            StrData.Position:=Data.ReadDWord;
            S:='';
            while StrData.Position < StrData.Size do begin
              B:=StrData.ReadByte;
              if B=0 then Break;
              S += Chr(B);
            end;
            Attr:=TDWARFEntryStringAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, S);
          end else raise EDWARFScannerError.Create('DW_FORM_strp form found but .debug_str section is missing');
        end;
        DW_FORM_udata: begin
          U8:=ReadUnsignedLEB128(Data);
          Attr:=TDWARFEntryDataAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm, U8);
        end;
        DW_FORM_ref_addr: begin
          if CurrentHeader.AddressSize=4 then
            Attr:=TDWARFEntryAddressReferenceAttribute.Create(AbbrevEntry^.AttrName[I], Data.ReadDWord)
          else if CurrentHeader.AddressSize=8 then
            Attr:=TDWARFEntryAddressReferenceAttribute.Create(AbbrevEntry^.AttrName[I], Data.ReadQWord);
        end;
        DW_FORM_ref1: begin
          Attr:=TDWARFEntryReferenceAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm);
          TDWARFEntryReferenceAttribute(Attr).Offset:=Data.ReadByte;
        end;
        DW_FORM_ref2: begin
          Attr:=TDWARFEntryReferenceAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm);
          TDWARFEntryReferenceAttribute(Attr).Offset:=Data.ReadWord;
        end;
        DW_FORM_ref4: begin
          Attr:=TDWARFEntryReferenceAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm);
          TDWARFEntryReferenceAttribute(Attr).Offset:=Data.ReadDWord;
        end;
        DW_FORM_ref8: begin
          Attr:=TDWARFEntryReferenceAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm);
          TDWARFEntryReferenceAttribute(Attr).Offset:=Data.ReadQWord;
        end;
        DW_FORM_ref_udata: begin
          U8:=ReadUnsignedLEB128(Data);
          Attr:=TDWARFEntryReferenceAttribute.Create(AbbrevEntry^.AttrName[I], AttrForm);
          TDWARFEntryReferenceAttribute(Attr).Offset:=U8;
        end;
      end;
      // Check if the attribute form wasn't handled
      if not Assigned(Attr) then
        raise EDWARFScannerError.Create('Unsupported form ' + HexStr(AttrForm, 8) +
          ' for attribute #' + IntToStr(I) + ' (' + HexStr(AbbrevEntry^.AttrName[I], 10) +
          ') in abbreviation ' + HexStr(AbbrevEntry^.Code, 10));
      // Store the attribute
      Entry.FAttributes[I]:=Attr;
    end;
  end;

begin
  // Reached the end of the unit?
  if Data.Position >= BasePosition + CurrentHeader.Size + 4 then begin
    Entry:=nil;
    Exit(dserEndOfUnit);
  end;
  // Read the code for the abbreviation entry that describes the forms for
  // the entry that follows
  AbbrevEntryCode:=ReadUnsignedLEB128(Data);
  if AbbrevEntryCode=0 then begin
    Entry:=nil;
    Exit(dserEndOfChain);
  end;
  // Find the abbreviation entry
  AbbrevEntry:=FindAbbrevEntry(AbbrevEntryCode);
  if not Assigned(AbbrevEntry) then
    raise EDWARFScannerError.Create('Parsing error, cannot find abbreviation entry for code ' + IntToStr(AbbrevEntryCode));
  // Create a new entry and store it for resolving references
  Entry:=TDWARFEntry.Create(AbbrevEntry^.Tag);
  Entry.FHasChildren:=AbbrevEntry^.Children;
  // Read the entry data
  try
    ReadEntryData;
  except
    FreeAndNil(Entry);
    raise;
  end;
  Result:=dserEntryRead;
end;

end.

