unit UFileList;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FGL;

type

  { TImageFileEntry }

  TImageFileEntry = class
  public
    FileName: string;
    FileSize: Integer;
    LocalPath: string;
    FirstCluster: Integer;
    Data: array of Byte;
  end;

  TImageFileEntryList = specialize TFPGObjectList<TImageFileEntry>;

  { TImageFileList }

  TImageFileList = class(TComponent)
  private
    // Property storage
    FFiles: TImageFileEntryList;
    function GetFile(AIndex: Integer): TImageFileEntry; inline;
    function GetFileCount: Integer; inline;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CalcDOSName(Original: string): string;
    procedure CreateDiskImage(AFileName: string);
    procedure AddFile(AFileName: string);
    procedure RemoveFile(AIndex: Integer);
    function IndexOfFileName(AFileName: string): Integer;
    function IndexOf(AEntry: TImageFileEntry): Integer;
    function TotalDataSize: Integer;
    procedure SaveToFile(AFileName: string);
    procedure LoadFromFile(AFileName: string);
    property Files[AIndex: Integer]: TImageFileEntry read GetFile;
    property FileCount: Integer read GetFileCount;
  end;

implementation

uses
  LazFileUtils;

{ TImageFileList }

function TImageFileList.GetFile(AIndex: Integer): TImageFileEntry;
begin
  Result:=FFiles[AIndex];
end;

function TImageFileList.GetFileCount: Integer;
begin
  Result:=FFiles.Count;
end;

function TImageFileList.CalcDOSName(Original: string): string;
var
  I: Integer;
  Ext: string;

  function Tildify: string;
  var
    J, K: Integer;
  begin
    K:=1;
    Result:='';
    for J:=Length(IntToStr(I)) + 2 to Length(OriginaL) do begin
      if K > Length(Original) then Break;
      if Length(Result + '~' + IntToStr(I))=8 then Break;
      if not (Original[K] in [' ', '.']) then
        Result += Original[K];
      Inc(K);
    end;
    Result += '~' +IntToStr(I);
  end;

begin
  Original:=UpperCase(Original);
  Result:=Original;
  if (Pos(' ', Result) <> 0) or (Length(ExtractFileNameWithoutExt(Result)) > 8) or
     (Pos('.', ExtractFileNameWithoutExt(Result)) <> 0) or
     (Length(ExtractFileExt(Result)) > 4) then begin
    Ext:=Copy(ExtractFileExt(Result), 1, 4);
    for I:=1 to 999999 do begin
      Result:=Tildify;
      if IndexOfFileName(Result + Ext)=-1 then Exit(Result + Ext);
    end;
  end;
end;

constructor TImageFileList.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFiles:=TImageFileEntryList.Create(True);
end;

destructor TImageFileList.Destroy;
begin
  FreeAndNil(FFiles);
  inherited Destroy;
end;

procedure TImageFileList.CreateDiskImage(AFileName: string);
type
  TDirEntry = packed record
    FileName: array [0..7] of Char;
    FileExt: array [0..2] of Char;
    Attr: Byte;
    Reserved: array [12..21] of Char;
    Time: Word;
    Date: Word;
    Cluster: Word;
    Size: DWord;
  end;

var
  BootSector: array [0..511] of Byte;
  FAT: array [0..4607] of Byte;
  RawData: array [0..1457663] of Byte;
  I, NextCluster, Clusters: Integer;
  S: string;
  F: File;
  E: TDirEntry;

  procedure SetFATEntry(Index, Value: Integer);
  begin
    if (Index and 1)=1 then begin
      FAT[(3*Index) div 2]:=(FAT[(3*Index) div 2] and $0F) or ((Value and $0F) shl 4);
      FAT[(3*Index) div 2 + 1]:=Value shr 4;
    end else begin
      FAT[(3*Index) div 2]:=Value and $FF;
      FAT[(3*Index) div 2 + 1]:=(FAT[(3*Index) div 2 + 1] and $F0) or (Value shr 8);
    end;
  end;

begin
  FillChar({%H-}BootSector[0], 512, 0);
  FillChar({%H-}RawData[0], SizeOf(RawData), 0);
  FillChar({%H-}FAT[0], SizeOf(FAT), 0);
  BootSector[0]:=$EB;
  BootSector[1]:=$3F;
  BootSector[2]:=$90;
  Move('MSWIN4.1', BootSector[3], 8);
  PWord(@BootSector[11])^:=512;
  BootSector[13]:=1;
  PWord(@BootSector[14])^:=1;
  BootSector[16]:=2;
  PWord(@BootSector[17])^:=224;
  PWord(@BootSector[19])^:=2880;
  BootSector[21]:=$F0;
  PWord(@BootSector[22])^:=9;
  PWord(@BootSector[24])^:=18;
  PWord(@BootSector[26])^:=2;
  Move('FAT12   ', BootSector[54], 8);
  BootSector[510]:=$55;
  BootSector[511]:=$AA;
  SetFATEntry(0, $FF0);
  SetFATEntry(1, $FFF);
  NextCluster:=2;
  for I:=0 to FileCount - 1 do with Files[I] do begin
    if FileSize=0 then Continue;
    Clusters:=(FileSize div 512) * 512;
    if Clusters < FileSize then Clusters += 512;
    Clusters:=Clusters div 512;
    FirstCluster:=NextCluster;
    while Clusters > 0 do begin
      if Clusters=1 then SetFATEntry(NextCluster, $FF8) else SetFATEntry(NextCluster, NextCluster + 1);
      Move(Data[(NextCluster - FirstCluster)*512], RawData[(NextCluster - 2)*512], 512);
      Inc(NextCluster);
      Dec(Clusters);
    end;
  end;
  AssignFile(F, AFileName);
  {$I-}
  ReWrite(F, 1);
  {$I+}
  if IOResult <> 0 then raise Exception.Create('Failed to create file');
  BlockWrite(F, BootSector, 512);
  BlockWrite(F, FAT, SizeOf(FAT));
  BlockWrite(F, FAT, SizeOf(FAT));
  for I:=0 to FileCount - 1 do begin
    FillChar(E, SizeOf(E), 0);
    FillChar(E.FileName, 8, ' ');
    S:=ExtractFileNameWithoutExt(Files[I].FileName);
    Move(S[1], E.FileName, Length(S));
    S:=ExtractFileExt(Files[I].FileName);
    S:=Copy(S, 2, Length(S));
    if S <> '' then Move(S[1], E.FileExt, Length(S));
    E.Cluster:=Files[I].FirstCluster;
    E.Size:=Files[I].FileSize;
    BlockWrite(F, E, SizeOf(E));
  end;
  FillChar(E, SizeOf(E), 0);
  for I:=FileCount to 223 do BlockWrite(F, E, SizeOf(E));
  BlockWrite(F, RawData, SizeOf(RawData));
  CloseFile(F);
end;

procedure TImageFileList.AddFile(AFileName: string);
var
  Entry: TImageFileEntry;
  F: File;
  FSize: Int64;
  Data: array of Byte;
  DOSFileName: String;
begin
  AssignFile(F, AFileName);
  {$I-}
  Reset(F, 1);
  {$I+}
  if IOResult <> 0 then raise Exception.Create('Failed to open file');
  FSize:=FileSize(F);
  if FSize + Int64(TotalDataSize) > 1457664 then begin
    CloseFile(F);
    raise Exception.Create('File too big');
  end;
  SetLength(Data, FSize + 512);
  if FSize > 0 then BlockRead(F, Data[0], FSize);
  CloseFile(F);
  AFileName:=ExpandFileName(AFileName);
  DOSFileName:=CalcDOSName(ExtractFileName(AFileName));
  Entry:=TImageFileEntry.Create;
  Entry.FileName:=DOSFileName;
  Entry.FileSize:=FSize;
  Entry.LocalPath:=AFileName;
  Entry.Data:=Data;
  FFiles.Add(Entry);
end;

procedure TImageFileList.RemoveFile(AIndex: Integer);
begin
  FFiles.Delete(AIndex);
end;

function TImageFileList.IndexOfFileName(AFileName: string): Integer;
var
  I: Integer;
begin
  AFileName:=UpperCase(AFileName);
  for I:=0 to FileCount - 1 do
    if Files[I].FileName=AFileName then Exit(I);
  Result:=-1;
end;

function TImageFileList.IndexOf(AEntry: TImageFileEntry): Integer;
var
  I: Integer;
begin
  for I:=0 to FileCount - 1 do
    if Files[I]=AEntry then Exit(I);
  Result:=-1;
end;

function TImageFileList.TotalDataSize: Integer;
var
  I, J: Integer;
begin
  Result:=0;
  for I:=0 to FileCount - 1 do if Files[I].FileSize > 0 then begin
    J:=(Files[I].FileSize div 512)*512;
    if J < Files[I].FileSize then Inc(J, 512);
    Result += J;
  end;
end;

procedure TImageFileList.SaveToFile(AFileName: string);
var
  Paths: TStringList;
  I: Integer;
begin
  Paths:=TStringList.Create;
  AFileName:=ExpandFileName(AFileName);
  try
    for I:=0 to FileCount - 1 do
      Paths.Add(ExtractRelativePath(AFileName, Files[I].LocalPath));
    Paths.SaveToFile(AFileName);
  finally
    Paths.Free;
  end;
end;

procedure TImageFileList.LoadFromFile(AFileName: string);
var
  Paths: TStringList;
  I: Integer;
begin
  FFiles.Clear;
  Paths:=TStringList.Create;
  AFileName:=ExpandFileName(AFileName);
  try
    Paths.LoadFromFile(AFileName);
    AFileName:=ExtractFilePath(AFileName);
    for I:=0 to Paths.Count - 1 do AddFile(CreateAbsolutePath(Paths[I], AFileName));
  finally
    Paths.Free;
  end;
end;

end.

