#ifndef ServerMeH
#define ServerMeH

// Single file HTTP server class by Kostas "Bad Sector" Michalopoulos
// This should work with most C++ compilers
//
// Usage:
//  At application startup:
//      CWebServer::Initialize()    (static)
//  At application shutdown:
//      CWebServer::Shutdown()      (static)
//  To start listening for connetions:
//      CWebServer::Listen(int port=80)
//  To stop listening:
//      CWebServer::Stop()
//  To setup a callback:
//      CWebServer::SetCallback(FWebServerCallback callback)
//  To pause and wait for connections (ms=0 dont wait, ms=-1 wait forever)
//      CWebServer::Wait(int ms)
//      (note: this will call any registered callback)
//  To avoid automatic closing of sockets (f.e. for responding in threads):
//      CWebServer::SetAutoClose(false)
//      (make sure you call request.Socket.Close() when you are done)
//

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#if (defined(_WIN32) || defined(WIN32)) && !defined(SMEH_NO_WINDOWS)
#include <winsock2.h>
#define SMEH_WINSOCK
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

// Socket wrapper class
class CWebServerSocket
{
public:
// Socket type alias
#ifdef SMEH_WINSOCK
    typedef SOCKET  ASocket;
#define SMEH_INVALID_SOCKET INVALID_SOCKET
#else
    typedef int     ASocket;
#define SMEH_INVALID_SOCKET -1
#endif
    inline CWebServerSocket()
        : m_socket(SMEH_INVALID_SOCKET)
    {
    }
    inline CWebServerSocket(ASocket socket)
        : m_socket(socket)
    {
    }

    // Create the socket (do not call for "request" in callback!)
    inline bool Create(int af, int type, int protocol)
    {
        m_socket = socket(af, type, protocol);
        if (m_socket != SMEH_INVALID_SOCKET)
        {
            // Setup lingering
            linger ling;
            ling.l_onoff = 1;
            ling.l_linger = 1;
            setsockopt(m_socket, SOL_SOCKET, SO_LINGER, (char*)&ling, sizeof(linger));
            return true;
        }
        return false;
    }

    // Setup the socket for incoming connections on the given port
    inline bool Listen(int port)
    {
        // Try to create the socket
        if (!Create(AF_INET, SOCK_STREAM, 0))
        {
            return false;
        }

        // Initialize the internet socket structure
        sockaddr_in sa;
        memset(&sa, 0, sizeof(sa));
        sa.sin_family = AF_INET;
        sa.sin_addr.s_addr = INADDR_ANY;
        sa.sin_port = htons((unsigned short)port);

        // Bind the address
        if (bind(m_socket, (sockaddr*)&sa, sizeof(sa)) != 0)
        {
            Close();
            return false;
        }

        // Listen for connections
        if (listen(m_socket, SOMAXCONN) != 0)
        {
            Close();
            return false;
        }

        return true;
    }

    // Accept a connection (assumes the socket is listening for them_
    inline bool Accept(CWebServerSocket& client)
    {
        // Ensure validity
        if (!IsValid()) return false;
        // Accept connection
        ASocket clientSocket = accept(m_socket, 0, 0);
        if (clientSocket != SMEH_INVALID_SOCKET)
        {
            client = CWebServerSocket(clientSocket);
            return true;
        }
        else
        {
            client = CWebServerSocket(SMEH_INVALID_SOCKET);
            return false;
        }
    }

    // Wait for data or connections (ms=0 do not wait, -1 wait forever)
    // This will return true for any data or false if expired or error
    inline bool Wait(int ms)
    {
        // Ensure validity
        if (!IsValid()) return false;
        
        // Setup the descriptor set
        fd_set fd;
        FD_ZERO(&fd);
        FD_SET(m_socket, &fd);

        // Setup the timeval
        timeval tv;
        if (ms > 0)
        {
            tv.tv_sec = ms/1000;
            tv.tv_usec = (ms%1000)*1000;
        }
        else
        {
            tv.tv_sec = 0;
            tv.tv_usec = 0;
        }

        // Wait for data/connections
        return select((int)(m_socket + 1), &fd, 0, 0, ms==-1 ? 0 : &tv) > 0;
    }

    // Returns the available bytes in the socket for receiving
    inline bool GetAvailableBytes(int& bytes)
    {
        bytes = -1;
        // Ensure validity
        if (!IsValid()) return false;
#ifdef SMEH_WINSOCK
        u_long b;
        if (ioctlsocket(m_socket, FIONREAD, &b) != 0) return false;
        bytes = (int)b;
#else
        if (ioctl(m_socket, FIONREAD, &bytes) != 0)
        {
            bytes = -1;
            return false;
        }
#endif
        return true;
    }

    // Close the socket (use on request.Socket only when autoClose is false)
    inline void Close()
    {
#ifdef SMEH_WINSOCK
        shutdown(m_socket, SD_BOTH);
        closesocket(m_socket);
#else
        shutdown(m_socket, SHUT_RDWR);
        close(m_socket);
#endif
        m_socket = SMEH_INVALID_SOCKET;
    }

    // Send data
    inline bool Send(const char* buffer, int count)
    {
        int sent = 0;

        // Ensure validity
        if (!IsValid()) return false;

        // Repeatedly sent the data in packets without exceeding message size
        while (sent < count)
        {
            int r = send(m_socket, buffer, count - sent, 0);
            if (r < 0)
            {
                m_socket = SMEH_INVALID_SOCKET;
                return false;
            }
            buffer += r;
            sent += r;
        }

        // Done
        return true;
    }

    // Send a string
    inline bool Send(const char* msg)
    {
        return Send(msg, strlen(msg));
    }

    // Send a string with content type
    inline bool SendContent(const char* msg, const char* contentType="text/html")
    {
        char* tmp = new char[strlen(contentType) + strlen(msg) + 20];
        sprintf(tmp, "Content-Type: %s\r\n\r\n%s", contentType, msg); 
        bool b = Send(tmp, strlen(tmp));
        delete[] tmp;
        return b;
    }

    // Send the contents of a file
    inline bool SendFile(const char* path, bool sendHeader=true)
    {
        char buffer[16384];
        size_t r;
        FILE* f = fopen(path, "rb");
        if (!f) return false;
        if (sendHeader)
        {
            char tmp[256];
            size_t size;
            fseek(f, 0, SEEK_END);
            size = ftell(f);
            fseek(f, 0, SEEK_SET);
            sprintf(tmp, "Content-Length: %u\r\n\r\n", (unsigned)size);
            Send(tmp, strlen(tmp));
        }
        while ((r = fread(buffer, 1, sizeof(buffer), f)) != 0)
        {                  
            if (!Send(buffer, r)) break;
        }
        fclose(f);
        return true;
    }

    // Receive data, returns false on error or if the connection should close
    inline bool Receive(char* buffer, int count, int& received)
    {
        received = 0;

        // Ensure validity
        if (!IsValid()) return false;

        // If a negative value for count was given, use whatever is available
        if (count < 0)
        {
            if (!GetAvailableBytes(count))
            {
                return false;
            }
        }

        // Special case for 0 bytes
        if (count == 0)
        {
            received = 0;
            return true;
        }

        // Request the received data
        received = recv(m_socket, buffer, count, 0);
        if (received < 1)
        {
            received = 0;
            m_socket = SMEH_INVALID_SOCKET;
            return false;
        }
 
        // Done
        return true;
    }

    // Returns true if this socket is valid
    inline bool IsValid() const { return m_socket != SMEH_INVALID_SOCKET; }

    // Returns the native socket
    inline ASocket GetSocket() const { return m_socket; }
private:
    ASocket     m_socket;       // Native socket
};

// Request information
struct SWebServerRequest
{
    CWebServerSocket    Socket;     // Client socket
    const char*         Address;    // Requested address
};

// Callback function for connections
typedef void (*FWebServerCallback)(SWebServerRequest& request, void* data);

// Provides a nice-ish werb server
class CWebServer
{
public:
    inline CWebServer()
        : m_autoClose(true)
    {
    }

    // One time initialization
    static bool Initialize()
    {
#ifdef SMEH_WINSOCK
        // Initialize Winsock 2.0
        WSADATA wsaData;
        if (WSAStartup(MAKEWORD(2, 0), &wsaData) != 0)
        {
            return false;
        }
        return true;
#else
        return true;
#endif
    }

    // One time shutdown
    static void Shutdown()
    {
#ifdef SMEH_WINSOCK
        // Shutdown Winsock
        WSACleanup();
#endif
    }

    // Listen for HTTP connections
    inline bool Listen(int port=80)
    {
        // Close existing socket (if any)
        if (m_socket.IsValid())
        {
            m_socket.Close();
        }

        // Start listening on given port
        return m_socket.Listen(port);
    }

    // Stop the server
    inline void Stop()
    {
        if (m_socket.IsValid())
        {
            m_socket.Close();
        }
    }

    // Wait for connections
    inline bool Wait(int ms)
    {
        // Do not wait at all if the socket is invalid
        if (!m_socket.IsValid()) return false;

        // Wait for a connection
        if (m_socket.Wait(ms))
        {
            // Accept the connection
            SWebServerRequest   request;
            if (m_socket.Accept(request.Socket))
            {
                // Get request data (the data should be enough for ~60K of encoded contents)
                char data[655360];
                int bytes, received = 0;

                // Read as much data as possible
                while (received < (int)sizeof(data))
                {
                    bytes = sizeof(data);
                    if (bytes + received > (int)sizeof(data)) bytes = (int)sizeof(data) - received;
                    if (!bytes) break;
                    int recbit;
                    if (!request.Socket.Wait(200)) break;
                    if (!request.Socket.Receive(data + received, bytes, recbit)) break;
                    received += recbit;
                }
                // Drain the rest of the data
                while (request.Socket.IsValid())
                {
                    char dump[1024];
                    int unused;
                    if (!request.Socket.GetAvailableBytes(bytes)) break;
                    if (!bytes) break;
                    request.Socket.Receive(dump, sizeof(dump), unused);
                }
                if (received < 100)
                {
                    request.Socket.Close();
                    return false;
                }

                // Find requested address position
                int pos = 0, addrStart = -1;
                while (pos < received && data[pos] == ' ') ++pos;
                if (pos + 4 >= received)
                {
                    return false;
                }
                if (data[pos] != 'G' ||
                    data[pos + 1] != 'E' ||
                    data[pos + 2] != 'T' ||
                    data[pos + 3] != ' ')
                {
                    request.Socket.Close();
                    return false;
                }
                pos += 4;
                while (pos < received && data[pos] == ' ') ++pos;
                if (pos + 4 < received &&
                    data[pos] == 'h' &&
                    data[pos + 1] == 't' &&
                    data[pos + 2] == 't' &&
                    data[pos + 3] == 'p')
                {
                    // first / in http://address/stuff
                    while (pos < received && data[pos] != '/') ++pos;
                    ++pos;
                    // second / in http://address/stuff
                    while (pos < received && data[pos] != '/') ++pos;
                    ++pos;
                    // third/ in http://address/stuff
                    while (pos < received && data[pos] != '/') ++pos;
                }
                addrStart = pos;
                while (pos < received && data[pos] != ' ') ++pos;
                if (addrStart == pos || addrStart >= received)
                {
                    request.Socket.Close();
                    return false;
                }
                data[pos] = 0;
                request.Address = data + addrStart;

                // Respond that everyhing is fine
                request.Socket.Send("HTTP/1.0 200 OK\r\n");

                // Do the callback
                if (m_callback)
                {
                    m_callback(request, m_cbdata);
                }

                // Drain any remaining data
                #if 0
                while (request.Socket.IsValid())
                {
                    char dump[1024];
                    int unused;
                    if (!request.Socket.GetAvailableBytes(bytes)) break;
                    if (!bytes)
                    {
                        request.Socket.Wait(100);
                        if (!request.Socket.GetAvailableBytes(bytes)) break;
                        if (!bytes) break;
                    }
                    request.Socket.Receive(dump, bytes, unused);
                }
                #endif

                // Close the connection
                if (m_autoClose)
                {
                    request.Socket.Close();
                }
                return true;
            }

            // Invalid socket
            return false;
        }

        // Timeout or error
        return false;
    }

    // Setup a callback
    inline void SetCallback(FWebServerCallback callback)
    {
        m_callback = callback;
    }

    // Setup automatic closing of sockets after the callback is called
    inline void SetAutoClose(bool autoClose)
    {
        m_autoClose = autoClose;
    }

private:
    CWebServerSocket        m_socket;       // Server socket
    FWebServerCallback      m_callback;     // User callback
    void*                   m_cbdata;       // User callback data
    bool                    m_autoClose;    // Automatically close sockets
};

#endif
