unit UndoManager;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FGL, ActnList, Nodes;

type

  { TUndoable }

  TUndoable = class
  protected
    function GetDescription: string; virtual; abstract;
    function Execute: Boolean; virtual; abstract;
    function Undo: Boolean; virtual; abstract;
    function Redo: Boolean; virtual; abstract;
  public
    property Description: string read GetDescription;
  end;

  TUndoableClass = class of TUndoable;
  TUndoableList = specialize TFPGObjectList<TUndoable>;

  { TUndoableGroup }

  TUndoableGroup = class(TUndoable)
  private
    FDescription: string;
    Undoables: TUndoableList;
    NextGroup: TUndoableGroup;
  protected
    function GetDescription: string; override;
    function Execute: Boolean; override;
    function Undo: Boolean; override;
    function Redo: Boolean; override;
  public
    constructor Create(ADescription: string);
    destructor Destroy; override;
  end;

  TUndoEvent = procedure(Sender: TObject; AUndoable: TUndoable) of object;

  { TUndoManager }

  TUndoManager = class(TComponent)
  private
    FOnExecute: TUndoEvent;
    FOnModified: TNotifyEvent;
    FOnRedo: TUndoEvent;
    FOnUndo: TUndoEvent;
    FRedoAction: TAction;
    FUndoAction: TAction;
    FUndoStack: TUndoableList;
    FRedoStack: TUndoableList;
    UndoableGroup: TUndoableGroup;
    procedure SetRedoAction(AValue: TAction);
    procedure SetUndoAction(AValue: TAction);
    procedure UpdateUserInterface;
    procedure StacksModified;
  public
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure ClearUndo;
    procedure ClearRedo;
    procedure Clear;
    function Undo: Boolean;
    function Redo: Boolean;
    function Execute(Undoable: TUndoable): Boolean;
    procedure BeginGroup(ADescription: string);
    procedure EndGroup;
    function CanUndo: Boolean;
    function CanRedo: Boolean;
    function GetUndoDescription: string;
    function GetRedoDescription: string;
  published
    property UndoAction: TAction read FUndoAction write SetUndoAction;
    property RedoAction: TAction read FRedoAction write SetRedoAction;
    property OnUndo: TUndoEvent read FOnUndo write FOnUndo;
    property OnRedo: TUndoEvent read FOnRedo write FOnRedo;
    property OnExecute: TUndoEvent read FOnExecute write FOnExecute;
    property OnModified: TNotifyEvent read FOnModified write FOnModified;
  end;

procedure Register;

implementation

uses
  LResources;

procedure Register;
begin
  {$I undomanager_icon.lrs}
  RegisterComponents('RTTK', [TUndoManager]);
end;

{ TUndoableGroup }

function TUndoableGroup.GetDescription: string;
begin
  Result:=FDescription;
end;

function TUndoableGroup.Execute: Boolean;
begin
  Result:=True;
end;

function TUndoableGroup.Undo: Boolean;
var
  I: Integer;
begin
  for I:=Undoables.Count - 1 downto 0 do
    if not Undoables[I].Undo then Exit(False);
  Result:=True;
end;

function TUndoableGroup.Redo: Boolean;
var
  I: Integer;
begin
  for I:=0 to Undoables.Count - 1 do
    if not Undoables[I].Redo then Exit(False);
  Result:=True;
end;

constructor TUndoableGroup.Create(ADescription: string);
begin
  FDescription:=ADescription;
  Undoables:=TUndoableList.Create(True);
end;

destructor TUndoableGroup.Destroy;
begin
  FreeAndNil(Undoables);
  inherited Destroy;
end;

{ TUndoManager }

procedure TUndoManager.UpdateUserInterface;
begin
  if Assigned(UndoAction) then begin
    UndoAction.Enabled:=FUndoStack.Count > 0;
    if UndoAction.Enabled then
      UndoAction.Hint:='Undo ' + FUndoStack.Last.Description
    else
      UndoAction.Hint:='Undo';
    UndoAction.Caption:=UndoAction.Hint;
  end;
  if Assigned(RedoAction) then begin
    RedoAction.Enabled:=FRedoStack.Count > 0;
    if RedoAction.Enabled then
      RedoAction.Hint:='Redo ' + FRedoStack.Last.Description
    else
      RedoAction.Hint:='Redo';
    RedoAction.Caption:=RedoAction.Hint;
  end;
end;

procedure TUndoManager.SetRedoAction(AValue: TAction);
begin
  if FRedoAction=AValue then Exit;
  FRedoAction:=AValue;
  UpdateUserInterface;
end;

procedure TUndoManager.SetUndoAction(AValue: TAction);
begin
  if FUndoAction=AValue then Exit;
  FUndoAction:=AValue;
  UpdateUserInterface;
end;

procedure TUndoManager.StacksModified;
begin
  UpdateUserInterface;
  if Assigned(FOnModified) then FOnModified(Self);
end;

procedure TUndoManager.AfterConstruction;
begin
  inherited AfterConstruction;
  FUndoStack:=TUndoableList.Create(False);
  FRedoStack:=TUndoableList.Create(False);
end;

procedure TUndoManager.BeforeDestruction;
begin
  Clear;
  FreeAndNil(FRedoStack);
  FreeAndNil(FUndoStack);
  inherited BeforeDestruction;
end;

procedure TUndoManager.ClearUndo;
var
  I: LongInt;
begin
  for I:=FUndoStack.Count - 1 downto 0 do FUndoStack[I].Free;
  FUndoStack.Clear;
  StacksModified;
end;

procedure TUndoManager.ClearRedo;
var
  I: LongInt;
begin
  for I:=FRedoStack.Count - 1 downto 0 do FRedoStack[I].Free;
  FRedoStack.Clear;
  StacksModified;
end;

procedure TUndoManager.Clear;
begin
  ClearRedo;
  ClearUndo;
  UndoableGroup:=nil;
end;

function TUndoManager.Undo: Boolean;
var
  Undoable: TUndoable;
begin
  if FUndoStack.Count=0 then Exit;
  Undoable:=FUndoStack.Last;
  FUndoStack.Remove(Undoable);
  FRedoStack.Add(Undoable);
  StacksModified;
  if not Undoable.Undo then begin
    FRedoStack.Remove(Undoable);
    FUndoStack.Add(Undoable);
    Result:=False;
    StacksModified;
  end else begin
    if Assigned(FOnUndo) then FOnUndo(Self, Undoable);
    Result:=True;
  end;
end;

function TUndoManager.Redo: Boolean;
var
  Undoable: TUndoable;
begin
  if FRedoStack.Count=0 then Exit;
  Undoable:=FRedoStack.Last;
  FRedoStack.Remove(Undoable);
  FUndoStack.Add(Undoable);
  StacksModified;
  if not Undoable.Redo then begin
    FUndoStack.Remove(Undoable);
    FRedoStack.Add(Undoable);
    Result:=False;
    StacksModified;
  end else begin
    if Assigned(FOnRedo) then FOnRedo(Self, Undoable);
    Result:=True;
  end;
end;

function TUndoManager.Execute(Undoable: TUndoable): Boolean;
begin
  Result:=False;
  if Assigned(UndoableGroup) then begin
    UndoableGroup.Undoables.Add(Undoable);
  end else begin
    FUndoStack.Add(Undoable);
    ClearRedo;
    StacksModified;
  end;
  if not Undoable.Execute then begin
    if Assigned(UndoableGroup) then begin
      UndoableGroup.Undoables.Remove(Undoable);
    end else begin
      FUndoStack.Remove(Undoable);
      Undoable.Free;
      StacksModified;
    end;
  end else begin
    if Assigned(FOnExecute) then FOnExecute(Self, Undoable);
  end;
end;

procedure TUndoManager.BeginGroup(ADescription: string);
var
  NewGroup: TUndoableGroup;
begin
  NewGroup:=TUndoableGroup.Create(ADescription);
  NewGroup.NextGroup:=UndoableGroup;
  UndoableGroup:=NewGroup;
end;

procedure TUndoManager.EndGroup;
var
  Group: TUndoableGroup;
begin
  Group:=UndoableGroup;
  UndoableGroup:=UndoableGroup.NextGroup;
  Execute(Group);
end;

function TUndoManager.CanUndo: Boolean;
begin
  Result:=FUndoStack.Count > 0;
end;

function TUndoManager.CanRedo: Boolean;
begin
  Result:=FRedoStack.Count > 0;
end;

function TUndoManager.GetUndoDescription: string;
begin
  if CanUndo then Result:=FUndoStack.Last.GetDescription else Result:='';
end;

function TUndoManager.GetRedoDescription: string;
begin
  if CanRedo then Result:=FRedoStack.Last.GetDescription else Result:='';
end;

end.

