unit SoundPlayback;

{$mode objfpc}{$H+}

interface

uses
  {$IFDEF WINDOWS}
  mmsystem,
  {$ENDIF}
  Classes, SysUtils;

type
  TSoundCallback = procedure(Samples: PInt16; Count: Integer) of object;

  { TSoundPlayer }

  TSoundPlayer = class(TComponent)
  private
    FBufferSize: Cardinal;
    FSoundCallback: TSoundCallback;
    Buffer: array [0..3] of PInt16;
    BufferNumber: Integer;
    {$IFDEF WINDOWS}
    Handle: THandle;
    Header: array [0..3] of TWAVEHDR;
    {$ENDIF}
    procedure SetBufferSize(AValue: Cardinal);
    procedure WriteSamples;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Initialize(ABufferSize: Cardinal): Boolean;
    procedure Shutdown;
    property SoundCallback: TSoundCallback read FSoundCallback write FSoundCallback;
    property BufferSize: Cardinal read FBufferSize write SetBufferSize;
  end;

procedure Register;

implementation

uses
  LResources;

procedure Register;
begin
  {$I soundplayback_icon.lrs}
  RegisterComponents('RTTK', [TSoundPlayer]);
end;

{ TSoundPlayer }

{$IFDEF WINDOWS}
{$DEFINE PLAYER_DEFINED}
procedure WaveOutCallback(hwo: HWAVEOUT; Msg: UInt32; hInstance, dwParam1, dwParam2: UIntPtr); stdcall;
begin
  if Msg=WOM_DONE then TSoundPlayer(hInstance).WriteSamples;
end;

procedure TSoundPlayer.WriteSamples;
begin
  if (Handle <> 0) then begin
    if Assigned(FSoundCallback) then FSoundCallback(Buffer[BufferNumber], FBufferSize);
    waveOutWrite(Handle, @Header[BufferNumber], SizeOf(Header[BufferNumber]));
    BufferNumber:=(BufferNumber + 1) mod Length(Buffer);
  end;
end;

procedure TSoundPlayer.SetBufferSize(AValue: Cardinal);
begin
  if FBufferSize=AValue then Exit;
  FBufferSize:=AValue;
end;

constructor TSoundPlayer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetBufferSize(4096);
end;

destructor TSoundPlayer.Destroy;
begin
  Shutdown;
  inherited Destroy;
end;

function TSoundPlayer.Initialize(ABufferSize: Cardinal): Boolean;
var
  Format: TWAVEFORMATEX;
  I: Integer;
begin
  with Format do begin
    wFormatTag:=WAVE_FORMAT_PCM;
    nChannels:=1;
    nSamplesPerSec:=44100;
    wBitsPerSample:=16;
    nBlockAlign:=nChannels*wBitsPerSample div 8;
    nAvgBytesPerSec:=nBlockAlign*nSamplesPerSec;
    cbSize:=0;
  end;
  Handle:=0;
  if waveOutOpen(@Handle, WAVE_MAPPER, @Format, {%H-}UIntPtr(@WaveOutCallback), UIntPtr(Self), CALLBACK_FUNCTION) <> MMSYSERR_NOERROR then Exit(False);
  waveOutPause(Handle);
  FBufferSize:=ABufferSize;
  BufferNumber:=0;
  for I:=0 to High(Buffer) do begin
    Buffer[I]:=GetMem(ABufferSize*2);
    Header[I].dwFlags:=0;
    Header[I].dwBufferLength:=ABufferSize*2;
    Header[I].lpData:=PChar(Buffer[I]);
    waveOutPrepareHeader(Handle, @Header[I], SizeOf(Header[I]));
    WriteSamples;
  end;
  waveOutRestart(Handle);
  Result:=True;
end;

procedure TSoundPlayer.Shutdown;
var
  I: Integer;
begin
  if Handle <> 0 then begin
    waveOutPause(Handle);
    for I:=High(Buffer) downto 0 do begin
      waveOutUnprepareHeader(Handle, @Header[I], SizeOf(Header[I]));
      FreeMem(Buffer[I]);
    end;
    waveOutClose(Handle);
    Handle:=0;
  end;
end;
{$ENDIF}

{$IFDEF linux}
{$DEFINE NO_SOUND}
{$ENDIF}

{$IFDEF NO_SOUND}
{$DEFINE PLAYER_DEFINED}
procedure TSoundPlayer.WriteSamples;
begin
end;

procedure TSoundPlayer.SetBufferSize(AValue: Cardinal);
begin
end;

constructor TSoundPlayer.Create(AOwner: TComponent);
begin
end;

destructor TSoundPlayer.Destroy;
begin
end;

function TSoundPlayer.Initialize(ABufferSize: Cardinal): Boolean;
begin
  Result:=False;
end;

procedure TSoundPlayer.Shutdown;
begin
end;
{$ENDIF}

{$IFNDEF PLAYER_DEFINED}
{$ERROR Operating system not supported yet}
{$ENDIF}

end.

