unit LILUtils;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FPLIL, FGL;

type
  TLILHandleIndexMap = specialize TFPGMap<Pointer, Integer>;
  TLILUtilCategory = (lucGUI, lucExec, lucDir);
  TLILUtilCategories = set of TLILUtilCategory;

const
  AllLILUtilCategories = [lucGUI, lucExec, lucDir];

type

  { TLILHandles }

  TLILHandles = class(TComponent)
  private
    FObjects: array of TObject;
    Handles: array of TLILValue;
    FreePos: array of Integer;
    FPCount: Integer;
    IndexMap: TLILHandleIndexMap;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Add(FObject: TObject): Integer;
    procedure Remove(FObject: TObject);
    function Has(AObject: TObject): Boolean;
    function GetHandle(AObject: TObject): TLILValue;
    function GetObject(AHandle: TLILValue; AClass: TClass=nil): TObject;
  end;

procedure RegisterLILUtilities(LIL: TLIL; Categories: TLILUtilCategories=AllLILUtilCategories);

procedure Register;

implementation

uses
  LResources, FileUtil, MiscUtils, Controls, Forms, Dialogs, Process;

procedure Register;
begin
  {$I lilutils_icon.lrs}
  RegisterComponents('RTTK', [TLILHandles]);
end;

{ GUI Functions }
function FncMessage(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Caption: String;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like message <message> [caption]');
    Exit(nil);
  end;
  if Length(Args) > 1 then
    Caption:=TLIL.ToString(Args[1])
  else
    Caption:='Script';
  MessageDlg(Caption, TLIL.ToString(Args[0]), mtCustom, [mbOK], 0);
  Result:=nil;
end;

function FncConfirm(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like confirm <message>');
    Exit(nil);
  end;
  Result:=TLIL.AllocBoolean(MessageDlg(TLIL.ToString(Args[0]), mtConfirmation, mbYesNo, 0)=mrYes);
end;

function FncAsk(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Default, Caption: String;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like ask <message> [default] [caption]');
    Exit(nil);
  end;
  if Length(Args) > 1 then
    Default:=TLIL.ToString(Args[1])
  else
    Default:='';
  if Length(Args) > 2 then
    Caption:=TLIL.ToString(Args[2])
  else
    Caption:='Script Query';
  Result:=TLIL.AllocString(InputBox(Caption, TLIL.ToString(Args[0]), Default));
end;

function FncAskPassword(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Caption: String;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like ask-password <message> [caption]');
    Exit(nil);
  end;
  if Length(Args) > 1 then
    Caption:=TLIL.ToString(Args[1])
  else
    Caption:='Script Password Query';
  Result:=TLIL.AllocString(PasswordBox(Caption, TLIL.ToString(Args[0])));
end;

function FncSaveDialog(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Ext: String;
  Dlg: TSaveDialog;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like save-dialog <message> [extension]');
    Exit(nil);
  end;
  if Length(Args) > 1 then
    Ext:=TLIL.ToString(Args[1])
  else
    Ext:='';
  Dlg:=TSaveDialog.Create(Application.MainForm);
  Dlg.Title:=TLIL.ToString(Args[0]);
  Dlg.DefaultExt:=Ext;
  Dlg.Options:=DefaultOpenDialogOptions + [ofFileMustExist];
  if Dlg.Execute then
    Result:=TLIL.AllocString(Dlg.FileName)
  else
    Result:=nil;
  FreeAndNil(Dlg);
end;

function FncOpenDialog(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Ext: String;
  Dlg: TOpenDialog;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like open-dialog <message> [extension]');
    Exit(nil);
  end;
  if Length(Args) > 1 then
    Ext:=TLIL.ToString(Args[1])
  else
    Ext:='';
  Dlg:=TOpenDialog.Create(Application.MainForm);
  Dlg.Title:=TLIL.ToString(Args[0]);
  Dlg.DefaultExt:=Ext;
  Dlg.Options:=DefaultOpenDialogOptions + [ofPathMustExist, ofOverwritePrompt];
  if Dlg.Execute then
    Result:=TLIL.AllocString(Dlg.FileName)
  else
    Result:=nil;
  FreeAndNil(Dlg);
end;

function FncSelectDirectory(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Dir, Default, Caption: String;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like select-directory <caption> [default]');
    Exit(nil);
  end;
  if Length(Args) > 1 then
    Default:=TLIL.ToString(Args[1])
  else
    Default:='.';
  if SelectDirectory(Caption, Default, Dir) then
    Result:=TLIL.AllocString(Dir)
  else
    Result:=nil;
end;

{ Execution functions }
function FncExec(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  Params: TStringArray;
  I: Integer;
  Res: string;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like exec <program> [param1] [param2] ...');
    Exit(nil);
  end;
  SetLength(Params, Length(Args) - 1);
  for I:=1 to High(Args) do Params[I - 1]:=TLIL.ToString(Args[I]);
  if RunCommand(TLIL.ToString(Args[0]), Params, Res) then
    Result:=TLIL.AllocString(Res)
  else
    Result:=nil;
end;

function FncLaunch(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  I: Integer;
  Proc: TProcess;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like launch <program> [param1] [param2] ...');
    Exit(nil);
  end;
  try
    Proc:=TProcess.Create(Application);
    Proc.InheritHandles:=False;
    Proc.Executable:=TLIL.ToString(Args[0]);
    for I:=1 to High(Args) do Proc.Parameters.Add(TLIL.ToString(Args[I]));
    Proc.Options:=[];
    Proc.ShowWindow:=swoShow;
    for I:=1 to GetEnvironmentVariableCount - 1 do
      Proc.Environment.Add(GetEnvironmentString(I));
    Proc.Execute;
  except
  end;
  FreeAndNil(Proc);
  Result:=nil;
end;

{ Dir functions }
function FncMkDir(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like mkdir <path>');
    Exit(nil);
  end;
  Result:=TLIL.AllocBoolean(ForceDirectories(TLIL.ToString(Args[0])));
end;

function FncRmDir(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like rmdir <path>');
    Exit(nil);
  end;
  Result:=TLIL.AllocBoolean(RemoveDir(TLIL.ToString(Args[0])));
end;

function FncChDir(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args) < 1 then begin
    LIL.SetError('Not enough parameters, use like chdir <path>');
    Exit(nil);
  end;
  Result:=TLIL.AllocBoolean(SetCurrentDir(TLIL.ToString(Args[0])));
end;

procedure RegisterLILUtilities(LIL: TLIL; Categories: TLILUtilCategories);

  procedure RegisterGUIFuncs;
  begin
    // Dialogs
    LIL.Register('message', @FncMessage);
    LIL.Register('confirm', @FncConfirm);
    LIL.Register('ask', @FncAsk);
    LIL.Register('ask-password', @FncAskPassword);
    LIL.Register('save-dialog', @FncSaveDialog);
    LIL.Register('open-dialog', @FncOpenDialog);
    LIL.Register('select-directory', @FncSelectDirectory);
  end;

  procedure RegisterExecFuncs;
  begin
    LIL.Register('exec', @FncExec);
    LIL.Register('launch', @FncLaunch);
  end;

  procedure RegisterDirFuncs;
  begin
    LIL.Register('mkdir', @FncMkDir);
    LIL.Register('rmdir', @FncRmDir);
    LIL.Register('chdir', @FncChDir);
  end;

begin
  if lucGUI in Categories then RegisterGUIFuncs;
  if lucExec in Categories then RegisterExecFuncs;
  if LucDir in Categories then RegisterDirFuncs;
end;

{ TLILHandles }

constructor TLILHandles.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  IndexMap:=TLILHandleIndexMap.Create;
  IndexMap.Sorted:=True;
  SetLength(FObjects, 1);
  SetLength(Handles, 1);
  FObjects[0]:=nil;
  Handles[0]:=TLIL.AllocInteger(0);
  IndexMap.Add(Nil, 0);
end;

destructor TLILHandles.Destroy;
var
  I: Integer;
begin
  for I:=0 to High(Handles) do
    if Assigned(Handles[I]) then
      Handles[I].Free;
  FreeAndNil(IndexMap);
  inherited Destroy;
end;

function TLILHandles.Add(FObject: TObject): Integer;
begin
  if Assigned(FObject) then begin
    if FPCount > 0 then begin
      Dec(FPCount);
      Result:=FreePos[FPCount];
    end else begin
      Result:=Length(FObjects);
      SetLength(FObjects, Result + 1);
      SetLength(Handles, Result + 1);
    end;
    FObjects[Result]:=FObject;
    if not Assigned(Handles[Result]) then
      Handles[Result]:=TLIL.AllocInteger(Result);
    IndexMap[FObject]:=Result;
  end else
    Result:=0;
end;

procedure TLILHandles.Remove(FObject: TObject);
var
  Index: Integer;
begin
  if Assigned(FObject) then begin
    if not IndexMap.Find(FObject, Index) then Exit;
    IndexMap.Remove(FObject);
    FObjects[Index]:=nil;
    if FPCount=Length(FreePos) then SetLength(FreePos, FPCount + 1024);
    FreePos[FPCount]:=Index;
    Inc(FPCount);
  end;
end;

function TLILHandles.Has(AObject: TObject): Boolean;
var
  Index: Integer;
begin
  if Assigned(AObject) then begin
    if not IndexMap.Find(AObject, Index) then Exit(False);
    Exit(True);
  end;
  Exit(False);
end;

function TLILHandles.GetHandle(AObject: TObject): TLILValue;
var
  Index: Integer;
begin
  if Assigned(AObject) then begin
    if not IndexMap.Find(AObject, Index) then
      Index:=Add(AObject);
    Result:=Handles[Index];
  end else
    Result:=nil;
end;

function TLILHandles.GetObject(AHandle: TLILValue; AClass: TClass): TObject;
var
  Index: Int64;
begin
  Index:=TLIL.ToInteger(AHandle);
  if (Index <= 0) or (Index >= Length(FObjects)) then Exit(nil);
  Result:=FObjects[Index];
  if not Assigned(Result) or not Assigned(AClass) then Exit;
  if not (FObjects[Index] is AClass) then Result:=nil;
end;

end.

