unit Containers;

{$mode objfpc}{$H+}
{$modeswitch advancedrecords}

interface

uses
  Classes, sysutils;

type
  EContainerError = class(Exception);

  { TDynArray }

  generic TDynArray<T> = record
  type
    PT = ^T;
    TSorter = function(const A, B: T; Data: Pointer): Integer;
  private
    FItems: Pointer;
    FCount: Integer;
    FCapacity: Integer;
    function GetFirst: T; inline;
    function GetFirstPointer: PT; inline;
    function GetItemPointers(AIndex: Integer): PT;
    function GetItems(AIndex: Integer): T; inline;
    function GetLast: T; inline;
    function GetLastPointer: PT; inline;
    function GetUsedBytes: Int64; inline;
    procedure SetCapacity(AValue: Integer);
    procedure SetCount(AValue: Integer);
    procedure SetItems(AIndex: Integer; AValue: T); inline;
    function GetRawData: PT; inline;
    // Used by Sort
    function Partition(L, H: Integer; ASorter: TSorter; AData: Pointer): Integer;
    procedure QuickSort(L, H: Integer; ASorter: TSorter; AData: Pointer);
  public
    // Initialize the container (not needed for zeroed out memory)
    procedure Init;
    // Shut down the container
    procedure Done;
    // Ensure the array has *at least* the given capacity
    procedure EnsureCapacity(ACapacity: Integer); inline;
    // Add an item at the end of the the array
    procedure Add(Item: T); inline;
    // Add a new item at the end of the array and return a pointer to it
    function AddNew: PT; inline;
    // Insert an item at the given position of the array (Count is same as Add)
    procedure Insert(Index: Integer; Item: T);
    // Returns the index of the given item or -1 if not found
    function IndexOf(Item: T): Integer;
    // Returns true if the container contains the given item
    function Contains(Item: T): Boolean; inline;
    // Removes the item at the given index
    procedure RemoveAt(Index: Integer);
    // Remove the given item if it exists in the container
    procedure Remove(Item: T); inline;
    // Removes all instances of the given item from the container
    procedure RemoveAll(Item: T);
    // Removes the item at the given index by swapping it with the last item and resizing
    procedure RemoveViaSwapAt(Index: Integer);
    // Removes the given item by swapping it with the last item and resizing
    procedure RemoveViaSwap(Item: T); inline;
    // Removes all items and releases any allocated memory
    procedure Clear;
    // Removes all items without releasing any memory
    procedure Reset;
    // Swaps the items at the given indices
    procedure Swap(Index1, Index2: Integer);
    // Removes the last item and returns it
    function Pop: T;
    // Removes the last item
    procedure PopAndIgnore;
    // Resize the capacity to be the same as the count of items in the container
    procedure Pack;
    // Sort the array using the given sorting function
    procedure Sort(ASorter: TSorter; AData: Pointer);
    // Set/Get the items in the container
    property Items[AIndex: Integer]: T read GetItems write SetItems; default;
    // Set/Get pointers to the items in the container
    property ItemPointers[AIndex: Integer]: PT read GetItemPointers;
    // Set/Get the number of items in the container
    property Count: Integer read FCount write SetCount;
    // Set/Get the amount of allocated memory in terms of items
    property Capacity: Integer read FCapacity write SetCapacity;
    // Get the first item in the container
    property First: T read GetFirst;
    // Get the last item in the container
    property Last: T read GetLast;
    // Get a pointer to the first item in the container
    property FirstPointer: PT read GetFirstPointer;
    // Get a pointer to the last item in the container
    property LastPointer: PT read GetLastPointer;
    // Get the number of used bytes by the container
    property UsedBytes: Int64 read GetUsedBytes;
    // Get the items as raw data
    property RawData: PT read GetRawData;
//    class operator=(const A, B: specialize TDynArray<T>): Boolean;
  end;

  TPointerDynArray = specialize TDynArray<Pointer>;
  TObjectDynArray = specialize TDynArray<TObject>;
  TStringDynArray = specialize TDynArray<string>;
  TIntegerDynArray = specialize TDynArray<Integer>;
  TBooleanDynArray = specialize TDynArray<Boolean>;
  TByteDynArray = specialize TDynArray<Byte>;
  TCharDynArray = specialize TDynArray<Char>;

  { TPair }

  generic TPair<F,S> = record
  private
    function GetUsedBytes: Int64;
  public
    First: F;
    Second: S;
    property UsedBytes: Int64 read GetUsedBytes;
    class operator=(const A, B: specialize TPair<F,S>): Boolean; inline;
  end;

  TBytePair = specialize TPair<Byte,Byte>;
  TStringPair = specialize TPair<string,string>;
  TPointerPair = specialize TPair<Pointer,Pointer>;
  TObjectPair = specialize TPair<TObject,TObject>;
  TStringPointerPair = specialize TPair<string,Pointer>;
  TStringObjectPair = specialize TPair<string,TObject>;

  { TLinearMap }
  generic TLinearMap<K,V> = object
  private
    FItemsStorage: array [1..SizeOf(TPointerDynArray)] of Byte;
    function GetCount: Integer;
    function GetKeys(AIndex: Integer): K;
    function GetValue(AKey: K): V;
    function GetValues(AIndex: Integer): V;
    procedure SetKeys(AIndex: Integer; AValue: K);
    procedure SetValue(AKey: K; AValue: V);
    procedure SetValues(AIndex: Integer; AValue: V);
  public
    procedure Init;
    procedure Done;
    procedure Clear;
    function IndexOfKey(AKey: K): Integer;
    function IndexOfValue(AValue: V): Integer;
    function HasKey(AKey: K): Boolean; inline;
    function HasValue(AValue: V): Boolean; inline;
    procedure RemoveKey(AKey: K);
    procedure RemoveValue(AValue: V);
    procedure RemoveAllValues(AValue: V);
    property Count: Integer read GetCount;
    property Keys[AIndex: Integer]: K read GetKeys write SetKeys;
    property Values[AIndex: Integer]: V read GetValues write SetValues;
    property Value[AKey: K]: V read GetValue write SetValue;
  end;

  TStringLinearMap = specialize TLinearMap<string,string>;
  TPointerLinearMap = specialize TLinearMap<Pointer,Pointer>;
  TObjectLinearMap = specialize TLinearMap<TObject,TObject>;
  TStringPointerLinearMap = specialize TLinearMap<string,Pointer>;
  TStringObjectLinearMap = specialize TLinearMap<string,TObject>;

implementation

uses
  typinfo;

{ TLinearMap }

function TLinearMap.GetCount: Integer;
type
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
begin
  Result:=Items.Count;
end;

function TLinearMap.GetKeys(AIndex: Integer): K;
type
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
begin
  Result:=Items[AIndex].First;
end;

function TLinearMap.GetValue(AKey: K): V;
begin

end;

function TLinearMap.GetValues(AIndex: Integer): V;
type
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
begin
  Result:=Items[AIndex].Second;
end;

procedure TLinearMap.SetKeys(AIndex: Integer; AValue: K);
type
  PKVPair = ^TKVPair;
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
begin
  if Items[AIndex].First=AValue then Exit;
  if IndexOfKey(AValue) <> -1 then raise EContainerError.Create('A key with that name already exists');
  PKVPair(Items.ItemPointers[AIndex])^.First:=AValue;
end;

procedure TLinearMap.SetValue(AKey: K; AValue: V);
type
  PKVPair = ^TKVPair;
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
  Index: Integer;
begin
  Index:=IndexOfKey(AKey);
  if Index=-1 then begin
    Items.Count:=Items.Count + 1;
    Initialize(PKVPair(Items.LastPointer)^);
    PKVPair(Items.LastPointer)^.First:=AKey;
    PKVPair(Items.LastPointer)^.Second:=AValue;
  end else begin
    PKVPair(Items.ItemPointers[Index])^.Second:=AValue;
  end;
end;

procedure TLinearMap.SetValues(AIndex: Integer; AValue: V);
begin

end;

procedure TLinearMap.Init;
begin
  FillChar(FItemsStorage, SizeOf(FItemsStorage), 0);
end;

procedure TLinearMap.Done;
type
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
begin
  Items.Done;
end;

procedure TLinearMap.Clear;
type
  TKVPair = specialize TPair<K,V>;
var
  Items: specialize TDynArray<TKVPair> absolute FItemsStorage;
begin
  Items.Clear;
end;

function TLinearMap.IndexOfKey(AKey: K): Integer;
begin

end;

function TLinearMap.IndexOfValue(AValue: V): Integer;
begin

end;

function TLinearMap.HasKey(AKey: K): Boolean;
begin

end;

function TLinearMap.HasValue(AValue: V): Boolean;
begin

end;

procedure TLinearMap.RemoveKey(AKey: K);
begin

end;

procedure TLinearMap.RemoveValue(AValue: V);
begin

end;

procedure TLinearMap.RemoveAllValues(AValue: V);
begin

end;

{ TPair }

function TPair.GetUsedBytes: Int64;
begin
  Result:=Int64(SizeOf(F)) + Int64(SizeOf(S));
end;

class operator TPair.=(const A, B: specialize TPair<F, S>): Boolean;
begin
  Result:=(A.First=B.First) and (A.Second=B.Second);
end;

{ TDynArray }

function TDynArray.GetItems(AIndex: Integer): T;
begin
  Assert((AIndex >= 0) and (AIndex < Count), 'Invalid index');
  Result:=PT(FItems)[AIndex];
end;

function TDynArray.GetFirst: T;
begin
  Assert(Count > 0, 'Empty container');
  Result:=Items[0];
end;

function TDynArray.GetFirstPointer: PT;
begin
  Assert(Count > 0, 'Empty container');
  Result:=@PT(FItems)[0];
end;

function TDynArray.GetItemPointers(AIndex: Integer): PT;
begin
  Assert((AIndex >= 0) and (AIndex < Count), 'Invalid index');
  Result:=@PT(FItems)[AIndex];
end;

function TDynArray.GetLast: T;
begin
  Assert(Count > 0, 'Empty container');
  Result:=Items[Count - 1];
end;

function TDynArray.GetLastPointer: PT;
begin
  Assert(Count > 0, 'Empty container');
  Result:=@PT(FItems)[Count - 1];
end;

function TDynArray.GetUsedBytes: Int64;
begin
  Result:=Int64(SizeOf(T))*Int64(FCapacity);
end;

procedure TDynArray.SetCapacity(AValue: Integer);
begin
  if FCapacity=AValue then Exit;
  if AValue < FCount then AValue:=FCount;
  ReAllocMem(FItems, AValue*SizeOf(T));
  FCapacity:=AValue;
end;

procedure TDynArray.SetCount(AValue: Integer);
begin
  if FCount=AValue then Exit;
  FCount:=AValue;
  if FCount > FCapacity then
    if Capacity=0 then
      SetCapacity(8)
    else if Capacity*SizeOf(T) < 16*1024*1024 then
      SetCapacity(FCapacity*2)
    else if SizeOf(T) < 64 then
      {%H-}SetCapacity(FCount + 256)
    else
      {%H-}SetCapacity(FCount + 16);
end;

procedure TDynArray.SetItems(AIndex: Integer; AValue: T);
begin
  Assert((AIndex >= 0) and (AIndex < Count), 'Invalid index');
  PT(FItems)[AIndex]:=AValue;
end;

function TDynArray.Partition(L, H: Integer; ASorter: TSorter; AData: Pointer): Integer;
var
  Pivot: T;
  J: Integer;
begin
  Pivot:=GetItems(H);
  Result:=L;
  for J:=L to H - 1 do
    if ASorter(GetItems(J), Pivot, AData) < 0 then begin
      if Result <> J then Swap(Result, J);
      Inc(Result);
    end;
  Swap(Result, H);
end;

procedure TDynArray.QuickSort(L, H: Integer; ASorter: TSorter; AData: Pointer);
var
  P: Integer;
begin
  if L < H then begin
    P:=Partition(L, H, ASorter, AData);
    QuickSort(L, P - 1, ASorter, AData);
    QuickSort(P + 1, H, ASorter, AData);
  end;
end;

function TDynArray.GetRawData: PT;
begin
  Result:=PT(FItems);
end;

procedure TDynArray.Init;
begin
  FItems:=nil;
  FCount:=0;
  FCapacity:=0;
end;

procedure TDynArray.Done;
begin
  Clear;
end;

procedure TDynArray.EnsureCapacity(ACapacity: Integer);
begin
  if FCapacity < ACapacity then SetCapacity(ACapacity);
end;

procedure TDynArray.Add(Item: T);
begin
  Insert(Count, Item);
end;

function TDynArray.AddNew: PT;
begin
  SetCount(Count + 1);
  Result:=LastPointer;
end;

procedure TDynArray.Insert(Index: Integer; Item: T);
//var
//  I: Integer;
begin
  Assert((Index >= 0) and (Index <= Count), 'Invalid insertion index');
  Count:=Count + 1;
  Move(PT(FItems)[Index], PT(FItems)[Index + 1], (Index - Count - 1)*SizeOf(T));
  //for I:=Count - 1 downto Index + 1 do
  //  PT(FItems)[I]:=PT(FItems)[I - 1];
  PT(FItems)[Index]:=Item;
end;

function TDynArray.IndexOf(Item: T): Integer;
var
  I: Integer;
begin
  for I:=0 to Count - 1 do
    if Item=Items[I] then
      Exit(I);
  Result:=-1;
end;

function TDynArray.Contains(Item: T): Boolean;
begin
  Result:=IndexOf(Item) <> -1;
end;

procedure TDynArray.RemoveAt(Index: Integer);
var
  I: Integer;
begin
  Assert((Index >= 0) and (Index < Count), 'Invalid index');
  for I:=Index to Count - 2 do Items[I]:=Items[I + 1];
  Count:=Count - 1;
end;

procedure TDynArray.Remove(Item: T);
var
  Index: Integer;
begin
  Index:=IndexOf(Item);
  if Index <> -1 then RemoveAt(Index);
end;

procedure TDynArray.RemoveAll(Item: T);
var
  I: Integer;
begin
  for I:=Count - 1 downto 0 do
    if Items[I]=Item then RemoveAt(I);
end;

procedure TDynArray.RemoveViaSwapAt(Index: Integer);
begin
  Assert((Index >= 0) and (Index < Count), 'Invalid index');
  if Index < Count - 1 then Swap(Index, Count - 1);
  Dec(FCount);
end;

procedure TDynArray.RemoveViaSwap(Item: T);
var
  Index: Integer;
begin
  Index:=IndexOf(Item);
  if Index <> -1 then RemoveViaSwapAt(Index);
end;

procedure TDynArray.Clear;
begin
  FreeMem(FItems);
  FItems:=nil;
  FCount:=0;
  FCapacity:=0;
end;

procedure TDynArray.Reset;
begin
  Count:=0;
end;

procedure TDynArray.Swap(Index1, Index2: Integer);
var
  Tmp: T;
begin
  Assert((Index1 >= 0) and (Index1 < Count), 'Invalid first index');
  Assert((Index2 >= 0) and (Index2 < Count), 'Invalid second index');
  Tmp:=PT(FItems)[Index1];
  PT(FItems)[Index1]:=PT(FItems)[Index2];
  PT(FItems)[Index2]:=Tmp;
end;

function TDynArray.Pop: T;
begin
  Assert(Count > 0, 'Tried to pop from an empty array');
  Result:=PT(FItems)[Count - 1];
  Dec(FCount);
end;

procedure TDynArray.PopAndIgnore;
begin
  Assert(Count > 0, 'Tried to pop from an empty array');
  Dec(FCount);
end;

procedure TDynArray.Pack;
begin
  Capacity:=Count;
end;

procedure TDynArray.Sort(ASorter: TSorter; AData: Pointer);
begin
  if Count > 1 then QuickSort(0, Count - 1, ASorter, AData);
end;

{class operator TDynArray.=(const A, B: specialize TDynArray<T>): Boolean;
type
  PT = ^T;
var
  I: Integer;
begin
  if A.Count <> B.Count then Exit(False);
  for I:=0 to A.Count - 1 do
    if PT(A.FItems)[I] <> PT(B.FItems)[I] then Exit(False);
  Result:=True;
end;}

end.


