unit WingedEdgeMeshes;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Maths, Meshes, FGL;

type
  TWEMeshEdgeSide = (esLeft, esRight);
  TWEMeshVertex = class;
  TWEMeshEdge = class;
  TWEMeshFace = class;
  TWEMesh = class;
  TWEMeshVertexList = specialize TFPGObjectList<TWEMeshVertex>;
  TWEMeshEdgeList = specialize TFPGObjectList<TWEMeshEdge>;
  TWEMeshFaceList = specialize TFPGObjectList<TWEMeshFace>;
  TWEMeshVertexEvent = procedure(Sender: TObject; AVertex: TWEMeshVertex) of object;
  TWEMeshEdgeEvent = procedure(Sender: TObject; AEdge: TWEMeshEdge) of object;
  TWEMeshFaceEvent = procedure(Sender: TObject; AFace: TWEMeshFace) of object;

  { TWEMeshVertex }

  TWEMeshVertex = class
  private
    FNormal: TVector;
    FX: Double;
    FY: Double;
    FZ: Double;
    FMesh: TWEMesh;
    FEdge: TWEMeshEdge;
    FUserData: IntPtr;
    function GetAsVector: TVector; inline;
    procedure SetAsVector(const AValue: TVector); inline;
  public
    procedure Assign(AValue: TWEMeshVertex);
    procedure UpdateNormals;
    procedure UpdateNormal;
    procedure GetEdges(EdgeList: TWEMeshEdgeList);
    procedure GetFaces(FaceList: TWEMeshFaceList);
    procedure CycleEdge;
    function Connect(Target: TWEMeshVertex): TWEMeshEdge;
    function GetConnection(Target: TWEMeshVertex): TWEMeshEdge;
    function Dissolve: Boolean;
    procedure Extrude(const Offset: TVector; BaseUnits: Double);
    procedure Move(const Offset: TVector);
    property X: Double read FX write FX;
    property Y: Double read FY write FY;
    property Z: Double read FZ write FZ;
    property AsVector: TVector read GetAsVector write SetAsVector;
    property Normal: TVector read FNormal;
    property Mesh: TWEMesh read FMesh;
    property Edge: TWEMeshEdge read FEdge;
    property UserData: IntPtr read FUserData write FUserData;
  end;

  { TWEMeshEdge }

  TWEMeshEdge = class
  private
    FFirstVertex: TWEMeshVertex;
    FLeftSideFace: TWEMeshFace;
    FLeftSideNextEdge: TWEMeshEdge;
    FLeftSidePreviousEdge: TWEMeshEdge;
    FMesh: TWEMesh;
    FNormal: TVector;
    FRightSideFace: TWEMeshFace;
    FRightSideNextEdge: TWEMeshEdge;
    FRightSidePreviousEdge: TWEMeshEdge;
    FSecondVertex: TWEMeshVertex;
    FUserData: IntPtr;
    function GetDirection: TVector; inline;
    function GetLength: Double; inline;
  public
    procedure UpdateNormals;
    procedure UpdateNormal;
    function CutAt(const Position: TVector): TWEMeshEdge;
    procedure Cut(Pieces: Integer);
    function Dissolve(ConvexOnly: Boolean): Boolean;
    procedure Move(const Offset: TVector);
    property Mesh: TWEMesh read FMesh;
    property Normal: TVector read FNormal;
    property FirstVertex: TWEMeshVertex read FFirstVertex;
    property SecondVertex: TWEMeshVertex read FSecondVertex;
    property Direction: TVector read GetDirection;
    property Length: Double read GetLength;
    property LeftSideFace: TWEMeshFace read FLeftSideFace;
    property LeftSidePreviousEdge: TWEMeshEdge read FLeftSidePreviousEdge;
    property LeftSideNextEdge: TWEMeshEdge read FLeftSideNextEdge;
    property RightSideFace: TWEMeshFace read FRightSideFace;
    property RightSidePreviousEdge: TWEMeshEdge read FRightSidePreviousEdge;
    property RightSideNextEdge: TWEMeshEdge read FRightSideNextEdge;
    property UserData: IntPtr read FUserData write FUserData;
  end;

  { TWEMeshFace }

  TWEMeshFace = class
  private
    FEdge: TWEMeshEdge;
    FMesh: TWEMesh;
    FNormal: TVector;
    FUserData: IntPtr;
  public
    procedure GetEdges(EdgeList: TWEMeshEdgeList);
    procedure GetVertices(VertexList: TWEMeshVertexList);
    function CalcPlane: TPlane;
    procedure UpdateNormal;
    procedure Extrude(const Offset: TVector);
    procedure Inset(Units: Double);
    procedure Move(const Offset: TVector);
    property Mesh: TWEMesh read FMesh;
    property Normal: TVector read FNormal;
    property Edge: TWEMeshEdge read FEdge;
    property UserData: IntPtr read FUserData write FUserData;
  end;

  { TWEMesh }

  TWEMesh = class
  private
    FVertices: TWEMeshVertexList;
    FEdges: TWEMeshEdgeList;
    FFaces: TWEMeshFaceList;
    function GetEdgeCount: Integer; inline;
    function GetEdges(AIndex: Integer): TWEMeshEdge; inline;
    function GetFaceCount: Integer; inline;
    function GetFaces(AIndex: Integer): TWEMeshFace; inline;
    function GetVertexCount: Integer; inline;
    function GetVertices(AIndex: Integer): TWEMeshVertex; inline;
    procedure SetVertices(AIndex: Integer; AValue: TWEMeshVertex); inline;
  public
    constructor Create;
    destructor Destroy; override;
    function IndexOfVertex(AVertex: TWEMeshVertex): Integer; inline;
    function IndexOfEdge(AEdge: TWEMeshEdge): Integer; inline;
    function IndexOfFace(AFace: TWEMeshFace): Integer; inline;
    procedure Clear;
    procedure AssignPolygonMesh(AMesh: TMesh);
    procedure UpdateNormals;
    procedure DissolveCoplanarFaceEdges;
    function VertexAt(const Position: TVector): TWEMeshVertex;
    function VertexIndexAt(const Position: TVector): Integer;
    property Vertices[AIndex: Integer]: TWEMeshVertex read GetVertices write SetVertices;
    property VertexCount: Integer read GetVertexCount;
    property Edges[AIndex: Integer]: TWEMeshEdge read GetEdges;
    property EdgeCount: Integer read GetEdgeCount;
    property Faces[AIndex: Integer]: TWEMeshFace read GetFaces;
    property FaceCount: Integer read GetFaceCount;
  end;

implementation


{ TWEMeshFace }

procedure TWEMeshFace.GetEdges(EdgeList: TWEMeshEdgeList);
var
  E: TWEMeshEdge;
begin
  E:=Edge;
  repeat
    if E.LeftSideFace=Self then begin
      EdgeList.Add(E);
      E:=E.LeftSideNextEdge;
    end else begin
      EdgeList.Add(E);
      E:=E.RightSideNextEdge;
    end;
  until E=Edge;
end;

procedure TWEMeshFace.GetVertices(VertexList: TWEMeshVertexList);
var
  E: TWEMeshEdge;
begin
  E:=Edge;
  repeat
    if VertexList.Count=200 then Break;
    if E.LeftSideFace=Self then begin
      VertexList.Add(E.FirstVertex);
      E:=E.LeftSideNextEdge;
    end else begin
      VertexList.Add(E.SecondVertex);
      E:=E.RightSideNextEdge;
    end;
  until E=Edge;
end;

function TWEMeshFace.CalcPlane: TPlane;
begin
  UpdateNormal;
  if Edge.LeftSideFace=Self then
    Result.FromPointAndNormal(Edge.FirstVertex.AsVector, Normal)
  else
    Result.FromPointAndNormal(Edge.SecondVertex.AsVector, Normal);
end;

procedure TWEMeshFace.UpdateNormal;
var
  Vertices: TWEMeshVertexList;
  I: Integer;
begin
  Vertices:=TWEMeshVertexList.Create(False);
  GetVertices(Vertices);
  FNormal.Zero;
  for I:=2 to Vertices.Count - 1 do
    FNormal.Add(TriangleNormal(Vertices[0].AsVector,
                               Vertices[I - 1].AsVector,
                               Vertices[I].AsVector));
  FNormal.Normalize;
  Vertices.Free;
{
  E:=Edge;
  if E.LeftSideFace=Self then begin
    A:=Vector(E.FirstVertex.X, E.FirstVertex.Y, E.FirstVertex.Z);
    E:=E.LeftSideNextEdge;
  end else begin
    A:=Vector(E.SecondVertex.X, E.SecondVertex.Y, E.SecondVertex.Z);
    E:=E.RightSideNextEdge;
  end;
  if E.LeftSideFace=Self then begin
    B:=Vector(E.FirstVertex.X, E.FirstVertex.Y, E.FirstVertex.Z);
    E:=E.LeftSideNextEdge;
  end else begin
    B:=Vector(E.SecondVertex.X, E.SecondVertex.Y, E.SecondVertex.Z);
    E:=E.RightSideNextEdge;
  end;
  if E.LeftSideFace=Self then begin
    C:=Vector(E.FirstVertex.X, E.FirstVertex.Y, E.FirstVertex.Z);
    E:=E.LeftSideNextEdge;
  end else begin
    C:=Vector(E.SecondVertex.X, E.SecondVertex.Y, E.SecondVertex.Z);
    E:=E.RightSideNextEdge;
  end;
  FNormal:=TriangleNormal(A, B, C);}
end;

procedure TWEMeshFace.Extrude(const Offset: TVector);
var
  Edges: TWEMeshEdgeList;
  Vertices, NewVertices: array of TWEMeshVertex;
  I, NextI, PrevI: Integer;
begin
  Edges:=TWEMeshEdgeList.Create(False);
  GetEdges(Edges);
  SetLength(NewVertices, Edges.Count);
  SetLength(Vertices, Edges.Count);
  for I:=0 to High(NewVertices) do begin
    NewVertices[I]:=TWEMeshVertex.Create;
    NewVertices[I].FMesh:=FMesh;
    if Edges[I].LeftSideFace=Self then
      Vertices[I]:=Edges[I].FirstVertex
    else
      Vertices[I]:=Edges[I].SecondVertex;
    NewVertices[I].AsVector:=Vertices[I].AsVector.Added(Offset);
    NewVertices[I].FEdge:=TWEMeshEdge.Create;
    NewVertices[I].FEdge.FMesh:=FMesh;
    NewVertices[I].FEdge.FFirstVertex:=NewVertices[I];
    NewVertices[I].FEdge.FLeftSideFace:=Self;
    FMesh.FVertices.Add(NewVertices[I]);
    FMesh.FEdges.Add(NewVertices[I].FEdge);
  end;
  for I:=0 to High(NewVertices) do begin
    NextI:=(I + 1) mod Length(NewVertices);
    PrevI:=(I + Length(NewVertices) - 1) mod Length(NewVertices);
    NewVertices[I].FEdge.FSecondVertex:=NewVertices[NextI].FEdge.FFirstVertex;
    NewVertices[I].FEdge.FLeftSideNextEdge:=NewVertices[NextI].FEdge;
    NewVertices[I].FEdge.FLeftSidePreviousEdge:=NewVertices[PrevI].FEdge;
  end;
  for I:=0 to High(NewVertices) do begin
    NewVertices[I].FEdge.FRightSideFace:=TWEMeshFace.Create;
    NewVertices[I].FEdge.FRightSideFace.FEdge:=NewVertices[I].FEdge;
    NewVertices[I].FEdge.FRightSideFace.FMesh:=Mesh;
    FMesh.FFaces.Add(NewVertices[I].FEdge.FRightSideFace);
  end;
  for I:=0 to High(NewVertices) do begin
    NewVertices[I].FEdge.FRightSideNextEdge:=TWEMeshEdge.Create;
    NewVertices[I].FEdge.FRightSideNextEdge.FMesh:=FMesh;
    NewVertices[I].FEdge.FRightSideNextEdge.FFirstVertex:=Vertices[I];
    NewVertices[I].FEdge.FRightSideNextEdge.FSecondVertex:=NewVertices[I];
    NewVertices[I].FEdge.FRightSideNextEdge.FRightSideFace:=NewVertices[I].FEdge.FRightSideFace;
    NewVertices[I].FEdge.FRightSideNextEdge.FRightSideNextEdge:=Edges[I];
    NewVertices[I].FEdge.FRightSideNextEdge.FRightSidePreviousEdge:=NewVertices[I].FEdge;
  end;
  for I:=0 to High(NewVertices) do begin
    NextI:=(I + 1) mod Length(NewVertices);
    PrevI:=(I + Length(NewVertices) - 1) mod Length(NewVertices);
    NewVertices[I].FEdge.FRightSideNextEdge.FLeftSideFace:=NewVertices[PrevI].FEdge.FRightSideFace;
    NewVertices[I].FEdge.FRightSideNextEdge.FLeftSideNextEdge:=NewVertices[PrevI].FEdge;
    NewVertices[I].FEdge.FRightSideNextEdge.FLeftSidePreviousEdge:=Edges[PrevI];
    NewVertices[I].FEdge.FRightSidePreviousEdge:=NewVertices[NextI].FEdge.FRightSideNextEdge;
  end;
  for I:=0 to High(NewVertices) do begin
    if Edges[I].LeftSideFace=Self then begin
      Edges[I].FLeftSideNextEdge:=NewVertices[I].FEdge.FRightSidePreviousEdge;
      Edges[I].FLeftSidePreviousEdge:=NewVertices[I].FEdge.FRightSideNextEdge;
      Edges[I].FLeftSideFace:=NewVertices[I].FEdge.FRightSideFace;
    end else begin
      Edges[I].FRightSideNextEdge:=NewVertices[I].FEdge.FRightSidePreviousEdge;
      Edges[I].FRightSidePreviousEdge:=NewVertices[I].FEdge.FRightSideNextEdge;
      Edges[I].FRightSideFace:=NewVertices[I].FEdge.FRightSideFace;
    end;
    if Edges[I]=Edge then FEdge:=NewVertices[I].Edge;

    FMesh.FEdges.Add(NewVertices[I].FEdge.FRightSideNextEdge);
  end;
  Edges.Free;
end;

procedure TWEMeshFace.Inset(Units: Double);
var
  Edges: TWEMeshEdgeList;
  Vertices: TWEMeshVertexList;
  Offsets: array of TVector;
  P: TPlane;
  I, PrevI: Integer;
  A, B, C, AB, CA, V: TVector;
begin
  Edges:=TWEMeshEdgeList.Create(False);
  GetEdges(Edges);
  SetLength(Offsets, Edges.Count);
  UpdateNormal;
  for I:=0 to Edges.Count - 1 do begin
    PrevI:=(I + Edges.Count - 1) mod Edges.Count;
    if Edges[I].LeftSideFace=Self then begin
      A:=Edges[I].FirstVertex.AsVector;
      B:=Edges[I].SecondVertex.AsVector;
    end else begin
      A:=Edges[I].SecondVertex.AsVector;
      B:=Edges[I].FirstVertex.AsVector;
    end;
    if Edges[PrevI].LeftSideFace=Self then
      C:=Edges[PrevI].FirstVertex.AsVector
    else
      C:=Edges[PrevI].SecondVertex.AsVector;

    AB:=Normal.Crossed(B.Subbed(A)).Normalized;
    CA:=Normal.Crossed(A.Subbed(C)).Normalized;
    V:=A;
    P:=CalcPlane;
    AB:=P.ProjectedNormal(AB);
    CA:=P.ProjectedNormal(CA);
    P.FromPointAndNormal(A.Added(AB.Scaled(Units)), AB);
    P.Project(V);
    P.FromPointAndNormal(A.Added(CA.Scaled(Units)), CA);
    P.Project(V);
    Offsets[I]:=V.Subbed(A);
  end;
  Edges.Free;
  Extrude(Vector(0, 0, 0));
  Vertices:=TWEMeshVertexList.Create(False);
  GetVertices(Vertices);
  for I:=0 to Vertices.Count - 1 do
    Vertices[I].Move(Offsets[I]);
  Vertices.Free;
end;

procedure TWEMeshFace.Move(const Offset: TVector);
var
  E: TWEMeshEdge;
begin
  E:=Edge;
  repeat
    if E.LeftSideFace=Self then begin
      E.Move(Offset);
      E:=E.LeftSideNextEdge;
    end else begin
      E.Move(Offset);
      E:=E.RightSideNextEdge;
    end;
  until E=Edge;
end;

{ TWEMeshEdge }

function TWEMeshEdge.GetDirection: TVector;
begin
  Result:=SecondVertex.AsVector.Subbed(FirstVertex.AsVector).Normalized;
end;

function TWEMeshEdge.GetLength: Double;
begin
  Result:=Distance(FirstVertex.AsVector, SecondVertex.AsVector);
end;

procedure TWEMeshEdge.UpdateNormals;
begin
  LeftSideFace.UpdateNormal;
  RightSideFace.UpdateNormal;
  UpdateNormal;
end;

procedure TWEMeshEdge.UpdateNormal;
begin
  FNormal:=FLeftSideFace.FNormal;
  FNormal.Add(FRightSideFace.FNormal);
  FNormal.Normalize;
end;

function TWEMeshEdge.CutAt(const Position: TVector): TWEMeshEdge;
var
  NewVertex: TWEMeshVertex;
begin
  NewVertex:=TWEMeshVertex.Create;
  NewVertex.FX:=Position.x;
  NewVertex.FY:=Position.y;
  NewVertex.FZ:=Position.z;
  NewVertex.FNormal:=FNormal;
  NewVertex.FMesh:=FMesh;
  Result:=TWEMeshEdge.Create;
  NewVertex.FEdge:=Result;
  Result.FFirstVertex:=NewVertex;
  Result.FSecondVertex:=FSecondVertex;
  Result.FMesh:=FMesh;
  Result.FNormal:=FNormal;
  Result.FLeftSideFace:=FLeftSideFace;
  Result.FRightSideFace:=FRightSideFace;

  Result.FLeftSideNextEdge:=FLeftSideNextEdge;
  Result.FLeftSidePreviousEdge:=Self;
  if FLeftSideNextEdge.FLeftSidePreviousEdge=Self then
    FLeftSideNextEdge.FLeftSidePreviousEdge:=Result
  else
    FLeftSideNextEdge.FRightSidePreviousEdge:=Result;
  FLeftSideNextEdge:=Result;

  Result.FRightSideNextEdge:=Self;
  Result.FRightSidePreviousEdge:=FRightSidePreviousEdge;
  if FRightSidePreviousEdge.FLeftSideNextEdge=Self then
    FRightSidePreviousEdge.FLeftSideNextEdge:=Result
  else
    FRightSidePreviousEdge.FRightSideNextEdge:=Result;
  FRightSidePreviousEdge:=Result;

  FSecondVertex:=NewVertex;
  FMesh.FVertices.Add(NewVertex);
  FMesh.FEdges.Add(Result);
end;

procedure TWEMeshEdge.Cut(Pieces: Integer);
var
  E: TWEMeshEdge;
  Delta: TVector;
  I: Integer;
begin
  if Pieces < 2 then Exit;
  E:=Self;
  Delta:=SecondVertex.AsVector.Subbed(FirstVertex.AsVector).Scaled(1/Pieces);
  for I:=1 to Pieces - 1 do
    E:=E.CutAt(FirstVertex.AsVector.Added(Delta.Scaled(I)));
end;

function TWEMeshEdge.Dissolve(ConvexOnly: Boolean): Boolean;
var
  Edges: TWEMeshEdgeList;
  I: Integer;
  F: TFace;
  E: TWEMeshEdge;
begin
  // Check if the new face would be convex
  if ConvexOnly then begin
    E:=Self.LeftSideNextEdge;
    repeat
      if E.LeftSideFace=LeftSideFace then begin
        F.AddVertex(E.FirstVertex.AsVector);
        E:=E.LeftSideNextEdge;
      end else begin
        F.AddVertex(E.SecondVertex.AsVector);
        E:=E.RightSideNextEdge;
      end;
    until E=Self;
    E:=Self.RightSideNextEdge;
    repeat
      if E.LeftSideFace=RightSideFace then begin
        F.AddVertex(E.FirstVertex.AsVector);
        E:=E.LeftSideNextEdge;
      end else begin
        F.AddVertex(E.SecondVertex.AsVector);
        E:=E.RightSideNextEdge;
      end;
    until E=Self;
    F.MergeZeroLengthEdges;
    if not F.IsConvex then Exit(False);
  end;
  // Merge face edges
  Edges:=TWEMeshEdgeList.Create(False);
  if FirstVertex.Edge=Self then begin
    FirstVertex.GetEdges(Edges);
    for I:=0 to Edges.Count - 1 do
      if Edges[I] <> FirstVertex.Edge then begin
        FirstVertex.FEdge:=Edges[I];
        Break;
      end;
    Edges.Count:=0;
  end;
  if SecondVertex.Edge=Self then begin
    SecondVertex.GetEdges(Edges);
    for I:=0 to Edges.Count - 1 do
      if Edges[I] <> SecondVertex.Edge then begin
        SecondVertex.FEdge:=Edges[I];
        Break;
      end;
    Edges.Count:=0;
  end;
  LeftSideFace.GetEdges(Edges);
  for I:=0 to Edges.Count - 1 do begin
    if Edges[I]=Self then Continue;
    if Edges[I].LeftSideFace=LeftSideFace then
      Edges[I].FLeftSideFace:=RightSideFace
    else
      Edges[I].FRightSideFace:=RightSideFace;
  end;
  if LeftSidePreviousEdge.FLeftSideNextEdge=Self then
    LeftSidePreviousEdge.FLeftSideNextEdge:=FRightSideNextEdge
  else
    LeftSidePreviousEdge.FRightSideNextEdge:=FRightSideNextEdge;
  if LeftSideNextEdge.FLeftSidePreviousEdge=Self then
    LeftSideNextEdge.FLeftSidePreviousEdge:=FRightSidePreviousEdge
  else
    LeftSideNextEdge.FRightSidePreviousEdge:=FRightSidePreviousEdge;
  if RightSidePreviousEdge.FLeftSideNextEdge=Self then
    RightSidePreviousEdge.FLeftSideNextEdge:=FLeftSideNextEdge
  else
    RightSidePreviousEdge.FRightSideNextEdge:=FLeftSideNextEdge;
  if RightSideNextEdge.FLeftSidePreviousEdge=Self then
    RightSideNextEdge.FLeftSidePreviousEdge:=FLeftSidePreviousEdge
  else
    RightSideNextEdge.FRightSidePreviousEdge:=FRightSidePreviousEdge;
  if RightSideFace.Edge=Self then RightSideFace.FEdge:=RightSideNextEdge;
  Mesh.FFaces.Remove(LeftSideFace);
  Mesh.FEdges.Remove(Self);
  FreeAndNil(Edges);
  FLeftSideFace.Free;
  Free;
  Result:=True;
end;

procedure TWEMeshEdge.Move(const Offset: TVector);
begin
  FirstVertex.Move(Offset);
  SecondVertex.Move(Offset);
end;


{ TWEMeshVertex }

function TWEMeshVertex.GetAsVector: TVector;
begin
  Result.x:=FX;
  Result.y:=FY;
  Result.z:=FZ;
end;

procedure TWEMeshVertex.SetAsVector(const AValue: TVector);
begin
  FX:=AValue.x;
  FY:=AValue.y;
  FZ:=AValue.z;
end;

procedure TWEMeshVertex.Assign(AValue: TWEMeshVertex);
begin
  FX:=AValue.FX;
  FY:=AValue.FY;
  FZ:=AValue.FZ;
end;

procedure TWEMeshVertex.UpdateNormals;
var
  E: TWEMeshEdge;
begin
  FNormal.Zero;
  E:=Edge;
  repeat
    E.UpdateNormals;
    FNormal.Add(E.Normal);
    if E.FirstVertex=Self then
      E:=E.RightSideNextEdge
    else
      E:=E.LeftSideNextEdge;
  until E=Edge;
  FNormal.Normalize;
end;

procedure TWEMeshVertex.UpdateNormal;
var
  E: TWEMeshEdge;
begin
  FNormal.Zero;
  E:=Edge;
  repeat
    FNormal.Add(E.Normal);
    if E.FirstVertex=Self then
      E:=E.RightSideNextEdge
    else
      E:=E.LeftSideNextEdge;
  until E=Edge;
  FNormal.Normalize;
end;

procedure TWEMeshVertex.GetEdges(EdgeList: TWEMeshEdgeList);
var
  E: TWEMeshEdge;
begin
  E:=Edge;
  repeat
    EdgeList.Add(E);
    if E.FirstVertex=Self then
      E:=E.RightSideNextEdge
    else
      E:=E.LeftSideNextEdge;
  until E=Edge;
end;

procedure TWEMeshVertex.GetFaces(FaceList: TWEMeshFaceList);
var
  E: TWEMeshEdge;
begin
  E:=Edge;
  repeat
    if E.FirstVertex=Self then begin
      FaceList.Add(E.LeftSideFace);
      E:=E.RightSideNextEdge
    end else begin
      FaceList.Add(E.RightSideFace);
      E:=E.LeftSideNextEdge;
    end;
  until E=Edge;
end;

procedure TWEMeshVertex.CycleEdge;
begin
  if Edge.FirstVertex=Self then
    FEdge:=Edge.RightSideNextEdge
  else
    FEdge:=Edge.LeftSideNextEdge;
end;

function TWEMeshVertex.Connect(Target: TWEMeshVertex): TWEMeshEdge;
var
  Face, NewFace: TWEMeshFace;
  Edges: TWEMeshEdgeList;
  I: Integer;
  E, NextE: TWEMeshEdge;
  Adding: Boolean;

  procedure FindCommonFace;
  var
    EL1, EL2: TWEMeshEdgeList;
    I, J: Integer;
  begin
    EL1:=TWEMeshEdgeList.Create(False);
    EL2:=TWEMeshEdgeList.Create(False);
    GetEdges(EL1);
    Target.GetEdges(EL2);
    Face:=nil;
    for I:=0 to EL1.Count - 1 do begin
      for J:=0 to EL2.Count - 1 do begin
        if (EL1[I].LeftSideFace=EL2[J].LeftSideFace) or
           (EL1[I].LeftSideFace=EL2[J].RightSideFace) then begin
          Face:=EL1[I].LeftSideFace;
          Break;
        end else if (EL1[I].RightSideFace=EL2[J].LeftSideFace) or
                    (EL1[I].RightSideFace=EL2[J].RightSideFace) then begin
          Face:=EL1[I].RightSideFace;
          Break;
        end;
      end;
      if Assigned(Face) then Break;
    end;
    EL2.Free;
    EL1.Free;
  end;

begin
  FindCommonFace;
  if not Assigned(Face) then Exit(nil);
  E:=Face.Edge;
  Result:=nil;
  Adding:=False;
  Edges:=TWEMeshEdgeList.Create(False);
  repeat
    if E.LeftSideFace=Face then begin
      if E.FirstVertex=Self then Adding:=True;
      NextE:=E.LeftSideNextEdge;
    end else begin
      if E.SecondVertex=Self then Adding:=True;
      NextE:=E.RightSideNextEdge;
    end;
    if Adding then begin
      repeat
        Edges.Add(E);
        if E.LeftSideFace=Face then begin
          if E.SecondVertex=Target then Break;
          NextE:=E.LeftSideNextEdge;
        end else begin
          if E.FirstVertex=Target then Break;
          NextE:=E.RightSideNextEdge;
        end;
        E:=NextE;
      until False;
      Break;
    end;
    E:=NextE;
  until E=Face.Edge;
  if Edges.Count > 1 then begin
    NewFace:=TWEMeshFace.Create;
    NewFace.FMesh:=Mesh;
    Result:=TWEMeshEdge.Create;
    NewFace.FEdge:=Result;
    Result.FMesh:=Mesh;
    Result.FNormal:=Face.Normal;
    Result.FFirstVertex:=Target;
    Result.FSecondVertex:=Self;
    Result.FLeftSideFace:=NewFace;
    Result.FRightSideFace:=Face;
    Result.FLeftSidePreviousEdge:=Edges.Last;
    Result.FLeftSideNextEdge:=Edges.First;
    if Edges.First.LeftSideFace=Face then begin
      Result.FRightSidePreviousEdge:=Edges.First.LeftSidePreviousEdge;
      Edges.First.FLeftSidePreviousEdge:=Result
    end else begin
      Result.FRightSidePreviousEdge:=Edges.First.RightSidePreviousEdge;
      Edges.First.FRightSidePreviousEdge:=Result;
    end;
    if Result.FRightSidePreviousEdge.LeftSideFace=Face then
      Result.FRightSidePreviousEdge.FLeftSideNextEdge:=Result
    else
      Result.FRightSidePreviousEdge.FRightSideNextEdge:=Result;
    if Edges.Last.LeftSideFace=Face then begin
      Result.FRightSideNextEdge:=Edges.Last.FLeftSideNextEdge;
      Edges.Last.FLeftSideNextEdge:=Result
    end else begin
      Result.FRightSideNextEdge:=Edges.Last.FRightSideNextEdge;
      Edges.Last.FRightSideNextEdge:=Result;
    end;
    if Result.FRightSideNextEdge.LeftSideFace=Face then
      Result.FRightSideNextEdge.FLeftSidePreviousEdge:=Result
    else
      Result.FRightSideNextEdge.FRightSidePreviousEdge:=Result;
    for I:=0 to Edges.Count - 1 do begin
      if Face.Edge=Edges[I] then
        Face.FEdge:=Result.FRightSidePreviousEdge;
      if Edges[I].LeftSideFace=Face then
        Edges[I].FLeftSideFace:=NewFace
      else
        Edges[I].FRightSideFace:=NewFace;
    end;
    Mesh.FFaces.Add(NewFace);
    Mesh.FEdges.Add(Result);
  end;
  Edges.Free;
end;

function TWEMeshVertex.GetConnection(Target: TWEMeshVertex): TWEMeshEdge;
var
  E: TWEMeshEdge;
begin
  E:=Edge;
  repeat
    if (E.FirstVertex=Target) or (E.SecondVertex=Target) then Exit(E);
    if E.FirstVertex=Self then
      E:=E.RightSideNextEdge
    else
      E:=E.LeftSideNextEdge;
  until E=Edge;
  Result:=nil;
end;

function TWEMeshVertex.Dissolve: Boolean;
var
  Edges: TWEMeshEdgeList;
  Faces: TWEMeshFaceList;
  Other: TWEMeshEdge;
  A, B: TWEMeshVertex;
  I, J: Integer;
  NewEdges: array of TWEMeshEdge;
  NewFace: TWEMeshFace;
begin
  Edges:=TWEMeshEdgeList.Create(False);
  GetEdges(Edges);
  if Edges.Count < 3 then begin
    Edges.Free;
    Exit(False);
  end;
  Faces:=TWEMeshFaceList.Create(False);
  SetLength(NewEdges, Edges.Count);
  for I:=0 to Edges.Count - 1 do begin
    Other:=Edges[(I + 1) mod Edges.Count];
    if Edges[I].FirstVertex=Self then
      A:=Edges[I].SecondVertex
    else
      A:=Edges[I].FirstVertex;
    if Other.FirstVertex=Self then
      B:=Other.SecondVertex
    else
      B:=Other.FirstVertex;
    NewEdges[I]:=A.GetConnection(B);
    if not Assigned(NewEdges[I]) then NewEdges[I]:=A.Connect(B);
  end;
  GetFaces(Faces);
  NewFace:=TWEMeshFace.Create;
  NewFace.FMesh:=FMesh;
  NewFace.FEdge:=NewEdges[0];
  for I:=0 to High(NewEdges) do begin
    for J:=0 to Faces.Count - 1 do begin
      if NewEdges[I].LeftSideFace=Faces[J] then begin
        NewEdges[I].FLeftSideFace:=NewFace;
        Break;
      end;
      if NewEdges[I].RightSideFace=Faces[J] then begin
        NewEdges[I].FRightSideFace:=NewFace;
        Break;
      end;
    end;
    if NewEdges[I].LeftSideFace=NewFace then begin
      NewEdges[I].FLeftSidePreviousEdge:=NewEdges[(I + 1) mod Length(NewEdges)];
      NewEdges[I].FLeftSideNextEdge:=NewEdges[(I + Length(NewEdges) - 1) mod Length(NewEdges)];
    end else begin
      NewEdges[I].FRightSidePreviousEdge:=NewEdges[(I + 1) mod Length(NewEdges)];
      NewEdges[I].FRightSideNextEdge:=NewEdges[(I + Length(NewEdges) - 1) mod Length(NewEdges)];
    end;
  end;
  FMesh.FFaces.Add(NewFace);
  for I:=0 to Faces.Count - 1 do begin
    FMesh.FFaces.Remove(Faces[I]);
    Faces[I].Free;
  end;
  for I:=0 to Edges.Count - 1 do begin
    if Edges[I].FirstVertex.Edge=Edges[I] then
      Edges[I].FirstVertex.CycleEdge;
    if Edges[I].SecondVertex.Edge=Edges[I] then
      Edges[I].SecondVertex.CycleEdge;
    FMesh.FEdges.Remove(Edges[I]);
    Edges[I].Free;
  end;
  FMesh.FVertices.Remove(Self);
  Faces.Free;
  Edges.Free;
  Free;
  Result:=True;
end;

procedure TWEMeshVertex.Extrude(const Offset: TVector; BaseUnits: Double);
var
  Edges: TWEMeshEdgeList;
  NewVertices: array of TWEMeshVertex;
  I: Integer;
begin
  Edges:=TWEMeshEdgeList.Create(False);
  GetEdges(Edges);
  if Edges.Count < 3 then begin
    Edges.Free;
    Exit;
  end;
  SetLength(NewVertices, Edges.Count);
  for I:=0 to Edges.Count - 1 do begin
    if Edges[I].FirstVertex=Self then
      NewVertices[I]:=Edges[I].CutAt(Edges[I].FirstVertex.AsVector.Added(Edges[I].Direction.Scaled(BaseUnits))).FirstVertex
    else
      NewVertices[I]:=Edges[I].CutAt(Edges[I].SecondVertex.AsVector.Added(Edges[I].Direction.Inverted.Scaled(BaseUnits))).FirstVertex
  end;
  for I:=0 to High(NewVertices) do
    NewVertices[I].Connect(NewVertices[(I + 1) mod Length(NewVertices)]);
  Move(Offset);
  Edges.Free;
end;

procedure TWEMeshVertex.Move(const Offset: TVector);
begin
  FX += Offset.x;
  FY += Offset.y;
  FZ += Offset.z;
end;

{ TWEMesh }

constructor TWEMesh.Create;
begin
  FVertices:=TWEMeshVertexList.Create(False);
  FEdges:=TWEMeshEdgeList.Create(False);
  FFaces:=TWEMeshFaceList.Create(False);
end;

destructor TWEMesh.Destroy;
begin
  Clear;
  FreeAndNil(FFaces);
  FreeAndNil(FEdges);
  FreeAndNil(FVertices);
  inherited Destroy;
end;

function TWEMesh.IndexOfVertex(AVertex: TWEMeshVertex): Integer;
begin
  Result:=FVertices.IndexOf(AVertex);
end;

function TWEMesh.IndexOfEdge(AEdge: TWEMeshEdge): Integer;
begin
  Result:=FEdges.IndexOf(AEdge);
end;

function TWEMesh.IndexOfFace(AFace: TWEMeshFace): Integer;
begin
  Result:=FFaces.IndexOf(AFace);
end;

procedure TWEMesh.Clear;
begin
  while FFaces.Count > 0 do begin
    FFaces.Last.Free;
    FFaces.Remove(FFaces.Last);
  end;
  while FEdges.Count > 0 do begin
    FEdges.Last.Free;
    FEdges.Remove(FEdges.Last);
  end;
  while FVertices.Count > 0 do begin
    FVertices.Last.Free;
    FVertices.Remove(FVertices.Last);
  end;
  FFaces.Clear;
  FEdges.Clear;
  FVertices.Clear;
end;

procedure TWEMesh.AssignPolygonMesh(AMesh: TMesh);

  function FindOrAddVertex(const V: TMeshVertex): TWEMeshVertex;
  begin
    Result:=VertexAt(V);
    if not Assigned(Result) then begin
      FVertices.Add(TWEMeshVertex.Create);
      Result:=FVertices.Last;
      Result.FMesh:=Self;
      Result.FX:=V.x;
      Result.FY:=V.y;
      Result.FZ:=V.z;
    end;
  end;

  function FindLeftOrCreateRight(const A, B: TMeshVertex; Face: TWEMeshFace; out WasLeft: Boolean): TWEMeshEdge;
  var
    I: Integer;
  begin
    for I:=0 to FEdges.Count - 1 do with FEdges[I] do begin
      if (FSecondVertex.X=A.x) and
         (FSecondVertex.Y=A.y) and
         (FSecondVertex.Z=A.z) and
         (FFirstVertex.X=B.x) and
         (FFirstVertex.Y=B.y) and
         (FFirstVertex.Z=B.z) and
         (not Assigned(FRightSideFace)) then begin
        WasLeft:=True;
        FEdges[I].FRightSideFace:=Face;
        Exit(FEdges[I]);
      end;
    end;
    WasLeft:=False;
    Result:=TWEMeshEdge.Create;
    Result.FMesh:=Self;
    Result.FFirstVertex:=VertexAt(A);
    Result.FSecondVertex:=VertexAt(B);
    Result.FLeftSideFace:=Face;
    Result.FFirstVertex.FEdge:=Result;
  end;

var
  A, B, C: TMeshVertex;
  WEF: TWEMeshFace;
  E1, E2, E3: TWEMeshEdge;
  WasLeft1, WasLeft2, WasLeft3: Boolean;
  Index: Integer;
begin
  Clear;
  // Add indices
  Index:=0;
  while Index < AMesh.IndexCount do begin
    A:=AMesh.Vertices[Index]; Inc(Index); if Index >= AMesh.IndexCount then Break;
    B:=AMesh.Vertices[Index]; Inc(Index); if Index >= AMesh.IndexCount then Break;
    C:=AMesh.Vertices[Index]; Inc(Index);
    FindOrAddVertex(A);
    FindOrAddVertex(B);
    FindOrAddVertex(C);
  end;
  // Add edges and faces
  Index:=0;
  while Index < AMesh.IndexCount do begin
    A:=AMesh.Vertices[Index]; Inc(Index); if Index >= AMesh.IndexCount then Break;
    B:=AMesh.Vertices[Index]; Inc(Index); if Index >= AMesh.IndexCount then Break;
    C:=AMesh.Vertices[Index]; Inc(Index);
    WEF:=TWEMeshFace.Create;
    WEF.FMesh:=Self;
    FFaces.Add(WEF);
    E1:=FindLeftOrCreateRight(A, B, WEF, WasLeft1);
    E2:=FindLeftOrCreateRight(B, C, WEF, WasLeft2);
    E3:=FindLeftOrCreateRight(C, A, WEF, WasLeft3);
    if WasLeft1 then begin
      E1.FRightSidePreviousEdge:=E3;
      E1.FRightSideNextEdge:=E2;
    end else begin
      E1.FLeftSidePreviousEdge:=E3;
      E1.FLeftSideNextEdge:=E2;
      FEdges.Add(E1);
    end;
    if WasLeft2 then begin
      E2.FRightSidePreviousEdge:=E1;
      E2.FRightSideNextEdge:=E3;
    end else begin
      E2.FLeftSidePreviousEdge:=E1;
      E2.FLeftSideNextEdge:=E3;
      FEdges.Add(E2);
    end;
    if WasLeft3 then begin
      E3.FRightSidePreviousEdge:=E2;
      E3.FRightSideNextEdge:=E1;
    end else begin
      E3.FLeftSidePreviousEdge:=E2;
      E3.FLeftSideNextEdge:=E1;
      FEdges.Add(E3);
    end;
    WEF.FEdge:=E1;
  end;
  UpdateNormals;
end;

procedure TWEMesh.UpdateNormals;
var
  I: Integer;
begin
  for I:=0 to FaceCount - 1 do Faces[I].UpdateNormal;
  for I:=0 to EdgeCount - 1 do Edges[I].UpdateNormal;
  for I:=0 to VertexCount - 1 do Vertices[I].UpdateNormal;
end;

procedure TWEMesh.DissolveCoplanarFaceEdges;
var
  I: Integer;
begin
  I:=0;
  while I < EdgeCount do with Edges[I] do begin
    LeftSideFace.UpdateNormal;
    RightSideFace.UpdateNormal;
    if LeftSideFace.Normal.Dot(RightSideFace.Normal) > 1-EPSILON then begin
      if not Dissolve(True) then Inc(I);
    end else
      Inc(I);
  end;
end;

function TWEMesh.VertexAt(const Position: TVector): TWEMeshVertex;
var
  I: Integer;
begin
  for I:=0 to FVertices.Count - 1 do with FVertices[I] do
    if (Position.x=X) and
       (Position.y=Y) and
       (Position.z=Z) then Exit(FVertices[I]);
  Result:=nil;
end;

function TWEMesh.VertexIndexAt(const Position: TVector): Integer;
var
  I: Integer;
begin
  for I:=0 to FVertices.Count - 1 do with FVertices[I] do
    if (Position.x=X) and
       (Position.y=Y) and
       (Position.z=Z) then Exit(I);
  Result:=-1;
end;

function TWEMesh.GetEdgeCount: Integer;
begin
  Result:=FEdges.Count;
end;

function TWEMesh.GetEdges(AIndex: Integer): TWEMeshEdge;
begin
  Result:=FEdges[AIndex];
end;

function TWEMesh.GetFaceCount: Integer;
begin
  Result:=FFaces.Count;
end;

function TWEMesh.GetFaces(AIndex: Integer): TWEMeshFace;
begin
  Result:=FFaces[AIndex];
end;

function TWEMesh.GetVertexCount: Integer;
begin
  Result:=FVertices.Count;
end;

function TWEMesh.GetVertices(AIndex: Integer): TWEMeshVertex;
begin
  Result:=FVertices[AIndex];
end;

procedure TWEMesh.SetVertices(AIndex: Integer; AValue: TWEMeshVertex);
begin
  FVertices[AIndex].Assign(AValue);
end;

end.

