unit Viewports;

{$mode objfpc}{$H+}
{$interfaces corba}

interface

uses
  Classes, SysUtils, Forms, Controls, Graphics, Dialogs, OpenGLContext, FGL,
  Math, Maths, Textures, Cameras, GLTextDrawers, LCLType, Types, LMessages,
  Widgets3d;

// TODO: reimplement the OpenGL context stuff (create custom opengl context
// control) so that viewports can keep sharing resources after the first
// viewport is destroyed

type
  TViewport = class;
  TViewportSide = (vsCustom, vsLeft, vsRight, vsTop, vsBottom, vsFront, vsBack);
  TViewportCameraMotion = (vcmNone, vcmFirstPerson);
  TViewportEvent = procedure(Sender: TObject; Viewport: TViewport) of object;
  TRenderTextEvent = procedure(Sender: TObject; Drawer: TGLTextDrawer) of object;
  TViewportRenderTextEvent = procedure(Sender: TObject; Viewport: TViewport; Drawer: TGLTextDrawer) of object;

const
  DefaultDrawAxes = [axX, axY, axZ];

type

  { TViewportRenderer }

  TViewportRenderer = class(TComponent)
  protected
    procedure BeginRender(Viewport: TViewport; Camera: TCamera); virtual;
    procedure FinishRender; virtual;
    procedure RenderBackground(Viewport: TViewport); virtual; abstract;
    procedure Render(Viewport: TViewport); virtual; abstract;
    procedure RenderOverlay(Viewport: TViewport); virtual; abstract;
    procedure RenderText(Viewport: TViewport; Drawer: TGLTextDrawer); virtual; abstract;
  end;

  { TSimpleViewportRenderer }

  TSimpleViewportRenderer = class(TViewportRenderer)
  private
    FOnRender: TViewportEvent;
    FOnRenderBackground: TViewportEvent;
    FOnRenderOverlay: TViewportEvent;
    FOnRenderText: TViewportRenderTextEvent;
  public
  protected
    procedure RenderBackground(Viewport: TViewport); override;
    procedure Render(Viewport: TViewport); override;
    procedure RenderOverlay(Viewport: TViewport); override;
    procedure RenderText(Viewport: TViewport; Drawer: TGLTextDrawer); override;
  published
    property OnRenderBackground: TViewportEvent read FOnRenderBackground write FOnRenderBackground;
    property OnRender: TViewportEvent read FOnRender write FOnRender;
    property OnRenderOverlay: TViewportEvent read FOnRenderOverlay write FOnRenderOverlay;
    property OnRenderText: TViewportRenderTextEvent read FOnRenderText write FOnRenderText;
  end;

  { TViewportPickable }

  TViewportPickable = class
  private
    FShadow: TObject;
  protected
    procedure GetBoundingSphere(out Center: TVector; out Radius: Double); virtual; abstract;
    function RayHitCheck(const ARay: TRay; out IP: TVector): Boolean; virtual; abstract;
  public
    property Shadow: TObject read FShadow write FShadow;
  end;

  TViewportPickableList = specialize TFPGObjectList<TViewportPickable>;

  { TGeometricViewportPickableElement }

  TGeometricViewportPickableElement = class
  protected
    function RayHitCheck(const ARay: TRay; out IP: TVector): Boolean; virtual; abstract;
    procedure GetBoundingSphere(out Center: TVector; out Radius: Double); virtual; abstract;
  end;

  { TGeometricViewportPickableSphereElement }

  TGeometricViewportPickableSphereElement = class(TGeometricViewportPickableElement)
  private
    FCenter: TVector;
    FRadius: Double;
  protected
    function RayHitCheck(const ARay: TRay; out IP: TVector): Boolean; override;
    procedure GetBoundingSphere(out XCenter: TVector; out XRadius: Double); override;
  public
    property Center: TVector read FCenter write FCenter;
    property Radius: Double read FRadius write FRadius;
  end;

  { TGeometricViewportPickableCapsuleElement }

  TGeometricViewportPickableCapsuleElement = class(TGeometricViewportPickableElement)
  private
    FHead: TVector;
    FTail: TVector;
    FWidth: Double;
  protected
    function RayHitCheck(const ARay: TRay; out IP: TVector): Boolean; override;
    procedure GetBoundingSphere(out Center: TVector; out Radius: Double); override;
  public
    property Head: TVector read FHead write FHead;
    property Tail: TVector read FTail write FTail;
    property Width: Double read FWidth write FWidth;
  end;

  { TGeometricViewportPickableAxisAlignedBoxElement }

  TGeometricViewportPickableAxisAlignedBoxElement = class(TGeometricViewportPickableElement)
  private
    FBox: TAABox;
  protected
    function RayHitCheck(const ARay: TRay; out IP: TVector): Boolean; override;
    procedure GetBoundingSphere(out Center: TVector; out Radius: Double); override;
  public
    property Box: TAABox read FBox write FBox;
  end;

  { TGeometricViewportPickable }

  TGeometricViewportPickable = class(TViewportPickable)
  private
    FElements: array of TGeometricViewportPickableElement;
    BSCenter: TVector;
    BSRadius: Double;
    DirtySphere: Boolean;
  protected
    procedure UpdateBoundingSphere;
    procedure GetBoundingSphere(out Center: TVector; out Radius: Double); override;
    function RayHitCheck(const ARay: TRay; out IP: TVector): Boolean; override;
  public
    constructor Create;
    destructor Destroy; override;
    procedure DestroyElements;
    procedure AddElement(AElement: TGeometricViewportPickableElement);
    procedure AddSphere(const ACenter: TVector; ARadius: Double);
    procedure AddCapsule(const AHead, ATail: TVector; AWidth: Double);
    procedure AddAxisAlignedBox(const ABox: TAABox);
  end;

  { TViewportPicker }

  TViewportPickableHoverEvent = procedure(Sender: TObject; Pickable: TViewportPickable; const ARay: TRay; const IP: TVector) of object;
  TViewportPickableMotionEvent = procedure(Sender: TObject; Pickable: TViewportPickable; const ARay: TRay; const IP: TVector; var BlockProcessing: Boolean) of object;
  TViewportPickableButtonEvent = procedure(Sender: TObject; Pickable: TViewportPickable; const ARay: TRay; const IP: TVector; Button: TMouseButton; Shift: TShiftState; var BlockProcessing: Boolean) of object;

  TViewportPicker = class(TComponent)
  private
    FPickables: TViewportPickableList;
    FOnMouseEnter: TViewportPickableHoverEvent;
    FOnMouseExit: TViewportPickableHoverEvent;
    FOnMouseMove: TViewportPickableMotionEvent;
    FOnMouseDown: TViewportPickableButtonEvent;
    FOnMouseUp: TViewportPickableButtonEvent;
    FHoverPickable: TViewportPickable;
    FAllowNullPickables: Boolean;
  protected
    function HandleMouseMotion(const ARay: TRay): Boolean;
    function HandleMouseDown(const ARay: TRay; Button: TMouseButton; Shift: TShiftState): Boolean;
    function HandleMouseUp(const ARay: TRay; Button: TMouseButton; Shift: TShiftState): Boolean;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure RemoveAllPickables;
    procedure AddPickable(APickable: TViewportPickable);
    procedure RemovePickable(APickable: TViewportPickable);
    function PickableAt(const ARay: TRay; out IP: TVector): TViewportPickable;
  published
    property AllowNullPickables: Boolean read FAllowNullPickables write FAllowNullPickables default True;
    property OnMouseEnter: TViewportPickableHoverEvent read FOnMouseEnter write FOnMouseEnter;
    property OnMouseExit: TViewportPickableHoverEvent read FOnMouseExit write FOnMouseExit;
    property OnMouseMove: TViewportPickableMotionEvent read FOnMouseMove write FOnMouseMove;
    property OnMouseDown: TViewportPickableButtonEvent read FOnMouseDown write FOnMouseDown;
    property OnMouseUp: TViewportPickableButtonEvent read FOnMouseUp write FOnMouseUp;
  end;

  { TViewport }

  TViewport = class(TWinControl,I3DWidgetManagerEnvironment)
  private
    FAutoUpdate: Boolean;
    FPicker: TViewportPicker;
    FTempAutoUpdate: Boolean;
    FCamera: TCamera;
    FCameraMotion: TViewportCameraMotion;
    FCameraSpeed: Double;
    FDrawAxes: TAxes;
    FDrawCaption: Boolean;
    FDrawFocusOutline: Boolean;
    FDrawGrid: Boolean;
    FDrawSide: Boolean;
    FFocusedColor: TColor;
    FGLControl: TOpenGLControl;
    FGridColor: TColor;
    FGridScale: Double;
    FGridSteps: Integer;
    FGridSubColor: TColor;
    FMultiSampling: Integer;
    FOnCameraChange: TNotifyEvent;
    FRenderer: TViewportRenderer;
    FUnfocusedColor: TColor;
    FTextDrawer: TGLTextDrawer;
    FOnRender: TNotifyEvent;
    FOnRenderBackground: TNotifyEvent;
    FOnRenderOverlay: TNotifyEvent;
    FOnRenderText: TRenderTextEvent;
    FOnUpdate: TNotifyEvent;
    FRendering: Boolean;
    FSetFocusLater: Boolean;
    FKeyState: packed array [0..255] of Boolean;
    FKeyDownCount: Integer;
    FWidgetManager: T3DWidgetManager;
    procedure CreateOpenGLControl;
    function GetCameraBookmark: string; inline;
    function GetCameraMode: TCameraMode; inline;
    function GetDirectionX: Double; inline;
    function GetDirectionY: Double; inline;
    function GetDirectionZ: Double; inline;
    function GetOrthoScale: Double; inline;
    function GetPositionX: Double; inline;
    function GetPositionY: Double; inline;
    function GetPositionZ: Double; inline;
    function GetSide: TViewportSide;
    procedure GLCClick(Sender: TObject);
    procedure GLCDblClick(Sender: TObject);
    procedure GLCDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure GLCDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure GLCEnter(Sender: TObject);
    procedure GLCExit(Sender: TObject);
    procedure GLCKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure GLCKeyPress(Sender: TObject; var Key: char);
    procedure GLCKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure GLCMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure GLCMouseEnter(Sender: TObject);
    procedure GLCMouseLeave(Sender: TObject);
    procedure GLCMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure GLCMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure GLCMouseWheel(Sender: TObject; Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure GLCMouseWheelDown(Sender: TObject; Shift: TShiftState; MousePos: TPoint; var Handled: Boolean);
    procedure GLCMouseWheelUp(Sender: TObject; Shift: TShiftState; MousePos: TPoint; var Handled: Boolean);
    procedure GLCPaint(Sender: TObject);
    procedure RenderGrid;
    procedure RenderAxes(ConfigGL: Boolean);
    procedure DoRenderViewport;
    procedure SetCameraBookmark(AValue: string);
    procedure SetCameraMode(AValue: TCameraMode); inline;
    procedure SetDirectionX(AValue: Double); inline;
    procedure SetDirectionY(AValue: Double); inline;
    procedure SetDirectionZ(AValue: Double); inline;
    procedure SetDrawAxes(AValue: TAxes); inline;
    procedure SetDrawCaption(AValue: Boolean); inline;
    procedure SetDrawFocusOutline(AValue: Boolean); inline;
    procedure SetDrawGrid(AValue: Boolean); inline;
    procedure SetDrawSide(AValue: Boolean); inline;
    procedure SetFocusedColor(AValue: TColor); inline;
    procedure SetGridColor(AValue: TColor); inline;
    procedure SetGridScale(AValue: Double); inline;
    procedure SetGridSteps(AValue: Integer); inline;
    procedure SetGridSubColor(AValue: TColor); inline;
    procedure SetMultiSampling(AValue: Integer); inline;
    procedure SetOrthoScale(AValue: Double); inline;
    procedure SetPicker(AValue: TViewportPicker); inline;
    procedure SetPositionX(AValue: Double); inline;
    procedure SetPositionY(AValue: Double); inline;
    procedure SetPositionZ(AValue: Double); inline;
    procedure SetRenderer(AValue: TViewportRenderer); inline;
    procedure SetSide(AValue: TViewportSide); inline;
    procedure SetUnfocusedColor(AValue: TColor); inline;
    procedure SetWidgetManager(AValue: T3DWidgetManager); inline;
  protected
    procedure WMPaint(var Msg: TLMPaint); message LM_PAINT;
    procedure WMEnter(var Message: TLMEnter); message LM_ENTER;
    procedure CameraChange(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Invalidate; override;
    procedure UpdateViewport;
    procedure RenderViewport;
    function IsKeyDown(Key: Integer): Boolean;
    function IsFocused: Boolean; inline;
    procedure TogglePerspective;
    function RayAt(X, Y: Integer): TRay;
    function WorldToView(const APoint: TVector): TVector;
    function ViewToWorld(const APoint: TVector): TVector;
    procedure SetFocus; override;
    property Camera: TCamera read FCamera;
  public
    // I3DWidgetManagerEnvironment
    function GetForwardDirection: TVector;
    function GetUpDirection: TVector;
    function GetRightDirection: TVector;
    function GetPixelScale(const APosition: TVector): Double;
    function IsActiveEnvironment: Boolean;
  published
    property AutoUpdate: Boolean read FAutoUpdate write FAutoUpdate default True;
    property DrawFocusOutline: Boolean read FDrawFocusOutline write SetDrawFocusOutline;
    property FocusedColor: TColor read FFocusedColor write SetFocusedColor default clWhite;
    property UnfocusedColor: TColor read FUnfocusedColor write SetUnfocusedColor default clGray;
    property DrawGrid: Boolean read FDrawGrid write SetDrawGrid default True;
    property DrawCaption: Boolean read FDrawCaption write SetDrawCaption default True;
    property DrawSide: Boolean read FDrawSide write SetDrawSide default True;
    property DrawAxes: TAxes read FDrawAxes write SetDrawAxes default DefaultDrawAxes;
    property GridColor: TColor read FGridColor write SetGridColor default $00663319;
    property GridSubColor: TColor read FGridSubColor write SetGridSubColor default $00231E19;
    property GridSteps: Integer read FGridSteps write SetGridSteps;
    property GridScale: Double read FGridScale write SetGridScale;
    property CameraMotion: TViewportCameraMotion read FCameraMotion write FCameraMotion default vcmFirstPerson;
    property CameraSpeed: Double read FCameraSpeed write FCameraSpeed;
    property Renderer: TViewportRenderer read FRenderer write SetRenderer;
    property Side: TViewportSide read GetSide write SetSide stored False default vsCustom;
    property CameraMode: TCameraMode read GetCameraMode write SetCameraMode;
    property PositionX: Double read GetPositionX write SetPositionX;
    property PositionY: Double read GetPositionY write SetPositionY;
    property PositionZ: Double read GetPositionZ write SetPositionZ;
    property DirectionX: Double read GetDirectionX write SetDirectionX;
    property DirectionY: Double read GetDirectionY write SetDirectionY;
    property DirectionZ: Double read GetDirectionZ write SetDirectionZ;
    property OrthoScale: Double read GetOrthoScale write SetOrthoScale;
    property CameraBookmark: string read GetCameraBookmark write SetCameraBookmark stored False;
    property WidgetManager: T3DWidgetManager read FWidgetManager write SetWidgetManager;
    property Picker: TViewportPicker read FPicker write SetPicker;
    property MultiSampling: Integer read FMultiSampling write SetMultiSampling;
    property Align;
    property Anchors;
    property BorderSpacing;
    property Caption;
    property Color default clBlack;
    property Constraints;
    property Cursor;
    property Font;
    property Width default 320;
    property Height default 240;
    property Hint;
    property ParentColor default False;
    property ParentShowHint;
    property ShowHint;
    property Visible;
    property OnUpdate: TNotifyEvent read FOnUpdate write FOnUpdate;
    property OnRenderBackground: TNotifyEvent read FOnRenderBackground write FOnRenderBackground;
    property OnRender: TNotifyEvent read FOnRender write FOnRender;
    property OnRenderOverlay: TNotifyEvent read FOnRenderOverlay write FOnRenderOverlay;
    property OnRenderText: TRenderTextEvent read FOnRenderText write FOnRenderText;
    property OnCameraChange: TNotifyEvent read FOnCameraChange write FOnCameraChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseEnter;
    property OnMouseLeave;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseWheel;
    property OnMouseWheelDown;
    property OnMouseWheelUp;
    property OnResize;
    property OnShowHint;
  end;

  { TViewportManager }

  TViewportManager = class(TComponent)
  private
    FOnUpdate: TNotifyEvent;
    function GetActiveViewport: TViewport;
    function GetUpdateRate: Integer;
    function GetViewportCount: Integer;
    function GetViewports(AIndex: Integer): TViewport;
    procedure SetUpdateRate(AValue: Integer);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Viewports[AIndex: Integer]: TViewport read GetViewports;
    property ViewportCount: Integer read GetViewportCount;
    property ActiveViewport: TViewport read GetActiveViewport;
  published
    property UpdateRate: Integer read GetUpdateRate write SetUpdateRate;
    property OnUpdate: TNotifyEvent read FOnUpdate write FOnUpdate;
  end;

procedure Register;

implementation

uses
  LResources, ExtCtrls, LCLIntf, GL, UIUtils;

type
  TViewportList = specialize TFPGObjectList<TViewport>;
  TGlobalViewportRegistryList = specialize TFPGObjectList<TViewportManager>;

  { TUpdateTimer }

  TUpdateTimer = class(TTimer)
  public
    constructor Create(AOwner: TComponent); override;
    procedure MyTimer(Sender: TObject);
  end;

var
  ViewportList: TViewportList;
  RegistryList: TGlobalViewportRegistryList;
  LastActiveViewport: TViewport;
  UpdateTimer: TTimer;
  UpdateFPS: Integer;
  LastTicks: Double;
  MX, MY: Integer;
  Panning, Rotating: Boolean;

procedure Register;
begin
  {$I viewports_icon.lrs}
  RegisterComponents('RTTK',[TViewport, TViewportManager, TSimpleViewportRenderer, TViewportPicker]);
end;

procedure NeedUpdateTimer; inline;
begin
  if not Assigned(UpdateTimer) then begin
    UpdateTimer:=TUpdateTimer.Create(Application);
    UpdateTimer.Interval:=8;
    UpdateTimer.Enabled:=True;
  end;
end;

{ TViewportPicker }

function TViewportPicker.HandleMouseMotion(const ARay: TRay): Boolean;
var
  Pickable: TViewportPickable;
  IP: TVector;
begin
  Result:=False;
  if Assigned(OnMouseMove) or Assigned(OnMouseEnter) or Assigned(OnMouseExit) then begin
    Pickable:=PickableAt(ARay, IP);
    if Pickable <> FHoverPickable then begin
      if Assigned(OnMouseExit) and (AllowNullPickables or Assigned(FHoverPickable)) then OnMouseExit(Self, FHoverPickable, ARay, IP);
      FHoverPickable:=Pickable;
      if Assigned(OnMouseEnter) and (AllowNullPickables or Assigned(FHoverPickable)) then OnMouseEnter(Self, FHoverPickable, ARay, IP);
    end;
    if Assigned(OnMouseMove) and (AllowNullPickables or Assigned(Pickable)) then
      OnMouseMove(Self, Pickable, ARay, IP, Result);
  end;
end;

function TViewportPicker.HandleMouseDown(const ARay: TRay;
  Button: TMouseButton; Shift: TShiftState): Boolean;
var
  Pickable: TViewportPickable;
  IP: TVector;
begin
  Result:=False;
  if Assigned(OnMouseDown) then begin
    Pickable:=PickableAt(ARay, IP);
    if AllowNullPickables or Assigned(Pickable) then
      OnMouseDown(Self, Pickable, ARay, IP, Button, Shift, Result);
  end;
end;

function TViewportPicker.HandleMouseUp(const ARay: TRay; Button: TMouseButton;
  Shift: TShiftState): Boolean;
var
  Pickable: TViewportPickable;
  IP: TVector;
begin
  Result:=False;
  if Assigned(OnMouseUp) then begin
    Pickable:=PickableAt(ARay, IP);
    if AllowNullPickables or Assigned(Pickable) then
      OnMouseUp(Self, Pickable, ARay, IP, Button, Shift, Result);
  end;
end;

constructor TViewportPicker.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAllowNullPickables:=True;
  FPickables:=TViewportPickableList.Create(False);
end;

destructor TViewportPicker.Destroy;
begin
  FreeAndNil(FPickables);
  inherited Destroy;
end;

procedure TViewportPicker.RemoveAllPickables;
var
  DummyRay: TRay;
begin
  if Assigned(FHoverPickable) then begin
    if Assigned(OnMouseExit) then begin
      DummyRay.o.Zero;
      DummyRay.d:=Vector(0, 1, 0);
      OnMouseExit(Self, FHoverPickable, DummyRay, Vector(0, 0, 0));
    end;
    FHoverPickable:=nil;
  end;
  FPickables.Clear;
end;

procedure TViewportPicker.AddPickable(APickable: TViewportPickable);
begin
  FPickables.Add(APickable);
end;

procedure TViewportPicker.RemovePickable(APickable: TViewportPickable);
var
  DummyRay: TRay;
begin
  if FHoverPickable=APickable then begin
    if Assigned(OnMouseExit) then begin
      DummyRay.o.Zero;
      DummyRay.d:=Vector(0, 1, 0);
      OnMouseExit(Self, FHoverPickable, DummyRay, Vector(0, 0, 0));
    end;
    FHoverPickable:=nil;
  end;
  FPickables.Remove(APickable);
end;

function TViewportPicker.PickableAt(const ARay: TRay; out IP: TVector): TViewportPickable;
var
  ClosestDistance, PickDistance: Double;
  PickIP: TVector;
  I: Integer;
begin
  Result:=nil;
  ClosestDistance:=MaxInt;
  for I:=0 to FPickables.Count - 1 do begin
    if FPickables[I].RayHitCheck(ARay, PickIP) then begin
      PickDistance:=DistanceSq(PickIP, ARay.o);
      if (not Assigned(Result)) or (PickDistance < ClosestDistance) then begin
        ClosestDistance:=PickDistance;
        IP:=PickIP;
        Result:=FPickables[I];
      end;
    end;
  end;
  if not Assigned(Result) then IP.Zero;
end;

{ TGeometricViewportPickableAxisAlignedBoxElement }

function TGeometricViewportPickableAxisAlignedBoxElement.RayHitCheck(
  const ARay: TRay; out IP: TVector): Boolean;
begin
  Result:=ARay.AABoxHit(Box, IP);
end;

procedure TGeometricViewportPickableAxisAlignedBoxElement.GetBoundingSphere(out
  Center: TVector; out Radius: Double);
begin
  Box.GetBoundingSphere(Center, Radius);
end;

{ TGeometricViewportPickableCapsuleElement }

function TGeometricViewportPickableCapsuleElement.RayHitCheck(const ARay: TRay;
  out IP: TVector): Boolean;
begin
  Result:=ARay.CapsuleHit(Head, Tail, Width, IP);
end;

procedure TGeometricViewportPickableCapsuleElement.GetBoundingSphere(out
  Center: TVector; out Radius: Double);
begin
  Center:=Head.Added(Tail).Scaled(0.5);
  Radius:=Distance(Head, Tail)/2 + Width;
end;

{ TGeometricViewportPickableSphereElement }

function TGeometricViewportPickableSphereElement.RayHitCheck(const ARay: TRay;
  out IP: TVector): Boolean;
begin
  Result:=ARay.SphereHit(Center, Radius, IP);
end;

procedure TGeometricViewportPickableSphereElement.GetBoundingSphere(out
  XCenter: TVector; out XRadius: Double);
begin
  XCenter:=Center;
  XRadius:=Radius;
end;

{ TGeometricViewportPickable }

procedure TGeometricViewportPickable.UpdateBoundingSphere;
var
  C: TVector;
  R, Dist: Double;
  I: Integer;
begin
  if Length(FElements)=0 then begin
    BSCenter.Zero;
    BSRadius:=0;
    DirtySphere:=False;
    Exit;
  end;
  BSCenter.Zero;
  for I:=0 to High(FElements) do begin
    FElements[I].GetBoundingSphere(C, R);
    BSCenter.Add(C);
  end;
  BSCenter.Scale(1/Length(FElements));
  BSRadius:=0;
  for I:=0 to High(FElements) do begin
    FElements[I].GetBoundingSphere(C, R);
    Dist:=Distance(C, BSCenter) + R*R;
    if Dist > BSRadius then BSRadius:=Dist;
  end;
  DirtySphere:=False;
end;

procedure TGeometricViewportPickable.GetBoundingSphere(out Center: TVector; out
  Radius: Double);
begin
  if DirtySphere then UpdateBoundingSphere;
  Center:=BSCenter;
  Radius:=BSRadius;
end;

function TGeometricViewportPickable.RayHitCheck(const ARay: TRay;
  out IP: TVector): Boolean;
var
  Closest: TGeometricViewportPickableElement;
  ElementIP: TVector;
  ClosestDistance, ElementDistance: Double;
  I: Integer;
begin
  Closest:=nil;
  ClosestDistance:=MaxInt;
  for I:=0 to High(FElements) do begin
    if FElements[I].RayHitCheck(ARay, ElementIP) then begin
      ElementDistance:=DistanceSq(ElementIP, ARay.o);
      if (not Assigned(Closest)) or (ElementDistance < ClosestDistance) then begin
        ClosestDistance:=ElementDistance;
        IP:=ElementIP;
        Closest:=FElements[I];
      end;
    end;
  end;
  Result:=Assigned(Closest);
  if not Result then IP.Zero;
end;

constructor TGeometricViewportPickable.Create;
begin
  DirtySphere:=True;
end;

destructor TGeometricViewportPickable.Destroy;
begin
  DestroyElements;
  inherited Destroy;
end;

procedure TGeometricViewportPickable.DestroyElements;
var
  I: Integer;
begin
  for I:=0 to High(FElements) do
    FElements[I].Free;
  SetLength(FElements, 0);
  DirtySphere:=True;
end;

procedure TGeometricViewportPickable.AddElement(AElement: TGeometricViewportPickableElement);
begin
  SetLength(FElements, Length(FElements) + 1);
  FElements[High(FElements)]:=AElement;
  DirtySphere:=True;
end;

procedure TGeometricViewportPickable.AddSphere(const ACenter: TVector;
  ARadius: Double);
var
  Element: TGeometricViewportPickableSphereElement;
begin
  Element:=TGeometricViewportPickableSphereElement.Create;
  Element.Center:=ACenter;
  Element.Radius:=ARadius;
  AddElement(Element);
end;

procedure TGeometricViewportPickable.AddCapsule(const AHead, ATail: TVector;
  AWidth: Double);
var
  Element: TGeometricViewportPickableCapsuleElement;
begin
  Element:=TGeometricViewportPickableCapsuleElement.Create;
  Element.Head:=AHead;
  Element.Tail:=ATail;
  Element.Width:=AWidth;
  AddElement(Element);
end;

procedure TGeometricViewportPickable.AddAxisAlignedBox(const ABox: TAABox);
var
  Element: TGeometricViewportPickableAxisAlignedBoxElement;
begin
  Element:=TGeometricViewportPickableAxisAlignedBoxElement.Create;
  Element.Box:=ABox;
  AddElement(Element);
end;

{ TViewportRenderer }

procedure TViewportRenderer.BeginRender(Viewport: TViewport; Camera: TCamera);
begin
end;

procedure TViewportRenderer.FinishRender;
begin
end;

{ TSimpleViewportRenderer }

procedure TSimpleViewportRenderer.RenderBackground(Viewport: TViewport);
begin
  if Assigned(FOnRenderBackground) then FOnRenderBackground(Self, Viewport);
end;

procedure TSimpleViewportRenderer.Render(Viewport: TViewport);
begin
  if Assigned(FOnRender) then FOnRender(Self, Viewport);
end;

procedure TSimpleViewportRenderer.RenderOverlay(Viewport: TViewport);
begin
  if Assigned(FOnRenderOverlay) then FOnRenderOverlay(Self, Viewport);
end;

procedure TSimpleViewportRenderer.RenderText(Viewport: TViewport;
  Drawer: TGLTextDrawer);
begin
  if Assigned(FOnRenderText) then FOnRenderText(Self, Viewport, Drawer);
end;

{ TUpdateTimer }

constructor TUpdateTimer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  OnTimer:=@MyTimer;
end;

procedure TUpdateTimer.MyTimer(Sender: TObject);
var
  Ticks: Double;
  I: Integer;
begin
  Ticks:=GetTickCount64;
  if Ticks - LastTicks > 2000 then LastTicks:=Ticks;
  while Ticks - LastTicks > 1000.0/UpdateFPS do begin
    if Assigned(RegistryList) then begin
      for I:=0 to RegistryList.Count - 1 do
        if Assigned(RegistryList[I].FOnUpdate) then
          RegistryList[I].FOnUpdate(RegistryList[I]);
    end;
    if Assigned(ViewportList) then begin
      for I:=0 to ViewportList.Count - 1 do
        if ViewportList[I].AutoUpdate or ViewportList[I].FTempAutoUpdate then
          ViewportList[I].UpdateViewport;
    end;
    LastTicks += 1000.0/UpdateFPS;
  end;
end;

{ TViewportManager }

function TViewportManager.GetViewportCount: Integer;
begin
  if not Assigned(ViewportList) then Exit(0);
  Result:=ViewportList.Count;
end;

function TViewportManager.GetUpdateRate: Integer;
begin
  Result:=UpdateFPS;
end;

function TViewportManager.GetActiveViewport: TViewport;
begin
  Result:=LastActiveViewport;
end;

function TViewportManager.GetViewports(AIndex: Integer): TViewport;
begin
  if not Assigned(ViewportList) then raise ERangeError.Create('Invalid viewport index ' + IntToStr(AIndex));
  if (AIndex < 0) or (AIndex >= ViewportCount) then raise ERangeError.Create('Invalid viewport index ' + IntToStr(AIndex));
  Result:=ViewportList[AIndex];
end;

procedure TViewportManager.SetUpdateRate(AValue: Integer);
begin
  UpdateFPS:=AValue;
  if UpdateFPS < 1 then UpdateFPS:=1;
  if UpdateFPS > 1000 then UpdateFPS:=1000;
  NeedUpdateTimer;
  UpdateTimer.Interval:=Max(1, Floor(500/UpdateFPS));
end;

constructor TViewportManager.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  if not (csDesigning in ComponentState) then begin
    if not Assigned(RegistryList) then RegistryList:=TGlobalViewportRegistryList.Create(False);
    RegistryList.Add(Self);
  end;
end;

destructor TViewportManager.Destroy;
begin
  if not (csDesigning in ComponentState) then begin
    RegistryList.Remove(Self);
  end;
  inherited Destroy;
end;

{ TViewport }

procedure TViewport.CreateOpenGLControl;
begin
  FGLControl:=TOpenGLControl.Create(Self);
  FGLControl.Align:=alClient;
  FGLControl.AutoResizeViewport:=True;
  FGLControl.DepthBits:=24;
  FGLControl.StencilBits:=8;
  FGLControl.CaptureMouseButtons:=[mbLeft, mbMiddle, mbRight];
  FGLControl.TabStop:=False;
  if Assigned(ViewportList) and (ViewportList.Count > 0) then
    FGLControl.SharedControl:=ViewportList[0].FGLControl;
  FGLControl.OnClick:=@GLCClick;
  FGLControl.OnDblClick:=@GLCDblClick;
  FGLControl.OnDragDrop:=@GLCDragDrop;
  FGLControl.OnDragOver:=@GLCDragOver;
  FGLControl.OnEnter:=@GLCEnter;
  FGLControl.OnExit:=@GLCExit;
  FGLControl.OnKeyDown:=@GLCKeyDown;
  FGLControl.OnKeyPress:=@GLCKeyPress;
  FGLControl.OnKeyUp:=@GLCKeyUp;
  FGLControl.OnMouseDown:=@GLCMouseDown;
  FGLControl.OnMouseEnter:=@GLCMouseEnter;
  FGLControl.OnMouseLeave:=@GLCMouseLeave;
  FGLControl.OnMouseMove:=@GLCMouseMove;
  FGLControl.OnMouseUp:=@GLCMouseUp;
  FGLControl.OnMouseWheel:=@GLCMouseWheel;
  FGLControl.OnMouseWheelDown:=@GLCMouseWheelDown;
  FGLControl.OnMouseWheelUp:=@GLCMouseWheelUp;
  FGLControl.OnPaint:=@GLCPaint;
  InsertControl(FGLControl);
end;

function TViewport.GetCameraBookmark: string;
begin
  Result:=Camera.Bookmark;
end;

function TViewport.GetCameraMode: TCameraMode;
begin
  Result:=Camera.Mode;
end;

function TViewport.GetDirectionX: Double;
begin
  Result:=Camera.DirectionX;
end;

function TViewport.GetDirectionY: Double;
begin
  Result:=Camera.DirectionY;
end;

function TViewport.GetDirectionZ: Double;
begin
  Result:=Camera.DirectionZ;
end;

function TViewport.GetOrthoScale: Double;
begin
  Result:=Camera.OrthoScale;
end;

function TViewport.GetPositionX: Double;
begin
  Result:=Camera.PositionX;
end;

function TViewport.GetPositionY: Double;
begin
  Result:=Camera.PositionY;
end;

function TViewport.GetPositionZ: Double;
begin
  Result:=Camera.PositionZ;
end;

function TViewport.GetSide: TViewportSide;
begin
  if Camera.Direction=Vector(1, 0, 0) then Result:=vsLeft else
  if Camera.Direction=Vector(-1, 0, 0) then Result:=vsRight else
  if Camera.Direction=Vector(0, -1, 0) then Result:=vsTop else
  if Camera.Direction=Vector(0, 1, 0) then Result:=vsBottom else
  if Camera.Direction=Vector(0, 0, -1) then Result:=vsFront else
  if Camera.Direction=Vector(0, 0, 1) then Result:=vsBack else
  Result:=vsCustom;
end;

procedure TViewport.GLCClick(Sender: TObject);
begin
  if Assigned(OnClick) then OnClick(Self);
end;

procedure TViewport.GLCDblClick(Sender: TObject);
begin
  if Assigned(OnDblClick) then OnDblClick(Self);
end;

procedure TViewport.GLCDragDrop(Sender, Source: TObject; X, Y: Integer);
begin
  if Assigned(OnDragDrop) then OnDragDrop(Self, Source, X, Y);
end;

procedure TViewport.GLCDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  if Assigned(OnDragOver) then OnDragOver(Sender, Source, X, Y, State, Accept);
end;

procedure TViewport.GLCEnter(Sender: TObject);
begin
  LastActiveViewport:=Self;
  if Assigned(OnEnter) then OnEnter(Self);
end;

procedure TViewport.GLCExit(Sender: TObject);
var
  I: Integer;
begin
  for I:=0 to 255 do FKeyState[I]:=False;
  FKeyDownCount:=0;
  if FTempAutoUpdate then FTempAutoUpdate:=False;
  Panning:=False;
  Rotating:=False;
  if Assigned(OnExit) then OnExit(Self);
end;

procedure TViewport.GLCKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key in [0..255]) and not FKeyState[Key] then begin
    FKeyState[Key]:=True;
    Inc(FKeyDownCount);
  end;
  if not AutoUpdate then FTempAutoUpdate:=True;
  if CameraMotion=vcmFirstPerson then begin
    case Key of
      VK_NUMPAD4: Camera.MoveTowardsInView(Vector(-CameraSpeed*10, 0, 0));
      VK_NUMPAD6: Camera.MoveTowardsInView(Vector(CameraSpeed*10, 0, 0));
      VK_NUMPAD2: Camera.MoveTowardsInView(Vector(0, -CameraSpeed*10, 0));
      VK_NUMPAD8: Camera.MoveTowardsInView(Vector(0, CameraSpeed*10, 0));
      VK_NUMPAD1: if ssCtrl in Shift then Side:=vsBack else Side:=vsFront;
      VK_NUMPAD3: if ssCtrl in Shift then Side:=vsLeft else Side:=vsRight;
      VK_NUMPAD7: if ssCtrl in Shift then Side:=vsBottom else Side:=vsTop;
      VK_NUMPAD5: begin
        Key:=0;
        TogglePerspective;
      end;
      VK_DECIMAL: Camera.Position:=Vector(0, 0, 0);
      VK_ADD: begin
        if CameraMode=cmPerspective then
          Camera.MoveForward(CameraSpeed*10)
        else
          Camera.OrthoScale:=Camera.OrthoScale - CameraSpeed*10;
      end;
      VK_SUBTRACT: begin
        if CameraMode=cmPerspective then
          Camera.MoveForward(-CameraSpeed*10)
        else
          Camera.OrthoScale:=Camera.OrthoScale + CameraSpeed*10;
      end;
    end;
  end;
  if Assigned(OnKeyDown) then OnKeyDown(Self, Key, Shift);
end;

procedure TViewport.GLCKeyPress(Sender: TObject; var Key: char);
begin
  if Assigned(OnKeyPress) then OnKeyPress(Self, Key);
end;

procedure TViewport.GLCKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(OnKeyUp) then OnKeyUp(Self, Key, Shift);
  Dec(FKeyDownCount);
  if (FKeyDownCount=0) and FTempAutoUpdate then FTempAutoUpdate:=False;
  if Key in [0..255] then FKeyState[Key]:=False;
end;

procedure TViewport.GLCMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  Ray: TRay;
begin
  FGLControl.SetFocus;
  if (CameraMotion=vcmFirstPerson) and ((not ModifiersInShift(Shift)) or (ssShift in Shift)) then begin
    if Button=mbMiddle then begin
      MX:=X;
      MY:=Y;
      Panning:=True;
      Exit;
    end;
    if Button=mbRight then begin
      MX:=X;
      MY:=Y;
      Rotating:=True;
      Exit;
    end;
  end;
  if Assigned(WidgetManager) or Assigned(Picker) then
    Ray:=RayAt(X, Y);
  if Assigned(WidgetManager) then begin
    WidgetManager.Environment:=Self;
    WidgetManager.MouseDown(Ray, Button, Shift);
    if Assigned(WidgetManager.ActiveWidget) then Exit;
  end;
  if Assigned(Picker) then begin
    if Picker.HandleMouseDown(Ray, Button, Shift) then Exit;
  end;
  if Assigned(OnMouseDown) then OnMouseDown(Self, Button, Shift, X, Y);
end;

procedure TViewport.GLCMouseEnter(Sender: TObject);
begin
  if Assigned(OnMouseEnter) then OnMouseEnter(Self);
end;

procedure TViewport.GLCMouseLeave(Sender: TObject);
begin
  if Assigned(OnMouseLeave) then OnMouseLeave(Self);
end;

procedure TViewport.GLCMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  Ray: TRay;
begin
  if Panning then begin
    Camera.MoveTowardsInView(Vector((MX - X)*CameraSpeed*0.5, (Y - MY)*CameraSpeed*0.5, 0));
    MX:=X;
    MY:=Y;
    Exit;
  end;
  if Rotating then begin
    Camera.RotateHorizontally((MX - X)*0.005);
    Camera.RotateVertically((MY - Y)*0.005);
    MX:=X;
    MY:=Y;
    Exit;
  end;
  if Assigned(WidgetManager) or Assigned(Picker) then
    Ray:=RayAt(X, Y);
  if Assigned(WidgetManager) then begin
    WidgetManager.Environment:=Self;
    WidgetManager.MouseMotion(Ray);
    if Assigned(WidgetManager.ActiveWidget) then Exit;
  end;
  if Assigned(Picker) then begin
    if Picker.HandleMouseMotion(Ray) then Exit;
  end;
  if Assigned(OnMouseMove) then OnMouseMove(Self, Shift, X, Y);
end;

procedure TViewport.GLCMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  Ray: TRay;
begin
  if CameraMotion=vcmFirstPerson then begin
    if Panning and (Button=mbMiddle) then begin
      Panning:=False;
      Exit;
    end;
    if Rotating and (Button=mbRight) then begin
      Rotating:=False;
      Exit;
    end;
  end;
  if Assigned(WidgetManager) or Assigned(Picker) then
    Ray:=RayAt(X, Y);
  if Assigned(Picker) then begin
    if Picker.HandleMouseUp(Ray, Button, Shift) then Exit;
  end;
  if Assigned(WidgetManager) then begin
    WidgetManager.Environment:=Self;
    WidgetManager.MouseUp(Ray, Button, Shift);
  end;
  if Assigned(OnMouseUp) then OnMouseUp(Self, Button, Shift, X, Y);
end;

procedure TViewport.GLCMouseWheel(Sender: TObject; Shift: TShiftState;
  WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
begin
  FGLControl.SetFocus;
  if CameraMotion=vcmFirstPerson then begin
    if IsKeyDown(VK_LSHIFT) or IsKeyDown(VK_RSHIFT) or IsKeyDown(VK_SHIFT) then FCameraSpeed *= 5;
    if CameraMode=cmPerspective then
      Camera.MoveForward(WheelDelta*CameraSpeed*0.01)
    else
      Camera.OrthoScale:=Camera.OrthoScale - WheelDelta*CameraSpeed*0.01;
    if IsKeyDown(VK_LSHIFT) or IsKeyDown(VK_RSHIFT) or IsKeyDown(VK_SHIFT) then FCameraSpeed /= 5;
  end;
  if Assigned(OnMouseWheel) then OnMouseWheel(Self, Shift, WheelDelta, MousePos, Handled);
end;

procedure TViewport.GLCMouseWheelDown(Sender: TObject; Shift: TShiftState;
  MousePos: TPoint; var Handled: Boolean);
begin
  if Assigned(OnMouseWheelDown) then OnMouseWheelDown(Self, Shift, MousePos, Handled);
end;

procedure TViewport.GLCMouseWheelUp(Sender: TObject; Shift: TShiftState;
  MousePos: TPoint; var Handled: Boolean);
begin
  if Assigned(OnMouseWheelUp) then OnMouseWheelUp(Self, Shift, MousePos, Handled);
end;

procedure TViewport.GLCPaint(Sender: TObject);
begin
  if csDesigning in ComponentState then Exit;
  if FSetFocusLater then begin
    FSetFocusLater:=False;
    SetFocus;
  end;
  if FGLControl.MakeCurrent then begin
    try
      FRendering:=True;
      DoRenderViewport;
    finally
      FRendering:=False;
    end;
    FGLControl.SwapBuffers;
  end;
end;

procedure TViewport.RenderGrid;
var
  CellX, CellZ: Integer;
  GridInterval: Double;
  Smoothing: Boolean;
  GC, SGC: TExtColor;

  procedure DrawGridCell(SubCell: Boolean);
  var
    I: Integer;
  begin
    if SubCell then begin
      glColor4f(SGC.r, SGC.g, SGC.b, 1.0);
      for I:=1 to GridSteps - 1 do begin
        glVertex3f(I*GridInterval, -0.002, 0);
        glVertex3f(I*GridInterval, -0.002, 1);
        glVertex3f(0, -0.002, I*GridInterval);
        glVertex3f(1, -0.002, I*GridInterval);
      end;
    end;
    glColor4f(GC.r, GC.g, GC.b, 1.0);
    glVertex3f(0, -0.001, 0);
    glVertex3f(1, -0.001, 0);
    glVertex3f(1, -0.001, 0);
    glVertex3f(1, -0.001, 1);
  end;

begin
  GridInterval:=1.0/GridSteps;
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDepthMask(GL_FALSE);
  glPushMatrix();
  GC.FromColor(GridColor);
  SGC.FromColor(GridSubColor);
  if GridScale <> 1 then glScaled(GridScale, GridScale, GridScale);
  Smoothing:=True;
  if CameraMode=cmOrthographic then begin
    Smoothing:=False;
    case Side of
      vsFront, vsBack: glRotated(90, 1, 0, 0);
      vsLeft, vsRight: glRotated(90, 0, 0, 1);
      vsCustom: Smoothing:=True;
    end;
  end;
  if Smoothing and (MultiSampling < 2) then begin
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_LINE_SMOOTH);
  end;
  glTranslatef(Trunc(Camera.PositionX/GridScale), 0, Trunc(Camera.PositionZ/GridScale));
  for CellX:=-16 to 16 do
    for CellZ:=-16 to 16 do begin
      glPushMatrix();
      glTranslatef(CellX, 0, CellZ);
      glBegin(GL_LINES);
      DrawGridCell((Abs(CellX) < 8) and (Abs(CellZ) < 8));
      glEnd();
      glPopMatrix();
    end;
  glPopMatrix();
  if DrawAxes <> [] then RenderAxes(False);
  glPopAttrib();
end;

procedure TViewport.RenderAxes(ConfigGL: Boolean);
var
  Smoothing: Boolean;
begin
  if ConfigGL then begin
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    glDepthMask(GL_FALSE);
    glPushMatrix();
    Smoothing:=True;
    if CameraMode=cmOrthographic then begin
      Smoothing:=False;
      case Side of
        vsFront, vsBack: glRotated(90, 1, 0, 0);
        vsLeft, vsRight: glRotated(90, 0, 0, 1);
        vsCustom: Smoothing:=True;
      end;
    end;
    if Smoothing and (MultiSampling < 2) then begin
      glEnable(GL_BLEND);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
      glEnable(GL_LINE_SMOOTH);
    end;
  end;
  glBegin(GL_LINES);
  if axX in DrawAxes then begin
    glColor3f(0.45, 0.15, 0.15);
    glVertex3f(-16*GridScale, 0, 0);
    glVertex3f(0, 0, 0);
    glColor3f(0.8, 0, 0);
    glVertex3f(0, 0, 0);
    glVertex3f(16*GridScale, 0, 0);
  end;
  if axY in DrawAxes then begin
    glColor3f(0.15, 0.45, 0.15);
    glVertex3f(0, -16*GridScale, 0);
    glVertex3f(0, 0, 0);
    glColor3f(0, 0.8, 0);
    glVertex3f(0, 0, 0);
    glVertex3f(0, 16*GridScale, 0);
  end;
  if axZ in DrawAxes then begin
    glColor3f(0.15, 0.15, 0.45);
    glVertex3f(0, 0, -16*GridScale);
    glVertex3f(0, 0, 0);
    glColor3f(0, 0, 0.8);
    glVertex3f(0, 0, 0);
    glVertex3f(0, 0, 16*GridScale);
  end;
  glEnd();
  if ConfigGL then begin
    glPopMatrix();
    glPopAttrib();
  end;
end;

procedure TViewport.DoRenderViewport;
var
  EC: TExtColor;
  SideStr: String;
  DoRenderGrid: Boolean;
begin
  if csDesigning in ComponentState then Exit;
  if (Width=0) or (Height=0) then Exit;
  Camera.Aspect:=Width/Height;
  FTextDrawer.ViewportSize:=Size(Width, Height);
  if Assigned(FRenderer) then FRenderer.BeginRender(Self, Camera);
  EC.FromColor(TColor(ColorToRGB(Color)));
  glClearColor(EC.r, EC.g, EC.b, EC.a);
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT or GL_STENCIL_BUFFER_BIT);
  glEnable(GL_DEPTH_TEST);
  glMatrixMode(GL_PROJECTION);
  glLoadMatrixd(Camera.ProjectionMatrix.ToArray);
  glMatrixMode(GL_MODELVIEW);
  glLoadMatrixd(Camera.ModelViewMatrix.ToArray);
  if Assigned(FRenderer) then FRenderer.RenderBackground(Self);
  if Assigned(FOnRenderBackground) then FOnRenderBackground(Self);
  glClear(GL_DEPTH_BUFFER_BIT or GL_STENCIL_BUFFER_BIT);
  DoRenderGrid:=True;
  if (DrawGrid or (DrawAxes <> [])) and (CameraMode=cmOrthographic) and (Side <> vsCustom) then begin
    if DrawGrid then
      RenderGrid
    else
      RenderAxes(True);
    DoRenderGrid:=False;
  end;
  if Assigned(FRenderer) then FRenderer.Render(Self);
  if Assigned(FOnRender) then FOnRender(Self);
  if DoRenderGrid then begin
    if DrawGrid then
      RenderGrid
    else
      RenderAxes(True);
  end;
  glClear(GL_DEPTH_BUFFER_BIT or GL_STENCIL_BUFFER_BIT);
  if Assigned(WidgetManager) then begin
    WidgetManager.Environment:=Self;
    WidgetManager.Render;
  end;
  if Assigned(FRenderer) then FRenderer.RenderOverlay(Self);
  if Assigned(FOnRenderOverlay) then FOnRenderOverlay(Self);
  FTextDrawer.Font:=Font;
  if Assigned(FRenderer) then FRenderer.RenderText(Self, FTextDrawer);
  if Assigned(FOnRenderText) then FOnRenderText(Self, FTextDrawer);
  if Assigned(FRenderer) then FRenderer.FinishRender;
  if DrawSide then begin
    case Side of
      vsCustom: SideStr:='Custom';
      vsLeft: SideStr:='Left';
      vsRight: SideStr:='Right';
      vsTop: SideStr:='Top';
      vsBottom: SideStr:='Bottom';
      vsFront: SideStr:='Front';
      vsBack: SideStr:='Back';
    end;
    case CameraMode of
      cmPerspective: SideStr += ' Perspective';
      cmOrthographic: SideStr += ' Orthographic';
    end;
  end else SideStr:='';
  if DrawFocusOutline then begin
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    if IsFocused then
      EC.FromColor(FocusedColor)
    else
      EC.FromColor(UnfocusedColor);
    glColor3d(EC.r, EC.g, EC.b);
    glPolygonMode(GL_FRONT, GL_LINE);
    glBegin(GL_QUADS);
    glVertex2d(-1 + 1/Width, -1 + 1/Height);
    glVertex2d(1, -1 + 1/Height);
    glVertex2d(1, 1);
    glVertex2d(-1 + 1/Width, 1);
    glEnd();
    glPolygonMode(GL_FRONT, GL_FILL);
    if DrawCaption and (Caption <> '') then begin
      if IsFocused then
        FTextDrawer.Draw(Width - 1 - FTextDrawer.GetTextWidth(Caption), 1, Caption, Color, clNone, FocusedColor)
      else
        FTextDrawer.Draw(Width - 1 - FTextDrawer.GetTextWidth(Caption), 1, Caption, UnfocusedColor);
    end;
    if SideStr <> '' then begin
      if IsFocused then
        FTextDrawer.Draw(1, 1, SideStr, Color, clNone, FocusedColor)
      else
        FTextDrawer.Draw(1, 1, SideStr, UnfocusedColor);
    end;
  end else begin
    if DrawCaption and (Caption <> '') then begin
      if IsFocused then
        FTextDrawer.Draw(Width - FTextDrawer.GetTextWidth(Caption), 0, Caption, Color, clNone, FocusedColor)
      else
        FTextDrawer.Draw(Width - FTextDrawer.GetTextWidth(Caption), 0, Caption, UnfocusedColor);
    end;
    if SideStr <> '' then begin
      if IsFocused then
        FTextDrawer.Draw(0, 0, SideStr, Color, clNone, FocusedColor)
      else
        FTextDrawer.Draw(0, 0, SideStr, UnfocusedColor);
    end;
  end;
end;

procedure TViewport.SetCameraBookmark(AValue: string);
begin
  Camera.Bookmark:=AValue;
end;

procedure TViewport.SetCameraMode(AValue: TCameraMode);
begin
  Camera.Mode:=AValue;
end;

procedure TViewport.SetDirectionX(AValue: Double);
begin
  Camera.DirectionX:=AValue;
end;

procedure TViewport.SetDirectionY(AValue: Double);
begin
  Camera.DirectionY:=AValue;
end;

procedure TViewport.SetDirectionZ(AValue: Double);
begin
  Camera.DirectionZ:=AValue;
end;

procedure TViewport.SetDrawAxes(AValue: TAxes);
begin
  if FDrawAxes=AValue then Exit;
  FDrawAxes:=AValue;
  Invalidate;
end;

procedure TViewport.SetDrawCaption(AValue: Boolean);
begin
  if FDrawCaption=AValue then Exit;
  FDrawCaption:=AValue;
  Invalidate;
end;

procedure TViewport.SetDrawFocusOutline(AValue: Boolean);
begin
  if FDrawFocusOutline=AValue then Exit;
  FDrawFocusOutline:=AValue;
  Invalidate;
end;

procedure TViewport.SetDrawGrid(AValue: Boolean);
begin
  if FDrawGrid=AValue then Exit;
  FDrawGrid:=AValue;
  Invalidate;
end;

procedure TViewport.SetDrawSide(AValue: Boolean);
begin
  if FDrawSide=AValue then Exit;
  FDrawSide:=AValue;
  Invalidate;
end;

procedure TViewport.SetFocusedColor(AValue: TColor);
begin
  if FFocusedColor=AValue then Exit;
  FFocusedColor:=AValue;
  Invalidate;
end;

procedure TViewport.SetGridColor(AValue: TColor);
begin
  if FGridColor=AValue then Exit;
  FGridColor:=AValue;
  Invalidate;
end;

procedure TViewport.SetGridScale(AValue: Double);
begin
  if FGridScale=AValue then Exit;
  FGridScale:=AValue;
  Invalidate;
end;

procedure TViewport.SetGridSteps(AValue: Integer);
begin
  if FGridSteps=AValue then Exit;
  FGridSteps:=AValue;
  Invalidate;
end;

procedure TViewport.SetGridSubColor(AValue: TColor);
begin
  if FGridSubColor=AValue then Exit;
  FGridSubColor:=AValue;
  Invalidate;
end;

procedure TViewport.SetMultiSampling(AValue: Integer);
begin
  if AValue < 0 then AValue:=0;
  if AValue > 16 then AValue:=16;
  if FMultiSampling=AValue then Exit;
  FMultiSampling:=AValue;
  if Assigned(FGLControl) then FGLControl.MultiSampling:=AValue;
end;

procedure TViewport.SetOrthoScale(AValue: Double);
begin
  Camera.OrthoScale:=AValue;
end;

procedure TViewport.SetPicker(AValue: TViewportPicker);
begin
  if FPicker=AValue then Exit;
  FPicker:=AValue;
end;

procedure TViewport.SetPositionX(AValue: Double);
begin
  Camera.PositionX:=AValue;
end;

procedure TViewport.SetPositionY(AValue: Double);
begin
  Camera.PositionY:=AValue;
end;

procedure TViewport.SetPositionZ(AValue: Double);
begin
  Camera.PositionZ:=AValue;
end;

procedure TViewport.SetRenderer(AValue: TViewportRenderer);
begin
  if FRenderer=AValue then Exit;
  FRenderer:=AValue;
  Invalidate;
end;

procedure TViewport.SetSide(AValue: TViewportSide);
begin
  case AValue of
    vsLeft: Camera.Direction:=Vector(1, 0, 0);
    vsRight: Camera.Direction:=Vector(-1, 0, 0);
    vsTop: Camera.Direction:=Vector(0, -1, 0);
    vsBottom: Camera.Direction:=Vector(0, 1, 0);
    vsFront: Camera.Direction:=Vector(0, 0, -1);
    vsBack: Camera.Direction:=Vector(0, 0, 1);
    vsCustom: begin end;
  end;
end;

procedure TViewport.SetUnfocusedColor(AValue: TColor);
begin
  if FUnfocusedColor=AValue then Exit;
  FUnfocusedColor:=AValue;
  Invalidate;
end;

procedure TViewport.SetWidgetManager(AValue: T3DWidgetManager);
begin
  if FWidgetManager=AValue then Exit;
  FWidgetManager:=AValue;
  Invalidate;
end;

procedure TViewport.WMPaint(var Msg: TLMPaint);
var
  Canvas: TControlCanvas;
  SideStr: String;
begin
  Include(FControlState, csCustomPaint);
  inherited WMPaint(Msg);
  if csDesigning in ComponentState then begin
    Canvas:=TControlCanvas.Create;
    with Canvas do begin
      if Msg.DC <> 0 then Handle:=Msg.DC;
      Brush.Color:=clBlack;
      Pen.Color:=clGray;
      Rectangle(0, 0, Self.Width, Self.Height);
      Line(0, 0, Self.Width, Self.Height);
      Line(Self.Width, 0, 0, Self.Height);
      Font.Color:=clRed;
      SetBkMode(Handle, TRANSPARENT);
      case Side of
        vsCustom: SideStr:='(Custom)';
        vsLeft: SideStr:='(Left)';
        vsRight: SideStr:='(Right)';
        vsTop: SideStr:='(Top)';
        vsBottom: SideStr:='(Bottom)';
        vsFront: SideStr:='(Front)';
        vsBack: SideStr:='(Back)';
      end;
      TextOut(5, 5, Caption + ' ' + SideStr);
      if Msg.DC <> 0 then Handle:=0;
    end;
  end;
  Exclude(FControlState, csCustomPaint);
end;

procedure TViewport.WMEnter(var Message: TLMEnter);
begin
  inherited WMEnter(Message);
  FGLControl.SetFocus;
end;

procedure TViewport.CameraChange(Sender: TObject);
begin
  if not (AutoUpdate or FTempAutoUpdate) then Invalidate;
  if (not FRendering) and Assigned(FOnCameraChange) then FOnCameraChange(Self);
end;

constructor TViewport.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width:=320;
  Height:=240;
  ParentColor:=False;
  Color:=clBlack;
  TabStop:=False;
  NeedUpdateTimer;
  FAutoUpdate:=True;
  FFocusedColor:=clWhite;
  FUnfocusedColor:=clGray;
  FDrawGrid:=True;
  FDrawCaption:=True;
  FDrawSide:=True;
  FDrawAxes:=DefaultDrawAxes;
  FGridColor:=$00663319;
  FGridSubColor:=$00231E19;
  FGridSteps:=8;
  FGridScale:=64;
  FCamera:=TCamera.Create(Self);
  FCamera.AddChangeListener(@CameraChange);
  FCamera.Position:=Vector(0, 64, 0);
  FCameraMotion:=vcmFirstPerson;
  FCameraSpeed:=1.0;
  FTextDrawer:=TGLTextDrawer.Create(Self);
  if not (csDesigning in ComponentState) then begin
    CreateOpenGLControl;
    if not Assigned(ViewportList) then ViewportList:=TViewportList.Create(False);
    ViewportList.Add(Self);
  end;
end;

destructor TViewport.Destroy;
begin
  FCamera.RemoveChangeListener(@CameraChange);
  if not (csDesigning in ComponentState) then ViewportList.Remove(Self);
  if LastActiveViewport=Self then LastActiveViewport:=nil;
  inherited Destroy;
end;

procedure TViewport.Invalidate;
begin
  inherited Invalidate;
  if Assigned(FGLControl) then FGLControl.Invalidate;
end;

procedure TViewport.UpdateViewport;
var
  Speed: Double;
begin
  if csDesigning in ComponentState then Exit;
  if CameraMotion=vcmFirstPerson then begin
    Speed:=CameraSpeed;
    if IsKeyDown(VK_LSHIFT) or IsKeyDown(VK_RSHIFT) or IsKeyDown(VK_SHIFT) then Speed *= 5;
    if IsKeyDown(65) then Camera.MoveTowardsInView(Vector(-Speed, 0, 0));
    if IsKeyDown(68) then Camera.MoveTowardsInView(Vector(Speed, 0, 0));
    if CameraMode=cmPerspective then begin
      if IsKeyDown(87) then Camera.MoveForward(Speed);
      if IsKeyDown(83) then Camera.MoveForward(-Speed);
    end else begin
      if IsKeyDown(87) then Camera.OrthoScale:=Camera.OrthoScale - Speed;
      if IsKeyDown(83) then Camera.OrthoScale:=Camera.OrthoScale + Speed;
    end;
    if IsKeyDown(81) then Camera.Position:=Camera.Position.Added(Vector(0, Speed, 0));
    if IsKeyDown(69) then Camera.Position:=Camera.Position.Added(Vector(0, -Speed, 0));
  end;
  if Assigned(FOnUpdate) then FOnUpdate(Self);
  Invalidate;
end;

procedure TViewport.RenderViewport;
begin
  {$IFDEF DARWIN}
  Invalidate;
  {$ELSE}
  GLCPaint(FGLControl);
  {$ENDIF}
end;

function TViewport.IsKeyDown(Key: Integer): Boolean;
begin
  if Key in [0..255] then Result:=FKeyState[Key] else Result:=False;
end;

function TViewport.IsFocused: Boolean;
begin
  Result:=FGLControl.Focused;
end;

procedure TViewport.TogglePerspective;
begin
  if CameraMode=cmPerspective then
    CameraMode:=cmOrthographic
  else
    CameraMode:=cmPerspective;
end;

function TViewport.RayAt(X, Y: Integer): TRay;
begin
  Result.FromSegment(ViewToWorld(Vector(X, Y, 0.1)),
                     ViewToWorld(Vector(X, Y, 0.9)));
end;

function TViewport.WorldToView(const APoint: TVector): TVector;
begin
  Result:=Camera.ProjectionMatrix.Multiplied(Camera.ModelViewMatrix).TransformedProj(APoint);
  Result.x:=(Result.x + 1.0)*0.5*Width;
  Result.y:=(1.0 - Result.y)*0.5*Height;
  Result.z:=(Result.z + 1.0)*0.5;
end;

function TViewport.ViewToWorld(const APoint: TVector): TVector;
begin
  Result.x:=APoint.x/Width*2.0-1.0;
  Result.y:=(Height - APoint.y)/Height*2.0-1.0;
  Result.z:=APoint.z*2.0-1.0;
  Camera.UnprojectionMatrix.TransformProj(Result);
end;

procedure TViewport.SetFocus;
begin
  if Assigned(FGLControl) then
    FGLControl.SetFocus
  else begin
    FSetFocusLater:=True;
    inherited SetFocus;
  end;
end;

function TViewport.GetForwardDirection: TVector;
begin
  Result:=Camera.Direction;
end;

function TViewport.GetUpDirection: TVector;
begin
  Result:=Camera.Up;
end;

function TViewport.GetRightDirection: TVector;
begin
  Result:=Camera.Right;
end;

function TViewport.GetPixelScale(const APosition: TVector): Double;
var
  PosInView, BackInWorld: TVector;
begin
  PosInView:=WorldToView(APosition);
  BackInWorld:=ViewToWorld(PosInView);
  PosInView.X += 1;
  Result:=Distance(BackInWorld, ViewToWorld(PosInView));
end;

function TViewport.IsActiveEnvironment: Boolean;
begin
  Result:=LastActiveViewport=Self;
end;

initialization
  UpdateFPS:=60;
  LastTicks:=GetTickCount64;
finalization
  FreeAndNil(RegistryList);
  FreeAndNil(ViewportList);
end.

