unit MiscUtils;

{$mode objfpc}{$H+}
{$hints off}
{$warnings off}

interface

uses
  Classes, SysUtils, Graphics, FGL;

type
  TStringArray = array of string;
  TNotifyEventList = specialize TFPGList<TNotifyEvent>;

var
  NormalFormatSettings: TFormatSettings;

function GetTextFileContents(FileName: string): string;
function SetTextFileContents(FileName, Data: string): Boolean;
function LoadImageFile(FileName: string): TGraphic;
function CanLoadImageFile(FileName: string): Boolean;
function GetFilesInDirectory(Path: string; Mask: string=AllFilesMask): TStringArray;
procedure MemZero(out V; Size: SizeUInt); inline;
procedure SplitString(S: string; Sep: Char; out Parts: TStringArray; SkipSpaces: Boolean=True);
function FloatToString(V: Double): string; inline;
function StringToFloat(s: string; DefValue: Double=0): Double; inline;
function TakeScreenshot: TBitmap;

implementation

uses
  LCLIntf, LCLType;

function GetTextFileContents(FileName: string): string;
var
  Strings: TStringList = nil;
begin
  try
    Strings:=TStringList.Create;
    Strings.LoadFromFile(FileName);
    Result:=Strings.Text;
  except
    Result:='';
  end;
  if Assigned(Strings) then FreeAndNil(Strings);
end;

function SetTextFileContents(FileName, Data: string): Boolean;
var
  Strings: TStringList = nil;
begin
  Result:=False;
  try
    Strings:=TStringList.Create;
    Strings.Text:=Data;
    Strings.SaveToFile(FileName);
    Result:=True;
  except
  end;
  if Assigned(Strings) then FreeAndNil(Strings);
end;

function LoadImageFile(FileName: string): TGraphic;
var
  Pic: TPicture;
  Gfx: TGraphic;
begin
  Pic:=TPicture.Create;
  Result:=nil;
  try
    Pic.LoadFromFile(FileName);
    if Assigned(Pic.Graphic) then begin
      Gfx:=TGraphic(Pointer(Pic.Graphic.NewInstance));
      Gfx.Create;
      Gfx.Assign(Pic.Graphic);
      Result:=Gfx;
    end;
  except
    FreeAndNil(Result);
  end;
end;

function CanLoadImageFile(FileName: string): Boolean;
var
  Pic: TPicture;
  Ext: string;
begin
  Ext:=ExtractFileExt(FileName);
  if Ext='' then Exit(False);
  if Ext[1]='.' then Ext:=Copy(Ext, 2, Length(Ext));
  if Ext='' then Exit(False);
  Pic:=TPicture.Create;
  try
    Result:=Assigned(Pic.FindGraphicClassWithFileExt(Ext, False));
  except
    Result:=False;
  end;
  FreeAndNil(Pic);
end;

function GetFilesInDirectory(Path: string; Mask: string): TStringArray;
var
  Fi: TRawByteSearchRec;
  R: LongInt;
begin
  if (Path <> '') and not (Path[Length(Path)] in ['\', '/']) then Path += DirectorySeparator;
  R:=FindFirst(Path + Mask, faAnyFile, Fi);
  SetLength(Result, 0);
  if R=0 then begin
    while R=0 do begin
      SetLength(Result, Length(Result) + 1);
      Result[High(Result)]:=ExpandFileName(Path + Fi.Name);
      R:=FindNext(Fi);
    end;
    FindClose(Fi);
  end;
end;

procedure MemZero(out V; Size: SizeUInt);
var
  Quads: SizeUInt;
begin
  // The weird (@V)^ is to avoid FPC's warning about unitialized parameter
  // since FillChar is defined to use "var" instead of "out"
  Quads:=Size div 8;
  FillQWord((@V)^, Quads, 0);
  FillChar(((@V) + Int64(Quads)*8)^, Int64(Size) - Int64(Quads)*8, 0);
end;

procedure SplitString(S: string; Sep: Char; out Parts: TStringArray; SkipSpaces: Boolean);
var
  I: Integer;
begin
  SetLength(Parts, 1);
  I:=1;
  while I <= Length(S) do begin
    if SkipSpaces and (I <= Length(S)) and (S[I] in [#9, ' ']) then begin
      while (I <= Length(S)) and (S[I] in [#9, ' ']) do Inc(I);
      if I > Length(S) then Break;
      if Sep in [#9, ' '] then begin
        SetLength(Parts, Length(Parts) + 1);
        Continue;
      end;
    end;
    if I > Length(S) then Break;
    if S[I]=Sep then begin
      Inc(I);
      SetLength(Parts, Length(Parts) + 1);
      Continue;
    end;
    Parts[High(Parts)]:=Parts[High(Parts)] + S[I];
    Inc(I);
  end;
end;

function FloatToString(V: Double): string;
begin
  Result:=FloatToStr(V, NormalFormatSettings);
end;

function StringToFloat(s: string; DefValue: Double): Double;
begin
  Result:=StrToFloatDef(s, DefValue, NormalFormatSettings);
end;

function TakeScreenshot: TBitmap;
var
  ScreenDC: HDC;
begin
  Result:=TBitmap.Create;
  ScreenDC:=GetDC(0);
  Result.LoadFromDevice(ScreenDC);
  ReleaseDC(0, ScreenDC);
end;

initialization
  NormalFormatSettings:=DefaultFormatSettings;
  NormalFormatSettings.DecimalSeparator:='.';
end.

