unit KeyValueStore;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

type
  TKeyValueStoreChangeType = (ctAdd, ctSet, ctRemove, ctRename);
  TKeyValueStoreChangeEvent = procedure(Sender: TObject; AType: TKeyValueStoreChangeType; AKeyName, AKeyValue: string; var Veto: Boolean) of object;

  { TKeyValueStore }

  TKeyValueStore = class(TComponent)
  private
    FKeys, FValues: array of string;
    FRemoveEmpty: Boolean;
    FOnChange: TKeyValueStoreChangeEvent;
    function GetCount: Integer; inline;
    function GetEntry(AKey: string): string;
    function GetKeys(AIndex: Integer): string; inline;
    function GetValues(AIndex: Integer): string; inline;
    procedure SetEntry(AKey: string; AValue: string);
    procedure SetKeys(AIndex: Integer; AValue: string); inline;
    procedure SetValues(AIndex: Integer; AValue: string); inline;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Clear;
    function HasKey(AKey: string): Boolean; inline;
    function IndexOf(AKey: string): Integer;
    procedure RemoveKey(AKey: string);
    procedure RemoveKeyAt(AIndex: Integer);
    property Keys[AIndex: Integer]: string read GetKeys write SetKeys;
    property Values[AIndex: Integer]: string read GetValues write SetValues;
    property Entry[AKey: string]: string read GetEntry write SetEntry; default;
    property Count: Integer read GetCount;
  published
    property RemoveEmpty: Boolean read FRemoveEmpty write FRemoveEmpty default True;
    property OnChange: TKeyValueStoreChangeEvent read FOnChange write FOnChange;
  end;

procedure Register;

implementation

uses
  LResources;

procedure Register;
begin
  {$I keyvaluestore_icon.lrs}
  RegisterComponents('RTTK', [TKeyValueStore]);
end;

{ TKeyValueStore }

function TKeyValueStore.GetCount: Integer;
begin
  Result:=Length(FKeys);
end;

function TKeyValueStore.GetEntry(AKey: string): string;
var
  I: Integer;
begin
  for I:=0 to High(FKeys) do
    if FKeys[I]=AKey then Exit(FValues[I]);
end;

function TKeyValueStore.GetKeys(AIndex: Integer): string;
begin
  Result:=FKeys[AIndex];
end;

function TKeyValueStore.GetValues(AIndex: Integer): string;
begin
  Result:=FValues[AIndex];
end;

procedure TKeyValueStore.SetEntry(AKey: string; AValue: string);
var
  I: Integer;
  Veto: Boolean = False;
begin
  if RemoveEmpty and (AValue='') then begin
    RemoveKey(AKey);
    Exit;
  end;
  for I:=0 to High(FKeys) do
    if FKeys[I]=AKey then begin
      if Assigned(FOnChange) then FOnChange(Self, ctSet, AKey, AValue, Veto);
      if Veto then Exit;
      FValues[I]:=AValue;
      Exit;
    end;
  if Assigned(FOnChange) then begin
    FOnChange(Self, ctAdd, AKey, AValue, Veto);
    if Veto then Exit;
    FOnChange(Self, ctSet, AKey, AValue, Veto);
    if Veto then Exit;
  end;
  SetLength(FKeys, Length(FKeys) + 1);
  SetLength(FValues, Length(FValues) + 1);
  FKeys[High(FKeys)]:=AKey;
  FValues[High(FKeys)]:=AValue;
end;

procedure TKeyValueStore.SetKeys(AIndex: Integer; AValue: string);
var
  OldName: string;
  Veto: Boolean = False;
begin
  if Assigned(FOnChange) then OldName:=FKeys[AIndex];
  FKeys[AIndex]:=AValue;
  if Assigned(FOnChange) then begin
    FOnChange(Self, ctRename, OldName, FKeys[AIndex], Veto);
    if Veto then FKeys[AIndex]:=OldName;
  end;
end;

procedure TKeyValueStore.SetValues(AIndex: Integer; AValue: string);
var
  Veto: Boolean;
begin
  if RemoveEmpty and (AValue='') then begin
    RemoveKeyAt(AIndex);
    Exit;
  end;
  if Assigned(FOnChange) then begin
    FOnChange(Self, ctSet, FKeys[AIndex], AValue, Veto);
    if Veto then Exit;
  end;
  FValues[AIndex]:=AValue;
end;

constructor TKeyValueStore.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FRemoveEmpty:=True;
end;

procedure TKeyValueStore.Clear;
var
  I: Integer;
  Veto: Boolean;
begin
  if Assigned(FOnChange) then begin
    for I:=High(FKeys) downto 0 do begin
      Veto:=False;
      FOnChange(Self, ctSet, FKeys[I], '', Veto);
      if Veto then Continue;
      Veto:=False;
      FOnChange(Self, ctRemove, FKeys[I], FValues[I], Veto);
      if Veto then Continue;
      RemoveKeyAt(I);
    end;
  end else begin
    SetLength(FKeys, 0);
    SetLength(FValues, 0);
  end;
end;

function TKeyValueStore.HasKey(AKey: string): Boolean;
begin
  Result:=IndexOf(AKey) <> -1;
end;

function TKeyValueStore.IndexOf(AKey: string): Integer;
var
  I: Integer;
begin
  for I:=0 to High(FKeys) do
    if FKeys[I]=AKey then Exit(I);
  Result:=-1;
end;

procedure TKeyValueStore.RemoveKey(AKey: string);
var
  Index: Integer;
begin
  Index:=IndexOf(AKey);
  if Index=-1 then Exit;
  RemoveKeyAt(Index);
end;

procedure TKeyValueStore.RemoveKeyAt(AIndex: Integer);
var
  I: Integer;
  Veto: Boolean = False;
begin
  if Assigned(FOnChange) then begin
    FOnChange(Self, ctSet, FKeys[AIndex], '', Veto);
    if Veto then Exit;
    FOnChange(Self, ctRemove, FKeys[AIndex], '', Veto);
    if Veto then Exit;
  end;
  for I:=AIndex to High(FKeys) - 1 do begin
    FKeys[I]:=FKeys[I + 1];
    FValues[I]:=FValues[I + 1];
  end;
  SetLength(FKeys, Length(FKeys) - 1);
  SetLength(FValues, Length(FKeys));
end;

end.

