unit GameInfos;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, IniFiles;

type

  { TGameInfo }

  TGameInfo = class(TComponent)
  private
    FDataPath: string;
    FDefaultGameInfo: Boolean;
    FINIFilePath: string;
    FTitle: string;
    FINI: TIniFile;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure ReadFromINIFile(AINIFilePath: string);

    property Title: string read FTitle;
    property DataPath: string read FDataPath;
    property INIFilePath: string read FINIFilePath;
    property INI: TIniFile read FINI;
    property DefaultGameInfo: Boolean read FDefaultGameInfo;
  end;

  { TGameInfoRegistry }

  TGameInfoRegistry = class(TComponent)
  private
    FGameInfos: array of TGameInfo;
    FDefaultGameInfo: TGameInfo;
    function GetCount: Integer;
    function GetGameInfo(AIndex: Integer): TGameInfo;
    procedure ScanGameInfosDirectory;
    procedure ProcessGameInfoFile(AINIFilePath: string);
    procedure UpdateDefaultGameInfo;
  public
    constructor Create(AOwner: TComponent); override;
    property Count: Integer read GetCount;
    property GameInfo[AIndex: Integer]: TGameInfo read GetGameInfo; default;
    property DefaultGameInfo: TGameInfo read FDefaultGameInfo;
  end;

procedure Register;

implementation

uses
  LResources, Dialogs;

procedure Register;
begin
  {$I gameinfos_icon.lrs}
  RegisterComponents('RTTK',[TGameInfoRegistry]);
end;

constructor TGameInfo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

destructor TGameInfo.Destroy;
begin
  FreeAndNil(FINI);
  inherited Destroy;
end;

procedure TGameInfo.ReadFromINIFile(AINIFilePath: string);
begin
  FreeAndNil(FINI);
  try
    FINI:=TIniFile.Create(AINIFilePath);
  except
    MessageDlg('Invalid INI', 'Failed to read ' + AINIFilePath, mtError, [mbOK], 0);
    FTitle:='(error) ' + ExtractFilename(AINIFilePath);
    FINI:=nil;
    Exit;
  end;
  try
    FINIFilePath:=AINIFilePath;
    FTitle:=FINI.ReadString('Game', 'Title', ExtractFileName(AINIFilePath));
    FDataPath:=FINI.ReadString('Data', 'Path', '');
    FDefaultGameInfo:=FINI.ReadBool('Game', 'Default', False);
  except
  end;
end;

{ TGameInfoRegistry }

function TGameInfoRegistry.GetCount: Integer;
begin
  Result:=Length(FGameInfos);
end;

function TGameInfoRegistry.GetGameInfo(AIndex: Integer): TGameInfo;
begin
  Result:=FGameInfos[AIndex];
end;

procedure TGameInfoRegistry.ScanGameInfosDirectory;
var
  r: LongInt;
  FI: TSearchRec;
begin
  r:=FindFirst('gameinfo' + DirectorySeparator + '*.ini', faAnyFile - faVolumeId, FI);
  if r=0 then begin
    while r=0 do begin
      ProcessGameInfoFile(ExpandFileName('gameinfo' + DirectorySeparator + FI.Name));
      r:=FindNext(FI);
    end;
    FindClose(FI);
  end;
  UpdateDefaultGameInfo;
end;

procedure TGameInfoRegistry.ProcessGameInfoFile(AINIFilePath: string);
var
  GI: TGameInfo;
begin
  GI:=TGameInfo.Create(Self);
  GI.ReadFromINIFile(AINIFilePath);
  SetLength(FGameInfos, Length(FGameInfos) + 1);
  FGameInfos[High(FGameInfos)]:=GI;
end;

procedure TGameInfoRegistry.UpdateDefaultGameInfo;
var
  I: Integer;
begin
  FDefaultGameInfo:=nil;
  for I:=High(FGameInfos) downto 0 do
    if FGameInfos[I].DefaultGameInfo then begin
      FDefaultGameInfo:=FGameInfos[I];
      Break;
    end;
end;

constructor TGameInfoRegistry.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ScanGameInfosDirectory;
end;

end.

