unit Events;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

type
  TEventHandler = procedure(ASender, ASubject, AObject: TObject; AMessage: string; var Handled: Boolean) of object;
  TEventHandlerProc = procedure(ASender, ASubject, AObject: TObject; AMessage: string; var Handled: Boolean);

procedure RegisterEventHandler(AMessage: string; AEventHandler: TEventHandler);
procedure UnregisterEventHandler(AMessage: string; AEventHandler: TEventHandler);
procedure RegisterEventHandler(AMessage: string; AEventHandler: TEventHandlerProc);
procedure UnregisterEventHandler(AMessage: string; AEventHandler: TEventHandlerProc);
procedure CallEventHandlers(ASender, ASubject, AObject: TObject; AMessage: string);

implementation

type
  THandlerEntry = record
    Handler: TEventHandler;
    Proc: TEventHandlerProc;
  end;

  TMessageHandlers = record
    Message: string;
    Handlers: array of THandlerEntry;
  end;

var
  Messages: array of TMessageHandlers;

function FindMessage(AMessage: string; Add: Boolean): Integer;
var
  i: Integer;
begin
  Result:=-1;
  for i:=0 to Length(Messages) - 1 do
    if Messages[i].Message=AMessage then Exit(i);
  if Add then begin
    Result:=Length(Messages);
    SetLength(Messages, Result + 1);
    Messages[Result].Message:=AMessage;
    SetLength(Messages[Result].Handlers, 0);
  end;
end;

procedure RegisterEventHandler(AMessage: string; AEventHandler: TEventHandler);
var
  MsgIndex, i: Integer;
begin
  MsgIndex:=FindMessage(AMessage, True);
  i:=Length(Messages[MsgIndex].Handlers);
  SetLength(Messages[MsgIndex].Handlers, i + 1);
  Messages[MsgIndex].Handlers[i].Handler:=AEventHandler;
  Messages[MsgIndex].Handlers[i].Proc:=nil;
end;

procedure UnregisterEventHandler(AMessage: string; AEventHandler: TEventHandler);
var
  MsgIndex, i, Idx: Integer;
begin
  MsgIndex:=FindMessage(AMessage, True);
  if MsgIndex=-1 then Exit;
  Idx:=-1;
  for i:=0 to Length(Messages[MsgIndex].Handlers) - 1 do
    if Messages[MsgIndex].Handlers[i].Handler=AEventHandler then begin
      Idx:=i;
      Break;
    end;
  if Idx=-1 then Exit;
  for i:=Idx to Length(Messages[MsgIndex].Handlers) - 2 do
    Messages[MsgIndex].Handlers[i]:=Messages[MsgIndex].Handlers[i + 1];
  SetLength(Messages[MsgIndex].Handlers, Length(Messages[MsgIndex].Handlers) - 1);
end;

procedure RegisterEventHandler(AMessage: string; AEventHandler: TEventHandlerProc);
var
  MsgIndex, i: Integer;
begin
  MsgIndex:=FindMessage(AMessage, False);
  i:=Length(Messages[MsgIndex].Handlers);
  SetLength(Messages[MsgIndex].Handlers, i + 1);
  Messages[MsgIndex].Handlers[i].Handler:=nil;
  Messages[MsgIndex].Handlers[i].Proc:=AEventHandler;
end;

procedure UnregisterEventHandler(AMessage: string; AEventHandler: TEventHandlerProc);
var
  MsgIndex, i, Idx: Integer;
begin
  MsgIndex:=FindMessage(AMessage, True);
  if MsgIndex=-1 then Exit;
  Idx:=-1;
  for i:=0 to Length(Messages[MsgIndex].Handlers) - 1 do
    if Messages[MsgIndex].Handlers[i].Proc=AEventHandler then begin
      Idx:=i;
      Break;
    end;
  if Idx=-1 then Exit;
  for i:=Idx to Length(Messages[MsgIndex].Handlers) - 1 do
    Messages[MsgIndex].Handlers[i]:=Messages[MsgIndex].Handlers[i + 1];
  SetLength(Messages[MsgIndex].Handlers, Length(Messages[MsgIndex].Handlers) - 1);
end;

procedure CallEventHandlers(ASender, ASubject, AObject: TObject; AMessage: string);
var
  MsgIndex, i: Integer;
  Handled: Boolean = False;
begin
  if IsConsole then Writeln('Event "', AMessage, '" from ', HexStr(ASender), ' about ', HexStr(ASubject), ' relating to ', HexStr(AObject));
  MsgIndex:=FindMessage(AMessage, False);
  if MsgIndex > -1 then begin
    for i:=Length(Messages[MsgIndex].Handlers) - 1 downto 0 do begin
      if Assigned(Messages[MsgIndex].Handlers[i].Handler) then
          Messages[MsgIndex].Handlers[i].Handler(ASender, ASubject, AObject, AMessage, Handled)
      else
          Messages[MsgIndex].Handlers[i].Proc(ASender, ASubject, AObject, AMessage, Handled);
      if Handled then Break;
    end;
  end;
end;

end.

