unit LazLILGUI;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, Controls, StdCtrls, ExtCtrls, ComCtrls, Spin, Dialogs,
  Forms, Graphics, FGL, FPLIL;

type
  TLILGUIContainerControl = class;
  TLazLILGUI = class;
  ELazLILGUIError = class(Exception);

  { TLILGUIControl }

  TLILGUIControl = class(TComponent)
  private
    FLCLControl: TControl;
    FControlName: string;
    FID: QWord;
    FGUI: TLazLILGUI;
    function GetParent: TLILGUIControl; inline;
  protected
    procedure SetupControl; virtual;
    function SetupVar(AVarName: string; AWatch: TNotifyEvent): TLILVariable;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ToValue: TLILValue;
    procedure SetBounds(X, Y, W, H: Integer); virtual;
    procedure GetPreferredSize(PW, PH: PInteger); virtual;
    property GUI: TLazLILGUI read FGUI;
    property Parent: TLILGUIControl read GetParent;
    property LCLControl: TControl read FLCLControl;
    property ID: QWord read FID;
    property ControlName: string read FControlName write FControlName;
  end;

  TLILGUIControlClass = class of TLILGUIControl;
  TLILGUIControlList = specialize TFPGObjectList<TLILGUIControl>;

  { TLILGUILCLHostControl }

  TLILGUILCLHostControl = class(TLILGUIControl)
  private
    Container: TLILGUIContainerControl;
    procedure OnControlDestroy(Sender: TObject);
  protected
    procedure SetupControl; override;
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUIContainerControl }

  TLILGUIContainerControl = class(TLILGUIControl)
  private
    FColumns: Integer;
    LayoutMode: (lmVStack, lmHStack, lmBorder, lmGrid);
    FSpacing: Integer;
    BLeft, BTop, BRight, BBottom, BCenter: TLILGUIControl;
    Expand: Boolean;
    procedure DoLayout;
    procedure DoFullLayout;
    function GetSpacing: Integer;
    procedure OnResize(Sender: TObject);
    procedure SetColumns(AValue: Integer);
    procedure SetSpacing(AValue: Integer);
  protected
    procedure SetupControl; override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure GetPreferredSize(PW, PH: PInteger); override;
    procedure BeforeDestruction; override;
    property Spacing: Integer read GetSpacing write SetSpacing;
    property Columns: Integer read FColumns write SetColumns;
  end;

  { TLILGUIButtonControl }

  TLILGUIButtonControl = class(TLILGUIControl)
  private
    Code: string;
    procedure OnClick(Sender: TObject);
  protected
    procedure SetupControl; override;
  end;

  { TLILGUIEditControl }

  TLILGUIEditControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnChange(Sender: TObject);
    procedure OnVarChange(Sender: TObject);
  public
    procedure GetPreferredSize(PW, PH: PInteger); override;
    procedure BeforeDestruction; override;
  end;

  { TLILGUICheckControl }

  TLILGUICheckControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnChange(Sender: TObject);
    procedure OnVarChange(Sender: TObject);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUIRadioControl }

  TLILGUIRadioControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    Value: string;
    procedure OnChange(Sender: TObject);
    procedure OnVarChange(Sender: TObject);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUIListBoxControl }

  TLILGUIListBoxControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnChange(Sender: TObject; User: Boolean);
    procedure OnVarChange(Sender: TObject);
    procedure SetItemsFromValue(AValue: TLILValue);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUIChoiceControl }

  TLILGUIChoiceControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnChange(Sender: TObject);
    procedure OnVarChange(Sender: TObject);
    procedure SetItemsFromValue(AValue: TLILValue);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUIProgressControl }

  TLILGUIProgressControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnVarChange(Sender: TObject);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUISliderControl }

  TLILGUISliderControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnChange(Sender: TObject);
    procedure OnVarChange(Sender: TObject);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUISpinnerControl }

  TLILGUISpinnerControl = class(TLILGUIControl)
  private
    ValueVar: TLILVariable;
    procedure OnChange(Sender: TObject);
    procedure OnVarChange(Sender: TObject);
  public
    procedure BeforeDestruction; override;
  end;

  { TLILGUIWindow }

  TLILGUIWindow = class(TForm)
  private
    ID: QWord;
    Persistent: Boolean;
    HostControl: TLILGUILCLHostControl;
  protected
    procedure DoClose(var CloseAction: TCloseAction); override;
  end;

  { TLazLILGUI }

  TLazLILGUI = class(TComponent)
  private
    FCurrentParent: TLILGUIControl;
    FLCLHost: TWinControl;
    FRoot: TLILGUIControl;
    FLastControl: TLILGUIControl;
    FLastLabel: TLILGUIControl;
    FLIL: TLIL;
    FMakingWindow: TLILGUIWindow;
    function GetCurrentParent: TLILGUIControl;
    procedure SetLCLHost(AValue: TWinControl);
    procedure SetLIL(AValue: TLIL);
    procedure RegisterFunctions;
    function FindControlForLCLHost(ALCLHost: TWinControl; AddNew: Boolean): TLILGUILCLHostControl;
    function FindWindowByID(ID: QWord): TLILGUIWindow;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ControlByID(ID: QWord): TLILGUIControl;
    function ControlByName(AName: string): TLILGUIControl;
    function ControlByValue(ALILValue: TLILValue): TLILGUIControl;
    function CreateGUIControl(ALCLControl: TControl; AClass: TLILGUIControlClass; OverrideParent: TLILGUIControl=nil): TLILGUIControl;
    function NameFromCaption(ACaption: string): string;
    procedure AddLCLHost(ALCLHost: TWinControl; MakeCurrent: Boolean);
    procedure RemoveLCLHost(ALCLHost: TWinControl);
    property Root: TLILGUIControl read FRoot;
    property CurrentParent: TLILGUIControl read GetCurrentParent;
    property LastControl: TLILGUIControl read FLastControl;
    property LastLabel: TLILGUIControl read FLastLabel;
  published
    property LIL: TLIL read FLIL write SetLIL;
    property LCLHost: TWinControl read FLCLHost write SetLCLHost;
  end;

procedure Register;

implementation

uses
  LResources;

type
  TLazLILGUIList = specialize TFPGObjectList<TLazLILGUI>;

var
  Instances: TLazLILGUIList;
  LastID: QWord;

procedure Register;
begin
  RegisterComponents('Misc', [TLazLILGUI]);
end;

{ Global }

function InstanceFromLIL(LIL: TLIL): TLazLILGUI;
var
  I: Integer;
begin
  for I:=0 to Instances.Count - 1 do
    if Instances[I].LIL=LIL then Exit(Instances[I]);
  raise ELazLILGUIError.Create('LazLILGUI instance for the given LIL instance not found');
  Result:=nil;
end;

function ControlValueOrNil(Control: TLILGUIControl): TLILValue;
begin
  if Assigned(Control) then
    Result:=Control.ToValue
  else
    Result:=nil;
end;

{ LIL functions }
function FncControl(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
begin
  GUI:=InstanceFromLIL(LIL);
  if Length(Args)=0 then
    Result:=ControlValueOrNil(GUI.LastControl)
  else
    Result:=ControlValueOrNil(GUI.ControlByValue(Args[0]));
end;

function FncParent(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Control: TLILGUIControl;
  SV: string;
begin
  GUI:=InstanceFromLIL(LIL);
  if Length(Args)=0 then Exit(ControlValueOrNil(GUI.CurrentParent));
  SV:=TLIL.ToString(Args[0]);
  if SV='up' then begin
    if Assigned(GUI.CurrentParent) and
       Assigned(GUI.CurrentParent.Parent) and
       Assigned(GUI.CurrentParent.Parent.LCLControl) then
      GUI.FCurrentParent:=GUI.CurrentParent.Parent;
    Exit(nil);
  end;
  if SV='of' then begin
    if Length(Args)=1 then begin
      LIL.SetError('missing control in parent of');
      Exit(nil);
    end;
    Control:=GUI.ControlByValue(Args[1]);
    if Assigned(Control) and Assigned(Control.Parent) then
      Exit(Control.Parent.ToValue)
    else
      Exit(nil);
  end;
  Control:=GUI.ControlByValue(Args[0]);
  if not Assigned(Control) then begin
    LIL.SetError('invalid control');
    Exit(nil);
  end;
  if not Assigned(Control.LCLControl) then begin
    LIL.SetError('cannot use the given control as a parent');
    Exit(nil);
  end;
  GUI.FCurrentParent:=Control;
  Result:=nil;
end;

function FncName(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Control: TLILGUIControl;
begin
  GUI:=InstanceFromLIL(LIL);
  if Length(Args)=1 then
    GUI.LastControl.ControlName:=TLIL.ToString(Args[0])
  else if Length(Args) > 1 then begin
    Control:=GUI.ControlByValue(Args[0]);
    if Assigned(Control) then
      Control.ControlName:=TLIL.ToString(Args[1])
    else
      LIL.SetError('Invalid control');
  end else
    LIL.SetError('Missing control name');
  Result:=nil;
end;

function FncShow(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Control: TLILGUIControl;
begin
  GUI:=InstanceFromLIL(LIL);
  if Length(Args)=0 then
    Control:=GUI.LastControl
  else
    Control:=GUI.ControlByValue(Args[0]);
  if Assigned(Control) then begin
    if Assigned(Control.LCLControl) then Control.LCLControl.Visible:=True;
  end else
    LIL.SetError('Invalid control');
  Result:=nil;
end;

function FncHide(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Control: TLILGUIControl;
begin
  GUI:=InstanceFromLIL(LIL);
  if Length(Args)=0 then
    Control:=GUI.LastControl
  else
    Control:=GUI.ControlByValue(Args[0]);
  if Assigned(Control) then begin
    if Assigned(Control.LCLControl) then Control.LCLControl.Visible:=False;
  end else
    LIL.SetError('Invalid control');
  Result:=nil;
end;

function FncButton(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Caption, Code: string;
  LCLControl: TButton;
  Control: TLILGUIControl;
begin
  GUI:=InstanceFromLIL(LIL);
  Caption:='';
  Code:='';
  if Length(Args) > 0 then Caption:=TLIL.ToString(Args[0]);
  if Length(Args) > 1 then Code:=TLIL.ToString(Args[1]);
  LCLControl:=TButton.Create(LIL);
  LCLControl.Caption:=Caption;
  Control:=GUI.CreateGUIControl(LCLControl, TLILGUIButtonControl);
  TLILGUIButtonControl(Control).Code:=Code;
  Result:=Control.ToValue;
end;

function FncLabel(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  LCLControl: TLabel;
  Control: TLILGUIControl;
begin
  GUI:=InstanceFromLIL(LIL);
  LCLControl:=TLabel.Create(LIL);
  if Length(Args) > 0 then
    LCLControl.Caption:=TLIL.ToString(Args[0])
  else
    LCLControl.Caption:='';
  LCLControl.AutoSize:=False;
  LCLControl.Layout:=tlCenter;
  Control:=GUI.CreateGUIControl(LCLControl, TLILGUIControl);
  GUI.FLastLabel:=Control;
  Result:=Control.ToValue;
end;

function FncTarget(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Target: TLILGUIControl;
begin
  GUI:=InstanceFromLIL(LIL);
  if Assigned(GUI.LastLabel) then begin
    if Length(Args)=0 then
      Target:=GUI.LastControl
    else
      Target:=GUI.ControlByValue(Args[0]);
    if Assigned(Target) then begin
      if Target <> GUI.LastLabel then begin
        if Target.LCLControl is TWinControl then
          TLabel(GUI.LastLabel.LCLControl).FocusControl:=TWinControl(Target.LCLControl)
        else
          LIL.SetError('The given control cannot be used as a target for a label');
      end else
        LIL.SetError('Cannot set the label as a target for itself');
    end else
      LIL.SetError('Invalid control given to target');
  end else
    LIL.SetError('No label created recently');
  Result:=nil;
end;

function FncEdit(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName: string;
  LCLControl: TEdit;
  Control: TLILGUIEditControl;
begin
  GUI:=InstanceFromLIL(LIL);
  VarName:='';
  if Length(Args) > 0 then VarName:=TLIL.ToString(Args[0]);
  LCLControl:=TEdit.Create(LIL);
  LCLControl.AutoSize:=False;
  Control:=TLILGUIEditControl(GUI.CreateGUIControl(LCLControl, TLILGUIEditControl));
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Text:=TLIL.ToString(Control.ValueVar.Value);
  end;
  LCLControl.OnChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncText(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName: string;
  LCLControl: TMemo;
  Control: TLILGUIEditControl;
begin
  GUI:=InstanceFromLIL(LIL);
  VarName:='';
  if Length(Args) > 0 then VarName:=TLIL.ToString(Args[0]);
  LCLControl:=TMemo.Create(LIL);
  LCLControl.WantReturns:=True;
  LCLControl.WantTabs:=True;
  LCLControl.ScrollBars:=ssAutoBoth;
  LCLControl.WordWrap:=False;
  LCLControl.AutoSize:=False;
  Control:=TLILGUIEditControl(GUI.CreateGUIControl(LCLControl, TLILGUIEditControl));
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Text:=TLIL.ToString(Control.ValueVar.Value);
  end;
  LCLControl.OnChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncCheckBox(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName, Caption: string;
  LCLControl: TCheckBox;
  Control: TLILGUICheckControl;
begin
  GUI:=InstanceFromLIL(LIL);
  Caption:='';
  VarName:='';
  if Length(Args) > 0 then Caption:=TLIL.ToString(Args[0]);
  if Length(Args) > 1 then VarName:=TLIL.ToString(Args[1]);
  LCLControl:=TCheckBox.Create(LIL);
  LCLControl.Caption:=Caption;
  LCLControl.AutoSize:=False;
  Control:=TLILGUICheckControl(GUI.CreateGUIControl(LCLControl, TLILGUICheckControl));
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Checked:=TLIL.ToBoolean(Control.ValueVar.Value);
  end;
  LCLControl.OnChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncRadio(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName, Caption, Value: string;
  LCLControl: TRadioButton;
  Control: TLILGUIRadioControl;
begin
  GUI:=InstanceFromLIL(LIL);
  Caption:='';
  VarName:='';
  Value:='';
  if Length(Args) > 0 then Caption:=TLIL.ToString(Args[0]);
  if Length(Args) > 1 then VarName:=TLIL.ToString(Args[1]);
  if Length(Args) > 2 then
    Value:=TLIL.ToString(Args[2])
  else
    Value:=Caption;
  LCLControl:=TRadioButton.Create(LIL);
  LCLControl.Caption:=Caption;
  LCLControl.AutoSize:=False;
  Control:=TLILGUIRadioControl(GUI.CreateGUIControl(LCLControl, TLILGUIRadioControl));
  Control.Value:=Value;
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Checked:=TLIL.ToString(Control.ValueVar.Value)=Control.Value;
  end;
  LCLControl.OnChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncListBox(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Items: TLILValue;
  VarName: string;
  LCLControl: TListBox;
  Control: TLILGUIListBoxControl;
begin
  GUI:=InstanceFromLIL(LIL);
  Items:=nil;
  VarName:='';
  if Length(Args) > 0 then Items:=Args[0];
  if Length(Args) > 1 then VarName:=TLIL.ToString(Args[1]);
  LCLControl:=TListBox.Create(LIL);
  Control:=TLILGUIListBoxControl(GUI.CreateGUIControl(LCLControl, TLILGUIListBoxControl));
  Control.SetItemsFromValue(Items);
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    Control.OnVarChange(Control.ValueVar);
  end;
  LCLControl.OnSelectionChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncChoice(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Items: TLILValue;
  VarName: string;
  LCLControl: TComboBox;
  Control: TLILGUIChoiceControl;
begin
  GUI:=InstanceFromLIL(LIL);
  Items:=nil;
  VarName:='';
  if Length(Args) > 0 then Items:=Args[0];
  if Length(Args) > 1 then VarName:=TLIL.ToString(Args[1]);
  LCLControl:=TComboBox.Create(LIL);
  LCLControl.Style:=csDropDownList;
  Control:=TLILGUIChoiceControl(GUI.CreateGUIControl(LCLControl, TLILGUIChoiceControl));
  Control.SetItemsFromValue(Items);
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    Control.OnVarChange(Control.ValueVar);
  end;
  LCLControl.OnSelect:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncProgress(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName: string;
  Min, Max: Integer;
  LCLControl: TProgressBar;
  Control: TLILGUIProgressControl;
begin
  GUI:=InstanceFromLIL(LIL);
  VarName:='';
  Min:=0;
  Max:=1000000;
  if Length(Args) > 0 then VarName:=TLIL.ToString(Args[0]);
  if Length(Args) > 1 then Min:=TLIL.ToInteger(Args[1]);
  if Length(Args) > 2 then Max:=TLIL.ToInteger(Args[2]);
  LCLControl:=TProgressBar.Create(LIL);
  LCLControl.Min:=Min;
  LCLControl.Max:=Max;
  LCLControl.Position:=Min;
  LCLControl.Smooth:=True;
  LCLControl.AutoSize:=False;
  Control:=TLILGUIProgressControl(GUI.CreateGUIControl(LCLControl, TLILGUIProgressControl));
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Position:=TLIL.ToInteger(Control.ValueVar.Value);
  end;
  Result:=Control.ToValue;
end;

function FncSlider(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName: string;
  Min, Max: Integer;
  LCLControl: TTrackBar;
  Control: TLILGUISliderControl;
begin
  GUI:=InstanceFromLIL(LIL);
  VarName:='';
  Min:=0;
  Max:=1000000;
  if Length(Args) > 0 then VarName:=TLIL.ToString(Args[0]);
  if Length(Args) > 1 then Min:=TLIL.ToInteger(Args[1]);
  if Length(Args) > 2 then Max:=TLIL.ToInteger(Args[2]);
  LCLControl:=TTrackBar.Create(LIL);
  LCLControl.Min:=Min;
  LCLControl.Max:=Max;
  LCLControl.Position:=Min;
  LCLControl.TickStyle:=tsNone;
  LCLControl.TickMarks:=tmBoth;
  LCLControl.ShowSelRange:=False;
  Control:=TLILGUISliderControl(GUI.CreateGUIControl(LCLControl, TLILGUISliderControl));
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Position:=TLIL.ToInteger(Control.ValueVar.Value);
  end;
  LCLControl.OnChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncSpinner(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  VarName: string;
  Min, Max, Incr: Integer;
  LCLControl: TSpinEdit;
  Control: TLILGUISpinnerControl;
begin
  GUI:=InstanceFromLIL(LIL);
  VarName:='';
  Min:=Low(Integer);
  Max:=High(Integer);
  Incr:=1;
  if Length(Args) > 0 then VarName:=TLIL.ToString(Args[0]);
  if Length(Args) > 1 then Min:=TLIL.ToInteger(Args[1]);
  if Length(Args) > 2 then Max:=TLIL.ToInteger(Args[2]);
  if Length(Args) > 3 then Incr:=TLIL.ToInteger(Args[3]);
  LCLControl:=TSpinEdit.Create(LIL);
  LCLControl.MinValue:=Min;
  LCLControl.MaxValue:=Max;
  LCLControl.Value:=0;
  LCLControl.Increment:=Incr;
  LCLControl.AutoSize:=False;
  Control:=TLILGUISpinnerControl(GUI.CreateGUIControl(LCLControl, TLILGUISpinnerControl));
  if VarName <> '' then begin
    Control.ValueVar:=Control.SetupVar(VarName, @Control.OnVarChange);
    LCLControl.Value:=TLIL.ToInteger(Control.ValueVar.Value);
  end;
  LCLControl.OnChange:=@Control.OnChange;
  Result:=Control.ToValue;
end;

function FncGroup(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  LCLControl: TPanel;
  Control: TLILGUIContainerControl;
begin
  GUI:=InstanceFromLIL(LIL);
  LCLControl:=TPanel.Create(LIL);
  LCLControl.AutoSize:=False;
  LCLControl.BevelOuter:=bvNone;
  Control:=TLILGUIContainerControl(GUI.CreateGUIControl(LCLControl, TLILGUIContainerControl));
  GUI.FCurrentParent:=Control;
  Result:=Control.ToValue;
end;

function FncSpace(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  LCLControl: TLabel;
  Width, Height: Integer;
  Control: TLILGUIControl;
begin
  if Length(Args)=0 then begin
    Width:=4;
    Height:=4;
  end else if Length(Args)=1 then begin
    Width:=TLIL.ToInteger(Args[0]);
    if Width < 1 then Width:=1;
    Height:=Width;
  end else begin
    Width:=TLIL.ToInteger(Args[0]);
    if Width < 1 then Width:=1;
    Height:=TLIL.ToInteger(Args[0]);
    if Height < 1 then Height:=1;
  end;
  GUI:=InstanceFromLIL(LIL);
  LCLControl:=TLabel.Create(LIL);
  LCLControl.Caption:='';
  LCLControl.AutoSize:=False;
  LCLControl.Width:=Width;
  LCLControl.Height:=Height;
  Control:=GUI.CreateGUIControl(LCLControl, TLILGUIControl);
  Result:=Control.ToValue;
end;

function FncSpacing(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Spacing: Integer;
begin
  GUI:=InstanceFromLIL(LIL);
  if Assigned(GUI.CurrentParent) and (GUI.CurrentParent is TLILGUIContainerControl) then begin
    if Length(Args) > 0 then
      Spacing:=TLIL.ToInteger(Args[0])
    else
      Spacing:=-1;
    TLILGUIContainerControl(GUI.CurrentParent).Spacing:=Spacing;
    TLILGUIContainerControl(GUI.CurrentParent).DoFullLayout;
  end else
    LIL.SetError('Cannot set the spacing for the current parent');
  Result:=nil;
end;

function FncGrid(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  Columns: Integer;
begin
  GUI:=InstanceFromLIL(LIL);
  if Assigned(GUI.CurrentParent) and (GUI.CurrentParent is TLILGUIContainerControl) then begin
    if Length(Args) > 0 then
      Columns:=TLIL.ToInteger(Args[0])
    else
      Columns:=2;
    TLILGUIContainerControl(GUI.CurrentParent).Columns:=Columns;
    TLILGUIContainerControl(GUI.CurrentParent).LayoutMode:=lmGrid;
    TLILGUIContainerControl(GUI.CurrentParent).DoFullLayout;
  end else
    LIL.SetError('Cannot set the layout for the current parent');
  Result:=nil;
end;

function FncStack(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  IsHorizontal: Boolean;
begin
  GUI:=InstanceFromLIL(LIL);
  if Assigned(GUI.CurrentParent) and (GUI.CurrentParent is TLILGUIContainerControl) then begin
    IsHorizontal:=False;
    if Length(Args) > 0 then
      if TLIL.ToString(Args[0])='horizontal' then
        IsHorizontal:=True;
    if IsHorizontal then
      TLILGUIContainerControl(GUI.CurrentParent).LayoutMode:=lmHStack
    else
      TLILGUIContainerControl(GUI.CurrentParent).LayoutMode:=lmVStack;
    TLILGUIContainerControl(GUI.CurrentParent).DoFullLayout;
  end else
    LIL.SetError('Cannot set the layout for the current parent');
  Result:=nil;
end;

function FncBorder(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
begin
  GUI:=InstanceFromLIL(LIL);
  if Assigned(GUI.CurrentParent) and (GUI.CurrentParent is TLILGUIContainerControl) then begin
    with TLILGUIContainerControl(GUI.CurrentParent) do begin
      if (Length(Args) > 0) and (TLIL.ToString(Args[0])='top') then begin
        if Length(Args) > 1 then
          BTop:=GUI.ControlByValue(Args[1])
        else
          BTop:=GUI.LastControl;
      end else if (Length(Args) > 0) and (TLIL.ToString(Args[0])='bottom') then begin
        if Length(Args) > 1 then
          BBottom:=GUI.ControlByValue(Args[1])
        else
          BBottom:=GUI.LastControl;
      end else if (Length(Args) > 0) and (TLIL.ToString(Args[0])='left') then begin
        if Length(Args) > 1 then
          BLeft:=GUI.ControlByValue(Args[1])
        else
          BLeft:=GUI.LastControl;
      end else if (Length(Args) > 0) and (TLIL.ToString(Args[0])='right') then begin
        if Length(Args) > 1 then
          BRight:=GUI.ControlByValue(Args[1])
        else
          BRight:=GUI.LastControl;
      end else if (Length(Args) > 0) and (TLIL.ToString(Args[0])='center') then begin
        if Length(Args) > 1 then
          BCenter:=GUI.ControlByValue(Args[1])
        else
          BCenter:=GUI.LastControl;
      end else begin
        BTop:=nil;
        BBottom:=nil;
        BLeft:=nil;
        BRight:=nil;
        BCenter:=nil;
        if Length(Args) > 0 then BTop:=GUI.ControlByValue(Args[0]);
        if Length(Args) > 1 then BBottom:=GUI.ControlByValue(Args[1]);
        if Length(Args) > 2 then BLeft:=GUI.ControlByValue(Args[2]);
        if Length(Args) > 3 then BRight:=GUI.ControlByValue(Args[3]);
        if Length(Args) > 4 then BCenter:=GUI.ControlByValue(Args[4]);
        LayoutMode:=lmBorder;
        DoFullLayout;
        Exit(nil);
      end;
      if LayoutMode=lmBorder then DoFullLayout;
    end;
  end else
    LIL.SetError('Cannot set the layout for the current parent');
  Result:=nil;
end;

function FncMsgBox(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
begin
  if Length(Args)=0 then Exit(nil);
  if Length(Args)=1 then
    MessageDlg('Message', TLIL.ToString(Args[0]), mtCustom, [mbOK], 0)
  else
    MessageDlg(TLIL.ToString(Args[1]), TLIL.ToString(Args[0]), mtCustom, [mbOK], 0);
  Result:=nil;
end;

function FncWindow(LIL: TLIL; Args: TLILFunctionProcArgs): TLILValue;
var
  GUI: TLazLILGUI;
  PH, PW: Integer;
  Win: TLILGUIWindow;
begin
  GUI:=InstanceFromLIL(LIL);
  if Length(Args)=0 then begin
    LIL.SetError('Missing window action');
    Exit(nil);
  end;
  if TLIL.ToString(Args[0])='begin' then begin
    if Assigned(GUI.FMakingWindow) then
      GUI.FMakingWindow.Free;
    Inc(LastID);
    GUI.FMakingWindow:=TLILGUIWindow.CreateNew(GUI);
    GUI.FMakingWindow.Visible:=False;
    GUI.FMakingWindow.ID:=LastID;
    if Length(Args) > 1 then
      GUI.FMakingWindow.Caption:=TLIL.ToString(Args[1])
    else
      GUI.FMakingWindow.Caption:=Application.Title;
    GUI.FMakingWindow.HostControl:=GUI.FindControlForLCLHost(GUI.FMakingWindow, True);
    GUI.FCurrentParent:=GUI.FMakingWindow.HostControl.Container;
    Exit(nil);
  end;
  if TLIL.ToString(Args[0])='show' then begin
    if (Length(Args)=1) or
       (
         (TLIL.ToString(Args[1])='dialog') or
         (TLIL.ToString(Args[1])='persistent')
       ) then begin
      if not Assigned(GUI.FMakingWindow) then begin
        LIL.SetError('Not making any window');
        Exit(nil);
      end;
      if (Length(Args) > 1) and (TLIL.ToString(Args[1])='dialog') then begin
        GUI.FMakingWindow.BorderStyle:=bsDialog;
        GUI.FMakingWindow.BorderIcons:=[biSystemMenu];
      end;
      GUI.FMakingWindow.HostControl.Container.DoFullLayout;
      GUI.FMakingWindow.HostControl.Container.GetPreferredSize(@PW, @PH);
      GUI.FMakingWindow.ClientWidth:=PW + 16;
      GUI.FMakingWindow.ClientHeight:=PH + 16;
      if Assigned(Screen.ActiveForm) then begin
        GUI.FMakingWindow.Left:=Screen.ActiveForm.Left + (Screen.ActiveForm.Width - GUI.FMakingWindow.Width) div 2;
        GUI.FMakingWindow.Top:=Screen.ActiveForm.Top + (Screen.ActiveForm.Height - GUI.FMakingWindow.Height) div 2;
      end else begin
        GUI.FMakingWindow.Left:=(Screen.Width - GUI.FMakingWindow.Width) div 2;
        GUI.FMakingWindow.Top:=(Screen.Height - GUI.FMakingWindow.Height) div 2;
      end;
      GUI.FMakingWindow.ChildSizing.LeftRightSpacing:=8;
      GUI.FMakingWindow.ChildSizing.TopBottomSpacing:=8;
      GUI.FMakingWindow.Persistent:=(Length(Args) > 1) and
                                    (TLIL.ToString(Args[1])='persistent');
      if (Length(Args)=1) or (GUI.FMakingWindow.Persistent) then begin
        GUI.FMakingWindow.Show;
        Result:=TLIL.AllocInteger(GUI.FMakingWindow.ID);
      end else begin
        GUI.FMakingWindow.Persistent:=True;
        GUI.FMakingWindow.ShowModal;
        GUI.FMakingWindow.Free;
        Result:=nil;
      end;
      GUI.FMakingWindow:=nil;
      Exit;
    end else begin
      Win:=GUI.FindWindowByID(TLIL.ToInteger(Args[1]));
      if Assigned(Win) then begin
        if not Win.Visible then Win.Show;
        Win.BringToFront;
      end else
        LIL.SetError('Invalid window id');
    end;
    Exit(nil);
  end;
  if TLIL.ToString(Args[0])='hide' then begin
    if Length(Args) > 1 then begin
      Win:=GUI.FindWindowByID(TLIL.ToInteger(Args[1]));
      if Assigned(Win) then begin
        if Win.Visible then Win.Hide;
      end else
        LIL.SetError('Invalid window id');
    end else
      LIL.SetError('Missing window id');
    Exit(nil);
  end;
  if TLIL.ToString(Args[0])='close' then begin
    if Length(Args) > 1 then begin
      Win:=GUI.FindWindowByID(TLIL.ToInteger(Args[1]));
      if Assigned(Win) then begin
        Win.Free;
      end else
        LIL.SetError('Invalid window id');
    end else
      LIL.SetError('Missing window id');
    Exit(nil);
  end;
  if TLIL.ToString(Args[0])='end' then begin
    if Assigned(GUI.FMakingWindow) then begin
      GUI.FMakingWindow.Close;
    end else begin
      LIL.SetError('There isn''t any active dialog');
    end;
    Exit(nil);
  end;
  if TLIL.ToString(Args[0])='valid' then begin
    if Length(Args) > 1 then begin
      Win:=GUI.FindWindowByID(TLIL.ToInteger(Args[1]));
      Exit(TLIL.AllocBoolean(Assigned(Win)));
    end else
      LIL.SetError('Missing window id');
    Exit(nil);
  end;
  LIL.SetError('Unknown window action');
  Result:=nil;
end;

{ TLILGUIWindow }

procedure TLILGUIWindow.DoClose(var CloseAction: TCloseAction);
begin
  inherited DoClose(CloseAction);
  if Persistent then CloseAction:=caHide else CloseAction:=caFree;
end;

{ TLILGUIChoiceControl }

procedure TLILGUIChoiceControl.OnChange(Sender: TObject);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) then begin
    Tmp:=TLIL.AllocInteger(TComboBox(LCLControl).ItemIndex);
    ValueVar.Value:=Tmp;
    Tmp.Free;
  end;
end;

procedure TLILGUIChoiceControl.OnVarChange(Sender: TObject);
var
  NewIndex: Int64;
begin
  TComboBox(LCLControl).OnSelect:=nil;
  NewIndex:=TLIL.ToInteger(ValueVar.Value);
  if (NewIndex >= 0) and (NewIndex < TComboBox(LCLControl).Items.Count) then
    TComboBox(LCLControl).ItemIndex:=NewIndex
  else
    TComboBox(LCLControl).ItemIndex:=-1;
  TComboBox(LCLControl).OnSelect:=@OnChange;
end;

procedure TLILGUIChoiceControl.SetItemsFromValue(AValue: TLILValue);
var
  List: TLILList;
  I, OldIndex: Integer;
begin
  OldIndex:=TComboBox(LCLControl).ItemIndex;
  TComboBox(LCLControl).OnSelect:=nil;
  TComboBox(LCLControl).Items.BeginUpdate;
  TComboBox(LCLControl).Items.Clear;
  List:=GUI.LIL.SubstituteToList(AValue);
  for I:=0 to List.Count - 1 do
    TComboBox(LCLControl).Items.Add(TLIL.ToString(List[I]));
  List.Free;
  TComboBox(LCLControl).Items.EndUpdate;
  TComboBox(LCLControl).OnSelect:=@OnChange;
  if OldIndex >= TComboBox(LCLControl).Items.Count then
    OldIndex:=TComboBox(LCLControl).Items.Count - 1;
  TComboBox(LCLControl).ItemIndex:=OldIndex;
end;

procedure TLILGUIChoiceControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUIListBoxControl }

procedure TLILGUIListBoxControl.OnChange(Sender: TObject; User: Boolean);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) then begin
    Tmp:=TLIL.AllocInteger(TListBox(LCLControl).ItemIndex);
    ValueVar.Value:=Tmp;
    Tmp.Free;
  end;
end;

procedure TLILGUIListBoxControl.OnVarChange(Sender: TObject);
var
  NewIndex: Int64;
begin
  TListBox(LCLControl).OnSelectionChange:=nil;
  NewIndex:=TLIL.ToInteger(ValueVar.Value);
  if (NewIndex >= 0) and (NewIndex < TListBox(LCLControl).Items.Count) then
    TListBox(LCLControl).ItemIndex:=NewIndex
  else
    TListBox(LCLControl).ItemIndex:=-1;
  TListBox(LCLControl).OnSelectionChange:=@OnChange;
end;

procedure TLILGUIListBoxControl.SetItemsFromValue(AValue: TLILValue);
var
  List: TLILList;
  I, OldIndex: Integer;
begin
  OldIndex:=TListBox(LCLControl).ItemIndex;
  TListBox(LCLControl).OnSelectionChange:=nil;
  TListBox(LCLControl).Items.BeginUpdate;
  TListBox(LCLControl).Items.Clear;
  List:=GUI.LIL.SubstituteToList(AValue);
  for I:=0 to List.Count - 1 do
    TListBox(LCLControl).Items.Add(TLIL.ToString(List[I]));
  List.Free;
  TListBox(LCLControl).Items.EndUpdate;
  TListBox(LCLControl).OnSelectionChange:=@OnChange;
  if OldIndex >= TListBox(LCLControl).Items.Count then
    OldIndex:=TListBox(LCLControl).Items.Count - 1;
  TListBox(LCLControl).ItemIndex:=OldIndex;
end;

procedure TLILGUIListBoxControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUIContainerControl }

procedure TLILGUIContainerControl.DoLayout;

  procedure DoVStackLayout;
  var
    Pos, I, PH: Integer;
    Control: TLILGUIControl;
  begin
    Pos:=0;
    for I:=0 to ComponentCount - 1 do begin
      if not (Components[I] is TLILGUIControl) then Continue;
      Control:=TLILGUIControl(Components[I]);
      if not Assigned(Control.LCLControl) then Continue;
      Control.GetPreferredSize(nil, @PH);
      Control.SetBounds(0, Pos, LCLControl.ClientWidth, PH);
      Inc(Pos, PH + Spacing);
    end;
  end;

  procedure DoHStackLayout;
  var
    Pos, I, PW: Integer;
    Control: TLILGUIControl;
  begin
    Pos:=0;
    for I:=0 to ComponentCount - 1 do begin
      if not (Components[I] is TLILGUIControl) then Continue;
      Control:=TLILGUIControl(Components[I]);
      if not Assigned(Control.LCLControl) then Continue;
      Control.GetPreferredSize(@PW, nil);
      Control.SetBounds(Pos, 0, PW, LCLControl.ClientHeight);
      Inc(Pos, PW + Spacing);
    end;
  end;

  procedure DoBorderLayout;
  var
    X1, Y1, X2, Y2, PW, PH: Integer;
  begin
    X1:=0;
    Y1:=0;
    X2:=LCLControl.ClientWidth - 1;
    Y2:=LCLControl.ClientHeight - 1;
    if Assigned(BTop) then begin
      BTop.GetPreferredSize(nil, @PH);
      BTop.SetBounds(0, 0, LCLControl.ClientWidth, PH);
      Y1:=PH;
    end;
    if Assigned(BBottom) then begin
      BBottom.GetPreferredSize(nil, @PH);
      Y2:=Y2 - PH;
      BBottom.SetBounds(0, Y2, LCLControl.ClientWidth, PH);
    end;
    if Assigned(BLeft) then begin
      BLeft.GetPreferredSize(@PW, nil);
      BLeft.SetBounds(0, Y1, PW, Y2 - Y1 + 1);
      X1:=PW;
    end;
    if Assigned(BRight) then begin
      BRight.GetPreferredSize(@PW, nil);
      X2:=X2 - PW;
      BRight.SetBounds(X2, Y1, PW, Y2 - Y1 + 1);
    end;
    if Assigned(BCenter) then begin
      BCenter.SetBounds(X1, Y1, X2 - X1 + 1, Y2 - Y1 + 1);
    end;
  end;

  procedure DoGridLayout;
  type
    TCell = record
      Control: TLILGUIControl;
      PW, PH: Integer;
    end;
  var
    Cells: array of TCell;
    ColumnLeft, ColumnWidth, RowTop, RowHeight: array of Integer;
    Rows, I, J: Integer;
    Scale: Extended;
  begin
    SetLength(Cells, 0);
    for I:=0 to ComponentCount - 1 do
      if (Components[I] is TLILGUIControl) and
         Assigned(TLILGUIControl(Components[I]).LCLControl) then begin
        SetLength(Cells, Length(Cells) + 1);
        Cells[High(Cells)].Control:=TLILGUIControl(Components[I]);
        Cells[High(Cells)].Control.GetPreferredSize(
          @Cells[High(Cells)].PW,
          @Cells[High(Cells)].PH);
      end;
    if Length(Cells)=0 then Exit;
    Rows:=Length(Cells) div Columns;
    if Rows*Columns < Length(Cells) then Inc(Rows);
    SetLength(Cells, Rows*Columns);
    SetLength(ColumnLeft, Columns);
    SetLength(ColumnWidth, Columns);
    SetLength(RowTop, Rows);
    SetLength(RowHeight, Rows);
    for I:=0 to Columns - 1 do begin
      ColumnWidth[I]:=0;
      if I > 0 then
        ColumnLeft[I]:=ColumnLeft[I - 1] + ColumnWidth[I - 1] + Spacing
      else
        ColumnLeft[I]:=0;
      for J:=0 to Rows - 1 do begin
        if Cells[J*Columns + I].PW > ColumnWidth[I] then
          ColumnWidth[I]:=Cells[J*Columns + I].PW;
      end;
    end;
    for J:=0 to Rows - 1 do begin
      RowHeight[J]:=0;
      if J > 0 then
        RowTop[J]:=RowTop[J - 1] + RowHeight[J - 1] + Spacing
      else
        RowTop[J]:=0;
      for I:=0 to Columns - 1 do begin
        if Cells[J*Columns + I].PH > RowHeight[J] then
          RowHeight[J]:=Cells[J*Columns + I].PH;
      end;
    end;
    if Expand then begin
      Scale:=LCLControl.ClientWidth/(ColumnLeft[Columns - 1] + ColumnWidth[Columns - 1]);
      for I:=0 to Columns - 1 do begin
        ColumnWidth[I]:=Round(ColumnWidth[I]*Scale);
        if I > 0 then
          ColumnLeft[I]:=ColumnLeft[I - 1] + ColumnWidth[I - 1] + Spacing;
      end;
    end;
    for J:=0 to Rows - 1 do begin
      for I:=0 to Columns - 1 do with Cells[J*Columns + I] do begin
        if not Assigned(Control) then Break;
        Control.SetBounds(
          ColumnLeft[I],
          RowTop[J] + (RowHeight[J] - PH) div 2,
          ColumnWidth[I],
          PH);
      end;
    end;
  end;

var
  I: Integer;
begin
  case LayoutMode of
    lmVStack: DoVStackLayout;
    lmHStack: DoHStackLayout;
    lmBorder: DoBorderLayout;
    lmGrid: DoGridLayout;
  end;
  for I:=0 to ComponentCount - 1 do
    if Components[I] is TLILGUIContainerControl then
      TLILGUIContainerControl(Components[I]).DoLayout;
end;

procedure TLILGUIContainerControl.DoFullLayout;
var
  Target: TComponent;
  TopMost: TLILGUIContainerControl;
begin
  Target:=Self;
  TopMost:=Self;
  while Assigned(Target) and (Target is TLILGUIControl) do begin
    if Target is TLILGUIContainerControl then
      TopMost:=TLILGUIContainerControl(Target);
    Target:=Target.Owner;
  end;
  TopMost.DoLayout;
end;

function TLILGUIContainerControl.GetSpacing: Integer;
begin
  if FSpacing=-1 then begin
    if LayoutMode in [lmVStack, lmHStack, lmGrid] then
      Result:=4
    else
      Result:=0;
  end else Result:=FSpacing;
end;

procedure TLILGUIContainerControl.OnResize(Sender: TObject);
begin
  DoLayout;
end;

procedure TLILGUIContainerControl.SetColumns(AValue: Integer);
begin
  if AValue < 1 then AValue:=1;
  if FColumns=AValue then Exit;
  FColumns:=AValue;
end;

procedure TLILGUIContainerControl.SetSpacing(AValue: Integer);
begin
  if AValue < 0 then AValue:=-1;
  if AValue=FSpacing then Exit;
  FSpacing:=AValue;
end;

procedure TLILGUIContainerControl.SetupControl;
begin
  inherited SetupControl;
  LCLControl.AddHandlerOnResize(@OnResize);
end;

constructor TLILGUIContainerControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  LayoutMode:=lmVStack;
  FColumns:=2;
  FSpacing:=-1;
  Expand:=True;
end;

procedure TLILGUIContainerControl.GetPreferredSize(PW, PH: PInteger);
var
  W, H: Integer;

  procedure CalcVStack;
  var
    I, CPH, CPW: Integer;
    Control: TLILGUIControl;
  begin
    for I:=0 to ComponentCount - 1 do begin
      if not (Components[I] is TLILGUIControl) then Continue;
      Control:=TLILGUIControl(Components[I]);
      if not Assigned(Control.LCLControl) then Continue;
      Control.GetPreferredSize(@CPW, @CPH);
      if I > 0 then Inc(H, Spacing);
      if W < CPW then W:=CPW;
      Inc(H, CPH);
    end;
  end;

  procedure CalcHStack;
  var
    I, CPW, CPH: Integer;
    Control: TLILGUIControl;
  begin
    for I:=0 to ComponentCount - 1 do begin
      if not (Components[I] is TLILGUIControl) then Continue;
      Control:=TLILGUIControl(Components[I]);
      if not Assigned(Control.LCLControl) then Continue;
      Control.GetPreferredSize(@CPW, @CPH);
      if I > 0 then Inc(W, Spacing);
      if H < CPH then H:=CPH;
      Inc(W, CPW);
    end;
  end;

  procedure CalcBorder;
  var
    CPH, CPW, TopEdge, BottomEdge, LeftEdge, RightEdge: Integer;
  begin
    if Assigned(BTop) then begin
      BTop.GetPreferredSize(@CPW, @CPH);
      Inc(W, CPW);
      Inc(H, CPH);
      TopEdge:=CPH;
    end else TopEdge:=0;
    if Assigned(BBottom) then begin
      BBottom.GetPreferredSize(@CPW, @CPH);
      if W < CPW then W:=CPW;
      Inc(H, CPH);
      BottomEdge:=CPH;
    end else BottomEdge:=0;
    if Assigned(BLeft) then begin
      BLeft.GetPreferredSize(@CPW, @CPH);
      Inc(W, CPW);
      Inc(H, CPH);
      LeftEdge:=CPW;
    end else LeftEdge:=0;
    if Assigned(BRight) then begin
      BRight.GetPreferredSize(@CPW, @CPH);
      if H < CPH then H:=CPH;
      Inc(W, CPW);
      RightEdge:=CPW;
    end else RightEdge:=0;
    if Assigned(BCenter) then begin
      BCenter.GetPreferredSize(@CPW, @CPH);
      if W - LeftEdge - RightEdge < CPW then
        W:=CPW + LeftEdge + RightEdge;
      if H - TopEdge - BottomEdge < CPH then
        H:=CPH + TopEdge + BottomEdge;
    end;
  end;

  procedure CalcGrid;
  var
    I: Integer;
  begin
    Expand:=False;
    DoLayout;
    for I:=0 to ComponentCount - 1 do
      if (Components[I] is TLILGUIControl) and
         Assigned(TLILGUIControl(Components[I]).LCLControl) then begin
        if W < TLILGUIControl(Components[I]).LCLControl.Left +
               TLILGUIControl(Components[I]).LCLControl.Width then
          W:=TLILGUIControl(Components[I]).LCLControl.Left +
             TLILGUIControl(Components[I]).LCLControl.Width;
        if H < TLILGUIControl(Components[I]).LCLControl.Top +
               TLILGUIControl(Components[I]).LCLControl.Height then
          H:=TLILGUIControl(Components[I]).LCLControl.Top +
             TLILGUIControl(Components[I]).LCLControl.Height;
      end;
    Expand:=True;
  end;

begin
  W:=0;
  H:=0;
  case LayoutMode of
    lmVStack: CalcVStack;
    lmHStack: CalcHStack;
    lmBorder: CalcBorder;
    lmGrid: CalcGrid;
  end;
  {$IFDEF DARWIN}
  // Workaround for off-by-one height
  Inc(H);
  {$ENDIF}
  if Assigned(PW) then PW^:=W;
  if Assigned(PH) then PH^:=H;
end;

procedure TLILGUIContainerControl.BeforeDestruction;
begin
  if Assigned(LCLControl) then LCLControl.RemoveHandlerOnResize(@OnResize);
  inherited BeforeDestruction;
end;

{ TLILGUISpinnerControl }

procedure TLILGUISpinnerControl.OnChange(Sender: TObject);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) then begin
    Tmp:=TLIL.AllocInteger(TSpinEdit(LCLControl).Value);
    ValueVar.Value:=Tmp;
    Tmp.Free;
  end;
end;

procedure TLILGUISpinnerControl.OnVarChange(Sender: TObject);
begin
  TSpinEdit(LCLControl).OnChange:=nil;
  TSpinEdit(LCLControl).Value:=TLIL.ToInteger(ValueVar.Value);
  TSpinEdit(LCLControl).OnChange:=@OnChange;
end;

procedure TLILGUISpinnerControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUISliderControl }

procedure TLILGUISliderControl.OnChange(Sender: TObject);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) then begin
    Tmp:=TLIL.AllocInteger(TTrackBar(LCLControl).Position);
    ValueVar.Value:=Tmp;
    Tmp.Free;
  end;
end;

procedure TLILGUISliderControl.OnVarChange(Sender: TObject);
begin
  TTrackBar(LCLControl).OnChange:=nil;
  TTrackBar(LCLControl).Position:=TLIL.ToInteger(ValueVar.Value);
  TTrackBar(LCLControl).OnChange:=@OnChange;
end;

procedure TLILGUISliderControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUIProgressControl }

procedure TLILGUIProgressControl.OnVarChange(Sender: TObject);
begin
  TProgressBar(LCLControl).Position:=TLIL.ToInteger(ValueVar.Value);
end;

procedure TLILGUIProgressControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUIRadioControl }

procedure TLILGUIRadioControl.OnChange(Sender: TObject);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) and TRadioButton(LCLControl).Checked then begin
    Tmp:=TLIL.AllocString(Value);
    ValueVar.Value:=Tmp;
    Tmp.Free;
  end;
end;

procedure TLILGUIRadioControl.OnVarChange(Sender: TObject);

  procedure SearchAndUpdate(Control: TLILGUIControl);
  var
    Save: TNotifyEvent;
    I: Integer;
  begin
    if (Control is TLILGUIRadioControl) and
       (TLILGUIRadioControl(Control).ValueVar=ValueVar) then begin
      Save:=TRadioButton(Control.LCLControl).OnChange;
      TRadioButton(Control.LCLControl).OnChange:=nil;
      TRadioButton(Control.LCLControl).Checked:=TLIL.ToString(ValueVar.Value)=TLILGUIRadioControl(Control).Value;
      TRadioButton(Control.LCLControl).OnChange:=Save;
    end;
    for I:=0 to Control.ComponentCount - 1 do
      if Control.Components[I] is TLILGUIControl then
        SearchAndUpdate(TLILGUIControl(Control.Components[I]));
  end;

begin
  SearchAndUpdate(GUI.Root);
end;

procedure TLILGUIRadioControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUICheckControl }

procedure TLILGUICheckControl.OnChange(Sender: TObject);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) then begin
    if TCheckBox(LCLControl).State=cbChecked then begin
      Tmp:=TLIL.AllocBoolean(True);
      ValueVar.Value:=Tmp;
      Tmp.Free;
    end else
      ValueVar.Value:=nil;
  end;
end;

procedure TLILGUICheckControl.OnVarChange(Sender: TObject);
begin
  TCheckBox(LCLControl).OnChange:=nil;
  TCheckBox(LCLControl).Checked:=TLIL.ToBoolean(ValueVar.Value);
  TCheckBox(LCLControl).OnChange:=@OnChange;
end;

procedure TLILGUICheckControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUIEditControl }

procedure TLILGUIEditControl.OnChange(Sender: TObject);
var
  Tmp: TLILValue;
begin
  if Assigned(ValueVar) then begin
    Tmp:=TLIL.AllocString(TCustomEdit(LCLControl).Text);
    ValueVar.Value:=Tmp;
    Tmp.Free;
  end;
end;

procedure TLILGUIEditControl.OnVarChange(Sender: TObject);
begin
  TCustomEdit(LCLControl).OnChange:=nil;
  TCustomEdit(LCLControl).Text:=TLIL.ToString(ValueVar.Value);
  TCustomEdit(LCLControl).OnChange:=@OnChange;
end;

procedure TLILGUIEditControl.GetPreferredSize(PW, PH: PInteger);
begin
  if LCLControl is TMemo then begin
    if Assigned(PW) then PW^:=150;
    if Assigned(PH) then PH^:=100;
  end else
    inherited GetPreferredSize(PW, PH);
end;

procedure TLILGUIEditControl.BeforeDestruction;
begin
  if Assigned(ValueVar) then ValueVar.RemoveWatch(@OnVarChange);
  inherited BeforeDestruction;
end;

{ TLILGUILCLHostControl }

procedure TLILGUILCLHostControl.OnControlDestroy(Sender: TObject);

  procedure NullifyControls(Target: TLILGUIControl);
  var
    I: Integer;
  begin
    if Target <> Self then Target.FLCLControl:=nil;
    for I:=0 to Target.ComponentCount - 1 do
      if Target.Components[I] is TLILGUIControl then
        NullifyControls(TLILGUIControl(Target.Components[I]));
  end;

begin
  NullifyControls(Self);
  GUI.RemoveLCLHost(TWinControl(LCLControl));
end;

procedure TLILGUILCLHostControl.SetupControl;
begin
  inherited SetupControl;
  LCLControl.AddHandlerOnBeforeDestruction(@OnControlDestroy);
end;

procedure TLILGUILCLHostControl.BeforeDestruction;
begin
  if Assigned(LCLControl) then LCLControl.RemoveHandlerOnBeforeDestruction(@OnControlDestroy);
  inherited BeforeDestruction;
end;

{ TLILGUIButtonControl }

procedure TLILGUIButtonControl.OnClick(Sender: TObject);
begin
  GUI.LIL.Parse(Code, True).Free;
end;

procedure TLILGUIButtonControl.SetupControl;
begin
  inherited SetupControl;
  TButton(LCLControl).OnClick:=@OnClick;
end;

{ TLILGUIControl }

function TLILGUIControl.GetParent: TLILGUIControl;
begin
  Result:=TLILGUIControl(Owner);
end;

procedure TLILGUIControl.SetupControl;
begin
end;

function TLILGUIControl.SetupVar(AVarName: string; AWatch: TNotifyEvent): TLILVariable;
begin
  if AVarName='' then Exit(nil);
  Result:=GUI.LIL.FindGlobal(AVarName);
  if not Assigned(Result) then Result:=GUI.LIL.SetVar(AVarName, nil, lsvlGlobal);
  Result.AddWatch(AWatch);
end;

constructor TLILGUIControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Inc(LastID);
  FID:=LastID;
end;

destructor TLILGUIControl.Destroy;
begin
  if Assigned(GUI) then begin
    if GUI.LastControl=Self then GUI.FLastControl:=nil;
    if GUI.LastLabel=Self then GUI.FLastLabel:=nil;
  end;
  if Assigned(LCLControl) and not (Self is TLILGUILCLHostControl) then
    FreeAndNil(FLCLControl);
  inherited Destroy;
end;

function TLILGUIControl.ToValue: TLILValue;
begin
  Result:=TLIL.AllocString('$' + IntToStr(ID));
end;

procedure TLILGUIControl.SetBounds(X, Y, W, H: Integer);
begin
  if W < 1 then W:=1;
  if H < 1 then H:=1;
  if Assigned(LCLControl) then begin
    LCLControl.SetBounds(X, Y, W, H);
  end;
end;

procedure TLILGUIControl.GetPreferredSize(PW, PH: PInteger);
var
  W, H: Integer;
begin
  W:=0;
  H:=0;
  if Assigned(LCLControl) then begin
    if LCLControl is TWinControl then
      TWinControl(LCLControl).HandleNeeded
    else
      LCLControl.InvalidatePreferredSize;
    LCLControl.GetPreferredSize(W, H);
    // macOS workarounds
    {$IFDEF DARWIN}
    // Tiny listbox
    if (H < 120) and (LCLControl is TListBox) then H:=120;
    {$ENDIF}
  end;
  if Assigned(PW) then PW^:=W;
  if Assigned(PH) then PH^:=H;
end;

{ TLazLILGUI }

procedure TLazLILGUI.SetLIL(AValue: TLIL);
begin
  if FLIL=AValue then Exit;
  FLastControl:=nil;
  FLastLabel:=nil;
  while Root.ComponentCount > 0 do
    Root.Components[Root.ComponentCount - 1].Free;
  FLIL:=AValue;
  if Assigned(FLIL) then RegisterFunctions;
  if Assigned(FLCLHost) then begin
    FCurrentParent:=FindControlForLCLHost(FLCLHost, True).Container;
  end else
    FCurrentParent:=nil;
end;

function TLazLILGUI.GetCurrentParent: TLILGUIControl;
begin
  if Assigned(FCurrentParent) then Exit(FCurrentParent);
  if Assigned(LCLHost) then FCurrentParent:=FindControlForLCLHost(LCLHost, True).Container;
  if Assigned(FCurrentParent) then Exit(FCurrentParent);
  Exit(Root);
end;

procedure TLazLILGUI.SetLCLHost(AValue: TWinControl);
begin
  if FLCLHost=AValue then Exit;
  FLCLHost:=AValue;
  if not (csDesigning in ComponentState) then
    FCurrentParent:=FindControlForLCLHost(FLCLHost, True).Container;
end;

procedure TLazLILGUI.RegisterFunctions;
begin
  LIL.Register('control', @FncControl);
  LIL.Register('parent', @FncParent);
  LIL.Register('name', @FncName);
  LIL.Register('show', @FncShow);
  LIL.Register('hide', @FncHide);
  LIL.Register('button', @FncButton);
  LIL.Register('label', @FncLabel);
  LIL.Register('target', @FncTarget);
  LIL.Register('edit', @FncEdit);
  LIL.Register('text', @FncText);
  LIL.Register('checkbox', @FncCheckBox);
  LIL.Register('radio', @FncRadio);
  LIL.Register('listbox', @FncListBox);
  LIL.Register('choice', @FncChoice);
  LIL.Register('progress', @FncProgress);
  LIL.Register('slider', @FncSlider);
  LIL.Register('spinner', @FncSpinner);
  LIL.Register('group', @FncGroup);
  LIL.Register('space', @FncSpace);
  LIL.Register('spacing', @FncSpacing);
  LIL.Register('grid', @FncGrid);
  LIL.Register('stack', @FncStack);
  LIL.Register('border', @FncBorder);
  LIL.Register('msgbox', @FncMsgBox);
  LIL.Register('window', @FncWindow);
end;

function TLazLILGUI.FindControlForLCLHost(ALCLHost: TWinControl; AddNew: Boolean): TLILGUILCLHostControl;
var
  I: Integer;
  Panel: TPanel;
begin
  for I:=0 to Root.ComponentCount - 1 do
    if (Root.Components[I] is TLILGUILCLHostControl) and
       ((TLILGUILCLHostControl(Root.Components[I]).LCLControl)=ALCLHost) then
      Exit(TLILGUILCLHostControl(Root.Components[I]));
  if not AddNew then Exit(nil);
  Result:=TLILGUILCLHostControl.Create(Root);
  Result.FGUI:=Self;
  Result.FLCLControl:=ALCLHost;
  Result.SetupControl;
  Panel:=TPanel.Create(Self);
  Panel.BevelOuter:=bvNone;
  Result.Container:=TLILGUIContainerControl(CreateGUIControl(Panel, TLILGUIContainerControl, Result));
  Panel.Align:=alClient;
end;

function TLazLILGUI.FindWindowByID(ID: QWord): TLILGUIWindow;
var
  I: Integer;
begin
  for I:=0 to ComponentCount - 1 do
    if (Components[I] is TLILGUIWindow) and
       (TLILGUIWindow(Components[I]).ID=ID) then
      Exit(TLILGUIWindow(Components[I]));
  Result:=nil;
end;

constructor TLazLILGUI.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Instances.Add(Self);
  FRoot:=TLILGUIControl.Create(nil);
end;

destructor TLazLILGUI.Destroy;
begin
  FreeAndNil(FMakingWindow);
  FreeAndNil(FRoot);
  Instances.Remove(Self);
  inherited Destroy;
end;

function TLazLILGUI.ControlByID(ID: QWord): TLILGUIControl;

   function Search(Control: TLILGUIControl): TLILGUIControl;
   var
     I: Integer;
   begin
     if Control.ID=ID then Exit(Control);
     for I:=0 to Control.ComponentCount - 1 do
       if Control.Components[I] is TLILGUIControl then begin
         Result:=Search(TLILGUIControl(Control.Components[I]));
         if Assigned(Result) then Exit;
       end;
     Result:=nil;
   end;

begin
  if ID > 0 then
    Result:=Search(Root)
  else
    Result:=nil;
end;

function TLazLILGUI.ControlByName(AName: string): TLILGUIControl;

   function Search(Control: TLILGUIControl): TLILGUIControl;
   var
     I: Integer;
   begin
     if Control.ControlName=AName then Exit(Control);
     for I:=0 to Control.ComponentCount - 1 do
       if Control.Components[I] is TLILGUIControl then begin
         Result:=Search(TLILGUIControl(Control.Components[I]));
         if Assigned(Result) then Exit;
       end;
     Result:=nil;
   end;

begin
  if AName='@last' then Exit(LastControl);
  if AName='@parent' then Exit(CurrentParent);
  Result:=Search(Root);
end;

function TLazLILGUI.ControlByValue(ALILValue: TLILValue): TLILGUIControl;
var
  SV: String;
begin
  if ALILValue=nil then Exit(nil);
  SV:=TLIL.ToString(ALILValue);
  if SV='' then Exit(nil);
  if SV[1]='@' then Exit(ControlByName(Copy(SV, 2, MaxInt)));
  if SV[1]='$' then Exit(ControlByID(StrToInt64Def(Copy(SV, 2, MaxInt), 0)));
  Result:=nil;
end;

function TLazLILGUI.CreateGUIControl(ALCLControl: TControl;
  AClass: TLILGUIControlClass; OverrideParent: TLILGUIControl): TLILGUIControl;
begin
  Result:=TLILGUIControl(AClass.NewInstance);
  if not Assigned(OverrideParent) then OverrideParent:=CurrentParent;
  Result.Create(OverrideParent);
  Result.FGUI:=Self;
  Result.FLCLControl:=ALCLControl;
  Result.FControlName:=NameFromCaption(ALCLControl.Caption);
  Result.SetupControl;
  FLastControl:=Result;
  if Assigned(Result.Parent) and
     Assigned(Result.Parent.LCLControl) and
     (Result.Parent.LCLControl is TWinControl) then begin
    TWinControl(Result.Parent.LCLControl).InsertControl(ALCLControl);
    if Result.Parent is TLILGUIContainerControl then
      TLILGUIContainerControl(Result.Parent).DoFullLayout;
  end;
end;

function TLazLILGUI.NameFromCaption(ACaption: string): string;
var
  I: Integer;
begin
  ACaption:=Trim(ACaption);
  if ACaption='' then Exit('');
  Result:='';
  for I:=1 to Length(ACaption) do begin
    if ACaption[I] in ['A'..'Z', 'a'..'z', '0'..'9'] then
      Result += LowerCase(ACaption[I])
    else if ACaption[I]=' ' then
      Result += '_';
  end;
  if Assigned(ControlByName(Result)) then Result:='';
end;

procedure TLazLILGUI.AddLCLHost(ALCLHost: TWinControl; MakeCurrent: Boolean);
var
  Control: TLILGUILCLHostControl;
begin
  Control:=FindControlForLCLHost(ALCLHost, True);
  if MakeCurrent then FCurrentParent:=Control.Container;
end;

procedure TLazLILGUI.RemoveLCLHost(ALCLHost: TWinControl);
var
  HostControl: TLILGUILCLHostControl;
begin
  HostControl:=FindControlForLCLHost(ALCLHost, False);
  if Assigned(HostControl) then
    HostControl.Free;
  if FLCLHost=ALCLHost then FLCLHost:=nil;
  if FCurrentParent=HostControl then FCurrentParent:=nil;
end;

initialization
  Instances:=TLazLILGUIList.Create(False);
  {$I ../pkg/lazlilguipackage.lrs}
finalization
  FreeAndNil(Instances);
end.

