#include "x11winsys.h"

#ifndef LFORMS_NO_OPENGL
#include <dlfcn.h>

static void* glso;
static GLXWindow (*L_glXCreateWindow)(Display*, GLXFBConfig, Window, const int*);
static GLXFBConfig* (*L_glXChooseFBConfig)(Display*, int screen, const int*, int*);
static XVisualInfo* (*L_glXGetVisualFromFBConfig)(Display *dpy, GLXFBConfig);
static GLXContext (*L_glXCreateNewContext)(Display*, GLXFBConfig, int, GLXContext, Bool);
static void (*L_glXDestroyWindow)(Display*, GLXWindow);
static void (*L_glXDestroyContext)(Display*, GLXContext);
Bool (*ff__x11_L_glXMakeContextCurrent)(Display*, GLXDrawable, GLXDrawable, GLXContext);
void (*ff__x11_L_glXSwapBuffers)(Display*, GLXDrawable);

static int load_glso_func(const char* name, void* func)
{
    void* f = dlsym(glso, name);
    if (!f) return FF_NO;
    memcpy(func, &f, sizeof(f));
    return FF_YES;
}

static int load_glso(void)
{
    if (glso) return FF_YES;
    glso = dlopen("libGL.so.1", RTLD_LAZY);
    if (!glso) return FF_NO;
    if (!load_glso_func("glXCreateWindow", (void*)&L_glXCreateWindow)) return 0;
    if (!load_glso_func("glXChooseFBConfig", (void*)&L_glXChooseFBConfig)) return 0;
    if (!load_glso_func("glXGetVisualFromFBConfig", (void*)&L_glXGetVisualFromFBConfig)) return 0;
    if (!load_glso_func("glXCreateNewContext", (void*)&L_glXCreateNewContext)) return 0;
    if (!load_glso_func("glXDestroyWindow", (void*)&L_glXDestroyWindow)) return 0;
    if (!load_glso_func("glXDestroyContext", (void*)&L_glXDestroyContext)) return 0;
    if (!load_glso_func("glXMakeContextCurrent", (void*)&ff__x11_L_glXMakeContextCurrent)) return 0;
    if (!load_glso_func("glXSwapBuffers", (void*)&ff__x11_L_glXSwapBuffers)) return 0;
    return FF_YES;
}
#endif /* FF_NO_OPENGL */

static Atom net_wm_icon, cardinal;

void* ff_ws_create_window(void* parent, int x, int y, int width, int height, unsigned flags)
{
    wswin_t* win;
    wswin_t* pwin = parent;
    XSizeHints* size_hints;
    XSetWindowAttributes swa;
    unsigned swa_mask;
#ifndef LFORMS_NO_OPENGL
    GLXFBConfig* fbconfigs = NULL;
#endif
    XVisualInfo* vinfo = NULL;
    XClassHint* class_hint;
    XWMHints* wm_hints;

    if (pwin && !FF_WS_VALIDWIN(pwin)) return NULL;

    if (width < 1) width = 1;
    if (height < 1) height = 1;

    win = calloc(1, sizeof(wswin_t));
    win->parent = pwin;

#ifndef LFORMS_NO_OPENGL
    if (flags & FF_OPENGL) {
        int attribs[] = {
            GLX_X_RENDERABLE, True,
            GLX_X_VISUAL_TYPE, GLX_TRUE_COLOR,
            GLX_DRAWABLE_TYPE, GLX_WINDOW_BIT,
            GLX_RENDER_TYPE, GLX_RGBA_BIT,
            GLX_RED_SIZE, 8,
            GLX_GREEN_SIZE, 8,
            GLX_BLUE_SIZE, 8,
            GLX_ALPHA_SIZE, 8,
            GLX_DEPTH_SIZE, 24,
            GLX_STENCIL_SIZE, 8,
            GLX_DOUBLEBUFFER, True,
            None, None
        };
        int returned;
        if (!load_glso()) goto fail;
        fbconfigs = L_glXChooseFBConfig(DISPLAY, DefaultScreen(DISPLAY), attribs, &returned);
        if (fbconfigs && returned)
            vinfo = L_glXGetVisualFromFBConfig(DISPLAY, fbconfigs[0]);
        if (vinfo) {
            win->glc = L_glXCreateNewContext(DISPLAY, fbconfigs[0], GLX_RGBA_TYPE, NULL, True);
        } else goto fail;
    }
#endif

    swa.border_pixel = 0;
    swa.event_mask = ExposureMask|ButtonPressMask|ButtonReleaseMask|KeyPressMask|KeyReleaseMask|StructureNotifyMask|FocusChangeMask|PointerMotionMask|EnterWindowMask|LeaveWindowMask;
    swa_mask = CWBorderPixel|CWEventMask;
    if (vinfo) {
        swa_mask |= CWColormap;
        swa.colormap = XCreateColormap(DISPLAY, RootWindow(DISPLAY, SCREEN), vinfo->visual, AllocNone);
        win->cmap = swa.colormap;
    }

    if (flags & FF_NOFRAME) {
        swa.override_redirect = True;
        swa_mask |= CWOverrideRedirect;
    }

    win->w = XCreateWindow(DISPLAY, (flags & FF_POPUP) ? RootWindow(DISPLAY, SCREEN) : (pwin ? pwin->w : RootWindow(DISPLAY, SCREEN)), x, y, (unsigned)width, (unsigned)height, 0, vinfo ? vinfo->depth : DefaultDepth(DISPLAY, SCREEN), InputOutput, vinfo ? vinfo->visual : DefaultVisual(DISPLAY, SCREEN), swa_mask, &swa);
    if (!win->w) goto fail;
    size_hints = XAllocSizeHints();
    size_hints->flags = 0;
    if (flags & FF_NORESIZE) {
        size_hints->flags |= PMinSize | PMaxSize;
        size_hints->min_width = size_hints->max_width = width;
        size_hints->min_height = size_hints->max_height = height;
    }
    XSetStandardProperties(DISPLAY, win->w, "", "", None, ff_argv(), ff_argc(), size_hints);
    XFree(size_hints);

    class_hint = XAllocClassHint();
    class_hint->res_name = ff_strdup(ff_appname());
    class_hint->res_class = ff_strdup(ff_appname());
    XSetClassHint(DISPLAY, win->w, class_hint);
    ff_free(class_hint->res_class);
    ff_free(class_hint->res_name);
    XFree(class_hint);

    if (!parent || (flags & FF_POPUP)) {
        wm_hints = XAllocWMHints();
        wm_hints->flags = WindowGroupHint;
        wm_hints->window_group = pwin ? pwin->w : win->w;
        XSetWMHints(DISPLAY, win->w, wm_hints);
        XFree(wm_hints);
    }

    XSetWindowBackground(DISPLAY, win->w, 0);
    XSetWindowBackgroundPixmap(DISPLAY, win->w, None);
    XSetWMProtocols(DISPLAY, win->w, &ff_ws_wm_delete_window, 1);
    XSelectInput(DISPLAY, win->w, swa.event_mask);
    ff_ws_regwin((void*)win->w, win);

#ifndef LFORMS_NO_OPENGL
    if (vinfo) win->glw = L_glXCreateWindow(DISPLAY, fbconfigs[0], win->w, NULL);
#endif

    if (flags & FF_CENTERED) win->wsflags |= FF_WSFLAG_CENTER;

    if (!(flags & FF_INVISIBLE)) XMapWindow(DISPLAY, win->w);

    win->flags = flags;
    if (!parent || (flags & FF_POPUP)) win->wsflags |= FF_WSFLAG_TOPLEVEL;

    return win;

fail:
#ifndef LFORMS_NO_OPENGL
    if (win->glw) L_glXDestroyWindow(DISPLAY, win->glw);
    if (win->glc) L_glXDestroyContext(DISPLAY, win->glc);
#endif
    if (win->w) XDestroyWindow(DISPLAY, win->w);
    if (win->cmap) XFreeColormap(DISPLAY, win->cmap);
    free(win);
    return NULL;
}

void ff_ws_destroy_window(void* window)
{
    wswin_t* win = window;
    if (!win || !ff_ws_native(window) || !FF_WS_VALIDWIN(win)) return;
    win->wsflags |= FF_WSFLAG_INVALID;
#ifndef LFORMS_NO_OPENGL
    if (win->glw) L_glXDestroyWindow(DISPLAY, win->glw);
    if (win->glc) L_glXDestroyContext(DISPLAY, win->glc);
#endif
    if (win->gc) ff_ws_free_gc(win->gc, win);
    if (win->db) XFreePixmap(DISPLAY, win->db);
    if (win->w) XDestroyWindow(DISPLAY, win->w);
    if (win->cmap) XFreeColormap(DISPLAY, win->cmap);
    XSync(DISPLAY, False);
}

void ff_ws_set_caption(void* window, const char* caption)
{
    wswin_t* win = window;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return;
    XStoreName(DISPLAY, win->w, caption);
    XSetIconName(DISPLAY, win->w, caption);
}

void ff_ws_set_icon(void* window, ff_bitmap_t icon)
{
    wswin_t* win = window;
    XImage* img;
    unsigned long* argb;
    int x, y;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return;
    img = XGetImage(DISPLAY, icon->pixmap, 0, 0, (unsigned)icon->width, (unsigned)icon->height, AllPlanes, ZPixmap);
    if (!img) return;
    argb = malloc(sizeof(unsigned long)*(size_t)(icon->width*icon->height + 2));
    argb[0] = (unsigned long)icon->width;
    argb[1] = (unsigned long)icon->height;
    for (y=0; y < icon->height; y++)
        for (x=0; x < icon->width; x++) {
            unsigned long pixel = img->f.get_pixel(img, x, y);
            if (pixel == 0xFF00FF)
                argb[2 + y*icon->width + x] = 0;
            else
                argb[2 + y*icon->width + x] = 0xFF000000U|pixel;
        }
    if (!net_wm_icon) net_wm_icon = XInternAtom(DISPLAY, "_NET_WM_ICON", False);
    if (!cardinal) cardinal = XInternAtom(DISPLAY, "CARDINAL", False);
    x = 2 + icon->width*icon->height;
    XChangeProperty(DISPLAY, win->w, net_wm_icon, cardinal, 32, PropModeReplace, (unsigned char*)argb, x);
    free(argb);
    XDestroyImage(img);
}

void ff_ws_show(void* window, int show)
{
    wswin_t* win = window;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return;
    if (show) {
        XMapWindow(DISPLAY, win->w);
        if (win->flags & FF_TOPMOST) XRaiseWindow(DISPLAY, win->w);
    } else
        XUnmapWindow(DISPLAY, win->w);
}

int ff_ws_shown(void* window)
{
    wswin_t* win = window;
    XWindowAttributes attr;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return FF_NO;
    if (!XGetWindowAttributes(DISPLAY, win->w, &attr)) return FF_NO;
    return attr.map_state != IsUnmapped;
}

void ff_ws_set_ffwindow(void* window, ff_window_t ffwindow)
{
    wswin_t* win = window;
    if (!win) return;
    win->ffwindow = ffwindow;
}

ff_window_t ff_ws_get_ffwindow(void* window)
{
    wswin_t* win = window;
    if (!win) return NULL;
    return win->ffwindow;
}

void ff_ws_move(void* window, int x, int y)
{
    wswin_t* win = window;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return;
    XMoveWindow(DISPLAY, win->w, x, y);
}

void ff_ws_resize(void* window, int width, int height)
{
    wswin_t* win = window;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return;
    if (width < 1) width = 1;
    if (height < 1) height = 1;
    if (win->flags & FF_NORESIZE) {
        char** fake_argv;
        XSizeHints* size_hints = XAllocSizeHints();
        size_hints->flags |= PMinSize | PMaxSize;
        size_hints->min_width = size_hints->max_width = width;
        size_hints->min_height = size_hints->max_height = height;
        fake_argv = malloc(sizeof(char*));
        fake_argv[0] = ff_strdup("fake");
        XSetStandardProperties(DISPLAY, win->w, "", "", None, fake_argv, 1, size_hints);
        free(fake_argv[0]);
        free(fake_argv);
        XFree(size_hints);
    }
    XResizeWindow(DISPLAY, win->w, (unsigned)width, (unsigned)height);
}

void ff_ws_translate(int* x, int* y, void* from, void* to)
{
    wswin_t* wsfrom = from;
    wswin_t* wsto = to;
    Window src, dst;
    int rx, ry;
    Window child;
    if (wsfrom && !FF_WS_VALIDWIN(wsfrom)) return;
    if (wsto && !FF_WS_VALIDWIN(wsto)) return;
    src = from ? ((wswin_t*)from)->w : RootWindow(DISPLAY, SCREEN);
    dst = to ? ((wswin_t*)to)->w : RootWindow(DISPLAY, SCREEN);
    if (!src || !dst) return;
    XTranslateCoordinates(DISPLAY, src, dst, x ? *x : 0, y ? *y : 0, &rx, &ry, &child);
    if (x) *x = rx;
    if (y) *y = ry;
}

void ff_ws_area(void* window, int* x, int* y, int* width, int* height)
{
    wswin_t* win = window;
    Window root;
    int rx, ry;
    unsigned rw, rh, bw, wd;
    if (!win || !win->w || !FF_WS_VALIDWIN(win)) return;
    if (x || y || !win->width || !win->height)
        XGetGeometry(DISPLAY, win->w, &root, &rx, &ry, &rw, &rh, &bw, &wd);
    if (!win->width || !win->height) {
        /* We need to make sure that if the win->width/height fields are
         * updated from here and a ConfigureEvent is received with the
         * same width/height, the window will receive the FF_AREA event
         * since some setup or layout might be depending on the event */
        win->width = (int)rw;
        win->height = (int)rh;
        ff_post(win->ffwindow, FF_AREA, win->width, win->height, 0, NULL, FF_NOFLAGS);
    }
    if (x) *x = rx;
    if (y) *y = ry;
    if (width) *width = win->width;
    if (height) *height = win->height;
}

void* ff_ws_pick(void* window, int x, int y)
{
    int rx, ry, wx, wy;
    wswin_t* wswin = window;
    Window win;
    Window root, child;
    unsigned m;
    if (wswin && !FF_WS_VALIDWIN(wswin)) return NULL;
    win = wswin ? wswin->w : RootWindow(DISPLAY, SCREEN);
    if (!win) return NULL;
    while (XQueryPointer(DISPLAY, win, &root, &child, &rx, &ry, &wx, &wy, &m))
        if (child)
            win = child;
        else break;
    return ff_ws_wswin((void*)win);
}

void ff_ws_pointer(int* x, int* y)
{
    Window root, child;
    int rx, ry, wx, wy;
    unsigned m;
    XQueryPointer(DISPLAY, RootWindow(DISPLAY, SCREEN), &root, &child, &rx, &ry, &wx, &wy, &m);
    if (x) *x = rx;
    if (y) *y = ry;
}

static void send_enable_event(ff_window_t win, int enable)
{
    ff_window_t* children;
    int i;
    ff_post(win, FF_ENABLE, enable, 0, 0, NULL, FF_NOFLAGS);
    ff_paint(win);
    children = ff_npchildren(win);
    for (i=0; children[i]; i++)
        send_enable_event(children[i], enable);
    ff_free_children(children);
}

void ff_ws_enable(void* window, int enable)
{
    wswin_t* wswin = window;
    if (!FF_WS_VALIDWIN(wswin)) return;
    if ((!enable && (wswin->wsflags & FF_WSFLAG_DISABLED)) ||
        (enable && !(wswin->wsflags & FF_WSFLAG_DISABLED))) return;
    if (enable) wswin->wsflags &= ~FF_WSFLAG_DISABLED;
    else wswin->wsflags |= FF_WSFLAG_DISABLED;
    send_enable_event(wswin->ffwindow, enable ? FF_YES : FF_NO);
}

int ff_ws_enabled(void* window)
{
    wswin_t* wswin = window;
    if (!FF_WS_VALIDWIN(wswin)) return FF_NO;
    if (wswin->parent && !(wswin->flags & FF_POPUP) &&
        !ff_ws_enabled(wswin->parent)) return FF_NO;
    return !(wswin->wsflags & FF_WSFLAG_DISABLED);
}
