#define LFORMS_SOURCE_CODE
#if defined(WIN32) || defined(_WIN32)
#include <winsock2.h>
#endif
#include "x11winsys.h"
#include <sys/time.h>
#include <unistd.h>
#include <limits.h>
#include <X11/XKBlib.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#ifndef LFORMS_NOXIM
#include <X11/Xlocale.h>
#endif

#define NO_MIN_TIMEOUT -1

struct timer {
    wswin_t* wswin;
    unsigned long time;
    int timeout;
};

struct werect {
    int x1, y1, x2, y2;
};

struct winexpose {
    struct werect* r;
    Window win;
    int x1, y1, x2, y2;
    unsigned rc:16;
    unsigned done:1;
};

static struct winexpose* winexp;
static size_t we_count;
static struct winexpose* we_current;
static struct timer* timer;
static size_t timers;
static struct modal* gmodal;
static int min_timeout = NO_MIN_TIMEOUT;
static wswin_t* focused;
static wswin_t* focused_toplevel;
static int app_focused;
static int check_input_focus;
static int ignore_errors;
static int show_errors;
#ifdef LFORMS_TRAPX11ERRORS
static int trap_errors = FF_YES;
#endif
static int show_damage;
static Cursor cursors[FF_CROSSHAIR + 1];
static Window last_click_win;
static int last_click_x, last_click_y;
static unsigned last_click_btn;
static unsigned long last_click_ticks;
#ifndef LFORMS_NOXIM
static XIM im;
static XIC ic;
static Window icwin;
#endif

Display* ff_ws_display;
int ff_ws_screen;
Atom ff_ws_wm_delete_window;

static int x11_error_handler(Display* dis, XErrorEvent* ev)
{
    char buff[1024];
    if (ignore_errors > 0) return 0;
    if (!show_errors) return 0;
    XGetErrorText(dis, ev->error_code, buff, sizeof(buff));
#ifdef LFORMS_DEBUG
    fprintf(stderr,
        "Little Forms X11 Backend Error:\n"
        "===============================\n"
        "Serial:          %lu\n"
        "Error Code:      %u (%s)\n"
        "Request Code:    %u\n"
        "Minor Code:      %u\n"
        "Resource ID:     %lx\n",
        ev->serial, ev->error_code, buff, ev->request_code,
        ev->minor_code, ev->resourceid);
#else
    fprintf(stderr, "Little Forms X11 Backend Error %u: %s\n",
        ev->error_code, buff);
#endif
#ifdef LFORMS_TRAPX11ERRORS
    if (trap_errors) __builtin_trap();
#endif
    return 0;
}

static int keycode(unsigned xcode)
{
    unsigned fromtable[] = {
        XK_BackSpace, XK_Tab, XK_Linefeed, XK_Clear, XK_Return, XK_Pause,
        XK_Sys_Req, XK_Escape, XK_Delete, XK_Home, XK_Left, XK_Right, XK_Up,
        XK_Down, XK_Page_Up, XK_Page_Down, XK_End, XK_Insert, XK_Num_Lock,
        XK_Scroll_Lock, XK_Caps_Lock, XK_F1, XK_F2, XK_F3, XK_F4, XK_F5, XK_F6,
        XK_F7, XK_F8, XK_F9, XK_F10, XK_F11, XK_F12, XK_F13, XK_F14,
        XK_Shift_L, XK_Shift_R, XK_Control_L, XK_Control_R, XK_Alt_L, XK_Alt_R,
        XK_Super_L, XK_Super_R, XK_space, XK_exclam, XK_quotedbl, XK_numbersign,
        XK_dollar, XK_percent, XK_ampersand, XK_apostrophe, XK_parenleft,
        XK_parenright, XK_asterisk, XK_plus, XK_comma, XK_minus, XK_period,
        XK_slash, XK_0, XK_1, XK_2, XK_3, XK_4, XK_5, XK_6, XK_7, XK_8, XK_9,
        XK_colon, XK_semicolon, XK_less, XK_equal, XK_greater, XK_question,
        XK_at, XK_a, XK_b, XK_c, XK_d, XK_e, XK_f, XK_g, XK_h, XK_i, XK_j,
        XK_k, XK_l, XK_m, XK_n, XK_o, XK_p, XK_q, XK_r, XK_s, XK_t, XK_u,
        XK_v, XK_w, XK_x, XK_y, XK_z, XK_bracketleft, XK_backslash,
        XK_bracketright, XK_asciicircum, XK_underscore, XK_grave, XK_braceleft,
        XK_bar, XK_braceright, XK_asciitilde,
        0
    };
    int totable[] = {
        FFK_BACKSPACE, FFK_TAB, FFK_LINEFEED, FFK_CLEAR, FFK_RETURN, FFK_PAUSE,
        FFK_SYSREQ, FFK_ESCAPE, FFK_DELETE, FFK_HOME, FFK_LEFT, FFK_RIGHT, FFK_UP,
        FFK_DOWN, FFK_PAGEUP, FFK_PAGEDOWN, FFK_END, FFK_INSERT, FFK_NUM_LOCK,
        FFK_SCROLL_LOCK, FFK_CAPS_LOCK, FFK_F1, FFK_F2, FFK_F3, FFK_F4, FFK_F5, FFK_F6,
        FFK_F7, FFK_F8, FFK_F9, FFK_F10, FFK_F11, FFK_F12, FFK_F13, FFK_F14,
        FFK_LSHIFT, FFK_RSHIFT, FFK_LCONTROL, FFK_RCONTROL, FFK_LALT, FFK_RALT,
        FFK_LSUPER, FFK_RSUPER, FFK_SPACE, FFK_EXCLAM, FFK_QUOTEDBL, FFK_NUMBERSIGN,
        FFK_DOLLAR, FFK_PERCENT, FFK_AMPERSAND, FFK_APOSTROPHE, FFK_PARENLEFT,
        FFK_PARENRIGHT, FFK_ASTERISK, FFK_PLUS, FFK_COMMA, FFK_MINUS, FFK_PERIOD,
        FFK_SLASH, FFK_0, FFK_1, FFK_2, FFK_3, FFK_4, FFK_5, FFK_6, FFK_7, FFK_8, FFK_9,
        FFK_COLON, FFK_SEMICOLON, FFK_LESS, FFK_EQUAL, FFK_GREATER, FFK_QUESTION,
        FFK_AT, FFK_A, FFK_B, FFK_C, FFK_D, FFK_E, FFK_F, FFK_G, FFK_H, FFK_I, FFK_J,
        FFK_K, FFK_L, FFK_M, FFK_N, FFK_O, FFK_P, FFK_Q, FFK_R, FFK_S, FFK_T, FFK_U,
        FFK_V, FFK_W, FFK_X, FFK_Y, FFK_Z, FFK_BRACKETLEFT, FFK_BACKSLASH,
        FFK_BRACKETRIGHT, FFK_CARET, FFK_UNDERSCORE, FFK_GRAVE, FFK_BRACELEFT,
        FFK_BAR, FFK_BRACERIGHT, FFK_TILDE,
        0
    };
    int i;
    if (xcode > 255) return FFK_NONE;
    xcode = (unsigned)XkbKeycodeToKeysym(DISPLAY, (unsigned char)xcode, 0, 0);
    for (i=0; fromtable[i]; i++)
        if (fromtable[i] == xcode)
            return totable[i];

    return (int)xcode + 100000;
}

static int keymask(unsigned mask)
{
    unsigned rmask = 0;
    if (mask & ShiftMask) rmask |= FF_SHIFT;
    if (mask & ControlMask) rmask |= FF_CONTROL;
    if (mask & Mod1Mask) rmask |= FF_ALT;
    return (int)rmask;
}

static int send_event(wswin_t* win, unsigned event, int x, int y, int z, void* p, unsigned flags)
{
    if (!win || !win->ffwindow || !FF_WS_VALIDWIN(win)) return 0;
    return ff_send(win->ffwindow, event, x, y, z, p, flags);
}

static void free_modal(struct modal* mdl)
{
    while (mdl) {
        struct modal* next = mdl->next;
        free(mdl);
        mdl = next;
    }
}

static void focus_modal(struct modal* mdl)
{
    while (mdl->window->modal) mdl = mdl->window->modal;
    XRaiseWindow(DISPLAY, mdl->window->w);
}

static int send_disableable_event(wswin_t* win, unsigned event, int x, int y, int z, void* p, unsigned flags, int translatexy)
{
    if (!win || !win->ffwindow || !FF_WS_VALIDWIN(win)) return 0;
    if (!ff_ws_enabled(win)) {
        if (gmodal) {
            focus_modal(gmodal);
            return 0;
        }
        if (win->modal) {
            focus_modal(win->modal);
            return 0;
        }
        if (win->parent && !ff_ispopup(win->parent->ffwindow)) {
            if (translatexy) ff_ws_translate(&x, &y, win, win->parent);
            return send_disableable_event(win->parent, event, x, y, z, p, flags, FF_YES);
        }
        return 0;
    }
    return ff_send(win->ffwindow, event, x, y, z, p, flags);
}

static void add_to_expose_later(Window window, int x1, int y1, int x2, int y2, unsigned done)
{
    size_t i, idx = we_count;
    for (i=0; i < we_count; i++)
        if (winexp[i].win == window) {
            idx = i;
            break;
        }
    if (idx == we_count) {
        winexp = realloc(winexp, sizeof(struct winexpose)*(we_count + 1));
        winexp[we_count].win = window;
        winexp[we_count].r = NULL;
        winexp[we_count].rc = 0;
        winexp[we_count].x1 = x1;
        winexp[we_count].y1 = y1;
        winexp[we_count].x2 = x2;
        winexp[we_count].y2 = y2;
        we_count++;
    }
    winexp[idx].r = realloc(winexp[idx].r, (winexp[idx].rc + 1U)*sizeof(struct werect));
    winexp[idx].r[winexp[idx].rc].x1 = x1;
    winexp[idx].r[winexp[idx].rc].y1 = y1;
    winexp[idx].r[winexp[idx].rc].x2 = x2;
    winexp[idx].r[winexp[idx].rc].y2 = y2;
    winexp[idx].rc++;
    if (x1 < winexp[idx].x1) winexp[idx].x1 = x1;
    if (y1 < winexp[idx].y1) winexp[idx].y1 = y1;
    if (x2 > winexp[idx].x2) winexp[idx].x2 = x2;
    if (y2 > winexp[idx].y2) winexp[idx].y2 = y2;
    winexp[idx].done = done&1;
}

static void invalidate_expose_later(Window window)
{
    size_t i;
    for (i=0; i < we_count; i++)
        if (winexp[i].win == window) {
            winexp[i].win = 0;
            free(winexp[i].r);
        }
}

static int find_timer(wswin_t* wswin)
{
    size_t i;
    for (i=0; i < timers; i++)
        if (timer[i].wswin == wswin)
            return (int)i;
    return -1;
}

static void update_min_timeout(void)
{
    size_t i;
    if (timers) {
        min_timeout = 10000;
        for (i=0; i < timers; i++)
            if (timer[i].timeout < min_timeout)
                min_timeout = timer[i].timeout;
    } else min_timeout = NO_MIN_TIMEOUT;
}

static void setup_timer(wswin_t* wswin, int timeout)
{
    int idx = find_timer(wswin);
    if (idx != -1) {
        timer[idx].time = ff_ticks() + (unsigned long)timeout;
        return;
    }
    if (min_timeout == NO_MIN_TIMEOUT || min_timeout > timeout) min_timeout = timeout;
    timer = realloc(timer, sizeof(struct timer)*(timers + 1));
    timer[timers].wswin = wswin;
    timer[timers].time = ff_ticks() + (unsigned long)timeout;
    timer[timers].timeout = timeout;
    timers++;
}

static void remove_timer(wswin_t* wswin)
{
    int idx = find_timer(wswin);
    size_t i;
    if (idx == -1) return;
    timers--;
    for (i=(size_t)idx; i < timers; i++)
        timer[i] = timer[i + 1];
    update_min_timeout();
}

static void process_timers(void)
{
    size_t i;
    unsigned long now;
    if (!timers) return;
    now = ff_ticks();
    i = timers - 1;
    for (;;i--) {
        if (timer[i].time <= now) {
            send_event(timer[i].wswin, FF_TIMER, 0, 0, (int)now, NULL, FF_NOFLAGS);
            timer[i].time = now + (unsigned long)timer[i].timeout;
        }
        if (!i) break;
    }
}

static void draw_damage_rects(struct winexpose* we, ff_gc_t gc)
{
    int i;
    ff_color(gc, FF_RGB(255, 0, 255));
    for (i=0; i < we->rc; i++)
        ff_rect(gc, we->r[i].x1, we->r[i].y1, we->r[i].x2, we->r[i].y2);
}

static void process_exposes(void)
{
    size_t i;
    int alldone = FF_YES;
    for (i=0; i < we_count; i++) if (winexp[i].win) {
        if (winexp[i].done) {
            wswin_t* wswin = ff_ws_wswin((void*)winexp[i].win);
            ff_window_t win = wswin ? wswin->ffwindow : NULL;
            /* This makes sure if the paint handler asks for a repaint
             * it will create a brand new entry instead of modifying
             * this one while it is being processed */
            winexp[i].win = 0;
            if (!(FF_WS_VALIDWIN(wswin))) {
                free(winexp[i].r);
                continue;
            }
            if (wswin && (wswin->flags & FF_OPENGL)) {
    #ifndef LFORMS_NO_OPENGL
                if (ff__x11_L_glXMakeContextCurrent) {
                    if (ff__x11_L_glXMakeContextCurrent(DISPLAY, wswin->glw, wswin->glw, wswin->glc)) {
                        ff__sendpaint(wswin->ffwindow, NULL);
                        ff__x11_L_glXSwapBuffers(DISPLAY, wswin->glw);
                    }
                }
    #else
                ff_gc_t gc = win ? ff_ws_create_gc(wswin->w, wswin) : NULL;
                if (gc) {
                    int w, h;
                    wswin->gc = gc;
                    ff_color(gc, 0);
                    ff_area(win, NULL, NULL, &w, &h);
                    ff_fill(gc, 0, 0, w, h);
                    ff_color(gc, FF_RGB(255, 255, 255));
                    ff_text(gc, 0, h/2, "OpenGL support is not compiled in Little Forms!");
                    ff_ws_free_gc(gc, wswin);
                    wswin->gc = NULL;
                }
    #endif
            } else {
                ff_gc_t gc = win ? ff_ws_create_gc(wswin->w, wswin) : NULL;
                if (gc) {
                    wswin->gc = gc;
                    we_current = winexp + i;
                    gc->dx1 = we_current->x1;
                    gc->dy1 = we_current->y1;
                    gc->dx2 = we_current->x2;
                    gc->dy2 = we_current->y2;
                    ff__sendpaint(wswin->ffwindow, gc);
                    if (show_damage) draw_damage_rects(winexp + i, gc);
                    ff_ws_free_gc(gc, wswin);
                    we_current = NULL;
                    wswin->gc = NULL;
                }
            }
            free(winexp[i].r);
        } else {
            alldone = FF_NO;
        }
    }
    if (alldone) {
        free(winexp);
        winexp = NULL;
        we_count = 0;
    }
}

#define PAH_APPBLUR 1
#define PAH_APPFOCUS 2
#define PAH_JUSTFOCUS 3

static void process_autohide_proc(void* native, void* wswin, void* data)
{
    wswin_t* win = wswin;
    ff_intptr_t op = (ff_intptr_t)data;
    switch (op) {
    case PAH_APPBLUR:
        if (win->flags & FF_AUTOHIDE)
            ff_ws_show(win, FF_NO);
        break;
    case PAH_APPFOCUS:
        if (win->flags & FF_AUTOHIDE)
            ff_ws_show(win, FF_YES);
        /* FALLTHRU */ /* GCC magic comment */
    case PAH_JUSTFOCUS:
        if (win->flags & FF_TOPMOST)
            XRaiseWindow(DISPLAY, win->w);
        break;
    }
}

static void process_autohide_topmost(ff_intptr_t op)
{
    ff_ws_enumreg(process_autohide_proc, (void*)op);
}

static wswin_t* toplevel_wswin_from_window(Window w)
{
    wswin_t* win = ff_ws_wswin((void*)w);
    while (win && win->parent && !(win->flags & FF_POPUP)) win = win->parent;
    return win;
}

static void process_focus(void)
{
    Window current_focus;
    wswin_t* toplevel;
    int whatever;
    if (!check_input_focus) return;
    /* We may get an error from Xlib due to delayed handling, to
     * avoid this, we just ignore errors here and assume no focus
     * changed */
    ignore_errors++;
    XGetInputFocus(DISPLAY, &current_focus, &whatever);
    ignore_errors--;
    toplevel = toplevel_wswin_from_window(current_focus);
    if (toplevel != focused_toplevel) {
        int pah = FF_NOTHING;
        if (toplevel && !focused_toplevel) {
            pah = PAH_APPFOCUS;
            app_focused = FF_YES;
            ff_send(NULL, FF_APPFOCUS, 0, 0, 0, NULL, FF_NOFLAGS);
        } else if (!toplevel && focused_toplevel) {
            pah = PAH_APPBLUR;
            app_focused = FF_NO;
            ff_send(NULL, FF_APPBLUR, 0, 0, 0, NULL, FF_NOFLAGS);
        }
        if (pah) process_autohide_topmost(pah);
        focused_toplevel = toplevel;
        if (toplevel) {
            if (!toplevel->focus_child) toplevel->focus_child = toplevel;
            ff_ws_focus(toplevel->focus_child);
        }
    }
    check_input_focus = FF_NO;
}

static void handle_x11_event(XEvent* event)
{
    unsigned long ticks;
    #if 0
    fprintf(stderr, "X11 event serial %lu type %i window %lx\n", event->xany.serial, event->type, event->xany.window);
    #endif
    switch (event->type) {
    case MapNotify:
        {
            wswin_t* wswin = ff_ws_wswin((void*)event->xmap.window);
            if (wswin->wsflags & FF_WSFLAG_CENTER) {
                /* TODO: handle multimonitor setups, this will most likely not
                 * work with multiple screens or any setup that treats multiple
                 * monitors as a huge screen */
                 int dw = WidthOfScreen(XDefaultScreenOfDisplay(DISPLAY));
                 int dh = HeightOfScreen(XDefaultScreenOfDisplay(DISPLAY));
                 wswin->wsflags &= ~FF_WSFLAG_CENTER;
                 XMoveWindow(DISPLAY, event->xmap.window, (dw - wswin->width)/2, (dh - wswin->height)/2);
            }
        }
        break;
    case Expose:
        add_to_expose_later(event->xexpose.window,
            event->xexpose.x, event->xexpose.y,
            event->xexpose.x + event->xexpose.width - 1,
            event->xexpose.y + event->xexpose.height - 1,
            event->xexpose.count == 0);
        break;
    case MappingNotify:
        XRefreshKeyboardMapping(&event->xmapping);
        break;
    case KeyPress:
        {
            char str[32];
            int len;
            send_disableable_event(focused, FF_KEYDOWN, keycode(event->xkey.keycode), keymask(event->xkey.state), 0, NULL, FF_BUBBLE, FF_NO);
#if !defined(LFORMS_NOXIM) && defined(X_HAVE_UTF8_STRING)
            if (im && ic) {
                Status status;
                len = Xutf8LookupString(ic, (XKeyPressedEvent*)event, str, 32, 0, &status);
                str[len] = 0;
            } else
#endif
            { /* (im && ic) or LFORMS_NOXIM not defined */
                len = XLookupString(&event->xkey, str, 16, NULL, NULL);
                str[1] = 0;
            }
            if (len) send_disableable_event(focused, FF_TEXT, 0, keymask(event->xkey.state), 0, str, FF_BUBBLE, FF_NO);
        }
        break;
    case KeyRelease:
        if (XEventsQueued(DISPLAY, QueuedAfterReading)) {
            XEvent nev;
            XPeekEvent(DISPLAY, &nev);
            if (nev.type == KeyPress && nev.xkey.time == event->xkey.time &&
                nev.xkey.keycode == event->xkey.keycode)
            {
                /* For repeated KeyPress/KeyRelease, only send KeyPress events */
                break;
            }
        }
        send_disableable_event(focused, FF_KEYUP, keycode(event->xkey.keycode), keymask(event->xkey.state), 0, NULL, FF_BUBBLE, FF_NO);
        break;
    case ButtonPress:
        send_disableable_event(ff_ws_wswin((void*)event->xbutton.window), FF_PRESS, event->xbutton.x, event->xbutton.y, (int)event->xbutton.button, NULL, FF_BUBBLE, FF_YES);
        ticks = ff_ticks();
        if (event->xbutton.button != 4 && event->xbutton.button != 5 &&
            last_click_btn == event->xbutton.button &&
            last_click_win == event->xbutton.window &&
            ticks - last_click_ticks < 400 &&
            abs(last_click_x - event->xbutton.x) < 16 &&
            abs(last_click_y - event->xbutton.y) < 16) {
            last_click_ticks = 0;
            send_disableable_event(ff_ws_wswin((void*)event->xbutton.window), FF_DBLCLICK, event->xbutton.x, event->xbutton.y, (int)event->xbutton.button, NULL, FF_BUBBLE, FF_YES);
        } else last_click_ticks = ticks;
        last_click_btn = event->xbutton.button;
        last_click_x = event->xbutton.x;
        last_click_y = event->xbutton.y;
        last_click_win = event->xbutton.window;
        break;
    case ButtonRelease:
        send_disableable_event(ff_ws_wswin((void*)event->xbutton.window), FF_RELEASE, event->xbutton.x, event->xbutton.y, (int)event->xbutton.button, NULL, FF_BUBBLE, FF_YES);
        break;
    case MotionNotify:
        send_disableable_event(ff_ws_wswin((void*)event->xmotion.window), FF_MOTION, event->xmotion.x, event->xmotion.y, 0, NULL, FF_BUBBLE, FF_YES);
        break;
    case ConfigureNotify:
        {
            wswin_t* wswin = ff_ws_wswin((void*)event->xmap.window);
            if (wswin->width != event->xconfigure.width ||
                wswin->height != event->xconfigure.height) {
                wswin->width = event->xconfigure.width;
                wswin->height = event->xconfigure.height;
                send_event(ff_ws_wswin((void*)event->xconfigure.window), FF_AREA, event->xconfigure.width, event->xconfigure.height, 0, NULL, FF_NOFLAGS);
            }
        }
        break;
    case ClientMessage:
        if ((Atom)event->xclient.data.l[0] == ff_ws_wm_delete_window) {
            wswin_t* wswin = ff_ws_wswin((void*)event->xclient.window);
            if (!wswin || !wswin->ffwindow || !wswin->w || !FF_WS_VALIDWIN(wswin))
                break;
            if (wswin->modal || (wswin->wsflags & FF_WSFLAG_DISABLED)) break;
            if (!send_event(wswin, FF_CLOSE, 0, 0, 0, NULL, FF_NOFLAGS))
                XDestroyWindow(DISPLAY, event->xclient.window);
        }
        break;
    case DestroyNotify:
        {
            wswin_t* wswin = ff_ws_wswin((void*)event->xdestroywindow.window);
            if (focused == wswin) {
                send_event(wswin, FF_BLUR, 0, 0, 0, NULL, FF_BUBBLE);
                focused = NULL;
            }
            if (focused_toplevel == wswin) {
                if (focused_toplevel) {
                    ff_send(NULL, FF_APPBLUR, 0, 0, 0, NULL, FF_NOFLAGS);
                    process_autohide_topmost(PAH_APPBLUR);
                }
                focused_toplevel = NULL;
            }
#ifndef LFORMS_NOXIM
            if (im && icwin == event->xdestroywindow.window) {
                XDestroyIC(ic);
                ic = NULL;
                icwin = 0;
            }
#endif
            invalidate_expose_later(event->xdestroywindow.window);
            if (wswin) remove_timer(wswin);
            if (!wswin || !wswin->ffwindow)
                break;
            free_modal(wswin->modal);
            ff__destroyed(wswin->ffwindow);
            ff_ws_unregwin(wswin);
            free(wswin);
        }
        break;
    case CirculateNotify:
        if (event->xcirculate.place == PlaceOnTop) {
            wswin_t* wswin = ff_ws_wswin((void*)event->xcirculate.window);
            if (wswin && (!wswin->parent || (wswin->flags & FF_POPUP)) &&
                !ff_ws_enabled(wswin)) {
                if (gmodal) focus_modal(gmodal);
                else if (wswin->modal) focus_modal(wswin->modal);
                break;
            }
        }
        /* FALLTHRU */ /* GCC magic comment */
    case FocusIn:
    case FocusOut:
        check_input_focus = FF_YES;
        break;
    case EnterNotify:
        send_event(ff_ws_wswin((void*)event->xcrossing.window), FF_HOVER, event->xcrossing.x, event->xcrossing.y, 0, NULL, FF_NOFLAGS);
        break;
    case LeaveNotify:
        send_event(ff_ws_wswin((void*)event->xcrossing.window), FF_LEAVE, event->xcrossing.x, event->xcrossing.y, 0, NULL, FF_NOFLAGS);
        break;
    }
}

int ff_ws_init(int argc, char** argv)
{
    int i;
    int sync = 0;
    DISPLAY = NULL;
#ifdef LFORMS_DEBUG
    show_errors = FF_YES;
#else
    show_errors = FF_NO;
#endif
    for (i=1; i < argc; i++)
        if (!strcmp(argv[i], "-display") && i < argc - 1) {
            i++;
            DISPLAY = XOpenDisplay(argv[i]);
            break;
        } else if (!strcmp(argv[i], "-x11-sync"))
            sync = FF_YES;
        else if (!strcmp(argv[i], "-show-x11-damage"))
            show_damage = FF_YES;
        else if (!strcmp(argv[i], "-show-x11-errors"))
            show_errors = FF_YES;
        else if (!strcmp(argv[i], "-hide-x11-errors"))
            show_errors = FF_NO;
#ifdef LFORMS_TRAPX11ERRORS
        else if (!strcmp(argv[i], "-no-x11-error-trap"))
            trap_errors = FF_NO;
#endif
    if (!DISPLAY)
        DISPLAY = XOpenDisplay(NULL);

    if (!DISPLAY)
        return 0;

    XSetErrorHandler(x11_error_handler);

    if (getenv("LFORMS_X11SYNC"))
        sync = atoi(getenv("LFORMS_X11SYNC"));

    if (sync) XSynchronize(DISPLAY, True);

    if (getenv("LFORMS_SHOWDAMAGE"))
        show_damage = atoi(getenv("LFORMS_SHOWDAMAGE"));

    if (getenv("LFORMS_SHOWERRORS"))
        show_errors = atoi(getenv("LFORMS_SHOWERRORS"));

#ifdef LFORMS_TRAPX11ERRORS
    if (getenv("LFORMS_TRAPERRORS"))
        trap_errors = atoi(getenv("LFORMS_TRAPERRORS"));
#endif

    SCREEN = DefaultScreen(DISPLAY);

    ff_ws_wm_delete_window = XInternAtom(DISPLAY, "WM_DELETE_WINDOW", 0);
    winexp = NULL;
    we_count = 0;

#ifndef LFORMS_NOXIM
    if (!(getenv("LFORMS_NOXIM") && getenv("LFORMS_NOXIM")[0])) {
        setlocale(LC_ALL, "");
        if (XSupportsLocale()) XSetLocaleModifiers("@im=local");
        im = XOpenIM(DISPLAY, NULL, NULL, NULL);
        if (!im) {
            if (XSupportsLocale()) XSetLocaleModifiers("@im=");
            im = XOpenIM(DISPLAY, NULL, NULL, NULL);
        }
    }
#endif

    XSync(DISPLAY, False);

    ff_ws_common_init();

    return 1;
}

void ff_ws_shutdown(void)
{
    size_t i;
    XSync(DISPLAY, False);
    while (ff_ws_has_events()) ff_ws_pump_events();
    for (i=0; i < we_count; i++) if (winexp[i].win) free(winexp[i].r);
    free(winexp);
    winexp = NULL;
    we_count = 0;
    free(timer);
    timer = NULL;
    for (i=1; i < sizeof(cursors)/sizeof(cursors[0]); i++)
        if (cursors[i]) XFreeCursor(DISPLAY, cursors[i]);
    ff_ws_common_shutdown();
    XCloseDisplay(DISPLAY);
}

void ff_ws_exit(int exitcode)
{
}

void ff_ws_sync(void)
{
    XSync(DISPLAY, False);
}

int ff_ws_has_events(void)
{
    return XPending(DISPLAY) != 0;
}

int ff_ws_pump_events(void)
{
    XEvent event;
    if (ff_running()) process_timers();
    if (!XPending(DISPLAY)) {
        if (ff_running()) {
            process_focus();
            process_exposes();
        }
        if (min_timeout != NO_MIN_TIMEOUT) {
            while (!XPending(DISPLAY)) {
                fd_set rs;
                struct timeval tv;
#if defined(WIN32) || defined(_WIN32)
                SOCKET fd = (SOCKET)ConnectionNumber(DISPLAY);
#else
                int fd = ConnectionNumber(DISPLAY);
#endif                
                FD_ZERO(&rs);
                FD_SET(fd, &rs);
                tv.tv_sec = (min_timeout/2)/1000;
                tv.tv_usec = ((min_timeout/2)%1000)*1000;
                select((int)(fd + 1), &rs, NULL, NULL, &tv);
                process_timers();
                process_exposes();
                ff_ws_process_laters();
            }
        }
    }
    while (1) {
        XNextEvent(DISPLAY, &event);
#ifndef LFORMS_NOXIM
        if (im && icwin) {
            if (XFilterEvent(&event, icwin))
                continue;
        }
#endif
        break;
    }
    handle_x11_event(&event);
    return 1;
}

void ff_ws_paint(void* window)
{
    ff_ws_damage(window, 0, 0, INT_MAX, INT_MAX);
}

void ff_ws_damage(void* window, int x1, int y1, int x2, int y2)
{
    wswin_t* wswin = window;
    if (wswin && wswin->w && FF_WS_VALIDWIN(wswin) && x1 <= x2 && y1 <= y2) {
        if (x1 < 0) x1 = 0;
        if (y1 < 0) y1 = 0;
        if (x2 > wswin->width - 1) x2 = wswin->width - 1;
        if (y2 > wswin->height - 1) y2 = wswin->height - 1;
        add_to_expose_later(wswin->w, x1, y1, x2, y2, FF_YES);
    }
}

void ff_damaged(ff_gc_t gc, int* x1, int* y1, int* x2, int* y2)
{
    if (we_current) {
        if (x1) *x1 = we_current->x1;
        if (y1) *y1 = we_current->y1;
        if (x2) *x2 = we_current->x2;
        if (y2) *y2 = we_current->y2;
    } else {
        if (x1) *x1 = gc->dx1;
        if (y1) *y1 = gc->dy1;
        if (x2) *x2 = gc->dx2;
        if (y2) *y2 = gc->dy2;
    }
}

int ff_isdamaged(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    if (x1 <= x2 && y1 <= y2) {
        if (we_current) {
            unsigned i;
            for (i=0; i < we_current->rc; i++)
                if (ff_rect_over_rect(
                    x1, y1, x2, y2,
                    we_current->r[i].x1,
                    we_current->r[i].y1,
                    we_current->r[i].x2,
                    we_current->r[i].y2)) return FF_YES;
            return FF_NO;
        } else return ff_rect_over_rect(x1, y1, x2, y2,
                                        gc->dx1, gc->dy1, gc->dx2, gc->dy2);
    } else return FF_NO;
}

int ff_ws_timer(void* window, int ms)
{
    wswin_t* wswin = window;
    if (wswin && wswin->w && FF_WS_VALIDWIN(wswin)) {
        if (ms < 0) remove_timer(wswin);
        else setup_timer(wswin, ms < 1 ? 1 : ms);
        return 1;
    }
    return 0;
}

void ff_ws_cursor(void* window, ff_uintptr_t cursor)
{
    static const unsigned map[sizeof(cursors)/sizeof(cursors[0])] = {
        None,                   /* FF_NOTHING */
        XC_left_ptr,            /* FF_ARROW */
        XC_sb_h_double_arrow,   /* FF_HARROWS */
        XC_sb_v_double_arrow,   /* FF_VARROWS */
        XC_sizing,              /* FF_TLBRARROWS */
        XC_top_right_corner,    /* FF_TRBLARROWS */
        XC_fleur,               /* FF_4DARROWS */
        XC_watch,               /* FF_WAIT */
        XC_pirate,              /* FF_UNAVAILABLE */
        XC_xterm,               /* FF_IBEAM */
        XC_crosshair
    };
    wswin_t* wswin = window;
    if (!wswin || !wswin->w || !FF_WS_VALIDWIN(wswin) ||
        cursor >= sizeof(cursors)/sizeof(cursors[0]))
        return;
    if (cursor > 0 && !cursors[cursor])
        cursors[cursor] = XCreateFontCursor(DISPLAY, map[cursor]);
    XDefineCursor(DISPLAY, wswin->w, cursors[cursor]);
}

void ff_ws_pushmodal(void* window, void* target)
{
    wswin_t* wswin = window;
    wswin_t* wstgt = target;
    struct modal* mdl;
    if (!wswin || !wswin->w || !FF_WS_VALIDWIN(wswin)) return;
    if (wstgt) {
        if (!wstgt->w || !FF_WS_VALIDWIN(wstgt)) return;
        mdl = malloc(sizeof(struct modal));
        mdl->window = window;
        mdl->next = wstgt->modal;
        wstgt->modal = mdl;
        XSetTransientForHint(DISPLAY, wswin->w, wstgt->w);
    } else {
        mdl = malloc(sizeof(struct modal));
        mdl->window = window;
        mdl->next = gmodal;
        gmodal = mdl;
        XSetTransientForHint(DISPLAY, wswin->w, RootWindow(DISPLAY, SCREEN));
    }
    /* This works around bugs in window managers that set up the dialog
     * status when the window is shown (e.g. Window Maker) */
    if (ff_ws_shown(window)) ff_ws_show(window, FF_NO);
    ff_ws_show(window, FF_YES);
}

void ff_ws_popmodal(void* target)
{
    wswin_t* wswin = target;
    struct modal* mdl = NULL;
    if (target) {
        if (!wswin || !wswin->w || !FF_WS_VALIDWIN(wswin)) return;
        mdl = wswin->modal;
        if (mdl) wswin->modal = mdl->next;
    } else if (gmodal) {
        mdl = gmodal;
        if (mdl) gmodal = mdl->next;
    }
    if (mdl) free(mdl);
}


void ff_ws_focus(void* window)
{
    wswin_t* wswin = window;
    wswin_t* prev = focused;
    if (wswin == focused) return;
    focused = FF_WS_VALIDWIN(wswin) ? wswin : NULL;
    if (focused) {
        wswin_t* toplevel = toplevel_wswin_from_window(focused->w);
        toplevel->focus_child = focused;
    }
#ifndef LFORMS_NOXIM
    if (im) {
        if (ic && prev && prev->w == icwin) {
            XDestroyIC(ic);
            ic = NULL;
            icwin = 0;
        }
        if (focused && (focused->flags & FF_TEXTINPUT)) {
            ic = XCreateIC(im, XNInputStyle, XIMPreeditNothing|XIMStatusNothing, XNClientWindow, focused->w, NULL);
            if (ic) {
                XSetICFocus(ic);
                icwin = focused->w;
            }
        }
    }
#endif
    if (app_focused) {
        if (prev) send_event(prev, FF_BLUR, 0, 0, 0, NULL, FF_BUBBLE);
        if (focused) send_event(prev, FF_FOCUS, 0, 0, 0, NULL, FF_BUBBLE);
    }
}

int ff_ws_focused(void* window)
{
    return app_focused && window == focused;
}

void* ff_ws_x11_display(void)
{
    return ff_ws_display;
}
