#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <lforms.h>
#include <lforms/winsys.h>

/*#define DUMP_SET*/

struct _ff_window_t
{
    void* wswin;
    ff_window_t parent;
    int children;
    ff_window_t* child;
    int npchildren;
    ff_window_t* npchild;
    ff_window_t defchild;
    ff_font_t font;
    ff_layout_proc_t layout;
    unsigned flags;
    int frozen;
    unsigned destroy_request:1;
    unsigned modal:1;
    unsigned modal_enable:1;
    unsigned paint_on_thaw:1;
    unsigned relayout_on_thaw:1;
    unsigned relayout_scheduled:1;
};

struct globalmodalenable
{
    ff_window_t modal;
    ff_window_t win;
    int enable;
    struct globalmodalenable* next;
};

static ff_window_t* window;
static int windows;
static struct globalmodalenable* globalmodalenables;

static void free_window_list(void)
{
    free(window);
}

static void globalmodaldisable_for(ff_window_t win)
{
    int i;
    for (i=0; i < windows; i++) {
        if (window[i] != win &&
            (!window[i]->parent || ff_ispopup(window[i]))) {
            struct globalmodalenable* md = malloc(sizeof(struct globalmodalenable));
            md->modal = win;
            md->win = window[i];
            md->enable = ff_ws_enabled(window[i]->wswin);
            md->next = globalmodalenables;
            globalmodalenables = md;
            ff_ws_enable(window[i]->wswin, FF_NO);
        }
    }
}

static void globalmodalenable_for(ff_window_t win)
{
    while (globalmodalenables && globalmodalenables->modal == win) {
        struct globalmodalenable* next = globalmodalenables->next;
        ff_ws_enable(globalmodalenables->win->wswin, globalmodalenables->enable);
        free(globalmodalenables);
        globalmodalenables = next;
    }
}

static void need_npchildren(ff_window_t win)
{
    int i;
    if (!win->npchild) {
        win->npchild = malloc(sizeof(ff_window_t)*(size_t)win->children);
        win->npchildren = 0;
        for (i=0; i < win->children; i++)
            if (!ff_ispopup(win->child[i]))
                win->npchild[win->npchildren++] = win->child[i];
    }
}

static void invalidate_npchildren(ff_window_t win)
{
    free(win->npchild);
    win->npchild = NULL;
    win->npchildren = 0;
}

static int child_index(ff_window_t parent, ff_window_t child)
{
    int i;
    for (i=0; i<parent->children; i++)
        if (parent->child[i] == child)
            return (int)i;
    return -1;
}

static void add_child(ff_window_t parent, ff_window_t child, int relayout)
{
    child->parent = parent;
    if (child_index(parent, child) != -1) return;
    parent->child = realloc(parent->child, sizeof(ff_window_t)*(size_t)(parent->children + 1));
    parent->child[parent->children++] = child;
    invalidate_npchildren(parent);
    ff_send(parent, FF_CHILD, FF_ADD, 0, 0, child, FF_NOFLAGS);
    if (ff_frozen(parent))
        ff_freeze(child);
    if (relayout)
        ff_relayout(parent);
}

static void remove_child(ff_window_t parent, ff_window_t child, int relayout)
{
    int idx, i;
    if (child->parent != parent) return;
    child->parent = NULL;
    idx = child_index(parent, child);
    if (idx == -1) return;
    parent->children--;
    for (i=idx; i<parent->children; i++)
        parent->child[i] = parent->child[i + 1];
    parent->child = realloc(parent->child, sizeof(ff_window_t)*(size_t)parent->children);
    invalidate_npchildren(parent);
    ff_send(parent, FF_CHILD, FF_REMOVE, 0, 0, child, FF_NOFLAGS);
    if (ff_frozen(parent))
        ff_thaw(child);
    if (relayout)
        ff_relayout(parent);
}

static int paint_handler(void* win, ff_event_t* event, void* data)
{
    int w, h;
    ff_gc_t gc = event->p;
    if (gc) {
        ff_area(win, NULL, NULL, &w, &h);
        ff_color_attr(gc, win, "background", FF_3DFACE_COLOR);
        ff_fill(gc, 0, 0, w, h);
    }
    return 1;
}

static int area_handler(void* win, ff_event_t* event, void* data)
{
    ff_relayout(win);
    return 1;
}

static int set_handler(void* win_, ff_event_t* event, void* data)
{
    ff_window_t const win = win_;
    ff_namevalue_t* nv = event->p;
    if (!strcmp(nv->name, "caption") && (!win->parent || ff_ispopup(win))) {
        ff_ws_set_caption(win->wswin, nv->value);
        return 0;
    } else if (!strcmp(nv->name, "font")) {
        if (win->font != nv->value) {
            win->font = nv->value;
            ff_send(win, FF_NEWFONT, 0, 0, 0, nv->value, FF_NOFLAGS);
        }
        return 1;
    }
    return 0;
}

static int get_handler(void* win, ff_event_t* event, void* data)
{
    ff_namevalue_t* nv = event->p;
    if (!strcmp(nv->name, "font")) {
        nv->value = ff_font_of(win);
        return 1;
    }
    return 0;
}

static int keydown_handler(void* win, ff_event_t* event, void* data)
{
    if (event->x == FFK_RETURN) {
        ff_window_t tgt = ff_default(win, win);
        if (tgt) {
            ff_post(tgt, FF_ACTIVATE, 0, 0, 0, NULL, FF_NOFLAGS);
            return 1;
        }
    }
    return 0;
}

static int newfont_handler(void* win, ff_event_t* event, void* data)
{
    ff_paint(win);
    return 0;
}

static int destroy_handler(void* win, ff_event_t* event, void* data)
{
    if (ff_geti(win, "exit-on-close")) {
        ff_exit(ff_geti(win, "exit-on-close") - 1);
        return 1;
    }
    return 0;
}

static int bubblemask_handler(void* win, ff_event_t* event, void* data)
{
    /* Disallow events from popup windows to reach the parent */
    return FF_YES;
}

static int window_destructor(void* obj)
{
    ff_window_t win = obj;
    if (win->destroy_request) {
        return FF_YES;
    } else {
        ff_ws_destroy_window(win->wswin);
        return FF_NO;
    }
}

ff_window_t ff_window(ff_window_t parent, int x, int y, int width, int height, unsigned flags)
{
    void* wswin;
    ff_window_t win = parent;
    while (win) {
        if (win->flags & FF_DOUBLEBUFFER)
            flags |= FF_DOUBLEBUFFER;
        win = win->parent;
    }
    if (width < 1) width = 1;
    if (height < 1) height = 1;
    if (parent && !(flags & FF_POPUP))
        flags &= ~(FF_NORESIZE|FF_NOFRAME|FF_CENTERED|FF_AUTOHIDE|FF_TOPMOST);
    wswin = ff_ws_create_window(parent ? parent->wswin : NULL, x, y, width, height, flags);
    if (!wswin) return NULL;
    win = ff_new(sizeof(struct _ff_window_t));
    ff_destructor(win, window_destructor);
    ff_superior(win, parent);
    win->wswin = wswin;
    win->flags = flags;
    ff_ws_set_ffwindow(wswin, win);
    ff_set(win, "class", (void*)"window");
    ff_link(win, FF_PAINT, paint_handler, NULL);
    ff_link(win, FF_AREA, area_handler, NULL);
    ff_link(win, FF_SET, set_handler, NULL);
    ff_link(win, FF_GET, get_handler, NULL);
    ff_link(win, FF_KEYDOWN, keydown_handler, NULL);
    ff_link(win, FF_NEWFONT, newfont_handler, NULL);
    ff_prefsize(win, width, height);
    if (!window) ff_atexit(free_window_list);
    window = realloc(window, sizeof(ff_window_t)*(size_t)(windows + 2));
    window[windows++] = win;
    window[windows] = NULL;
    if (parent) {
        add_child(parent, win, 1);
        if (flags & FF_POPUP) ff_link(win, FF_BUBBLEMASK, bubblemask_handler, NULL);
    } else ff_link(win, FF_DESTROY, destroy_handler, NULL);
    ff_ws_cursor(win->wswin, FF_ARROW);
    if (flags & FF_DISABLED) {
        ff_enable(win, FF_NO);
        win->flags &= ~FF_DISABLED;
    }
    if (flags & FF_FROZEN) {
        ff_freeze(win);
        win->flags &= ~FF_FROZEN;
    }
    return win;
}

void ff__destroyed(ff_window_t win)
{
    int i;
    if (!win) return;
    if (win->modal) {
        if (win->parent) {
            ff_ws_popmodal(win->parent->wswin);
            ff_ws_enable(win->parent->wswin, win->modal_enable);
        } else {
            ff_ws_popmodal(NULL);
            globalmodalenable_for(win);
        }
        win->modal = FF_NO;
    }
    while (win->children > 0) {
        ff_window_t child = win->child[win->children - 1];
        remove_child(win, child, 0);
        ff_destroy(child);
    }
    if (win->parent) remove_child(win->parent, win, 0);
    for (i=((int)windows) - 1; i >= 0; i--)
        if (window[i] == win) {
            windows--;
            memmove(window + i, window + i + 1, sizeof(ff_window_t)*(size_t)(windows - i));
            window[windows] = NULL;
            break;
        }
    win->destroy_request = FF_YES;
    invalidate_npchildren(win);
    ff_destroy(win);
}

int ff_valid(ff_window_t win)
{
    ff_window_t* w;
    for (w=window; *w; w++)
        if (*w == win) return FF_YES;
    return FF_NO;
}

ff_window_t* ff_all_windows(void)
{
    return window;
}

unsigned ff_flags(ff_window_t win)
{
    return win->flags;
}

int ff_ispopup(ff_window_t win)
{
    return (win->flags & FF_POPUP) ? FF_YES : FF_NO;
}

void ff_move(ff_window_t win, int x, int y)
{
    if (!win) return;
    ff_ws_move(win->wswin, x, y);
}

void ff_resize(ff_window_t win, int width, int height)
{
    if (!win) return;
    if (width < 1) width = 1;
    if (height < 1) height = 1;
    ff_ws_resize(win->wswin, width, height);
}

void ff_place(ff_window_t win, int x, int y, int width, int height)
{
    if (!win) return;
    if (width < 1) width = 1;
    if (height < 1) height = 1;
    ff_ws_move(win->wswin, x, y);
    ff_ws_resize(win->wswin, width, height);
}

void ff_paint(ff_window_t win)
{
    if (!win) return;
    if (win->paint_on_thaw)
        return;
    if (ff_frozen(win)) {
        win->paint_on_thaw = FF_YES;
        return;
    }
    ff_ws_paint(win->wswin);
}

void ff_damage(ff_window_t win, int x1, int y1, int x2, int y2)
{
    if (!win) return;
    if (win->paint_on_thaw)
        return;
    if (ff_frozen(win)) {
        win->paint_on_thaw = FF_YES;
        return;
    }
    ff_ws_damage(win, x1, y1, x2, y2);
}

int ff__sendpaint(ff_window_t win, ff_gc_t gc)
{
    if (ff_frozen(win)) {
        win->paint_on_thaw = FF_YES;
        return FF_YES;
    }
    return ff_send(win, FF_PAINT, 0, 0, 0, gc, FF_NOFLAGS);
}

ff_window_t ff_parent(ff_window_t win)
{
    return win ? win->parent : NULL;
}

ff_window_t ff_toplevel(ff_window_t win)
{
    while (win && win->parent && !ff_ispopup(win))
        win = win->parent;
    return win;
}

ff_window_t* ff_children(ff_window_t win)
{
    ff_window_t* r;
    if (!win) return NULL;
    r = malloc((size_t)(win->children + 1)*sizeof(ff_window_t));
    if (win->children > 0)
        memcpy(r, win->child, (size_t)win->children*sizeof(ff_window_t));
    r[win->children] = 0;
    return r;
}

ff_window_t* ff_npchildren(ff_window_t win)
{
    ff_window_t* r;
    if (!win) return NULL;
    need_npchildren(win);
    r = malloc((size_t)(win->npchildren + 1)*sizeof(ff_window_t));
    if (win->npchildren > 0)
        memcpy(r, win->npchild, (size_t)win->npchildren*sizeof(ff_window_t));
    r[win->npchildren] = 0;
    return r;
}

void ff_free_children(ff_window_t* children)
{
    free(children);
}

ff_window_t ff_default(ff_window_t win, ff_window_t child)
{
    if (!win) return NULL;
    if (child != win && !ff_ispopup(child)) {
        if (win->defchild) ff_seti(win->defchild, "default", 0);
        win->defchild = child;
        if (child) ff_seti(child, "default", 1);
        return NULL;
    }
    return win->defchild ? win->defchild
        : (win->parent ? ff_default(win->parent, win->parent) : NULL);
}

void ff_translate(int* x, int* y, ff_window_t from, ff_window_t to)
{
    ff_ws_translate(x, y, from ? from->wswin : NULL, to ? to->wswin : NULL);
}

void ff_area(ff_window_t win, int* x, int* y, int* width, int* height)
{
    if (!win) return;
    ff_ws_area(win->wswin, x, y, width, height);
}

ff_window_t ff_pick(ff_window_t win, int x, int y)
{
    void* wswin = ff_ws_pick(win ? win->wswin : NULL, x, y);
    if (!wswin) return NULL;
    return ff_ws_get_ffwindow(wswin);
}

static int find_match(ff_window_t win, const char* name, int cmp, const void* value)
{
    void* v = ff_get(win, name);
    switch (cmp) {
    case FF_NO:
    case FF_YES:
        return cmp;
    case FF_IS:
        return v == value;
    case FF_ISNOT:
        return v != value;
    case FF_ISINT:
        return ((int)(ff_intptr_t)v) == ((int)(ff_intptr_t)value);
    case FF_ISNOTINT:
        return ((int)(ff_intptr_t)v) != ((int)(ff_intptr_t)value);
    case FF_ISSTR:
        return (v && !strcmp(v, value)) || (!v && !value) ? FF_YES : FF_NO;
    case FF_ISNOTSTR:
        return (v && !strcmp(v, value)) || (!v && !value) ? FF_NO : FF_YES;
    case FF_ISABOVE:
        return ((int)(ff_intptr_t)v) > ((int)(ff_intptr_t)value);
    case FF_ISBELOW:
        return ((int)(ff_intptr_t)v) < ((int)(ff_intptr_t)value);
    case FF_ISEQUALORABOVE:
        return ((int)(ff_intptr_t)v) >= ((int)(ff_intptr_t)value);
    case FF_ISEQUALORBELOW:
        return ((int)(ff_intptr_t)v) <= ((int)(ff_intptr_t)value);
    }
    return FF_NO;
}

ff_window_t ff_find(ff_window_t win, const char* name, int cmp, const void* value)
{
    ff_window_t r;
    int i;
    if (!win) {
        for (i=0; i < windows; i++)
            if (!window[i]->parent) {
                r = ff_find(window[i], name, cmp, value);
                if (r) return r;
            }
        return NULL;
    }
    if (find_match(win, name, cmp, value)) return win;
    for (i=0; i < win->children; i++) {
        r = ff_find(win->child[i], name, cmp, value);
        if (r) return r;
    }
    return NULL;
}

ff_window_t ff_findi(ff_window_t win, const char* name, int cmp, int value)
{
    return ff_find(win, name, cmp, (void*)(ff_intptr_t)value);
}

void ff_pointer(int* x, int* y)
{
    ff_ws_pointer(x, y);
}

void ff_layout(ff_window_t win, ff_layout_proc_t layout)
{
    if (win && win->layout != layout) {
        win->layout = layout;
        ff_relayout(win);
    }
}

static void do_relayout(ff_window_t win)
{
    int w, h;
    if (!win->relayout_scheduled) return;
    win->relayout_scheduled = FF_NO;
    ff_area(win, NULL, NULL, &w, &h);
    need_npchildren(win);
    win->layout(win, win->npchild, win->npchildren, w, h);
}

void ff_relayout(ff_window_t win)
{
    if (!win || win->relayout_on_thaw ||
        win->relayout_scheduled || !win->layout)
        return;
    if (ff_frozen(win)) {
        win->relayout_on_thaw = FF_YES;
        return;
    }
    win->relayout_scheduled = FF_YES;
    ff_later((ff_later_proc_t)do_relayout, win);
}

void ff_prefsize(ff_window_t win, int width, int height)
{
    int layout_parent = 0;
    if (!win) return;
    if (width < 0) width = 0;
    if (height < 0) height = 0;
    if (width) {
        int cwidth = ff_geti(win, "preferred-width");
        if (cwidth != width) {
            ff_seti(win, "preferred-width", width);
            layout_parent = 1;
        }
    }
    if (height) {
        int cheight = ff_geti(win, "preferred-height");
        if (cheight != height) {
            ff_seti(win, "preferred-height", height);
            layout_parent = 1;
        }
    }
    if (layout_parent && ff_parent(win)) ff_relayout(ff_parent(win));
}

static void do_pack(ff_window_t win, int horizontal, int vertical)
{
    int w, h;
    if (win->layout) {
        win->relayout_scheduled = FF_YES;
        do_relayout(win);
    }

    ff_lh_prefsize(win, &w, &h, FF_LHINVISIBLE);
    if (!horizontal) ff_area(win, NULL, NULL, &w, NULL);
    if (!vertical) ff_area(win, NULL, NULL, NULL, &h);
    ff_resize(win, w, h);
}

void ff_pack(ff_window_t win, int direction)
{
    int i, horizontal = FF_NO, vertical = FF_NO;
    if (!win) return;
    switch (direction) {
    case FF_HORIZONTAL: horizontal = FF_YES; break;
    case FF_VERTICAL: vertical = FF_YES; break;
    case FF_BOTH: horizontal = vertical = FF_YES; break;
    default:
        return;
    }
    for (i=0; i<win->children; i++)
        if (!ff_ispopup(win->child[i]))
            ff_pack(win->child[i], direction);
    do_pack(win, horizontal, vertical);
}

void ff_padding(ff_window_t win, int left, int top, int right, int bottom)
{
    if (!win) return;
    ff_seti(win, "padding-left", left);
    ff_seti(win, "padding-top", top);
    ff_seti(win, "padding-right", right);
    ff_seti(win, "padding-bottom", bottom);
}

void ff_margin(ff_window_t win, int left, int top, int right, int bottom)
{
    if (!win) return;
    ff_seti(win, "margin-left", left);
    ff_seti(win, "margin-top", top);
    ff_seti(win, "margin-right", right);
    ff_seti(win, "margin-bottom", bottom);
}

void ff_show(ff_window_t win, int show)
{
    if (!win) return;
    ff_ws_show(win->wswin, show);
}

int ff_shown(ff_window_t win)
{
    if (!win) return FF_NO;
    return ff_ws_shown(win->wswin);
}

void ff_modal(ff_window_t win)
{
    if (!win || (win->parent && !ff_ispopup(win)) || win->modal) return;
    win->modal = FF_YES;
    if (win->parent) {
        win->modal_enable = ff_ws_enabled(win->parent->wswin) ? FF_YES : FF_NO;
        ff_ws_enable(win->parent->wswin, FF_NO);
        ff_ws_pushmodal(win->wswin, win->parent->wswin);
    } else {
        globalmodaldisable_for(win);
        ff_ws_pushmodal(win->wswin, NULL);
    }
}

void ff_focus(ff_window_t win)
{
    if (!win) return;
    ff_ws_focus(win->wswin);
}

int ff_focused(ff_window_t win)
{
    if (!win) return 0;
    return ff_ws_focused(win->wswin);
}

void ff_enable(ff_window_t win, int enable)
{
    if (!win) return;
    ff_ws_enable(win->wswin, enable);
}

int ff_enabled(ff_window_t win)
{
    if (!win) return FF_NO;
    return ff_ws_enabled(win->wswin);
}

int ff_timer(ff_window_t win, int msorremove)
{
    if (!win) return 0;
    return ff_ws_timer(win->wswin, msorremove);
}

void ff_freeze(ff_window_t win)
{
    int i;
    if (!win) return;
    win->frozen++;
    if (win->frozen == 1) ff_send(win, FF_FREEZE, 0, 0, 0, NULL, FF_NOFLAGS);
    for (i=0; i < win->children; i++)
        if (!ff_ispopup(win->child[i]))
            ff_freeze(win->child[i]);
}

void ff_thaw(ff_window_t win)
{
    int i;
    if (!win) return;
    for (i=0; i < win->children; i++)
        if (!ff_ispopup(win->child[i]))
            ff_thaw(win->child[i]);
    win->frozen--;
    if (win->frozen == 0) {
        ff_send(win, FF_THAW, 0, 0, 0, NULL, FF_NOFLAGS);
        if (win->paint_on_thaw) {
            win->paint_on_thaw = FF_NO;
            ff_paint(win);
        }
        if (win->relayout_on_thaw) {
            win->relayout_on_thaw = FF_NO;
            ff_relayout(win);
        }
    }
}

int ff_frozen(ff_window_t win)
{
    return win && win->frozen > 0;
}

ff_font_t ff_font_of(ff_window_t win)
{
    return win && win->font ? win->font : ff_default_font(FF_STANDARD_FONT);
}

void ff_cursor(ff_window_t win, ff_uintptr_t cursor)
{
    if (!win) return;
    ff_ws_cursor(win->wswin, cursor);
}
