#define _WIN32_WINNT 0x0500
#include "win32winsys.h"

static ff_font_t standard_font, bold_font, italic_font, mono_font, big_font, small_font;
static ff_font_t* fonts;
static size_t font_count;

static void ff_font_free(ff_font_t font);

void ff_ws_gfx_init(void)
{
    if (!standard_font) {
        standard_font = ff_font_search("MS Sans Serif", 11, FF_PROPORTIONAL);
        bold_font = ff_font_search("MS Sans Serif", 11, FF_PROPORTIONAL|FF_BOLD);
        italic_font = ff_font_search("MS Sans Serif", 11, FF_PROPORTIONAL|FF_ITALIC);
        mono_font = ff_font_search("Courier New", 11, FF_PROPORTIONAL);
        big_font = ff_font_search("Arial", 24, FF_PROPORTIONAL|FF_BOLD);
        small_font = ff_font_search("Small Fonts", 9, FF_PROPORTIONAL);
    }
}

void ff_ws_gfx_shutdown(void)
{
    size_t i;
    for (i=0; i < font_count; i++) ff_font_free(fonts[i]);
    free(fonts);
    font_count = 0;
}

ff_gc_t ff_ws_create_gc(HWND w, wswin_t* win)
{
    ff_gc_t gc;
    if (win->flags & FF_OPENGL)
        return NULL;
    gc = calloc(1, sizeof(struct _ff_gc_t));
    gc->w = w;
    if (win->flags & FF_DOUBLEBUFFER) {
        RECT r;
        GetClientRect(w, &r);
        if (!win->db || r.right != win->dbw || r.bottom != win->dbh) {
            HDC ddc = GetDC(NULL);
            if (win->db) DeleteObject(win->db);
            win->dbw = r.right;
            win->dbh = r.bottom;
            win->db = CreateCompatibleBitmap(ddc, r.right, r.bottom);
            ReleaseDC(NULL, ddc);
        }
        win->dbdc = CreateCompatibleDC(NULL);
        win->dbold = SelectObject(win->dbdc, win->db);
        gc->dc = win->dbdc;
    } else {
        gc->dc = GetDC(w);
        if (!gc->dc) {
            free(gc);
            return NULL;
       }
    }
    ff_font(gc, ff_font_of(win->ffwindow));
    return gc;
}

void ff_ws_free_gc(ff_gc_t gc, wswin_t* win)
{
    if (gc) {
        if (win->flags & FF_DOUBLEBUFFER) {
            HDC dc = GetDC(win->w);
            BitBlt(dc, 0, 0, win->dbw, win->dbh, win->dbdc, 0, 0, SRCCOPY);
            ReleaseDC(win->w, dc);
            SelectObject(win->dbdc, win->dbold);
            DeleteDC(win->dbdc);
            win->dbdc = 0;
            win->dbold = 0;
        } else {
            ReleaseDC(gc->w, gc->dc);
        }
        free(gc);
    }
}

void ff_color(ff_gc_t gc, ff_color_t color)
{
    if (!gc) return;
    gc->color = (COLORREF)((color&0xFF)<<16)|(color&0xFF00)|((color&0xFF0000)>>16);
}

void ff_font(ff_gc_t gc, ff_font_t font)
{
    if (!gc) return;
    if (!font) font = standard_font;
    gc->font = font;
}

void ff_fill(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    HGDIOBJ prevbrush, prevpen;
    if (!gc) return;
    prevbrush = SelectObject(gc->dc, GetStockObject(DC_BRUSH));
    prevpen = SelectObject(gc->dc, GetStockObject(DC_PEN));
    SetDCBrushColor(gc->dc, gc->color);
    SetDCPenColor(gc->dc, gc->color);
    Rectangle(gc->dc, x1, y1, x2 + 1, y2 + 1);
    SelectObject(gc->dc, prevpen);
    SelectObject(gc->dc, prevbrush);
}

void ff_rect(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    HGDIOBJ prevbrush, prevpen;
    if (!gc) return;
    prevbrush = SelectObject(gc->dc, GetStockObject(HOLLOW_BRUSH));
    prevpen = SelectObject(gc->dc, GetStockObject(DC_PEN));
    SetDCPenColor(gc->dc, gc->color);
    Rectangle(gc->dc, x1, y1, x2 + 1, y2 + 1);
    SelectObject(gc->dc, prevpen);
    SelectObject(gc->dc, prevbrush);
}

void ff_line(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    HGDIOBJ prevpen;
    if (!gc) return;
    prevpen = SelectObject(gc->dc, GetStockObject(DC_PEN));
    SetDCPenColor(gc->dc, gc->color);
    MoveToEx(gc->dc, x1, y1, NULL);
    LineTo(gc->dc, x2, y2);
    SetPixel(gc->dc, x2, y2, gc->color);
    SelectObject(gc->dc, prevpen);
}

void ff_oval(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    HGDIOBJ prevbrush, prevpen;
    if (!gc) return;
    prevbrush = SelectObject(gc->dc, GetStockObject(HOLLOW_BRUSH));
    prevpen = SelectObject(gc->dc, GetStockObject(DC_PEN));
    SetDCPenColor(gc->dc, gc->color);
    Ellipse(gc->dc, x1, y1, x2 + 1, y2 + 1);
    SelectObject(gc->dc, prevpen);
    SelectObject(gc->dc, prevbrush);
}

void ff_disc(ff_gc_t gc, int x1, int y1, int x2, int y2)
{
    HGDIOBJ prevbrush, prevpen;
    if (!gc) return;
    prevbrush = SelectObject(gc->dc, GetStockObject(DC_BRUSH));
    prevpen = SelectObject(gc->dc, GetStockObject(DC_PEN));
    SetDCBrushColor(gc->dc, gc->color);
    SetDCPenColor(gc->dc, gc->color);
    Ellipse(gc->dc, x1, y1, x2 + 1, y2 + 1);
    SelectObject(gc->dc, prevpen);
    SelectObject(gc->dc, prevbrush);
}

void ff_pixel(ff_gc_t gc, int x, int y)
{
    if (!gc) return;
    SetPixel(gc->dc, x, y, gc->color);
}

void ff_text(ff_gc_t gc, int x, int y, const char* text)
{
    int prevmode;
    HGDIOBJ prevfont;
    TEXTMETRIC tm;
#ifdef UNICODE
    unsigned short* utf16;
#endif
    if (!gc) return;
    prevfont = SelectObject(gc->dc, gc->font->font);
    prevmode = GetBkMode(gc->dc);
    SetBkMode(gc->dc, TRANSPARENT);
    SetTextColor(gc->dc, gc->color);
    GetTextMetrics(gc->dc, &tm);
#ifdef UNICODE
    utf16 = ff_8to16((unsigned char*)text);
    TextOut(gc->dc, x, y - tm.tmHeight, utf16, ff_strlen16(utf16));
    free(utf16);
#else
    if (!ff_isascii(text)) {
        char* ascii = ff_asciify(text);
        TextOut(gc->dc, x, y - tm.tmHeight, ascii, strlen(ascii));
        free(ascii);
    } else TextOut(gc->dc, x, y - tm.tmHeight, text, strlen(text));
#endif
    SetBkMode(gc->dc, prevmode);
    SelectObject(gc->dc, prevfont);
}

void ff_text_size(ff_gc_t gc, const char* text, int* width, int* height)
{
    SIZE size;
    HGDIOBJ prevfont;
#ifdef UNICODE
    unsigned short* utf16;
#endif
    if (!gc) {
        if (width) *width = 0;
        if (height) *height = 0;
        return;
    }
    prevfont = SelectObject(gc->dc, gc->font->font);
#ifdef UNICODE
    utf16 = ff_8to16((unsigned char*)text);
    GetTextExtentPoint32(gc->dc, utf16, ff_strlen16(utf16), &size);
    free(utf16);
#else
    if (!ff_isascii(text)) {
        char* ascii = ff_asciify(text);
        GetTextExtentPoint32(gc->dc, ascii, strlen(ascii), &size);
        free(ascii);
    } else
        GetTextExtentPoint32(gc->dc, text, strlen(text), &size);
#endif
    SelectObject(gc->dc, prevfont);
    if (width) *width = size.cx;
    if (height) *height = size.cy;
}

void ff_draw(ff_gc_t gc, int x, int y, ff_bitmap_t bmp)
{
    if (!gc || !bmp) return;
    if (bmp->mask) {
        SelectObject(bmp->gc.dc, bmp->mask);
        BitBlt(gc->dc, x, y, bmp->width, bmp->height, bmp->gc.dc, 0, 0, SRCAND);
        SelectObject(bmp->gc.dc, bmp->bmp);
        BitBlt(gc->dc, x, y, bmp->width, bmp->height, bmp->gc.dc, 0, 0, SRCPAINT);
    } else BitBlt(gc->dc, x, y, bmp->width, bmp->height, bmp->gc.dc, 0, 0, SRCCOPY);
}

ff_bitmap_t ff_bitmap(int width, int height, const ff_color_t* pixels)
{
    /* TODO: use DIBs for faster pixel access */
    HDC ddc;
    ff_bitmap_t bmp;
    if (width < 1 || height < 1) return NULL;
    bmp = calloc(1, sizeof(struct _ff_bitmap_t));
    memset(bmp, 0, sizeof(struct _ff_bitmap_t));
    ff_refobj_init(FF_REFOBJ(bmp), (ff_refobj_dtor_proc_t)ff_bitmap_free);

    ddc = GetDC(NULL);
    bmp->width = width;
    bmp->height = height;
    bmp->bmp = CreateCompatibleBitmap(ddc, width, height);
    bmp->gc.dc = CreateCompatibleDC(NULL);
    bmp->gc.font = standard_font;
    bmp->gc.bmpw = width;
    bmp->gc.bmph = height;
    bmp->oldbmp = SelectObject(bmp->gc.dc, bmp->bmp);
    ReleaseDC(NULL, ddc);

    if (pixels) ff_bitmap_pixels(bmp, pixels);

    return bmp;
}

void ff_bitmap_free(ff_bitmap_t bmp)
{
    if (!bmp) return;
    SelectObject(bmp->gc.dc, bmp->oldbmp);
    DeleteObject(bmp->bmp);
    DeleteDC(bmp->gc.dc);
    if (bmp->mask) DeleteObject(bmp->mask);
    free(bmp);
}

void ff_bitmap_pixels(ff_bitmap_t bmp, const ff_color_t* pixels)
{
    int x, y, hastc = 0;
    const ff_color_t* p = pixels;
    for (y=0; y<bmp->height; y++)
        for (x=0; x<bmp->width; x++,pixels++) {
            if (*pixels == 0xFF00FF) {
                hastc = 1;
                SetPixelV(bmp->gc.dc, x, y, 0);
            } else
                SetPixel(bmp->gc.dc, x, y, (COLORREF)(((*pixels)&0xFF)<<16)|((*pixels)&0xFF00)|(((*pixels)&0xFF0000)>>16));
        }
    if (hastc) {
        if (!bmp->mask) bmp->mask = CreateBitmap(bmp->width, bmp->height, 1, 1, NULL);
        if (bmp->mask) {
            HDC dc = CreateCompatibleDC(NULL);
            if (dc) {
                HBITMAP oldbmp = SelectObject(dc, bmp->mask);
                for (pixels=p,y=0; y<bmp->height; y++)
                    for (x=0; x<bmp->width; x++,pixels++)
                        SetPixel(dc, x, y, *pixels != 0xFF00FF ? 0 : 0xFFFFFF);
                SelectObject(dc, oldbmp);
                DeleteDC(dc);
            } else {
                DeleteObject(bmp->mask);
                bmp->mask = NULL;
            }
        }
    } else if (bmp->mask) {
        DeleteObject(bmp->mask);
        bmp->mask = NULL;
    }
}

void ff_bitmap_size(ff_bitmap_t bmp, int* width, int* height)
{
    if (width) *width = bmp ? bmp->width : 0;
    if (height) *height = bmp ? bmp->height : 0;
}

ff_gc_t ff_bitmap_gc(ff_bitmap_t bmp)
{
    return bmp ? &bmp->gc : NULL;
}

static ff_font_t find_exact_font(const char* name, int size, unsigned flags)
{
    size_t i;
    flags &= ~FF_EXACT_FONT;
    for (i=0; i < font_count; i++)
        if ((!name || ff_streq_nocase(name, fonts[i]->name)) &&
            size == fonts[i]->size && flags == fonts[i]->flags)
            return fonts[i];
    return NULL;
}

ff_font_t ff_font_search(const char* name, int size, unsigned flags)
{
    ff_font_t font = find_exact_font(name, size, flags);
    DWORD pitch = DEFAULT_PITCH;
#ifdef UNICODE
    unsigned short* utf16;
#endif
    if (font) return font;
    if (flags & FF_PROPORTIONAL) pitch = VARIABLE_PITCH;
    else if (flags & FF_MONOSPACE) pitch = FIXED_PITCH;
    if (size < 0) size = 0;
    font = calloc(1, sizeof(struct _ff_font_t));
#ifdef UNICODE
    utf16 = ff_8to16((unsigned char*)name);
    font->font = CreateFont(-size, 0, 0, 0,
        (flags & FF_BOLD) ? 700 : 400,
        (flags & FF_ITALIC) ? TRUE : FALSE,
        (flags & FF_UNDERLINE) ? TRUE : FALSE,
        (flags & FF_STRIKEOUT) ? TRUE : FALSE,
        DEFAULT_CHARSET,
        OUT_DEFAULT_PRECIS,
        CLIP_DEFAULT_PRECIS,
        DRAFT_QUALITY, /* Synthesizes missing bits where necessary */
        pitch|FF_DONTCARE,
        utf16);
    free(utf16);
#else
    font->font = CreateFont(-size, 0, 0, 0,
        (flags & FF_BOLD) ? 700 : 400,
        (flags & FF_ITALIC) ? TRUE : FALSE,
        (flags & FF_UNDERLINE) ? TRUE : FALSE,
        (flags & FF_STRIKEOUT) ? TRUE : FALSE,
        DEFAULT_CHARSET,
        OUT_DEFAULT_PRECIS,
        CLIP_DEFAULT_PRECIS,
        DRAFT_QUALITY, /* Synthesizes missing bits where necessary */
        pitch|FF_DONTCARE,
        name);
#endif
    if (!font->font && name)
        font->font = CreateFont(-size, 0, 0, 0,
            (flags & FF_BOLD) ? 700 : 400,
            (flags & FF_ITALIC) ? TRUE : FALSE,
            (flags & FF_UNDERLINE) ? TRUE : FALSE,
            (flags & FF_STRIKEOUT) ? TRUE : FALSE,
            DEFAULT_CHARSET,
            OUT_DEFAULT_PRECIS,
            CLIP_DEFAULT_PRECIS,
            DRAFT_QUALITY, /* Synthesizes missing bits where necessary */
            pitch|FF_DONTCARE,
            NULL);
    if (!font->font) {
        free(font);
        font = NULL;
    } else {
        font->name = ff_strdup(name);
        font->size = size;
        font->flags = flags & (~FF_EXACT_FONT);
        fonts = realloc(fonts, sizeof(ff_font_t)*(font_count + 1));
        fonts[font_count++] = font;
    }
    return font;
}

ff_font_t ff_default_font(int which)
{
    switch (which) {
    case FF_BOLD_FONT: return bold_font;
    case FF_ITALIC_FONT: return italic_font;
    case FF_MONOSPACE_FONT: return mono_font;
    case FF_BIG_FONT: return big_font;
    case FF_SMALL_FONT: return small_font;
    }
    return standard_font;
}

static void ff_font_free(ff_font_t font)
{
    if (font) {
        free(font->name);
        DeleteObject(font->font);
        free(font);
    }
}

const char* ff_font_name(ff_font_t font)
{
    return font ? font->name : standard_font->name;
}

int ff_font_size(ff_font_t font)
{
    return font ? font->size : standard_font->size;
}

unsigned ff_font_flags(ff_font_t font)
{
    return font ? font->flags : standard_font->flags;
}

struct fontenum_data_s {
    ff_font_enum_proc proc;
    void* data;
};

static int CALLBACK font_enum_proc(const LOGFONT* lf, const TEXTMETRIC* tm, DWORD dwType, LPARAM lpData)
{
    /* TODO: do a better job at this, especially when it comes to bitmap fonts
       that do not support arbitrary sizes */
    struct fontenum_data_s* fed = (struct fontenum_data_s*)lpData;
    unsigned flags = FF_BOLD|FF_ITALIC;
    int sizes[] = {6, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 20, 22, 24,
        28, 32, 36, 48, 64, 72, 80, 108, 110, 120, 135, 160};
#ifdef UNICODE
    unsigned char* utf8 = ff_16to8(lf->lfFaceName);
    if (!fed->proc((char*)utf8, sizes, sizeof(sizes)/sizeof(int), flags, fed->data)) {
        free(utf8);
        return 0;
    }
    free(utf8);
#else
    if (!fed->proc(lf->lfFaceName, sizes, sizeof(sizes)/sizeof(int), flags, fed->data))
        return 0;
#endif
    return 1;
}

void ff_font_enum(ff_font_enum_proc proc, void* data, unsigned enum_flags)
{
    HWND wnd = GetDesktopWindow();
    HDC dc = GetDC(wnd);
    struct fontenum_data_s fed;
    if (!dc) return;
    fed.proc = proc;
    fed.data = data;
    EnumFonts(dc, NULL, font_enum_proc, (LPARAM)&fed);
    ReleaseDC(wnd, dc);
}
