#define LFORMS_SOURCE_CODE
#include <lforms.h>

typedef struct _state_t
{
    char** tab;
    int tabs;
    int* visx1;
    int* visx2;
    int selection;
} state_t;

static char* itableft[] = {
    "16 16 4 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "* s 3d-face",
    "             ***",
    "           **---",
    "          *-----",
    "         *------",
    "         *------",
    "        *-------",
    "        *-------",
    "       *--------",
    "       *--------",
    "       *--------",
    "      *---------",
    "      *---------",
    "     *----------",
    "     *----------",
    "    *-----------",
    "   *------------"};
static char* itabright[] = {
    "16 16 4 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "* s 3d-face",
    "*               ",
    "---.            ",
    "----.           ",
    "----..          ",
    "-----.          ",
    "-----..         ",
    "------.         ",
    "------..        ",
    "-------.        ",
    "-------..       ",
    "--------.       ",
    "--------..      ",
    "---------.      ",
    "---------..     ",
    "----------.     ",
    "-----------.    "};
static char* atableft[] = {
    "16 16 3 1",
    "  s none",
    "@ s 3d-light",
    "* s 3d-face",
    "            *@@@",
    "          *@@***",
    "          @*****",
    "         @******",
    "         @******",
    "        @*******",
    "        @*******",
    "       @********",
    "       @********",
    "      *@********",
    "      @*********",
    "     *@*********",
    "     @**********",
    "    *@**********",
    "   *@***********",
    "  *@************"};
static char* atabright[] = {
    "16 16 5 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "@*              ",
    "***.            ",
    "****.           ",
    "****-.          ",
    "*****.          ",
    "*****-.         ",
    "******.         ",
    "******-.        ",
    "*******.        ",
    "*******.        ",
    "********.       ",
    "********.       ",
    "*********.      ",
    "*********.      ",
    "**********.     ",
    "***********.    "};

static ff_bitmap_t itableftbmp, itabrightbmp, atableftbmp, atabrightbmp;

static void free_bitmaps(void)
{
    ff_bitmap_free(atabrightbmp);
    ff_bitmap_free(atableftbmp);
    ff_bitmap_free(itabrightbmp);
    ff_bitmap_free(itableftbmp);
}

static void create_bitmaps(void)
{
    itableftbmp = ff_xpm(itableft, NULL, NULL);
    itabrightbmp = ff_xpm(itabright, NULL, NULL);
    atableftbmp = ff_xpm(atableft, NULL, NULL);
    atabrightbmp = ff_xpm(atabright, NULL, NULL);
    ff_atexit(free_bitmaps);
}

static void update_prefsize(ff_window_t ts)
{
    state_t* state = ff_get(ts, "-state");
    int tw, i, w = 28;
    ff_font_t font = ff_font_of(ts);
    for (i=0; i < state->tabs; i++) {
        ff_font_text_size(font, state->tab[i], &tw, NULL);
        w += tw + 16;
    }
    ff_prefsize(ts, w, 23);
}

static int set_handler(void* ts, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(ts, "-state");
    if (!strcmp(nv->name, "selection")) {
        int selection = (int)(ff_intptr_t)nv->value;
        if (selection < 0) selection = -1;
        if (selection >= state->tabs) selection = state->tabs - 1;
        if (selection != state->selection) {
            state->selection = selection;
            ff_paint(ts);
            ff_post(ts, FF_SELECT, selection, 0, 0, NULL, FF_BUBBLE);
        }
        return 1;
    } else if (!strcmp(nv->name, "text")) {
        int i, newsel = -1;
        for (i=0; i < state->tabs; i++)
            if (!strcmp(nv->value, state->tab[i])) {
                newsel = i;
                break;
            }
        if (newsel != -1 && newsel != state->selection) {
            state->selection = newsel;
            ff_paint(ts);
            ff_post(ts, FF_SELECT, newsel, 0, 0, NULL, FF_BUBBLE);
        }
        return 1;
    }
    return 0;
}

static int get_handler(void* ts, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(ts, "-state");
    if (!strcmp(nv->name, "selection")) {
        nv->value = (void*)(ff_intptr_t)state->selection;
        return 1;
    } else if (!strcmp(nv->name, "text")) {
        if (state->selection >= 0 && state->selection < state->tabs)
            nv->value = state->tab[state->selection];
        else
            nv->value = "";
        return 1;
    }
    return 0;
}

static int destroy_handler(void* ts, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ts, "-state");
    int i;
    for (i=0; i < state->tabs; i++)
        free(state->tab[i]);
    free(state->tab);
    free(state->visx1);
    free(state->visx2);
    free(state);
    return 0;
}

static int paint_handler(void* ts, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, ty, tw = 0, th = 0, i, x = 6;
    state_t* state = ff_get(ts, "-state");
    if (!itableftbmp) create_bitmaps();
    ff_area(ts, NULL, NULL, &w, &h);
    ff_color(gc, FF_3DSHADOW_COLOR);
    ff_fill(gc, 0, 0, w, 21);
    for (i=0; i < state->tabs; i++) {
        int tabw;
        ff_text_size(gc, state->tab[i], &tw, &th);
        tabw = tw;
        ty = 8 + th;
        if (ty > 22) ty = 22;
        if (i == state->selection) {
            ff_draw(gc, x + tabw + 16, 6, atabrightbmp);
            ff_color(gc, FF_3DFACE_COLOR);
            ff_fill(gc, x + 16, 7, x + tabw + 16, 21);
            ff_color(gc, FF_3DLIGHT_COLOR);
            ff_line(gc, x + 16, 6, x + tabw + 15, 6);
        } else {
            ff_draw(gc, x, 6, itableftbmp);
            ff_color(gc, FF_3DSHADOW_COLOR);
            ff_fill(gc, x + 16, 6, x + tabw + 16, 21);
            ff_color(gc, FF_3DFACE_COLOR);
            ff_line(gc, x + 16, 6, x + tabw + 15, 6);
        }
        if (i > 0 && i - 1 != state->selection) ff_draw(gc, x, 6, itabrightbmp);
        if (i > 0 && i - 1 == state->selection) ff_draw(gc, x, 6, atabrightbmp);
        if (i == state->selection) ff_draw(gc, x, 6, atableftbmp);
        ff_color(gc, FF_3DTEXT_COLOR);
        ff_text(gc, x + 16, ty, state->tab[i]);
        state->visx1[i] = x;
        x += tabw + 16;
        state->visx2[i] = x;
    }
    if (state->tabs > 0) {
        if (state->selection == state->tabs - 1)
            ff_draw(gc, x, 6, atabrightbmp);
        else
            ff_draw(gc, x, 6, itabrightbmp);
    }
    ff_color(gc, FF_3DLIGHT_COLOR);
    if (state->selection == -1)
        ff_line(gc, 0, 22, w, 22);
    else {
        ff_line(gc, 0, 22, state->visx1[state->selection] + 2, 22);
        ff_line(gc, state->visx2[state->selection] + 14, 22, w, 22);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_line(gc, state->visx1[state->selection] + 3, 22, state->visx2[state->selection] + 13, 22);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_pixel(gc, state->visx2[state->selection] + 12, 22);
    }
    return 1;
}

static int press_handler(void* ts, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ts, "-state");
    if (ev->z == 1) {
        int i, newsel = -1;
        for (i=0; i < state->tabs; i++)
            if (ev->x >= state->visx1[i] + 8 &&
                ev->x <= state->visx2[i] + 8) {
                newsel = i;
                break;
            }
        if (newsel != state->selection && newsel != -1) {
            state->selection = newsel;
            ff_paint(ts);
            ff_post(ts, FF_SELECT, newsel, 0, 0, NULL, FF_BUBBLE);
        }
        return FF_YES;
    } else if (ev->z == 4) {
        if (state->selection > 0) {
            state->selection--;
            ff_paint(ts);
            ff_post(ts, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
            return FF_YES;
        }
    } else if (ev->z == 5) {
        if (state->selection < state->tabs - 1) {
            state->selection++;
            ff_paint(ts);
            ff_post(ts, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
            return FF_YES;
        }
    }
    return FF_NO;
}

ff_window_t ff_tabstrip(ff_window_t parent, const char** tab, int count, int selection)
{
    ff_window_t ts = ff_window(parent, 0, 0, 100, 23, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    ff_set(ts, "-state", state);
    ff_set(ts, "class", (void*)"tabs");
    ff_link(ts, FF_SET, set_handler, NULL);
    ff_link(ts, FF_GET, get_handler, NULL);
    ff_link(ts, FF_DESTROY, destroy_handler, NULL);
    ff_link(ts, FF_PAINT, paint_handler, NULL);
    ff_link(ts, FF_PRESS, press_handler, NULL);
    if (tab && count)
        ff_tabstrip_set(ts, tab, count, selection);
    else {
        state->selection = -1;
        update_prefsize(ts);
    }
    return ts;

}

void ff_tabstrip_set(ff_window_t ts, const char** tab, int count, int selection)
{
    state_t* state = ff_get(ts, "-state");
    int i;
    if (count < 0) count = 0;
    if (selection < -1) selection = -1;
    if (selection >= count) selection = count - 1;
    for (i=0; i < state->tabs; i++)
        free(state->tab[i]);
    free(state->tab);
    free(state->visx1);
    free(state->visx2);
    if (count && tab) {
        state->tab = malloc(sizeof(char*)*(size_t)count);
        state->visx1 = malloc(sizeof(int)*(size_t)count);
        state->visx2 = malloc(sizeof(int)*(size_t)count);
        state->tabs = count;
        for (i=0; i < count; i++)
            state->tab[i] = ff_strdup(tab[i]);
        state->selection = selection;
    } else {
        state->tabs = 0;
        state->tab = NULL;
        state->visx1 = NULL;
        state->visx2 = NULL;
        state->selection = -1;
    }
    update_prefsize(ts);
    ff_paint(ts);
    ff_post(ts, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
}
