#define LFORMS_SOURCE_CODE
#include <lforms.h>

enum { LEFT, RIGHT, BOTH };

static char* holeglyph[] = {
    "6 6 5 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    " -... ",
    "-.----",
    ".--***",
    ".-**@@",
    ".-*@@@",
    " -*@@ "};

typedef struct _state_t
{
    ff_window_t a, b;
    int position, oldsize;
    unsigned grow:2;
    unsigned ofs:4;
    unsigned orientation:1;
    unsigned moving:1;
} state_t;

static ff_bitmap_t holebmp;

static void free_bitmap(void)
{
    if (!holebmp) return;
    ff_bitmap_free(holebmp);
}

static void update_prefsize(void* spl, state_t* state)
{
    int paw, pah, pbw, pbh;
    if (!state->a || !state->b) return;
    ff_lh_prefsize(state->a, &paw, &pah, FF_LHINVISIBLE);
    ff_lh_prefsize(state->b, &pbw, &pbh, FF_LHINVISIBLE);
    if (state->orientation == FF_HORIZONTAL) {
        if (pah < pbh) pah = pbh;
        ff_prefsize(spl, paw + pbw + 8, pah);
    } else {
        if (paw < pbw) paw = pbw;
        ff_prefsize(spl, paw, pah + pbh + 8);
    }
}

static void update_layout(void* spl, state_t* state, int w, int h, int growth)
{
    if (!state->a || !state->b) return;
    switch (state->grow) {
    case LEFT: state->position += growth; break;
    case BOTH: state->position += growth/2; break;
    }
    if (state->orientation == FF_HORIZONTAL) {
        if (state->position > w - 8) state->position = w - 8;
        if (state->position < 0) state->position = 0;
        if (state->position > 0) {
            ff_show(state->a, FF_YES);
            ff_place(state->a, 0, 0, state->position, h);
        } else ff_show(state->a, FF_NO);
        if (state->position < w - 9) {
            ff_show(state->b, w - (state->position + 8) > 0);
            ff_place(state->b, state->position + 8, 0, w - (state->position + 8), h);
        } else ff_show(state->b, FF_NO);
    } else {
        if (state->position > h - 8) state->position = h - 8;
        if (state->position < 0) state->position = 0;
        if (state->position > 0) {
            ff_show(state->a, FF_YES);
            ff_place(state->a, 0, 0, w, state->position);
        } else ff_show(state->a, FF_NO);
        if (state->position < h - 9) {
            ff_show(state->b, h - (state->position + 8) > 0);
            ff_place(state->b, 0, state->position + 8, w, h - (state->position + 8));
        } else ff_show(state->b, FF_NO);
    }
}

static int set_handler(void* spl, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    if (!strcmp(nv->name, "position")) {
        state_t* state = ff_get(spl, "-state");
        int w, h, newpos = (int)(ff_intptr_t)nv->value;
        if (newpos != state->position) {
            state->position = newpos;
            ff_area(spl, NULL, NULL, &w, &h);
            update_layout(spl, state, w, h, 0);
            ff_paint(spl);
        }
        return FF_YES;
    } else if (!strcmp(nv->name, "grow")) {
        state_t* state = ff_get(spl, "-state");
        if (!strcmp(nv->value, "left") || !strcmp(nv->value, "top")) state->grow = LEFT;
        else if (!strcmp(nv->value, "right") || !strcmp(nv->value, "bottom")) state->grow = RIGHT;
        else if (!strcmp(nv->value, "both")) state->grow = BOTH;
        return FF_YES;
    }
    return FF_NO;
}

static int get_handler(void* spl, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    if (!strcmp(nv->name, "position")) {
        state_t* state = ff_get(spl, "-state");
        nv->value = (void*)(ff_intptr_t)state->position;
        return FF_YES;
    } else if (!strcmp(nv->name, "grow")) {
        state_t* state = ff_get(spl, "-state");
        switch (state->grow) {
        case LEFT: nv->value = state->orientation == FF_HORIZONTAL ? "left" : "top"; break;
        case RIGHT: nv->value = state->orientation == FF_HORIZONTAL ? "right" : "bottom"; break;
        case BOTH: nv->value = "both"; break;
        }
        return FF_YES;
    }
    return FF_NO;
}

static int area_handler(void* spl, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(spl, "-state");
    if (state->orientation == FF_HORIZONTAL) {
        if (!state->oldsize) state->oldsize = ev->x;
        update_layout(spl, state, ev->x, ev->y, ev->x - state->oldsize);
        state->oldsize = ev->x;
    } else {
        if (!state->oldsize) state->oldsize = ev->y;
        update_layout(spl, state, ev->x, ev->y, ev->y - state->oldsize);
        state->oldsize = ev->y;
    }
    return FF_YES;
}

static int press_handler(void* spl, ff_event_t* ev, void* data)
{
    if (ev->z == 1) {
        state_t* state = ff_get(spl, "-state");
        if (state->orientation == FF_HORIZONTAL &&
            ev->x >= state->position && ev->x < state->position + 9) {
            state->moving = FF_YES;
            state->ofs = (unsigned)(ev->x - state->position)&0x0F;
            return FF_YES;
        } else if (state->orientation == FF_VERTICAL &&
            ev->y >= state->position && ev->y < state->position + 9) {
            state->moving = FF_YES;
            state->ofs = (unsigned)(ev->y - state->position)&0x0F;
            return FF_YES;
        }
    }
    return FF_NO;
}

static int release_handler(void* spl, ff_event_t* ev, void* data)
{
    if (ev->z == 1) {
        state_t* state = ff_get(spl, "-state");
        state->moving = FF_NO;
        return FF_YES;
    }
    return FF_NO;
}

static int motion_handler(void* spl, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(spl, "-state");
    if (state->moving) {
        int w, h;
        state->position = (state->orientation == FF_HORIZONTAL ? ev->x : ev->y) - state->ofs;
        ff_area(spl, NULL, NULL, &w, &h);
        update_layout(spl, state, w, h, 0);
        ff_paint(spl);
        return FF_YES;
    }
    return FF_NO;
}

static int dblclick_handler(void* spl, ff_event_t* ev, void* data)
{
    if (ev->z == 1) {
        int w, h;
        state_t* state = ff_get(spl, "-state");
        ff_area(spl, NULL, NULL, &w, &h);
        state->moving = FF_NO;
        if ((state->orientation == FF_HORIZONTAL &&
             ev->x >= state->position &&
             ev->x < state->position + 8) ||
            (state->orientation == FF_VERTICAL &&
             ev->y >= state->position &&
             ev->y < state->position + 8))
            ff_splitter_fit(spl);
        return FF_YES;
    }
    return FF_NO;
}

static int paint_handler(void* spl, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h;
    state_t* state = ff_get(spl, "-state");
    ff_area(spl, NULL, NULL, &w, &h);

    ff_color_attr(gc, spl, "background", FF_3DFACE_COLOR);
    ff_fill(gc, 0, 0, w - 1, h - 1);
    if (!holebmp) {
        holebmp = ff_xpm(holeglyph, NULL, NULL);
        ff_atexit(free_bitmap);
    }
    switch (state->orientation) {
    case FF_HORIZONTAL:
        ff_draw(gc, state->position + 1, h/2-3, holebmp);
        break;
    case FF_VERTICAL:
        ff_draw(gc, w/2-3, state->position + 1, holebmp);
        break;
    }
    return FF_YES;
}

static int child_handler(void* spl, ff_event_t* ev, void* data)
{
    if (ev->x == FF_ADD || ev->x == FF_REMOVE) {
        state_t* state = ff_get(spl, "-state");
        ff_window_t* children = ff_children(spl);
        state->a = children[0] ? children[0] : NULL;
        state->b = children[0] && children[1] ? children[1] : NULL;
        ff_free_children(children);
        if (state->a && state->b) update_prefsize(spl, state);
    }
    return FF_NO;
}

static int destroy_handler(void* spl, ff_event_t* ev, void* data)
{
    free(ff_get(spl, "-state"));
    return FF_NO;
}

static void splitter_layout(ff_window_t spl, ff_window_t* child, int children, int width, int height)
{
    state_t* state = ff_get(spl, "-state");
    update_layout(spl, state, width, height, 0);
    update_prefsize(spl, state);
}

ff_window_t ff_splitter(ff_window_t parent, int orientation)
{
    ff_window_t spl = ff_window(parent, 0, 0, 1, 1, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    state->orientation = orientation == FF_VERTICAL ? FF_VERTICAL : FF_HORIZONTAL;
    state->grow = RIGHT;
    ff_set(spl, "-state", state);
    ff_link(spl, FF_SET, set_handler, NULL);
    ff_link(spl, FF_GET, get_handler, NULL);
    ff_link(spl, FF_AREA, area_handler, NULL);
    ff_link(spl, FF_PRESS, press_handler, NULL);
    ff_link(spl, FF_RELEASE, release_handler, NULL);
    ff_link(spl, FF_MOTION, motion_handler, NULL);
    ff_link(spl, FF_DBLCLICK, dblclick_handler, NULL);
    ff_link(spl, FF_PAINT, paint_handler, NULL);
    ff_link(spl, FF_CHILD, child_handler, NULL);
    ff_link(spl, FF_DESTROY, destroy_handler, NULL);
    ff_layout(spl, splitter_layout);
    ff_cursor(spl, orientation == FF_HORIZONTAL ? FF_HARROWS : FF_VARROWS);
    return spl;
}

void ff_splitter_fit(ff_window_t spl)
{
    state_t* state = ff_get(spl, "-state");
    int as, bs, size;
    if (!state->a || !state->b) return;
    if (state->orientation == FF_HORIZONTAL) {
        switch (state->grow) {
        case LEFT:
            ff_lh_prefsize(state->b, &bs, NULL, FF_LHINVISIBLE);
            ff_area(spl, NULL, NULL, &size, NULL);
            ff_seti(spl, "position", size - bs - 9);
            break;
        case RIGHT:
            ff_lh_prefsize(state->a, &as, NULL, FF_LHINVISIBLE);
            ff_seti(spl, "position", as);
            break;
        case BOTH:
            ff_lh_prefsize(state->a, &as, NULL, FF_LHINVISIBLE);
            ff_lh_prefsize(state->b, &bs, NULL, FF_LHINVISIBLE);
            ff_area(spl, NULL, NULL, &size, NULL);
            size = as + (size - as - bs)/2;
            if (size < as) size = as;
            ff_seti(spl, "position", size);
            break;
        }
    } else {
        switch (state->grow) {
        case LEFT:
            ff_lh_prefsize(state->b, NULL, &bs, FF_LHINVISIBLE);
            ff_area(spl, NULL, NULL, NULL, &size);
            ff_seti(spl, "position", size - bs - 9);
            break;
        case RIGHT:
            ff_lh_prefsize(state->a, NULL, &as, FF_LHINVISIBLE);
            ff_seti(spl, "position", as);
            break;
        case BOTH:
            ff_lh_prefsize(state->a, NULL, &as, FF_LHINVISIBLE);
            ff_lh_prefsize(state->b, NULL, &bs, FF_LHINVISIBLE);
            ff_area(spl, NULL, NULL, NULL, &size);
            size = as + (size - as - bs)/2;
            if (size < as) size = as;
            ff_seti(spl, "position", size);
            break;
        }
    }
}
