#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <string.h>
#include <lforms.h>

typedef struct _state_t
{
    int min, max, pos, change;
    int tp;
    int ofs;
    unsigned down:1;
    unsigned horizontal:1;
} state_t;

static int pos_at_xy(ff_window_t sld, state_t* state, int xy)
{
    int w, h, vsize;
    ff_area(sld, NULL, NULL, &w, &h);
    vsize = state->horizontal ? w - 21 : h - 21;
    if (vsize < 0) return state->pos;
    xy -= 9;
    if (xy < 0) xy = 0;
    if (xy > vsize) xy = vsize;
    return vsize > 0 ? (xy*(state->max - state->min)/vsize + state->min) : state->pos;
}

static int xy_at_pos(ff_window_t sld, state_t* state, int pos)
{
    int w, h, vsize;
    if (state->max <= state->min) return 1;
    ff_area(sld, NULL, NULL, &w, &h);
    vsize = state->horizontal ? w - 21 : h - 21;
    if (vsize < 0) return 9;
    return vsize*(pos - state->min)/(state->max - state->min) + 1;
}

static int set_handler(void* sld, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(sld, "-state");
    if (!strcmp(nv->name, "min")) {
        state->min = (int)(ff_intptr_t)nv->value;
        if (state->min > state->max)
            state->min = state->max;
        if (state->pos < state->min)
            ff_seti(sld, "position", state->min);
        else
            ff_paint(sld);
        return 1;
    }
    if (!strcmp(nv->name, "max")) {
        state->max = (int)(ff_intptr_t)nv->value;
        if (state->max < state->min)
            state->max = state->min;
        if (state->pos > state->max)
            ff_seti(sld, "position", state->max);
        else
            ff_paint(sld);
        return 1;
    }
    if (!strcmp(nv->name, "position")) {
        int oldpos = state->pos, oldtp;
        state->pos = (int)(ff_intptr_t)nv->value;
        if (oldpos != state->pos) {
            if (state->pos < state->min)
                state->pos = state->min;
            if (state->pos > state->max)
                state->pos = state->max;
            oldtp = state->tp;
            state->tp = xy_at_pos(sld, state, state->pos);
            ff_post(sld, FF_CHANGE, state->pos, 0, 0, NULL, FF_BUBBLE);
            if (oldtp != state->tp) ff_paint(sld);
        }
        return 1;
    }
    if (!strcmp(nv->name, "orientation")) {
        const char* value = nv->value;
        state->horizontal = !(value && !strcmp(value, "vertical")) ? FF_YES : FF_NO;
        ff_paint(sld);
        return 1;
    }
    return 0;
}

static int get_handler(void* sld, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(sld, "-state");
    if (!strcmp(nv->name, "min")) {
        nv->value = (void*)(ff_intptr_t)state->min;
        return 1;
    }
    if (!strcmp(nv->name, "max")) {
        nv->value = (void*)(ff_intptr_t)state->max;
        return 1;
    }
    if (!strcmp(nv->name, "position")) {
        nv->value = (void*)(ff_intptr_t)state->pos;
        return 1;
    }
    if (!strcmp(nv->name, "change")) {
        nv->value = (void*)(ff_intptr_t)state->change;
        return 1;
    }
    if (!strcmp(nv->name, "orientation")) {
        if (state->horizontal)
            nv->value = "horizontal";
        else
            nv->value = "vertical";
        return 1;
    }
    return 0;
}

static int destroy_handler(void* sld, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sld, "-state");
    free(state);
    return 0;
}

static int paint_handler(void* sld, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, p;
    state_t* state = ff_get(sld, "-state");
    ff_area(sld, NULL, NULL, &w, &h);
    ff_color_attr(gc, sld, "background", FF_TRACK_COLOR);
    ff_fill(gc, 1, 1, w - 2, h - 2);
    ff_color(gc, FF_3DDARK_COLOR);
    ff_line(gc, 0, 0, w - 1, 0);
    ff_line(gc, 0, 0, 0, h - 1);
    ff_color(gc, FF_3DLIGHT_COLOR);
    ff_line(gc, 1, h - 1, w - 1, h - 1);
    ff_line(gc, w - 1, 1, w - 1, h - 1);

    p = state->tp;
    if (state->horizontal) {
        ff_color(gc, FF_3DFACE_COLOR);
        ff_fill(gc, p, 1, p + 18, h - 4);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, p, 1, p + 18, 1);
        ff_line(gc, p, 1, p, h - 2);
        ff_line(gc, p + 1, 1, p + 1, h - 2);
        ff_line(gc, p + 9, 2, p + 9, h - 2);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, p + 18, 1, p + 18, h - 2);
        ff_line(gc, p, h - 2, p + 18, h - 2);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, p + 17, 1, p + 17, h - 3);
        ff_line(gc, p + 1, h - 3, p + 17, h - 3);
        ff_line(gc, p + 8, 2, p + 8, h - 3);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_pixel(gc, p + 9, h - 3);
        if (p + 19 < w - 1) {
            ff_color(gc, FF_DARKTRACK_COLOR);
            ff_line(gc, p + 19, 1, p + 19, h - 2);
        }
    } else {
        ff_color(gc, FF_3DFACE_COLOR);
        ff_fill(gc, 1, p, w - 4, p + 16);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, 1, p, 1, p + 16);
        ff_line(gc, 2, p, 2, p + 16);
        ff_line(gc, 1, p, w - 3, p);
        ff_line(gc, 2, p + 9, w - 4, p + 9);
        ff_color(gc, FF_3DDARK_COLOR);
        ff_line(gc, w - 2, p, w - 2, p + 18);
        ff_line(gc, 1, p + 18, w - 2, p + 18);
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, w - 3, p + 1, w - 3, p + 17);
        ff_line(gc, 1, p + 17, w - 3, p + 17);
        ff_line(gc, 2, p + 8, w - 3, p + 8);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_pixel(gc, w - 3, p + 9);
        if (p + 19 < h - 1) {
            ff_color(gc, FF_DARKTRACK_COLOR);
            ff_line(gc, 1, p + 19, w - 2, p + 19);
        }
    }
    return 1;
}

static int area_handler(void* sld, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sld, "-state");
    state->tp = xy_at_pos(sld, state, state->pos);
    return 0;
}

static int press_handler(void* sld, ff_event_t* ev, void* data)
{
    if (ev->z == 1) {
        state_t* state = ff_get(sld, "-state");
        if (state->horizontal) {
            if (ev->x >= state->tp && ev->x < state->tp + 20) {
                state->down = 1;
                state->ofs = ev->x - state->tp - 9;
            }
        } else {
            if (ev->y >= state->tp && ev->y < state->tp + 20) {
                state->down = 1;
                state->ofs = ev->y - state->tp - 9;
            }
        }
        return 1;
    } else if (ev->z == 4) {
        state_t* state = ff_get(sld, "-state");
        if (state->pos > state->min)
            ff_seti(sld, "position", state->pos - state->change);
        return 1;
    } else if (ev->z == 5) {
        state_t* state = ff_get(sld, "-state");
        if (state->pos < state->max)
            ff_seti(sld, "position", state->pos + state->change);
        return 1;
    }
    return 0;
}

static int release_handler(void* sld, ff_event_t* ev, void* data)
{
    if (ev->z == 1) {
        state_t* state = ff_get(sld, "-state");
        state->down = 0;
        return 1;
    }
    return 0;
}

static int motion_handler(void* sld, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(sld, "-state");
    if (state->down) {
        ff_seti(sld, "position", pos_at_xy(sld, state, (state->horizontal ? ev->x : ev->y) - state->ofs));
        return 1;
    }
    return 0;
}

ff_window_t ff_slider(ff_window_t parent, int min, int max, int position, int horizontal)
{
    ff_window_t sld = ff_window(parent, 0, 0, 100, 25, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    ff_set(sld, "-state", state);
    ff_set(sld, "class", (void*)"slider");
    ff_link(sld, FF_SET, set_handler, NULL);
    ff_link(sld, FF_GET, get_handler, NULL);
    ff_set(sld, "orientation", horizontal ? "horizontal" : "vertical");
    ff_link(sld, FF_DESTROY, destroy_handler, NULL);
    ff_link(sld, FF_PAINT, paint_handler, NULL);
    ff_link(sld, FF_AREA, area_handler, NULL);
    ff_link(sld, FF_PRESS, press_handler, NULL);
    ff_link(sld, FF_RELEASE, release_handler, NULL);
    ff_link(sld, FF_MOTION, motion_handler, NULL);
    if (min > max) min = max;
    if (max < min) max = min;
    if (position < min) position = min;
    if (position > max) position = max;
    state->min = min;
    state->max = max;
    state->pos = position;
    state->change = (max - min)/25;
    if (!state->change) state->change = 1;
    ff_prefsize(sld, horizontal ? 32 : 16, horizontal ? 16 : 32);
    return sld;
}
