#define LFORMS_SOURCE_CODE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <lforms.h>

typedef struct _line_t
{
    char* text;
    int len;
    int* x;
    int y, h;
    int xclean;
} line_t;

typedef struct _state_t
{
    ff_sbinfo_t sb;
    line_t** line;
    char* text;
    int lines;
    int cx, cy;
    int vcx;
    int sx, sy;
    int cx2, cy2;
    int monoline;
    int readonly;
} state_t;

static void go_left(ff_window_t ed, state_t* state, int paint);
static void go_right(ff_window_t ed, state_t* state, int paint);

static void update_prefsize(ff_window_t ed)
{
    state_t* state = ff_get(ed, "-state");
    if (state->monoline) {
        int tw, th, border = 0;
        ff_bitmap_t bmp = ff_bitmap(1, 1, NULL);
        ff_gc_t gc = ff_bitmap_gc(bmp);
        const char* borderattr = ff_get(ed, "border");
        if (!strcmp(borderattr, "none"))
            border = 0;
        else if (!strcmp(borderattr, "normal"))
            border = 1;
        ff_text_size(gc, "A", &tw, &th);
        ff_bitmap_free(bmp);
        ff_prefsize(ed, 150, th + 2 + (border ? 4 : 0));
    } else {
        ff_prefsize(ed, 150, 60);
    }
}

static int pos_at(state_t* state, int x, int y, int* cx, int* cy)
{
    int i;
    for (i=0; i<state->lines; i++) {
        line_t* l = state->line[i];
        if (y >= l->y && y < l->y + l->h) {
            int j;
            if (!l->xclean) return FF_NO;
            *cx = 0;
            *cy = i;
            for (j=l->len; j >= 0; j--) {
                if (x >= l->x[j] - state->sx) {
                    while (j > 0 && l->x[j - 1] == l->x[j]) j--;
                    *cx = j;
                    break;
                }
            }
            return 1;
        }
    }
    if (!state->monoline && state->lines > 0 && y > state->line[state->lines - 1]->y + state->line[state->lines - 1]->h) {
        *cx = 0;
        *cy = state->lines;
        return 1;
    }
    return 0;
}

static int show_cursor(ff_window_t ed, state_t* state)
{
    int rcx = state->vcx;
    int repaint = 0;
    int rcy = state->lines > 0 ? (state->cy < state->lines ? state->line[state->cy]->y : state->line[state->lines - 1]->y + state->line[state->lines - 1]->h) : 0;
    int lh = state->lines > 0 ? (state->cy < state->lines ? state->line[state->cy]->h : state->line[state->lines - 1]->h) : 16;
    int tmp;
    if (rcx < 16) {
        tmp = state->sx;
        state->sx -= 16-rcx;
        if (state->sx < 0)
            state->sx = 0;
        repaint |= state->sx != tmp;
    }
    if (rcx > state->cx2 - 16) {
        tmp = state->sx;
        state->sx += rcx - state->cx2 + 16;
        repaint |= tmp != state->sx;
    }
    if (state->monoline) {
        if (state->sy != 0) {
            state->sy = 0;
            state->sb.max = 1;
            ff_sbinfo_set(&state->sb, ed, state->sy);
            repaint = 1;
        }
    } else {
        if (rcy < 4) {
            tmp = state->sy;
            state->sy -= 4-rcy;
            if (state->sy < 0)
                state->sy = 0;
            repaint |= tmp != state->sy;
        }
        if (rcy > state->cy2 - lh - 4) {
            tmp = state->sy;
            state->sy += rcy - state->cy2 + lh + 4;
            repaint |= tmp != state->sy;
        }
        state->sb.max = state->lines*16;
        ff_sbinfo_set(&state->sb, ed, state->sy);
    }
    return repaint;
}

static void insert_line(state_t* state, int index, const char* line)
{
    int i;
    state->line = realloc(state->line, sizeof(line_t*)*(size_t)(state->lines + 1));
    for (i=state->lines; i>index; i--)
        state->line[i] = state->line[i - 1];
    state->line[index] = calloc(1, sizeof(line_t));
    state->line[index]->text = ff_strdup(line);
    state->line[index]->len = (int)strlen(line);
    state->line[index]->xclean = FF_NO;
    state->lines++;
}

static int insert_text(state_t* state, const char* txt)
{
    char* new;
    size_t txtlen = strlen(txt);
    if (state->cy >= state->lines) {
        insert_line(state, state->lines, "");
    }
    while (state->cx < state->line[state->cy]->len &&
        ff_in8seq((unsigned char)state->line[state->cy]->text[state->cx]))
        state->cx++;
    new = malloc((size_t)state->line[state->cy]->len + txtlen + 1U);
    memcpy(new, state->line[state->cy]->text, (size_t)state->cx);
    memcpy(new + state->cx, txt, (size_t)txtlen);
    memcpy(new + state->cx + txtlen, state->line[state->cy]->text + state->cx, (size_t)(state->line[state->cy]->len - state->cx + 1));
    free(state->line[state->cy]->text);
    state->line[state->cy]->text = new;
    state->line[state->cy]->len += (int)txtlen;
    new[state->line[state->cy]->len] = 0;
    state->line[state->cy]->xclean = FF_NO;
    return 1;
}

static int delete_char(state_t* state, int goleft)
{
    if (state->cy >= state->lines) {
        if (goleft) {
            go_left(NULL, state, 0);
            return delete_char(state, 0);
        }
        return 0;
    }

    if (goleft) {
        if (state->cx > 0)
            go_left(NULL, state, 0);
        else {
            char* new;
            int i;
            if (state->cy == 0) return 0;
            state->cx = state->line[state->cy - 1]->len;
            new = malloc((size_t)(state->line[state->cy - 1]->len + state->line[state->cy]->len + 1));
            memcpy(new, state->line[state->cy - 1]->text, (size_t)state->line[state->cy - 1]->len);
            memcpy(new + state->line[state->cy - 1]->len, state->line[state->cy]->text, (size_t)state->line[state->cy]->len + 1);
            free(state->line[state->cy - 1]->text);
            state->line[state->cy - 1]->text = new;
            state->line[state->cy - 1]->len += state->line[state->cy]->len;
            state->line[state->cy - 1]->xclean = FF_NO;
            free(state->line[state->cy]->text);
            free(state->line[state->cy]->x);
            free(state->line[state->cy]);
            for (i=state->cy; i<state->lines - 1; i++)
                state->line[i] = state->line[i + 1];
            state->lines--;
            state->cy--;
            return 1;
        }
    }

    if (state->cx >= state->line[state->cy]->len) {
        if (state->cy >= state->lines - 1) {
            if (state->cx > 0) return 0;
            free(state->line[state->lines - 1]->text);
            free(state->line[state->lines - 1]->x);
            free(state->line[state->lines - 1]);
            state->lines--;
            if (state->cy > 0) {
                state->cy--;
                state->cx = state->line[state->cy]->len;
            }
            return 1;
        }
        go_right(NULL, state, 0);
        return delete_char(state, 1);
    }
    while (state->cx > 0 && ff_in8seq((unsigned char)state->line[state->cy]->text[state->cx]))
        state->cx--;
    memmove(state->line[state->cy]->text + state->cx, state->line[state->cy]->text + state->cx + 1, (size_t)(state->line[state->cy]->len - state->cx));
    state->line[state->cy]->len--;
    while (ff_in8seq((unsigned char)state->line[state->cy]->text[state->cx])) {
        memmove(state->line[state->cy]->text + state->cx, state->line[state->cy]->text + state->cx + 1, (size_t)(state->line[state->cy]->len - state->cx));
        state->line[state->cy]->len--;
    }
    state->line[state->cy]->xclean = FF_NO;
    return 1;
}

static int set_handler(void* ed, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(ed, "-state");
    if (!strcmp(nv->name, "read-only")) {
        state->readonly = nv->value ? 1 : 0;
        return 1;
    }
    if (!strcmp(nv->name, "monoline")) {
        state->monoline = nv->value ? 1 : 0;
        if (state->monoline && state->lines > 1) {
            int i;
            for (i=1; i<state->lines; i++) {
                free(state->line[i]->text);
                free(state->line[i]->x);
                free(state->line[i]);
            }
            state->lines = 1;
        }
        update_prefsize(ed);
        ff_paint(ed);
        return 1;
    }
    if (!strcmp(nv->name, "border")) {
        update_prefsize(ed);
        ff_paint(ed);
        return 0;
    }
    if (!strcmp(nv->name, "text")) {
        int i, len;
        char* line = NULL;
        const char* text = nv->value ? nv->value : "";
        for (i=0; i<state->lines; i++) {
            free(state->line[i]->text);
            free(state->line[i]->x);
            free(state->line[i]);
        }
        free(state->line);
        state->cx = state->cy = state->sx = state->sy = 0;
        state->line = NULL;
        state->lines = 0;
        len = 0;
        for (i=0; text[i]; i++) {
            if (text[i] == '\r') continue;
            line = realloc(line, (size_t)len + 1U);
            if (text[i] == '\n') {
                if (!state->monoline) {
                    line[len] = 0;
                    insert_line(state, state->lines, line);
                    free(line);
                    line = NULL;
                    len = 0;
                }
            } else {
                line[len++] = text[i];
            }
        }
        if (line) {
            line = realloc(line, (size_t)len + 1U);
            line[len] = 0;
            insert_line(state, state->lines, line);
            free(line);
        }
        state->sb.max = state->lines*16;
        ff_sbinfo_set(&state->sb, ed, 0);
        ff_paint(ed);
        ff_post(ed, FF_CHANGE, 0, 0, 0, NULL, FF_BUBBLE);
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
        return 1;
    }
    return 0;
}

static int get_handler(void* ed, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(ed, "-state");
    if (!strcmp(nv->name, "read-only")) {
        nv->value = state->readonly ? (void*)1 : NULL;
        return 1;
    }
    if (!strcmp(nv->name, "monoline")) {
        nv->value = state->monoline ? (void*)1 : NULL;
        return 1;
    }
    if (!strcmp(nv->name, "text")) {
        int i, len = 0;
        free(state->text);
        state->text = calloc(1, 1);
        for (i=0; i<state->lines; i++) {
            len += state->line[i]->len;
            state->text = realloc(state->text, (size_t)len + 2U);
            strcat(state->text, state->line[i]->text);
            if (i < state->lines - 1) {
                state->text[len++] = '\n';
                state->text[len] = 0;
            }
        }
        nv->value = state->text;
        return 1;
    }
    return 0;
}

static int destroy_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    int i;
    for (i=0; i<state->lines; i++) {
        free(state->line[i]->text);
        free(state->line[i]->x);
        free(state->line[i]);
    }
    free(state->line);
    free(state->text);
    free(state);
    return 0;
}

static int paint_handler(void* ed, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h, tw, th = 0;
    const char* borderattr = ff_get(ed, "border");
    int border = 0;
    state_t* state = ff_get(ed, "-state");
    int x, y, i, sbpad, focused = ff_focused(ed);

    if (!strcmp(borderattr, "none"))
        border = 0;
    else if (!strcmp(borderattr, "normal"))
        border = 1;

    ff_area(ed, NULL, NULL, &w, &h);
    ff_color_attr(gc, ed, "background", FF_EDITOR_COLOR);

    sbpad = state->monoline ? 0 : 21;
    state->sy = state->sb.pos;
    x = 2 - state->sx + sbpad;
    y = 1 - state->sy;
    if (border == 1) {
        x += 2;
        y += 2;
        ff_fill(gc, 2, 2, w - 2, h - 2);
        state->cx2 = w - 2;
        state->cy2 = h - 2;
    } else {
        ff_fill(gc, 0, 0, w, h);
        state->cx2 = w - 2;
        state->cy2 = h - 2;
     }

    ff_text_size(gc, "A", &tw, &th);
    ff_color(gc, ff_enabled(ed) ? FF_EDTEXT_COLOR : FF_EDDISABLEDTEXT_COLOR);
    for (i=0; i<state->lines; i++) {
        line_t* l = state->line[i];
        int j = 0;
        l->y = y;
        l->h = th;
        y += th;
        if (l->len) {
            if (!l->xclean) {
                l->xclean = FF_YES;
                free(l->x);
                l->x = malloc((size_t)(l->len + 1)*sizeof(int));
                for (j=0; j<=l->len; j++) {
                    char save_ch = l->text[j];
                    l->text[j] = 0;
                    ff_text_size(gc, l->text, &tw, &th);
                    l->text[j] = save_ch;
                    l->x[j] = x + tw + state->sx;
                }
            }
            ff_text(gc, x, y, l->text);
            if (focused && state->cy == i) {
                ff_line(gc, l->x[state->cx] - state->sx, y - th, l->x[state->cx] - state->sx, y - 1);
                state->vcx = l->x[state->cx] - state->sx;
            }
        } else {
            l->y = y;
            l->h = th;
            if (focused && state->cy == i) {
                ff_line(gc, x, y - th, x, y - 1);
                state->vcx = x;
            }
        }
        if (focused && state->cx == j && state->cy == i) {
            ff_line(gc, x, y - th, x, y - 1);
            state->vcx = x;
        }
    }
    if (!th)
        ff_text_size(gc, "A", &tw, &th);
    if (focused && state->cy == state->lines) {
        ff_line(gc, x, y, x, y + th - 1);
        state->vcx = x;
    }

    if (show_cursor(ed, state)) {
        static int fail = 0;
        if (++fail < 16) {
            int r = paint_handler(ed, ev, data);
            fail--;
            return r;
        } else
            fail = 0;
    }

    if (border == 1) {
        ff_color(gc, FF_3DSHADOW_COLOR);
        ff_line(gc, 0, 0, w - 2, 0);
        ff_line(gc, 0, 0, 0, h - 2);
        ff_line(gc, 1, 1, w - 3, 1);
        ff_line(gc, 1, 1, 1, h - 3);
        ff_color(gc, FF_3DLIGHT_COLOR);
        ff_line(gc, w - 1, 0, w - 1, h - 1);
        ff_line(gc, 0, h - 1, w - 1, h - 1);
        ff_color(gc, FF_3DFACE_COLOR);
        ff_line(gc, w - 2, 1, w - 2, h - 2);
        ff_line(gc, 1, h - 2, w - 2, h - 2);
    }

    if (!state->monoline) {
        ff_sbinfo_paint(&state->sb, ed, gc);
    }

    return 1;
}

static int area_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    int border = 0;
    const char* borderattr;
    if (state->monoline) return 0;
    borderattr = ff_get(ed, "border");
    if (!strcmp(borderattr, "none"))
        border = 0;
    else if (!strcmp(borderattr, "normal"))
        border = 1;
    state->sb.page = 0;
    if (border)
        ff_sbinfo_place(&state->sb, ed, 1, 1, 20, ev->y - 2);
    else
        ff_sbinfo_place(&state->sb, ed, 0, 0, 20, ev->y);
    ff_sbinfo_update(&state->sb, ed);
    return 0;
}

static int press_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    ff_focus(ed);
    if (ff_sbinfo_press(&state->sb, ed, ev->x, ev->y, ev->z)) {
        if (ff_sbinfo_needs_timer(&state->sb))
            ff_timer(ed, 5);
        return 1;
    }
    if (ev->z == 1) {
        int cx, cy;
        if (pos_at(state, ev->x, ev->y, &cx, &cy)) {
            state->cx = cx;
            state->cy = cy;
            ff_paint(ed);
            ff_post(ed, FF_CARET, cx, cy, 0, NULL, FF_BUBBLE);
        }
        return 1;
    }
    return 0;
}

static int release_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    if (ff_sbinfo_release(&state->sb, ed, ev->x, ev->y, ev->z)) {
        if (ff_sbinfo_needs_timer(&state->sb))
            ff_timer(ed, 5);
        return 1;
    }
    /*state_t* state = ff_get(ed, "-state");
    if (ev->z == 1) {
    }*/
    return 0;
}

static int motion_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    return ff_sbinfo_motion(&state->sb, ed, ev->x, ev->y);
}

static void go_left(ff_window_t ed, state_t* state, int paint)
{
    if (state->cx > 0) {
        state->cx--;
        while (state->cx > 0 && ff_in8seq((unsigned char)state->line[state->cy]->text[state->cx]))
            state->cx--;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else {
        if (state->cy > 0) {
            state->cy--;
            state->cx = state->line[state->cy]->len;
            ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
        } else paint = 0;
    }
    if (paint) ff_paint(ed);
}

static void go_right(ff_window_t ed, state_t* state, int paint)
{
    if (state->cy >= state->lines) return;
    if (state->cx < state->line[state->cy]->len) {
        state->cx++;
        while (state->cx < state->line[state->cy]->len &&
               ff_in8seq((unsigned char)state->line[state->cy]->text[state->cx]))
            state->cx++;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else if (!state->monoline) {
        state->cx = 0;
        state->cy++;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else paint = 0;
    if (paint) ff_paint(ed);
}

static void go_up(ff_window_t ed, state_t* state, int paint)
{
    if (state->cy > 0) {
        state->cy--;
        if (state->cx > state->line[state->cy]->len)
            state->cx = state->line[state->cy]->len;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else if (state->cx > 0) {
        state->cx = 0;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else paint = 0;
    if (paint) ff_paint(ed);
}

static void go_down(ff_window_t ed, state_t* state, int paint)
{
    if (state->monoline) {
        if (state->cy < state->lines)
            state->cx = state->line[state->cy]->len;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else if (state->cy < state->lines) {
        state->cy++;
        if (state->cy < state->lines) {
            if (state->cx > state->line[state->cy]->len)
                state->cx = state->line[state->cy]->len;
        } else {
            state->cx = 0;
        }
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else paint = 0;
    if (paint) ff_paint(ed);
}

static void go_bol(ff_window_t ed, state_t* state, int paint)
{
    if (state->cy < state->lines) {
        state->cx = 0;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else paint = 0;
    if (paint) ff_paint(ed);
}

static void go_eol(ff_window_t ed, state_t* state, int paint)
{
    if (state->cy < state->lines) {
        state->cx = state->line[state->cy]->len;
        ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    } else paint = 0;
    if (paint) ff_paint(ed);
}

static void insert_return(ff_window_t ed, state_t* state, int paint)
{
    if (state->cy == state->lines) {
        insert_line(state, state->lines, "");
    } else {
        insert_line(state, state->cy + 1, state->line[state->cy]->text + state->cx);
        state->line[state->cy]->text[state->cx] = 0;
        state->line[state->cy]->len = state->cx;
        state->line[state->cy]->xclean = FF_NO;
    }
    state->cy++;
    state->cx = 0;
    ff_post(ed, FF_CHANGE, 0, 0, 0, NULL, FF_BUBBLE);
    ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
    if (paint) ff_paint(ed);
}

static int keydown_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    if (!ff_focused(ed)) return 0;

    switch (ev->x) {
    case FFK_DELETE:
        if (state->readonly) return 0;
        if (delete_char(state, 0)) {
            ff_paint(ed);
            ff_post(ed, FF_CHANGE, 0, 0, 0, NULL, FF_BUBBLE);
        }
        return 1;
    case FFK_BACKSPACE:
        if (state->readonly) return 0;
        if (delete_char(state, 1)) {
            ff_paint(ed);
            ff_post(ed, FF_CHANGE, 0, 0, 0, NULL, FF_BUBBLE);
            ff_post(ed, FF_CARET, state->cx, state->cy, 0, NULL, FF_BUBBLE);
        }
        return 1;
    case FFK_RETURN:
        if (state->readonly) return 0;
        if (!state->monoline) {
            insert_return(ed, state, 1);
            return 1;
        }
        return 0;
    case FFK_HOME:
        go_bol(ed, state, 1);
        return 1;
    case FFK_END:
        go_eol(ed, state, 1);
        return 1;
    case FFK_LEFT:
        go_left(ed, state, 1);
        return 1;
    case FFK_RIGHT:
        go_right(ed, state, 1);
        break;
    case FFK_UP:
        if (state->monoline) return 0;
        go_up(ed, state, 1);
        return 1;
    case FFK_DOWN:
        if (state->monoline) return 0;
        go_down(ed, state, 1);
        return 1;
    }
    return 0;
}

static int keyup_handler(void* ed, ff_event_t* ev, void* data)
{
    return 0;
}

static int text_handler(void* ed, ff_event_t* ev, void* data)
{
    const char* text = ev->p;
    state_t* state = ff_get(ed, "-state");
    if (state->readonly || (!text[1] && text[0] < 32)) return 0;
    if (text[0] == 127 && !text[1]) return 0; /* ignore del char */
    if (insert_text(state, text)) {
        go_right(ed, state, 1);
        ff_post(ed, FF_CHANGE, 0, 0, 0, NULL, FF_BUBBLE);
    }
    return 1;
}

static int timer_handler(void* ed, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(ed, "-state");
    return ff_sbinfo_timer(&state->sb, ed);
}

static int focus_handler(void* ed, ff_event_t* ev, void* data)
{
    ff_paint(ed);
    return 0;
}

ff_window_t ff_editor(ff_window_t parent, const char* text)
{
    ff_window_t ed = ff_window(parent, 0, 0, 150, 20, FF_DOUBLEBUFFER|FF_TEXTINPUT);
    state_t* state = calloc(1, sizeof(state_t));
    ff_sbinfo_init(&state->sb, FF_SBVERTICAL);
    ff_set(ed, "-state", state);
    ff_set(ed, "class", (void*)"editor");
    ff_set(ed, "border", "normal");
    ff_set(ed, "read-only", 0);
    ff_link(ed, FF_SET, set_handler, NULL);
    ff_link(ed, FF_GET, get_handler, NULL);
    ff_link(ed, FF_DESTROY, destroy_handler, NULL);
    ff_link(ed, FF_PAINT, paint_handler, NULL);
    ff_link(ed, FF_AREA, area_handler, NULL);
    ff_link(ed, FF_PRESS, press_handler, NULL);
    ff_link(ed, FF_RELEASE, release_handler, NULL);
    ff_link(ed, FF_MOTION, motion_handler, NULL);
    ff_link(ed, FF_KEYDOWN, keydown_handler, NULL);
    ff_link(ed, FF_KEYUP, keyup_handler, NULL);
    ff_link(ed, FF_TEXT, text_handler, NULL);
    ff_link(ed, FF_TIMER, timer_handler, NULL);
    ff_link(ed, FF_FOCUS, focus_handler, NULL);
    ff_link(ed, FF_BLUR, focus_handler, NULL);
    if (text) ff_set(ed, "text", (char*)text);
    ff_cursor(ed, FF_IBEAM);
    update_prefsize(ed);

    return ed;
}
