#define LFORMS_SOURCE_CODE
#include <lforms.h>

static char* menuglyph[] = {
    "12 8 5 1",
    "  s none",
    ". s 3d-dark",
    "- s 3d-shadow",
    "@ s 3d-light",
    "* s 3d-face",
    "@@@@@@@@.   ",
    "@******-.   ",
    "@******-.-- ",
    "@******-.-- ",
    "@-------.-- ",
    ".........-- ",
    "  --------- ",
    "  --------- "};
static ff_bitmap_t menubmp;

typedef struct _item_t
{
    char* text;
    void* data;
} item_t;

typedef struct _state_t
{
    item_t** item;
    ff_window_t popup;
    int items;
    int selection;
    int highlight;
} state_t;

static void free_glyph(void)
{
    ff_bitmap_free(menubmp);
    menubmp = NULL;
}

static void paint_item(ff_gc_t gc, int x, int y, int w, int h, const char* text, ff_color_t bg, ff_color_t fg, int selected)
{
    int th;
    ff_color(gc, bg);
    ff_fill(gc, x + 1, y + 1, x + w - 3, y + h - 3);
    ff_color(gc, FF_3DLIGHT_COLOR);
    ff_line(gc, x, y, x + w - 2, y);
    ff_line(gc, x, y, x, y + h - 2);
    ff_color(gc, FF_3DSHADOW_COLOR);
    ff_line(gc, x + w - 2, y + 1, x + w - 2, y + h - 2);
    ff_line(gc, x + 1, y + h - 2, x + w - 2, y + h - 2);
    ff_color(gc, FF_3DDARK_COLOR);
    ff_line(gc, x + w - 1, y, x + w - 1, y + h - 1);
    ff_line(gc, x, y + h - 1, x + w - 1, y + h - 1);
    ff_color(gc, fg);
    ff_text_size(gc, text, NULL, &th);
    ff_text(gc, x + 6, y + (h - th)/2 + th, text);

    if (selected) {
        if (!menubmp) {
            menubmp = ff_xpm(menuglyph, NULL, NULL);
            ff_atexit(free_glyph);
        }
        ff_draw(gc, x + w - 15, y + h/2 - 4, menubmp);
    }
}

static void calc_prefsize(ff_window_t cb, int* w, int* h)
{
    state_t* state = ff_get(cb, "-state");
    int tw, th;
    int i;
    ff_font_t font = ff_font_of(cb);
    *w = 0;
    for (i=0; i < state->items; i++) {
        ff_font_text_size(font, state->item[i]->text, &tw, &th);
        if (*w < tw) *w = tw;
    }
    if (!state->items) ff_font_text_size(font, "A", &tw, &th);
    *w += 26;
    *h = th + 4;
    if (*h < 20) *h = 20;
}

static void update_prefsize(ff_window_t cb)
{
    if (!ff_frozen(cb)) {
        int w, h;
        calc_prefsize(cb, &w, &h);
        ff_prefsize(cb, w, h);
    }
}

static int popup_item_height(ff_window_t pp, ff_window_t cb)
{
    int r;
    if (!cb) cb = ff_get(pp, "-choice");
    ff_area(cb, NULL, NULL, NULL, &r);
    return r;
}

static int popup_paint(void* pp, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, ih, i;
    state_t* state = ff_get(pp, "-state");
    ff_window_t cb = ff_get(pp, "-choice");
    ff_color_t bg = ff_color_get(cb, "background", FF_3DFACE_COLOR),
               fg = ff_color_get(cb, "foreground", FF_3DTEXT_COLOR);
    ff_area(pp, NULL, NULL, &w, NULL);
    ih = popup_item_height(pp, cb);
    for (i=0; i < state->items; i++) {
        paint_item(gc, 0, i*ih, w, ih, state->item[i]->text,
                i == state->highlight ? FF_3DLIGHT_COLOR : bg,
                i == state->highlight ? FF_3DTEXT_COLOR : fg,
                i == state->selection);
    }
    return 1;
}

static int set_handler(void* cb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(cb, "-state");
    if (!strcmp(nv->name, "selection")) {
        int newsel = (int)(ff_intptr_t)nv->value;
        if (newsel != state->selection) {
            state->selection = newsel;
            ff_paint(cb);
            ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
        }
        return 1;
    } else if (!strcmp(nv->name, "text")) {
        int i, newsel = -1;
        const char* text = nv->value;
        for (i=0; i < state->items; i++)
            if (!strcmp(text, state->item[i]->text)) {
                newsel = i;
                break;
            }
        if (newsel != -1 && newsel != state->selection) {
            state->selection = newsel;
            ff_paint(cb);
            ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
        }
        return 1;
    }
    return 0;
}

static int get_handler(void* cb, ff_event_t* ev, void* data)
{
    ff_namevalue_t* nv = ev->p;
    state_t* state = ff_get(cb, "-state");
    if (!strcmp(nv->name, "selection")) {
        nv->value = (void*)(ff_intptr_t)state->selection;
        return 1;
    } else if (!strcmp(nv->name, "text")) {
        if (state->selection < 0 || state->selection >= state->items)
            nv->value = "";
        else
            nv->value = state->item[state->selection]->text;
        return 1;
    }
    return 0;
}

static int destroy_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    int i;
    for (i=0; i<state->items; i++) {
        free(state->item[i]->text);
        free(state->item[i]);
    }
    free(state->item);
    free(state);
    return 0;
}

static int newfont_handler(void* cb, ff_event_t* ev, void* data)
{
    update_prefsize(cb);
    return 0;
}

static int paint_handler(void* cb, ff_event_t* ev, void* data)
{
    ff_gc_t gc = ev->p;
    int w, h;
    state_t* state = ff_get(cb, "-state");
    ff_color_t bg = ff_color_get(cb, "background", FF_3DFACE_COLOR);
    ff_color_t fg = ff_color_get(cb, "foreground", ff_enabled(cb) ?
                                                    FF_3DTEXT_COLOR :
                                                    FF_3DDISABLEDTEXT_COLOR);
    ff_area(cb, NULL, NULL, &w, &h);
    paint_item(gc, 0, 0, w, h, state->selection >= 0 ? state->item[state->selection]->text : "", bg, fg, 1);
    return 1;
}

static int press_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    if (ev->z == 1) {
        int px = 0, py = 0, pw, ph;
        if (state->popup) {
            ff_destroy(state->popup);
            state->popup = 0;
        }
        if (!state->items) return 1;
        state->highlight = state->selection;
        ff_translate(&px, &py, cb, NULL);
        ff_area(cb, NULL, NULL, &pw, &ph);
        if (state->selection >= 0) py -= state->selection*ph;
        state->popup = ff_window(cb, px, py, pw, state->items*ph, FF_NOFRAME|FF_POPUP|FF_DOUBLEBUFFER|FF_INVISIBLE);
        ff_set(state->popup, "-state", state);
        ff_set(state->popup, "-choice", cb);
        ff_set(state->popup, "font", ff_font_of(cb));
        ff_link(state->popup, FF_PAINT, popup_paint, NULL);
        ff_show(state->popup, FF_YES);
        return 1;
    } else if (ev->z == 4) {
        if (state->selection > 0) {
            state->selection--;
            ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
            ff_paint(cb);
        }
        return 1;
    } else if (ev->z == 5) {
        if (state->selection < state->items - 1) {
            state->selection++;
            ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
            ff_paint(cb);
        }
        return 1;
    }
    return 0;
}

static int release_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    if (ev->z == 1) {
        if (state->popup) {
            ff_destroy(state->popup);
            state->popup = NULL;
            if (state->highlight != -1) {
                state->selection = state->highlight;
                ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
            }
            ff_paint(cb);
        }
        return 1;
    }
    return 0;
}

static int motion_handler(void* cb, ff_event_t* ev, void* data)
{
    state_t* state = ff_get(cb, "-state");
    if (state->popup) {
        int newsel = ev->y, ph = popup_item_height(state->popup, cb);
        ff_translate(NULL, &newsel, cb, state->popup);
        if (newsel < 0 || newsel >= state->items*ph)
            newsel = -1;
        else
            newsel = newsel/ph;
        if (newsel != state->highlight) {
            state->highlight = newsel;
            ff_paint(state->popup);
        }
        return 1;
    }
    return 0;
}

static int thaw_handler(void* cb, ff_event_t* ev, void* data)
{
    update_prefsize(cb);
    return 0;
}

ff_window_t ff_choice(ff_window_t parent, const char** item, void** data, int items, int selection)
{
    ff_window_t cb = ff_window(parent, 0, 0, 150, 150, FF_NOFLAGS);
    state_t* state = calloc(1, sizeof(state_t));
    if (selection < 0) selection = -1;
    if (items > 0 && (selection < 0 || selection >= items)) selection = 0;
    if (selection >= items) selection = items - 1;
    state->selection = -1;
    ff_set(cb, "-state", state);
    ff_set(cb, "class", (void*)"choice");
    ff_seti(cb, "selection", items > 0 ? -1 : selection);
    ff_link(cb, FF_SET, set_handler, NULL);
    ff_link(cb, FF_GET, get_handler, NULL);
    ff_link(cb, FF_DESTROY, destroy_handler, NULL);
    ff_link(cb, FF_NEWFONT, newfont_handler, NULL);
    ff_link(cb, FF_PAINT, paint_handler, NULL);
    ff_link(cb, FF_PRESS, press_handler, NULL);
    ff_link(cb, FF_RELEASE, release_handler, NULL);
    ff_link(cb, FF_MOTION, motion_handler, NULL);
    ff_link(cb, FF_THAW, thaw_handler, NULL);
    update_prefsize(cb);
    if (items > 0 && (item || data)) ff_choice_set(cb, item, data, items, selection);
    return cb;
}

void ff_choice_clear(ff_window_t cb)
{
    state_t* state = ff_get(cb, "-state");
    int i;
    if (!state->items) return;
    for (i=0; i<state->items; i++) {
        free(state->item[i]->text);
        free(state->item[i]);
    }
    free(state->item);
    state->item = NULL;
    state->items = 0;
    state->selection = -1;
    update_prefsize(cb);
    ff_paint(cb);
    ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
}

void ff_choice_set(ff_window_t cb, const char** item, void** data, int items, int selection)
{
    state_t* state = ff_get(cb, "-state");
    int i;
    if (items < 0) items = 0;
    if (items > 0 && (selection < 0 || selection >= items)) selection = 0;
    if (selection >= items) selection = items - 1;
    if (state->items) {
        for (i=0; i<state->items; i++) {
            free(state->item[i]->text);
            free(state->item[i]);
        }
        free(state->item);
        state->item = NULL;
    }
    state->items = items;
    if (items) {
        state->item = malloc(sizeof(item_t*)*(size_t)items);
        for (i=0; i < items; i++) {
            state->item[i] = malloc(sizeof(item_t));
            state->item[i]->text = ff_strdup(item[i]);
            state->item[i]->data = data ? data[i] : NULL;
        }
    }
    state->selection = selection;
    update_prefsize(cb);
    ff_paint(cb);
    ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
}

int ff_choice_add(ff_window_t cb, const char* item, void* data)
{
    state_t* state = ff_get(cb, "-state");
    state->item = realloc(state->item, sizeof(item_t*)*(size_t)(state->items + 1));
    state->item[state->items] = malloc(sizeof(item_t));
    state->item[state->items]->text = ff_strdup(item);
    state->item[state->items]->data = data;
    state->items++;
    update_prefsize(cb);
    ff_paint(cb);
    return state->items - 1;
}

void ff_choice_remove(ff_window_t cb, int index)
{
    state_t* state = ff_get(cb, "-state");
    int i;
    if (index < 0 || index >= state->items) return;
    free(state->item[index]->text);
    free(state->item[index]);
    state->items--;
    for (i=index; i < state->items; i++)
        state->item[i] = state->item[i + 1];
    update_prefsize(cb);
    ff_paint(cb);
    if (state->selection >= state->items) state->selection = state->items - 1;
    ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
}

int ff_choice_items(ff_window_t cb)
{
    state_t* state = ff_get(cb, "-state");
    return state->items;
}

int ff_choice_itemindex(ff_window_t cb, const char* item)
{
    state_t* state = ff_get(cb, "-state");
    int i;
    for (i=0; i < state->items; i++)
        if (!strcmp(state->item[i]->text, item))
            return i;
    return -1;
}

int ff_choice_dataindex(ff_window_t cb, void* data)
{
    state_t* state = ff_get(cb, "-state");
    int i;
    for (i=0; i < state->items; i++)
        if (state->item[i]->data == data)
            return i;
    return -1;
}

const char* ff_choice_text(ff_window_t cb, int index)
{
    state_t* state = ff_get(cb, "-state");
    if (index >= 0 && index < state->items)
        return state->item[index]->text;
    return NULL;
}

void* ff_choice_data(ff_window_t cb, int index)
{
    state_t* state = ff_get(cb, "-state");
    if (index >= 0 && index < state->items)
        return state->item[index]->data;
    return NULL;
}

void ff_choice_replace(ff_window_t cb, int index, const char* text, void* data)
{
    state_t* state = ff_get(cb, "-state");
    if (index >= 0 && index < state->items) {
        free(state->item[index]->text);
        state->item[index]->text = ff_strdup(text);
        state->item[index]->data = data;
        if (index == state->selection)
            ff_post(cb, FF_SELECT, state->selection, 0, 0, NULL, FF_BUBBLE);
    }
}
