unit RBVM;
{$MODE OBJFPC}{$H+}
interface
uses
  SysUtils, RBOwn, RBCode, RBProg, RBError, RBVMStor, RBNative, RBUtil;

type
  TVirtualMachine = class;

  ERuntimeError = class(Exception)
  public
    Prog: TProgram;
    Code: TCode;
    Method: TMethod;
    Addr: Integer;
    constructor New(AProgram: TProgram; AMethod: TMethod; ACode: TCode; AAddr: Integer; AMessage: string);
  end;

  { TVirtualMachineIO }

  TVirtualMachineIO = class
  public
    procedure OutputString(AString: string); virtual; abstract;
    procedure OutputTab; virtual; abstract;
    procedure OutputNewLine; virtual; abstract;
  end;

  { TVirtualMachineStandardIO }

  TVirtualMachineStandardIO = class(TVirtualMachineIO)
  public
    procedure OutputString(AString: string); override;
    procedure OutputTab; override;
    procedure OutputNewLine; override;
  end;

  { TVMStack }

  TVMStack = class
  private
    Owner: TVirtualMachine;
    Items: array of TVMValue;
    Head: Integer;
  public
    procedure Clear;
    procedure Push(const AValue: TVMValue);
    procedure Pop(out AValue: TVMValue);
    function Empty: Boolean; inline;
    procedure PushInteger(AValue: Int64);
    procedure PushReal(AValue: Double);
    procedure PushString(AValue: string);
    procedure PushObject(AValue: INativeObject);
    procedure Drop;
    procedure Duplicate;
    procedure Swap;
  end;

  { TVMDisposal }
  TVMDisposal = record
    Arr: PVMArray;
    ICnt: Integer;
  end;

  { TVirtualMachine }

  TVirtualMachine = class(TOwnable)
  private
    FIO: TVirtualMachineIO;
    FProg: TProgram;
    Method: TMethod;
    Code: TCode;
    Running: Boolean;
    Stack, Args: TVMStack;
    Global, Local: TVMStorage;
    JumpStack: array [0..1023] of Integer;
    JSHead: Integer;
    Disposals: array of TVMDisposal;
    DispCount: Integer;
    procedure SetProg(AValue: TProgram);
    procedure RunOpCode(OpCode: TOpCode);
    procedure RunCode(ACode: TCode; AMethod: TMethod=nil);
    procedure ProcessDisposals;
    procedure CleanDisposals;
  public
    constructor Create(AOwner: TOwnable); override;
    destructor Destroy; override;
    procedure Reset;
    procedure RunMethod(AMethod: TMethod);
    procedure Run(AProgram: TProgram);
  public
    procedure DisposeArray(Arr: PVMArray);
    procedure RuntimeError(AMessage: string);
    property IO: TVirtualMachineIO read FIO write FIO;
    property GlobalStorage: TVMStorage read Global;
  end;

var
  VirtualMachineStandardIO: TVirtualMachineStandardIO;

function NativeVM(ANativeInterface: TNativeInterface): TVirtualMachine; inline;

implementation

function NativeVM(ANativeInterface: TNativeInterface): TVirtualMachine; inline;
begin
  Result:=TVirtualMachine(ANativeInterface.VMP);
end;

procedure ToBool(var V: TVMValue; B: Boolean); inline;
begin
  V.ValueType:=vtInteger;
  if B then V.IntValue:=-1 else V.IntValue:=0;
end;

function DoPower(A, B: Double): Double; inline;
begin
  Result:=0;
  if Abs(A) < 1e-15 then A:=0;
  if Abs(B) < 1e-15 then B:=0;
  if B=0.0 then Result:=1 else
  if A=0.0 then Result:=0 else
  if (A < 0) and (B < 0) then Result:=1.0/Exp(-B*Ln(-A)) else
  if (A < 0) and (B >= 0) then Result:=Exp(B*Ln(-A)) else
  if (A > 0) and (B < 0) then Result:=1.0/Exp(-B*Ln(A)) else
  if (A > 0) and (B > 0) then Result:=Exp(B*Ln(A));
  if (A < 0) and (Frac(B/2.0) <> 0.0) then Result:=-Result;
end;

{ TVirtualMachineStandardIO }

procedure TVirtualMachineStandardIO.OutputString(AString: string);
begin
  if IsConsole then Write(AString);
end;

procedure TVirtualMachineStandardIO.OutputTab;
begin
  if IsConsole then Write(#9);
end;

procedure TVirtualMachineStandardIO.OutputNewLine;
begin
  if IsConsole then Writeln;
end;

{ ERuntimeError }
constructor ERuntimeError.New(AProgram: TProgram; AMethod: TMethod; ACode: TCode; AAddr: Integer; AMessage: string);
begin
  Prog:= AProgram;
  Code:=ACode;
  Method:=AMethod;
  Addr:=AAddr;
  inherited Create(AMessage);
end;

{ TVMStack }

procedure TVMStack.Clear;
begin
  Head:=0;
end;

procedure TVMStack.Push(const AValue: TVMValue);
begin
  if Head=Length(Items) then SetLength(Items, Head + 64);
  Items[Head]:=AValue;
  Inc(Head);
end;

procedure TVMStack.Pop(out AValue: TVMValue);
begin
  if Head=0 then Owner.RuntimeError(emVMStackUnderflow);
  Dec(Head);
  AValue:=Items[Head];
end;

function TVMStack.Empty: Boolean;
begin
  Result:=Head=0;
end;

procedure TVMStack.PushInteger(AValue: Int64);
var
  Item: TVMValue;
begin
  Item.ValueType:=vtInteger;
  Item.IntValue:=AValue;
  Push(Item);
end;

procedure TVMStack.PushReal(AValue: Double);
var
  Item: TVMValue;
begin
  Item.ValueType:=vtReal;
  Item.RealValue:=AValue;
  Push(Item);
end;

procedure TVMStack.PushString(AValue: string);
var
  Item: TVMValue;
begin
  Item.ValueType:=vtString;
  Item.StrValue:=AValue;
  Push(Item);
end;

procedure TVMStack.PushObject(AValue: INativeObject);
var
  Item: TVMValue;
begin
  Item.ValueType:=vtObject;
  Item.ObjValue:=AValue;
  Push(Item);
end;

procedure TVMStack.Drop;
begin
  if Head=0 then Owner.RuntimeError(emVMStackUnderflow);
  Dec(Head);
end;

procedure TVMStack.Duplicate;
var
  Item: TVMValue;
begin
  Pop(Item);
  Push(Item);
  Push(Item);
end;

procedure TVMStack.Swap;
var
  A, B: TVMValue;
begin
  if Head < 2 then Owner.RuntimeError(emVMStackUnderflow);
  Pop(A);
  Pop(B);
  Push(A);
  Push(B);
end;

{ TVirtualMachine }
procedure TVirtualMachine.SetProg(AValue: TProgram);
begin
  if FProg=AValue then Exit;
  FProg:=AValue;
  Reset;
end;

var
  A, B, C: TVMValue;

    function ToDouble(VM: TVirtualMachine; const V: TVMValue): Double;
    begin
      case V.ValueType of
        vtInteger: Result:=V.IntValue;
        vtReal: Result:=V.RealValue;
        else VM.RuntimeError(emArithmeticValueExpected);
      end;
    end;


procedure TVirtualMachine.RunOpCode(OpCode: TOpCode);
var
  I, J: Integer;

  function ForDoneCheck(const A, B, C: TVMValue): Boolean;
  var
    Enum, Step, Target: Double;
  begin
    Enum:=ToDouble(Self, A);
    Step:=ToDouble(Self, B);
    Target:=ToDouble(Self, C);
    Result:=((Step > 0) and (Enum > Target)) or ((Step < 0) and (Enum < Target));
  end;

  procedure GetArrayNumbers(out Dimensions, Slot: Integer; out DimSizes: TVMArraySize);
  var
    I: Integer;
  begin
    Dimensions:=Code.GetByte;
    Slot:=Integer(Code.GetCardinal);
    for I:=Dimensions - 1 downto 0 do begin
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: DimSizes[I]:=A.IntValue;
        vtReal: begin
          if Round(A.RealValue) <> A.RealValue then RuntimeError(emIntegerValueExpected);
          DimSizes[I]:=Round(A.RealValue);
        end;
        else RuntimeError(emIntegerValueExpected);
      end;
    end;
  end;

  procedure DoDim(Keep, Global: Boolean);
  var
    Storage: TVMStorage;
    I, Dimensions, Slot: Integer;
    DimSizes: TVMArraySize;
  begin
    if Global then Storage:=Self.Global else Storage:=Local;
    GetArrayNumbers(Dimensions, Slot, DimSizes);
    for I:=0 to Dimensions - 1 do
      if DimSizes[I] <= 0 then RuntimeError(emDimSizeMustBeGreaterThanZero);
    CreateArray(Self, DimSizes, Dimensions, A);
    Storage.Write(Slot, A);
  end;

  procedure DoDimWrite(Global: Boolean);
  var
    Dimensions, Slot: Integer;
    Indices: TVMArraySize;
  begin
    Stack.Pop(B);
    GetArrayNumbers(Dimensions, Slot, Indices);
    if Global then Self.Global.Read(Slot, A) else Local.Read(Slot, A);
    if A.ValueType <> vtArray then RuntimeError(emValueIsNotAnArray);
    if Dimensions <> A.ArrValue^.Dims then RuntimeError(emInvalidArraySubscriptCount);
    A.ArrValue^.Write(Indices, B);
  end;

  procedure DoDimRead(Global: Boolean);
  var
    Dimensions, Slot: Integer;
    Indices: TVMArraySize;
  begin
    GetArrayNumbers(Dimensions, Slot, Indices);
    if Global then Self.Global.Read(Slot, A) else Local.Read(Slot, A);
    if A.ValueType <> vtArray then RuntimeError(emValueIsNotAnArray);
    if Dimensions <> A.ArrValue^.Dims then RuntimeError(emInvalidArraySubscriptCount);
    A.ArrValue^.Read(Indices, B);
    Stack.Push(B);
  end;

  procedure DoNativeCall;
  var
    Native: TNativeMethodInfo;
    Args: TNativeArguments;
  begin
    J:=Integer(Code.GetCardinal);
    I:=Integer(Code.GetCardinal);
    Native:=FProg.NativeInterface.Methods[I];
    SetLength(Args, J);
    while J > 0 do begin
      Self.Args.Pop(Args[J - 1]);
      Dec(J);
    end;
    try
      Native.Method(Args, A);
    except
      on ENativeError do RuntimeError(ENativeError(ExceptObject).Message);
    end;
    Stack.Push(A);
  end;

  procedure DoObjectCall;
  var
    MetName: Integer;
    Obj: TVMValue;
    Args: TNativeArguments;
  begin
    Stack.Pop(Obj);
    MetName:=Integer(Code.GetCardinal);
    J:=Integer(Code.GetCardinal);
    if Obj.ValueType <> vtObject then RuntimeError(emValueIsNotAnObject);
    SetLength(Args, J);
    while J > 0 do begin
      Self.Args.Pop(Args[J - 1]);
      Dec(J);
    end;
    SetInt(A, 0);
    try
      INativeObject(Obj.ObjValue).MethodCall(MetName, Args, A);
    except
      on ENativeError do RuntimeError(ENativeError(ExceptObject).Message);
    end;
    Stack.Push(A);
  end;

  procedure DoPropertyWrite;
  var
    PropertyName: Integer;
    Obj, Value: TVMValue;
  begin
    Stack.Pop(Value);
    Stack.Pop(Obj);
    PropertyName:=Integer(Code.GetCardinal);
    if Obj.ValueType <> vtObject then RuntimeError(emValueIsNotAnObject);
    try
      INativeObject(Obj.ObjValue).SetProperty(PropertyName, Value);
    except
      on ENativeError do RuntimeError(ENativeError(ExceptObject).Message);
    end;
  end;

begin
  //Writeln ('OpCode: ' , OpCode);
  case OpCode of
    opNop: Exit;
    opHello: begin
      IO.OutputString('HELLO Opcode');
      IO.OutputNewLine;
    end;
    opPushZero: Stack.PushInteger(0);
    opPushInt: Stack.PushInteger(Code.GetInteger);
    opPushReal: Stack.PushReal(Code.GetReal);
    opPushStr: Stack.PushString(Code.GetString);
    opDrop: Stack.Drop;
    opNeg: begin
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: A.IntValue:=-A.IntValue;
        vtReal: A.RealValue:=-A.RealValue;
        else RuntimeError(emArithmeticValueExpected);
      end;
      Stack.Push(A);
    end;
    opNot: begin
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: A.IntValue:=not A.IntValue;
        vtReal: begin A.ValueType:=vtInteger; A.IntValue:=not Round(A.RealValue); end;
        else RuntimeError(emArithmeticValueExpected);
      end;
      Stack.Push(A);
    end;
    opAnd: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue and B.IntValue;
          vtReal: A.IntValue:=A.IntValue and Round(B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: begin
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) and B.IntValue;
          end;
          vtReal: begin
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) and Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opOr: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue or B.IntValue;
          vtReal: A.IntValue:=A.IntValue or Round(B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: begin
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) or B.IntValue;
          end;
          vtReal: begin
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) or Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opXOr: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue xor B.IntValue;
          vtReal: A.IntValue:=A.IntValue xor Round(B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: begin
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) xor B.IntValue;
          end;
          vtReal: begin
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) xor Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opEq: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: ToBool(A, A.IntValue = B.IntValue);
          vtReal: ToBool(A, A.IntValue = B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: ToBool(A, A.RealValue = B.IntValue);
          vtReal: ToBool(A, A.RealValue = B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtString: ToBool(A, A.StrValue = B.StrValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opNEq: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: ToBool(A, A.IntValue <> B.IntValue);
          vtReal: ToBool(A, A.IntValue <> B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: ToBool(A, A.RealValue <> B.IntValue);
          vtReal: ToBool(A, A.RealValue <> B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtString: ToBool(A, A.StrValue <> B.StrValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opLess: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: ToBool(A, A.IntValue < B.IntValue);
          vtReal: ToBool(A, A.IntValue < B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: ToBool(A, A.RealValue < B.IntValue);
          vtReal: ToBool(A, A.RealValue < B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtString: ToBool(A, A.StrValue < B.StrValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opGreat: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: ToBool(A, A.IntValue > B.IntValue);
          vtReal: ToBool(A, A.IntValue > B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: ToBool(A, A.RealValue > B.IntValue);
          vtReal: ToBool(A, A.RealValue > B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtString: ToBool(A, A.StrValue > B.StrValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opLEqual: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: ToBool(A, A.IntValue <= B.IntValue);
          vtReal: ToBool(A, A.IntValue <= B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: ToBool(A, A.RealValue <= B.IntValue);
          vtReal: ToBool(A, A.RealValue <= B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtString: ToBool(A, A.StrValue <= B.StrValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opGEqual: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: ToBool(A, A.IntValue >= B.IntValue);
          vtReal: ToBool(A, A.IntValue >= B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: ToBool(A, A.RealValue >= B.IntValue);
          vtReal: ToBool(A, A.RealValue >= B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtString: ToBool(A, A.StrValue >= B.StrValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opAdd: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue + B.IntValue;
          vtReal: begin A.ValueType:=vtReal; A.RealValue:=A.IntValue + B.RealValue; end;
          vtString: begin A.ValueType:=vtString; A.StrValue:=IntToStr(A.IntValue) + B.StrValue; end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: A.RealValue:=A.RealValue + B.IntValue;
          vtReal: A.RealValue:=A.RealValue + B.RealValue;
          vtString: begin A.ValueType:=vtString; A.StrValue:=FloatStr(A.RealValue) + B.StrValue; end;
          else RuntimeError(emInvalidType);
        end;
        vtString: case B.ValueType of
          vtInteger: A.StrValue:=A.StrValue + IntToStr(B.IntValue);
          vtReal: A.StrValue:=A.StrValue + FloatStr(B.RealValue);
          vtString: A.StrValue:=A.StrValue + B.StrValue;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opSub: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue - B.IntValue;
          vtReal: begin A.ValueType:=vtReal; A.RealValue:=A.IntValue - B.RealValue; end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: A.RealValue:=A.RealValue - B.IntValue;
          vtReal: A.RealValue:=A.RealValue - B.RealValue;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opDiv: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: begin
            if B.IntValue=0 then RuntimeError(emDivisionByZero);
            A.IntValue:=A.IntValue div B.IntValue;
          end;
          vtReal: begin
            if B.RealValue=0.0 then RuntimeError(emDivisionByZero);
            A.ValueType:=vtReal;
            A.RealValue:=A.IntValue / B.RealValue;
          end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: begin
            if B.IntValue=0 then RuntimeError(emDivisionByZero);
            A.RealValue:=A.RealValue / B.IntValue;
          end;
          vtReal: begin
            if B.RealValue=0.0 then RuntimeError(emDivisionByZero);
            A.RealValue:=A.RealValue / B.RealValue;
          end;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opIntDiv: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: begin
            if B.IntValue=0 then RuntimeError(emDivisionByZero);
            A.IntValue:=A.IntValue div B.IntValue;
          end;
          vtReal: begin
            if Round(B.RealValue)=0 then RuntimeError(emDivisionByZero);
            A.IntValue:=A.IntValue div Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: begin
            if B.IntValue=0 then RuntimeError(emDivisionByZero);
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) div B.IntValue;
          end;
          vtReal: begin
            if Round(B.RealValue)=0 then RuntimeError(emDivisionByZero);
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) div Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opMod: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: begin
            if B.IntValue=0 then RuntimeError(emDivisionByZero);
            A.IntValue:=A.IntValue mod B.IntValue;
          end;
          vtReal: begin
            if Round(B.RealValue)=0 then RuntimeError(emDivisionByZero);
            A.IntValue:=A.IntValue mod Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: begin
            if B.IntValue=0 then RuntimeError(emDivisionByZero);
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) mod B.IntValue;
          end;
          vtReal: begin
            if Round(B.RealValue)=0 then RuntimeError(emDivisionByZero);
            A.ValueType:=vtInteger;
            A.IntValue:=Round(A.RealValue) mod Round(B.RealValue);
          end;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opMul: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue * B.IntValue;
          vtReal: begin A.ValueType:=vtReal; A.RealValue:=A.IntValue * B.RealValue; end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: A.RealValue:=A.RealValue * B.IntValue;
          vtReal: A.RealValue:=A.RealValue * B.RealValue;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opPow: begin
      Stack.Pop(B);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: begin
            if B.IntValue < 0 then begin
              A.ValueType:=vtReal;
              A.RealValue:=DoPower(A.IntValue, B.IntValue);
            end else A.IntValue:=Round(DoPower(A.IntValue, B.IntValue));
          end;
          vtReal: begin A.ValueType:=vtReal; A.RealValue:=DoPower(A.IntValue, B.RealValue); end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: A.RealValue:=DoPower(A.RealValue, B.IntValue);
          vtReal: A.RealValue:=DoPower(A.RealValue, B.RealValue);
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      Stack.Push(A);
    end;
    opJump: Code.Address:=Integer(Code.GetCardinal);
    opCall: begin
      if JSHead=Length(JumpStack) then RuntimeError(emJumpstackOverflow);
      I:=Integer(Code.GetCardinal);
      JumpStack[JSHead]:=Code.Address;
      Code.Address:=I;
      Inc(JSHead);
    end;
    opRet: begin
      if JSHead=0 then RuntimeError(emJumpstackUnderflow);
      Code.Address:=JumpStack[JSHead];
      Dec(JSHead);
    end;
    opIfNot: begin
      I:=Integer(Code.GetCardinal);
      Stack.Pop(A);
      case A.ValueType of
        vtInteger: if A.IntValue=0 then Code.Address:=I;
        vtReal: if A.RealValue=0.0 then Code.Address:=I;
        else RuntimeError(emInvalidType);
      end;
    end;
    opLFor, opGFor: begin
      Stack.Pop(C);
      Stack.Pop(B);
      Stack.Pop(A);
      I:=Integer(Code.GetCardinal);
      J:=Integer(Code.GetCardinal);
      case A.ValueType of
        vtInteger: case B.ValueType of
          vtInteger: A.IntValue:=A.IntValue + B.IntValue;
          vtReal: begin A.ValueType:=vtReal; A.RealValue:=A.IntValue + B.RealValue; end;
          else RuntimeError(emInvalidType);
        end;
        vtReal: case B.ValueType of
          vtInteger: A.RealValue:=A.RealValue + B.IntValue;
          vtReal: A.RealValue:=A.RealValue + B.RealValue;
          else RuntimeError(emInvalidType);
        end;
        else RuntimeError(emInvalidType);
      end;
      if OpCode=opLFor then Local.Write(J, A) else Global.Write(J, A);
      if not ForDoneCheck(A, B, C) then Code.Address:=I;
    end;
    opEnd: Running:=False;
    opMetCall: begin
      I:=Integer(Code.GetCardinal);
      if (I < 0) or (I >= FProg.MethodCount) then RuntimeError(emInvalidMethodID);
      RunMethod(FProg.Methods[I]);
    end;
    opNatCall: DoNativeCall;
    opObjCall: DoObjectCall;
    opPushArg: begin
      Stack.Pop(A);
      Args.Push(A);
    end;
    opPopArg: begin
      Args.Pop(A);
      Stack.Push(A);
    end;
    opLRead: begin
      Local.Read(Integer(Code.GetCardinal), A);
      Stack.Push(A);
    end;
    opGRead: begin
      Global.Read(Integer(Code.GetCardinal), A);
      Stack.Push(A);
    end;
    opLWrite: begin
      Stack.Pop(A);
      Local.Write(Integer(Code.GetCardinal), A);
    end;
    opGWrite: begin
      Stack.Pop(A);
      Global.Write(Integer(Code.GetCardinal), A);
    end;
    opDimLWrite, opDimGWrite: DoDimWrite(OpCode=opDimGWrite);
    opDimLRead, opDimGRead: DoDimRead(OpCode=opDimGRead);
    opLDim: DoDim(False, False);
    opGDim: DoDim(False, True);
    opLReDim: DoDim(True, False);
    opGReDim: DoDim(True, True);
    opPWrite: DoPropertyWrite;
    opPrint: begin
      Stack.Pop(A);
      IO.OutputString(ValueToString(A));
    end;
    opTab: IO.OutputTab;
    opNewLine: IO.OutputNewLine;
    else RuntimeError(emFmt(emInvalidOpCode, IntToStr(Ord(OpCode))));
  end;
end;

procedure TVirtualMachine.RunCode(ACode: TCode; AMethod: TMethod);
var
  SaveRunning: Boolean;
  SaveCode: TCode;
  SaveMethod: TMethod;
  SaveAddress: Integer;
begin
  SaveRunning:=Running;
  SaveCode:=Code;
  SaveMethod:=Method;

  Code:=ACode;
  Method:=AMethod;

  Running:=True;

  SaveAddress:=Code.Address;
  Code.Address:=0;
  while Running and Code.HasMore do begin
    RunOpCode(Code.GetOpCode);
    ProcessDisposals;
  end;
  Code.Address:=SaveAddress;

  Running:=SaveRunning;
  Code:=SaveCode;
  Method:=SaveMethod;
end;

procedure TVirtualMachine.ProcessDisposals;
var
  I, J: Integer;
begin
  I:=0;
  while I < DispCount do begin
    Dec(Disposals[I].ICnt);
    if (Disposals[I].Arr^.Refs > 0) or (Disposals[I].ICnt < 0) then begin
      if Disposals[I].ICnt < 0 then Dispose(Disposals[I].Arr, Done);
      for J:=I to DispCount - 2 do Disposals[J]:=Disposals[J + 1];
      Dec(DispCount);
      Continue;
    end;
    Inc(I);
  end;
end;

procedure TVirtualMachine.CleanDisposals;
var
  I: Integer;
begin
  for I:=0 to DispCount - 1 do Dispose(Disposals[I].Arr, Done);
  DispCount:=0;
  SetLength(Disposals, 0);
end;

constructor TVirtualMachine.Create(AOwner: TOwnable);
begin
  inherited Create(AOwner);
  Stack:=TVMStack.Create;
  Stack.Owner:=Self;
  Args:=TVMStack.Create;
  Args.Owner:=Self;
  Global:=TVMStorage.Create(Self);
  FIO:=VirtualMachineStandardIO;
  Reset;
end;

destructor TVirtualMachine.Destroy;
begin
  CleanDisposals;
  FreeAndNil(Args);
  FreeAndNil(Stack);
  inherited Destroy;
end;

procedure TVirtualMachine.Reset;
begin
  Method:=nil;
  Code:=nil;
  Stack.Clear;
  Args.Clear;
  Global.Clear;
  Local:=nil;
  JSHead:=0;
end;

procedure TVirtualMachine.RunMethod(AMethod: TMethod);
var
  I: Integer;
  V: TVMValue;
  Store: TVMStorage = nil;
  SaveLocal: TVMStorage;
  SaveActive: Boolean;
begin
  if not Assigned(FProg) then RuntimeError(emNoProgramForMethod);
  if not Assigned(AMethod.Local) then AMethod.Local:=TVMStorage.Create(Self);
  SaveLocal:=Local;
  Local:=AMethod.Local;
  if Local.Active then begin
    Store:=TVMStorage.Create(Self);
    Store.Copy(Local);
  end;
  Local.Clear;
  SaveActive:=Local.Active;
  Local.Active:=True;
  for I:=AMethod.ArgCount - 1 downto 0 do begin
    Args.Pop(V);
    Local.Write(I, V);
  end;
  RunCode(AMethod.Code, AMethod);
  Local.Active:=SaveActive;
  Local.Clear;
  if Assigned(Store) then begin
    Local.Copy(Store);
    FreeAndNil(Store);
  end;
  Local:=SaveLocal;
end;

procedure TVirtualMachine.Run(AProgram: TProgram);
begin
  FProg:=AProgram;
  FProg.NativeInterface.VMP:=Self;
  RunCode(FProg.Code, nil);
  CleanDisposals;
end;

procedure TVirtualMachine.DisposeArray(Arr: PVMArray);
var
  I: Integer;
begin
  for I:=0 to High(Disposals) do if Disposals[I].Arr=Arr then Exit;
  if DispCount >= Length(Disposals) then SetLength(Disposals, Length(Disposals) + 256);
  Disposals[DispCount].Arr:=Arr;
  Disposals[DispCount].ICnt:=100;
  Inc(DispCount);
end;

procedure TVirtualMachine.RuntimeError(AMessage: string);
begin
  raise ERuntimeError.New(FProg, Method, Code, Code.Address, AMessage);
end;

initialization
  VirtualMachineStandardIO:=TVirtualMachineStandardIO.Create;
finalization
  FreeAndNil(VirtualMachineStandardIO);
end.

