unit RBNative;
{$MODE OBJFPC}{$H+}
interface

uses
  SysUtils, RBOwn, RBVMStor;

type
  ENativeError = class(Exception);
  TNativeArguments = array of TVMValue;
  TNativeMethod = procedure(Args: TNativeArguments; var Result: TVMValue) of object;
  TNativeMethodType = (mtSub, mtFunction);

  { TNativeMethodInfo }

  TNativeMethodInfo = class(TOwnable)
  public
    Index: Integer;
    MetName: string;
    FmtName: string;
    Method: TNativeMethod;
    MethodType: TNativeMethodType;
  end;

  { TNativeInterface }

  TNativeInterface = class(TOwnable)
  private
    FMethods: array of TNativeMethodInfo;
    function GetMethods(AIndex: Integer): TNativeMethodInfo; inline;
    function GetMethodCount: Integer; inline;
  public
    VMP: Pointer;
    procedure Register(AName: string; AType: TNativeMethodType; ANativeMethod: TNativeMethod);
    function Find(AName: string; out Info: TNativeMethodInfo): Boolean;
    procedure RegisterAll(Source: TNativeInterface);
    procedure Assign(Source: TObject);
    property Methods[AIndex: Integer]: TNativeMethodInfo read GetMethods;
    property MethodCount: Integer read GetMethodCount;
  end;

procedure RuntimeError(AMessage: string); inline;
procedure ArgumentCheck(const Args: TNativeArguments; Min, Max: Integer; MetName: string); inline;

implementation

uses
  RBError;

procedure RuntimeError(AMessage: string);
begin
  raise ENativeError.Create(AMessage);
end;

procedure ArgumentCheck(const Args: TNativeArguments; Min, Max: Integer;
  MetName: string);
begin
  if (Length(Args) < Min) or (Length(Args) > Max) then RuntimeError(emFmt(emInvalidArgumentCountToX, MetName));
end;

{ TNativeInterface }
function TNativeInterface.GetMethods(AIndex: Integer): TNativeMethodInfo; inline;
begin
  Result:=FMethods[AIndex];
end;

function TNativeInterface.GetMethodCount: Integer;
begin
  Result:=Length(FMethods);
end;

procedure TNativeInterface.Register(AName: string; AType: TNativeMethodType; ANativeMethod: TNativeMethod);
begin
  SetLength(FMethods, Length(FMethods) + 1);
  FMethods[High(FMethods)]:=TNativeMethodInfo.Create(Self);
  with FMethods[High(FMethods)] do begin
    Index:=High(FMethods);
    MetName:=UpperCase(AName);
    FmtName:=AName;
    Method:=ANativeMethod;
    MethodType:=AType;
  end;
end;

function TNativeInterface.Find(AName: string; out Info: TNativeMethodInfo): Boolean;
var
  I: Integer;
begin
  for I:=0 to High(FMethods) do
    if FMethods[I].MetName=AName then begin
      Info:=FMethods[I];
      Exit(True);
    end;
  Info:=nil;
  Result:=False;
end;

procedure TNativeInterface.RegisterAll(Source: TNativeInterface);
var
  I: Integer;
begin
  for I:=0 to High(Source.FMethods) do
    Register(Source.FMethods[I].MetName, Source.FMethods[I].MethodType, Source.FMethods[I].Method);
end;

procedure TNativeInterface.Assign(Source: TObject);
var
  Src: TNativeInterface;
  I: Integer;
begin
  if not (Source is TNativeInterface) then raise ENativeError.Create('TNativeInterface cannot be assigned to ' + Source.ClassName);
  Src:=TNativeInterface(Source);
  for I:=0 to High(FMethods) do FMethods[I].Free;
  SetLength(FMethods, 0);
  RegisterAll(Src);
end;

end.
