unit RBUI;
{$MODE OBJFPC}{$H+}
interface

uses
  Classes, SysUtils, Math,
  RBOwn, RBCpl, RBProg, RBVM, RBScan, RBCode, RBSrc, RBFmt;

type
  TMessageType = (mtInfo, mtWarning, mtError);

  { TCodeLine }

  PCodeLine = ^TCodeLine;
  TCodeLine = object
    Code: string;
    procedure Reset;
  end;

  TCodeLines = array of TCodeLine;

  { TMethodCode }

  TMethodCode = class(TOwnable)
  public
    Name: string;
    Lines: TCodeLines;
    CursorX, CursorY: Integer;
    function ToSource: TSource;
  end;

  { TCodeEditor }

  TCodeEditor = class(TOwnable)
  private
    FLines: TCodeLines;
    FCursorX, FCursorY: Integer;
    FCurrentLineWidth: Integer;
    FModified: Boolean;
    SelAX, SelAY, SelBX, SelBY, SelMinY, SelMaxY: Integer;
    BeforeEdit: string;
    DisableChangingLine: Boolean;
    procedure SetCursorX(AValue: Integer);
    procedure SetCursorY(AValue: Integer);
    procedure SetModified(AValue: Boolean);
    function GetLines(AIndex: Integer): PCodeLine; inline;
    function GetLineCount: Integer; inline;
    procedure SetCode(AValue: string);
    function GetCode: string;
    procedure SaveToMethod(AMethod: TMethodCode);
    procedure ApplyMethod(AMethod: TMethodCode);
  protected
    procedure MarkModified; virtual;
    function ChangingLine: Boolean; virtual;
    procedure LineChanged; virtual;
  public
    procedure Reset;
    procedure Refresh; inline;
    procedure RefreshLine(Y: Integer); inline;
    procedure RefreshSelectionLines;
    procedure Insert(Text: string);
    procedure Delete;
    procedure Erase;
    procedure DeleteLine;
    procedure BreakLine;
    procedure MergeLine;
    procedure AppendNewLine(Text: string);
    procedure MoveLeft(ByWord: Boolean);
    procedure MoveRight(ByWord: Boolean);
    procedure MoveUp(Lines: Integer);
    procedure MoveDown(Lines: Integer);
    procedure MoveHome;
    procedure MoveEnd;
    procedure ClearSelection;
    procedure UpdateSelection(LastCursorX, LastCursorY: Integer);
    function HasSelection: Boolean; inline;
    function LineInSelection(Y: Integer): Boolean; inline;
    function WholeLineInSelection(Y: Integer): Boolean; inline;
    function CharInSelection(X, Y: Integer): Boolean; inline;
    function AtEndOfDocument: Boolean;
    procedure DeleteSelection;
    function GetSelectionText: string;
    procedure SetSelectionText(NewText: string);
    procedure IndentLine(Y: Integer);
    procedure ShiftIndent(Left: Boolean);
    property CurrentLineWidth: Integer read FCurrentLineWidth;
    property CursorX: Integer read FCursorX write SetCursorX;
    property CursorY: Integer read FCursorY write SetCursorY;
    property Modified: Boolean read FModified write SetModified;
    property Lines[AIndex: Integer]: PCodeLine read GetLines;
    property LineCount: Integer read GetLineCount;
    property Code: string read GetCode write SetCode;
  end;

  { TUserInterface }

  TUserInterface = class(TOwnable)
  protected
    FCodeEditor: TCodeEditor;
    FProg: TProgram;
    FMethods: array of TMethodCode;
    FCurrentMethod: TMethodCode;
    FFileName: string;

    procedure CommonInit;
    procedure CommonShutdown;
    function GetMethods(AIndex: Integer): TMethodCode; inline;
    function GetMethodCount: Integer; inline;
    procedure SetCurrentMethod(AMethod: TMethodCode);
    function AddNewMethod(AName: string): TMethodCode;
    procedure BreakProgramCodeToMethods(ACode: string);
    function MergeMethodsToProgramCode: string;
    procedure SetFileName(AFileName: string);
    procedure FileNameChanged; virtual; abstract;
    function ShowFileDialog(ForSave: Boolean; Description, Extension: string; var SelectedFile: string): Boolean; virtual; abstract;
  public
    function Initialize: Boolean; virtual; abstract;
    procedure Run; virtual; abstract;
    procedure Quit; virtual; abstract;
    procedure MessageBox(ACaption, AMessage: string; MessageType: TMessageType); virtual; abstract;
    function ConfirmBox(ACaption, AMessage: string; DefValue: Boolean): Boolean; virtual; abstract;
    procedure RefreshEditor; virtual; abstract;
    procedure RefreshEditorLine(Y: Integer); virtual; abstract;
    procedure NewProgram; virtual;
    procedure OpenProgram; virtual;
    procedure SaveProgram; virtual;
    procedure SaveProgramAs; virtual;
    procedure RunProgram; virtual;
    property CodeEditor: TCodeEditor read FCodeEditor;
    property Prog: TProgram read FProg;
    property Methods[AIndex: Integer]: TMethodCode read GetMethods;
    property MethodCount: Integer read GetMethodCount;
    property CurrentMethod: TMethodCode read FCurrentMethod write SetCurrentMethod;
    property FileName: string read FFileName write SetFileName;
  end;

var
  UserInterface: TUserInterface;

implementation

uses
  RBUtil, RBUTF8;

{ TCodeLine }
procedure TCodeLine.Reset;
begin
  Code:='';
end;

{ TMethodCode }

function TMethodCode.ToSource: TSource;
var
  Code: string;
  I: Integer;
begin
  Code:='';
  for I:=0 to High(Lines) do Code:=Code + Lines[I].Code + LineEnding;
  Result:=TSource.Create(Self);
  Result.SourceCode:=Code;
end;

{ TCodeEditor }
procedure TCodeEditor.SetCursorX(AValue: Integer);
begin
  if AValue=FCursorX then Exit;
  if AValue < 0 then AValue:=0;
  FCursorX:=AValue;
  RefreshLine(CursorY);
end;

procedure TCodeEditor.SetCursorY(AValue: Integer);
var
  PrevY: Integer;
begin
  if AValue=FCursorY then Exit;
  if FCursorY=-1 then FCursorY:=0;
  if not DisableChangingLine and not ChangingLine then Exit;
  if AValue < 0 then AValue:=0
  else if AValue > Length(FLines) then AValue:=Length(FLines);
  PrevY:=FCursorY;
  FCursorY:=AValue;
  LineChanged;
  RefreshLine(FCursorY);
  RefreshLine(PrevY);
end;

procedure TCodeEditor.SetModified(AValue: Boolean);
begin
  if AValue=FModified then Exit;
  FModified:=AValue;
  Refresh;
end;

procedure TCodeEditor.MarkModified;
begin
  Modified:=True;
end;

function TCodeEditor.GetLines(AIndex: Integer): PCodeLine; inline;
begin
  Result:=@FLines[AIndex];
end;

function TCodeEditor.GetLineCount: Integer; inline;
begin
  Result:=Length(FLines);
end;

procedure TCodeEditor.SetCode(AValue: string);
var
  I: Integer;
  Line: string;
begin
  Reset;
  DisableChangingLine:=True;
  Line:='';
  for I:=1 to Length(AValue) do begin
    if AValue[I]=#10 then begin
      AppendNewLine(Line);
      Line:='';
    end else if AValue[I]=#13 then begin
      if (I < Length(AValue)) and (AValue[I + 1] <> #10) then begin
        AppendNewLine(Line);
        Line:='';
      end;
    end else Line:=Line + AValue[I];
  end;
  if Line <> '' then AppendNewLine(Line);
  CursorY:=0;
  CursorX:=0;
  MarkModified;
  DisableChangingLine:=False;
end;

function TCodeEditor.GetCode: string;
var
  I: Integer;
begin
  Result:='';
  for I:=0 to High(FLines) do Result:=Result + FLines[I].Code + LineEnding;
end;

procedure TCodeEditor.SaveToMethod(AMethod: TMethodCode);
begin
  AMethod.Lines:=FLines;
  AMethod.CursorX:=CursorX;
  AMethod.CursorY:=CursorY;
end;

procedure TCodeEditor.ApplyMethod(AMethod: TMethodCode);
begin
  FLines:=AMethod.Lines;
  FCursorY:=AMethod.CursorY;
  FCursorX:=AMethod.CursorX;
  LineChanged;
end;

function TCodeEditor.ChangingLine: Boolean;
var
  Fmt: TFormatter;
  Previous, Formatted: string;

  function GetMethodName: string;
  begin
    if Copy(Formatted, 1, 4)='SUB ' then
      Result:=Copy(Trim(FLines[CursorY].Code), 5, Length(Formatted))
    else
      Result:=Copy(Trim(FLines[CursorY].Code), 10, Length(Formatted));
    if Pos('(', Result) <> 0 then Result:=Copy(Result, 1, Pos('(', Result) - 1);
    Result:=Trim(Result);
  end;

  procedure MethodLineModified;
  var
    NewName: string;
  begin
    NewName:=GetMethodName;
    if NewName='' then Exit;
    UserInterface.CurrentMethod.Name:=NewName;
  end;

  procedure NewMethodTyped;
  var
    NewName: string;
    NewMet: TMethodCode;
  begin
    NewName:=GetMethodName;
    if NewName='' then Exit;
    NewMet:=UserInterface.AddNewMethod(NewName);
    SetLength(NewMet.Lines, 3);
    NewMet.Lines[0].Code:=Trim(FLines[CursorY].Code);
    FLines[CursorY].Code:='';
    if Copy(Formatted, 1, 4)='SUB ' then
      NewMet.Lines[2].Code:='End Sub'
    else
      NewMet.Lines[2].Code:='End Function';
    DisableChangingLine:=True;
    UserInterface.CurrentMethod:=NewMet;
    CursorY:=1;
    CursorX:=2;
    Refresh;
    DisableChangingLine:=False;
  end;

begin
  if CursorY < Length(FLines) then begin
    Formatted:=TrimRight(FLines[CursorY].Code);
    try
      Fmt:=TFormatter.Create(Self);
      Fmt.Prog:=UserInterface.Prog;
      Formatted:=Fmt.FormatLine(Formatted);
    finally
      FreeAndNil(Fmt);
    end;
    Previous:=FLines[FCursorY].Code;
    if Previous <> Formatted then begin
      FLines[CursorY].Code:=Formatted;
      MarkModified;
    end;
    Formatted:=UpperCase(Trim(Formatted));
    Previous:=UpperCase(Trim(BeforeEdit));
    if Previous <> Formatted then begin
      if (Copy(Formatted, 1, 4)='SUB ') or
         (Copy(Formatted, 1, 9)='FUNCTION ') then begin
        if (Copy(Previous, 1, 4)='SUB ') or
           (Copy(Previous, 1, 9)='FUNCTION ') then begin
          MethodLineModified;
        end else begin
          NewMethodTyped;
          Exit(False);
        end;
      end;
    end;
  end;
  Result:=True;
end;

procedure TCodeEditor.LineChanged;
begin
  if CursorY < Length(FLines) then begin
    FCurrentLineWidth:=UTF8Length(FLines[CursorY].Code);
    BeforeEdit:=FLines[CursorY].Code;
  end else begin
    FCurrentLineWidth:=0;
    BeforeEdit:='';
  end;
end;

procedure TCodeEditor.Reset;
begin
  SetLength(FLines, 0);
  CursorY:=0;
  CursorX:=0;
  ClearSelection;
  Refresh;
  Modified:=False;
end;

procedure TCodeEditor.Refresh; inline;
begin
  UserInterface.RefreshEditor;
end;

procedure TCodeEditor.RefreshLine(Y: Integer); inline;
begin
  if (Y >= 0) and (Y <= Length(FLines)) then
    UserInterface.RefreshEditorLine(Y);
end;

procedure TCodeEditor.RefreshSelectionLines;
var
  Y: Integer;
begin
  if SelAY < SelBY then
    for Y:=SelAY to SelBY do RefreshLine(Y)
  else
    for Y:=SelBY to SelAY do RefreshLine(Y);
end;

procedure TCodeEditor.Insert(Text: string);
var
  I: Integer;
begin
  MarkModified;
  if HasSelection then DeleteSelection;
  if AtEndOfDocument then begin
    I:=CursorX;
    AppendNewLine('');
    CursorX:=I;
    CursorY:=CursorY - 1;
    Insert(Text);
    Exit;
  end;
  with FLines[CursorY] do begin
    for I:=UTF8Length(Code) to CursorX - 1 do Code:=Code + ' ';
    Code:=UTF8Copy(Code, 1, CursorX) + Text + UTF8Copy(Code, CursorX + 1, Length(Code));
    FCurrentLineWidth:=UTF8Length(Code);
  end;
  CursorX:=CursorX + UTF8Length(Text);
  RefreshLine(CursorY);
end;

procedure TCodeEditor.Delete;
begin
  if HasSelection then begin
    DeleteSelection;
    Exit;
  end;
  if AtEndOfDocument then Exit;
  if CursorX >= CurrentLineWidth then begin
    MergeLine;
    Exit;
  end;
  with FLines[CursorY] do begin
    Code:=UTF8Copy(Code, 1, CursorX) + UTF8Copy(Code, CursorX + 2, Length(Code));
    FCurrentLineWidth:=UTF8Length(Code);
  end;
  RefreshLine(CursorY);
  MarkModified;
end;

procedure TCodeEditor.Erase;
var
  PrevY: Integer;
begin
  if HasSelection then begin
    DeleteSelection;
    Exit;
  end;
  if CursorX=0 then begin
    PrevY:=CursorY;
    CursorY:=CursorY - 1;
    if PrevY=CursorY then Exit;
    CursorX:=FCurrentLineWidth;
    MergeLine;
    Exit;
  end;
  CursorX:=CursorX - 1;
  Delete;
end;

procedure TCodeEditor.BreakLine;
var
  I: Integer;
begin
  if HasSelection then DeleteSelection;
  if AtEndOfDocument then begin
    AppendNewLine('');
    Exit;
  end;
  SetLength(FLines, Length(FLines) + 1);
  for I:=High(FLines) downto CursorY + 1 do FLines[I]:=FLines[I - 1];
  with FLines[CursorY] do Code:=UTF8Copy(Code, 1, CursorX);
  with FLines[CursorY + 1] do Code:=UTF8Copy(Code, CursorX + 1, Length(Code));
  CursorX:=0;
  CursorY:=CursorY + 1;
  MarkModified;
  Refresh;
end;

procedure TCodeEditor.DeleteLine;
var
  I: Integer;
begin
  if AtEndOfDocument then Exit;
  for I:=CursorY to High(FLines) - 1 do FLines[I]:=FLines[I + 1];
  SetLength(FLines, Length(FLines) - 1);
  if AtEndOfDocument then FCurrentLineWidth:=0 else FCurrentLineWidth:=UTF8Length(FLines[CursorY].Code);
  Refresh;
  MarkModified;
end;

procedure TCodeEditor.MergeLine;
var
  I: Integer;
begin
  if CursorY >= High(FLines) then Exit;
  with FLines[CursorY] do begin
    for I:=Length(Code) to CursorX - 1 do Code:=Code + ' ';
    Code:=Code + FLines[CursorY + 1].Code;
  end;
  for I:=CursorY + 1 to High(FLines) - 1 do FLines[I]:=FLines[I + 1];
  SetLength(FLines, Length(FLines) - 1);
  FCurrentLineWidth:=UTF8Length(FLines[CursorY].Code);
  Refresh;
  MarkModified;
end;

procedure TCodeEditor.AppendNewLine(Text: string);
begin
  SetLength(FLines, Length(FLines) + 1);
  FLines[High(FLines)].Reset;
  FLines[High(FLines)].Code:=Text;
  CursorX:=0;
  CursorY:=Length(FLines);
  MarkModified;
end;

procedure TCodeEditor.MoveLeft(ByWord: Boolean);
var
  PrevY: Integer;
begin
  if (CursorX=0) and (CursorY=0) then Exit;
  PrevY:=CursorY;
  if CursorX=0 then begin
    CursorY:=CursorY - 1;
    if PrevY=CursorY then Exit;
    MoveEnd;
  end else begin
    CursorX:=CursorX - 1;
  end;
  if ByWord then begin
    if AtEndOfDocument then begin
      CursorX:=0;
      MoveLeft(True);
    end;
    if (CursorX >= UTF8Length(FLines[CursorY].Code)) or (FLines[CursorY].Code[CursorX + 1] in [#9, #32]) then begin
      while (not ((CursorX=0) and (CursorY=0))) and
            ((CursorX >= UTF8Length(FLines[CursorY].Code)) or
             (FLines[CursorY].Code[CursorX + 1] in [#9, #32])) do begin
        PrevY:=CursorY;
        MoveLeft(False);
        if PrevY <> CursorY then begin
          MoveLeft(False);
          Break;
        end;
      end;
    end else begin
      while (not ((CursorX=0) and (CursorY=0))) and
            (not ((CursorX >= UTF8Length(FLines[CursorY].Code)) or
                 (FLines[CursorY].Code[CursorX + 1] in [#9, #32]))) do begin
        PrevY:=CursorY;
        MoveLeft(False);
        if CursorX=0 then Exit;
        if PrevY <> CursorY then begin
          MoveLeft(False);
          Break;
        end;
      end;
    end;
    if not ((CursorX=0) and (CursorY=0)) then CursorX:=CursorX + 1;
  end;
end;

procedure TCodeEditor.MoveRight(ByWord: Boolean);
var
  PrevY: Integer;
begin
  CursorX:=CursorX + 1;
  if ByWord and not AtEndOfDocument then begin
    if (CursorX >= UTF8Length(FLines[CursorY].Code)) or (FLines[CursorY].Code[CursorX + 1] in [#9, #32]) then begin
      while not AtEndOfDocument and
            ((CursorX >= UTF8Length(FLines[CursorY].Code)) or
             (FLines[CursorY].Code[CursorX + 1] in [#9, #32])) do begin
        CursorX:=CursorX + 1;
        if CursorX >= UTF8Length(FLines[CursorY].Code) then begin
          CursorX:=0;
          PrevY:=CursorY;
          CursorY:=CursorY + 1;
          if PrevY=CursorY then Break;
          if AtEndOfDocument then begin
            CursorY:=CursorY - 1;
            MoveEnd;
            Break;
          end;
        end;
      end;
    end else begin
      if CursorX >= UTF8Length(FLines[CursorY].Code) then begin
        CursorX:=0;
        PrevY:=CursorY;
        CursorY:=CursorY + 1;
        if PrevY=CursorY then Exit;
        if AtEndOfDocument then begin
          CursorY:=CursorY - 1;
          MoveEnd;
          Exit;
        end;
      end;
      while not AtEndOfDocument and
            (not ((CursorX >= UTF8Length(FLines[CursorY].Code)) or
                 (FLines[CursorY].Code[CursorX + 1] in [#9, #32]))) do begin
        CursorX:=CursorX + 1;
      end;
    end;
  end;
end;

procedure TCodeEditor.MoveUp(Lines: Integer);
begin
  CursorY:=CursorY - Lines;
end;

procedure TCodeEditor.MoveDown(Lines: Integer);
begin
  MoveUp(-Lines);
end;

procedure TCodeEditor.MoveHome;
var
  FirstNonSpace: Integer;
begin
  if AtEndOfDocument then begin
    CursorX:=0;
    Exit;
  end;
  FirstNonSpace:=1;
  with FLines[CursorY] do begin
    while FirstNonSpace <= Length(Code) do begin
      if not (Code[FirstNonSpace] in [#9, #32]) then Break;
      Inc(FirstNonSpace);
    end;
  end;
  Dec(FirstNonSpace);
  if CursorX=0 then
    CursorX:=FirstNonSpace
  else
    if CursorX <= FirstNonSpace then
      CursorX:=0
    else
      CursorX:=FirstNonSpace;
end;

procedure TCodeEditor.MoveEnd;
begin
  if not AtEndOfDocument then CursorX:=UTF8Length(FLines[CursorY].Code) else CursorX:=0;
end;

procedure TCodeEditor.ClearSelection;
begin
  RefreshSelectionLines;
  SelAX:=0;
  SelAY:=0;
  SelBX:=0;
  SelBY:=0;
  SelMinY:=-1;
  SelMaxY:=-1;
end;

procedure TCodeEditor.UpdateSelection(LastCursorX, LastCursorY: Integer);
begin
  if HasSelection then begin
    if (SelBX <> CursorX) and (SelBY <> CursorY) then begin
      SelBX:=CursorX;
      if SelBY=CursorY then begin
        RefreshLine(CursorY);
      end else if (SelBY=CursorY - 1) or (SelBY=CursorY + 1) then begin
        RefreshLine(CursorY);
        RefreshLine(SelBY);
        SelBY:=CursorY;
      end else begin
        SelBY:=CursorY;
        RefreshSelectionLines;
      end;
    end else if SelBX <> CursorX then begin
      SelBX:=CursorX;
      RefreshLine(CursorY);
    end else if SelBY <> CursorY then begin
      if (SelBY=CursorY - 1) or (SelBY=CursorY + 1) then begin
        RefreshLine(CursorY);
        RefreshLine(SelBY);
        SelBY:=CursorY;
      end else begin
        SelBY:=CursorY;
        RefreshSelectionLines;
      end;
    end;
  end else begin
    SelAX:=LastCursorX;
    SelAY:=LastCursorY;
    SelBX:=CursorX;
    SelBY:=CursorY;
    RefreshSelectionLines;
  end;
  SelMinY:=Min(SelAY, SelBY);
  SelMaxY:=Max(SelAY, SelBY);
end;

function TCodeEditor.HasSelection: Boolean; inline;
begin
  Result:=(SelMinY <> -1) or (SelMaxY <> -1) or (SelAX <> 0) or (SelAY <> 0) or (SelBX <> 0) or (SelBY <> 0);
end;

function TCodeEditor.LineInSelection(Y: Integer): Boolean; inline;
begin
  Result:=HasSelection and (Y >= SelMinY) and (Y <= SelMaxY);
end;

function TCodeEditor.WholeLineInSelection(Y: Integer): Boolean; inline;
begin
  Result:=(Y > SelMinY) and (Y < SelMaxY);
end;

function TCodeEditor.CharInSelection(X, Y: Integer): Boolean; inline;
begin
  if LineInSelection(Y) then begin
    if (Y=SelMinY) and (Y=SelMaxY) then begin
      Result:=(X >= Min(SelAX, SelBX)) and (X < Max(SelAX, SelBX));
    end else if Y=SelMinY then begin
      if SelMinY=SelAY then
        Result:=X >= SelAX
      else
        Result:=X >= SelBX;
    end else if Y=SelMaxY then begin
      if SelMaxY=SelAY then
        Result:=X < SelAX
      else
        Result:=X < SelBX;
    end else
      Result:=True;
  end else
    Result:=False;
end;

function TCodeEditor.AtEndOfDocument: Boolean;
begin
  Result:=CursorY=Length(FLines);
end;

procedure TCodeEditor.DeleteSelection;
var
  I: Integer;
begin
  if not HasSelection then Exit;
  if (SelMaxY=SelMinY) and (SelMaxY >= Length(FLines)) then Exit;
  if SelMaxY - SelMinY > 1 then begin
    CursorY:=SelMinY + 1;
    for I:=SelMinY + 1 to SelMaxY - 1 do DeleteLine;
  end;
  if SelMaxY=SelMinY then with FLines[SelMinY] do begin
    Code:=UTF8Copy(Code, 1, Min(SelAX, SelBX)) + UTF8Copy(Code, Max(SelAX, SelBX) + 1, Length(Code));
  end else begin
    with FLines[SelMinY] do begin
      if SelMinY=SelAY then
        Code:=UTF8Copy(Code, 1, SelAX)
      else
        Code:=UTF8Copy(Code, 1, SelBX);
    end;
    if SelMinY < High(FLines) then
      with FLines[SelMinY + 1] do begin
        if SelMinY=SelAY then
          Code:=UTF8Copy(Code, SelBX + 1, Length(Code))
        else
          Code:=UTF8Copy(Code, SelAX + 1, Length(Code));
      end;
    CursorY:=SelMinY;
    MergeLine;
  end;
  CursorY:=SelMinY;
  if SelMinY=SelMaxY then
    CursorX:=Min(SelAX, SelBX)
  else
    if SelMinY=SelAY then CursorX:=SelAX else CursorX:=SelBX;
  ClearSelection;
  MarkModified;
  Refresh;
end;

function TCodeEditor.GetSelectionText: string;
var
  X, Y: Integer;
begin
  Result:='';
  if not HasSelection then Exit;
  for Y:=SelMinY to SelMaxY do if Y < Length(FLines) then with FLines[Y] do begin
    for X:=1 to UTF8Length(Code) do
      if CharInSelection(X, Y) then Result:=Result + UTF8Copy(Code, X, 1);
    if Y < SelMaxY then Result:=Result + LineEnding;
  end;
end;

procedure TCodeEditor.SetSelectionText(NewText: string);
var
  I: Integer;
begin
  DeleteSelection;
  for I:=1 to Length(NewText) do begin
    if NewText[I]=#10 then BreakLine
    else if NewText[I]=#13 then begin
      if (I < Length(NewText)) and (NewText[I + 1] <> #10) then BreakLine;
    end else Insert(NewText[I]);
  end;
end;

procedure TCodeEditor.IndentLine(Y: Integer);
var
  I, Spaces, NonEmpty: Integer;
begin
  if Y > High(FLines) then Exit;
  Spaces:=0;
  if Y > 1 then begin
    NonEmpty:=0;
    for I:=Y - 1 downto 1 do
      if TrimLeft(FLines[I].Code) <> '' then begin
        NonEmpty:=I;
        Break;
      end;
    if NonEmpty <> 0 then begin
      for I:=1 to Length(FLines[NonEmpty].Code) do
        if FLines[NonEmpty].Code[I]=' ' then Inc(Spaces) else Break;
    end;
  end;
  with FLines[Y] do begin
    Code:=TrimLeft(Code);
    for I:=1 to Spaces do Code:=' ' + Code;
  end;
  if Y=CursorY then CursorX:=CursorX + Spaces else RefreshLine(Y);
  MarkModified;
end;

procedure TCodeEditor.ShiftIndent(Left: Boolean);
var
  I, NewCursorX: Integer;
begin
  if HasSelection then begin
    for I:=SelMinY to SelMaxY do
      if I < Length(FLines) then with FLines[I] do begin
        if Left then begin
          if Code[1]=' ' then Code:=Copy(Code, 2, Length(Code) - 1);
          if Code[1]=' ' then Code:=Copy(Code, 2, Length(Code) - 1);
        end else Code:='  ' + Code;
        RefreshLine(I);
      end;
    if Left then I:=-2 else I:=2;
    SelAX:=SelAX + I;
    SelBX:=SelBX + I;
    CursorX:=CursorX + I;
  end else begin
    if Left then begin
      NewCursorX:=(CursorX div 2)*2 - 1;
      for I:=CursorX downto NewCursorX do begin
        if (I=0) or ((CursorY < Length(FLines)) and (CursorX <= Length(FLines[CursorY].Code)) and (FLines[CursorY].Code[CursorX] <> #32)) then Break;
        Erase;
      end;
    end else begin
      NewCursorX:=(CursorX div 2)*2 + 1;
      for I:=CursorX to NewCursorX do Insert(' ');
    end;
  end;
  MarkModified;
end;

{ TUserInterface }
procedure TUserInterface.CommonInit;
begin
  NewProgram;
end;

procedure TUserInterface.CommonShutdown;
begin
  FreeAndNil(FProg);
end;

function TUserInterface.GetMethods(AIndex: Integer): TMethodCode; inline;
begin
  Result:=FMethods[AIndex];
end;

function TUserInterface.GetMethodCount: Integer; inline;
begin
  Result:=Length(FMethods);
end;

procedure TUserInterface.SetCurrentMethod(AMethod: TMethodCode);
begin
  if AMethod=FCurrentMethod then Exit;
  CodeEditor.SaveToMethod(FCurrentMethod);
  FCurrentMethod:=AMethod;
  CodeEditor.ApplyMethod(FCurrentMethod);
end;

function TUserInterface.AddNewMethod(AName: string): TMethodCode;
begin
  Result:=TMethodCode.Create(Self);
  Result.Name:=AName;
  SetLength(FMethods, Length(FMethods) + 1);
  FMethods[High(FMethods)]:=Result;
end;

// TODO: This method is really bad, i should rewrite it later when more
//       and better parsing features are implemented...
procedure TUserInterface.BreakProgramCodeToMethods(ACode: string);
var
  Lines: TStringList = nil;
  Line: Integer;
  Main, MetName, Sub: string;
  Fmt: TFormatter;
  TmpEd: TCodeEditor;
  SubMet: TMethodCode;
begin
  try
    Lines:=TStringList.Create;
    Lines.Text:=ACode;
    Fmt:=TFormatter.Create(Self);
    Fmt.Prog:=Prog;
    Line:=0;
    Main:='';
    while Line < Lines.Count do begin
      try
        Lines[Line]:=Fmt.FormatLine(Lines[Line]);
      except
      end;
      if Copy(UpperCase(Trim(Lines[Line])), 1, 4)='SUB ' then begin
        Sub:=Lines[Line] + LineEnding;
        Inc(Line);
        MetName:=Copy(Sub, 5, Length(Sub));
        if Pos('(', MetName) <> 0 then MetName:=Copy(MetName, 1, Pos('(', MetName) - 1);
        MetName:=Trim(MetName);
        while Line < Lines.Count do begin
          try
            Lines[Line]:=Fmt.FormatLine(Lines[Line]);
          except
          end;
          Sub:=Sub + Lines[Line] + LineEnding;
          if Copy(UpperCase(Trim(Lines[Line])), 1, 7)='END SUB' then Break;
          Inc(Line);
        end;
        TmpEd:=TCodeEditor.Create(Self);
        TmpEd.Code:=TrimRight(Sub) + LineEnding;
        SubMet:=AddNewMethod(MetName);
        SubMet.Lines:=TmpEd.FLines;
        FreeAndNil(TmpEd);
        Inc(Line);
        while (Line < Lines.Count) and (Trim(Lines[Line])='') do Inc(Line);
        Continue;
      end;
      if Copy(UpperCase(Trim(Lines[Line])), 1, 9)='FUNCTION ' then begin
        Sub:=Lines[Line] + LineEnding;
        Inc(Line);
        MetName:=Copy(Sub, 10, Length(Sub));
        if Pos('(', MetName) <> 0 then MetName:=Copy(MetName, 1, Pos('(', MetName) - 1);
        MetName:=Trim(MetName);
        while Line < Lines.Count do begin
          try
            Lines[Line]:=Fmt.FormatLine(Lines[Line]);
          except
          end;
          Sub:=Sub + Lines[Line] + LineEnding;
          if Copy(UpperCase(Trim(Lines[Line])), 1, 12)='END FUNCTION' then Break;
          Inc(Line);
        end;
        TmpEd:=TCodeEditor.Create(Self);
        TmpEd.Code:=TrimRight(Sub) + LineEnding;
        SubMet:=AddNewMethod(MetName);
        SubMet.Lines:=TmpEd.FLines;
        FreeAndNil(TmpEd);
        Inc(Line);
        while (Line < Lines.Count) and (Trim(Lines[Line])='') do Inc(Line);
        Continue;
      end;
      Main:=Main + Lines[Line] + LineEnding;
      Inc(Line);
    end;
    TmpEd:=TCodeEditor.Create(Self);
    TmpEd.Code:=TrimRight(Main) + LineEnding;
    Methods[0].Lines:=TmpEd.FLines;
  finally
    FreeAndNil(TmpEd);
    FreeAndNil(Fmt);
    FreeAndNil(Lines);
  end;
  CodeEditor.Refresh;
end;

function TUserInterface.MergeMethodsToProgramCode: string;
var
  I, J: Integer;
begin
  Result:='';
  for I:=0 to MethodCount - 1 do with Methods[I] do begin
    for J:=0 to High(Lines) do Result:=Result + Lines[J].Code + LineEnding;
    Result:=Result + LineEnding;
  end;
end;

procedure TUserInterface.SetFileName(AFileName: string);
begin
  if AFileName=FFileName then Exit;
  FFileName:=AFileName;
  FileNameChanged;
end;

procedure TUserInterface.NewProgram;
var
  I: Integer;
begin
  if CodeEditor.Modified and not ConfirmBox('Modified Program', 'The program has been modified. If you create a new program now, you will lose these modifications. Do you want to proceed?', False) then Exit;
  if Assigned(FProg) then FreeAndNil(FProg);
  FProg:=TProgram.Create(Self);
  CodeEditor.Reset;
  for I:=0 to High(FMethods) do FMethods[I].Free;
  SetLength(FMethods, 0);
  AddNewMethod('');
  FCurrentMethod:=FMethods[0];
  FileName:='';
end;

procedure TUserInterface.OpenProgram;
var
  NewFileName, Code: string;
begin
  if CodeEditor.Modified and not ConfirmBox('Modified Program', 'The program has been modified. If you create a new program now, you will lose these modifications. Do you want to proceed?', False) then Exit;
  NewFileName:=FileName;
  if not ShowFileDialog(False, 'Runtime BASIC Program', 'bas', NewFileName) then Exit;
  try
    Code:=LoadTextFile(NewFileName);
  except
    MessageBox('Failed to load the file', 'Failed to load "' + NewFileName + '"' + LineEnding + LineEnding + Exception(ExceptObject).Message, mtError);
    Exit;
  end;
  CodeEditor.Reset;
  CodeEditor.Modified:=False;
  SetLength(FMethods, 0);
  AddNewMethod('');
  FCurrentMethod:=FMethods[0];
  BreakProgramCodeToMethods(Code);
  CodeEditor.FLines:=FCurrentMethod.Lines;
  FileName:=NewFileName;
end;

procedure TUserInterface.SaveProgram;
begin
  if FileName='' then begin
    SaveProgramAs;
    Exit;
  end;
  try
    SaveTextFile(FileName, MergeMethodsToProgramCode);
  except
    MessageBox('Failed to save the file', 'Failed to save "' + FileName + '"', mtError);
    Exit;
  end;
  CodeEditor.Modified:=False;
end;

procedure TUserInterface.SaveProgramAs;
var
  NewFileName: string;
begin
  NewFileName:=FileName;
  if not ShowFileDialog(True, 'Runtime BASIC Program', 'bas', NewFileName) then Exit;
  try
    SaveTextFile(NewFileName, MergeMethodsToProgramCode);
  except
    MessageBox('Failed to save the file', 'Failed to save "' + NewFileName + '"', mtError);
    Exit;
  end;
  CodeEditor.Modified:=False;
  FileName:=NewFileName;
end;

procedure TUserInterface.RunProgram;
var
  VM: TVirtualMachine;
  CPos: TCodePosition;
  I, X, Y: Integer;
  Sources: array of TSource;
  Cpl: TCompiler;

  function FindSource(ASource: TSource): TMethodCode;
  var
    I: Integer;
  begin
    for I:=0 to High(Sources) do
      if Sources[I]=ASource then Exit(Methods[I]);
    Result:=nil;
  end;

begin
  CodeEditor.SaveToMethod(CurrentMethod);
  VM:=TVirtualMachine.Create(Prog);
  Cpl:=TCompiler.Create(VM);

  try
    SetLength(Sources, MethodCount);
    for I:=0 to MethodCount - 1 do begin
      Sources[I]:=Methods[I].ToSource;
      Cpl.AddSource(Sources[I]);
    end;
    Cpl.Prog:=Prog;
    Cpl.Compile;
    VM.Run(Prog);
  except
    on ECompileError do begin
      CurrentMethod:=FindSource(ECompileError(ExceptObject).Source);
      CodeEditor.CursorX:=ECompileError(ExceptObject).Col - 1;
      CodeEditor.CursorY:=ECompileError(ExceptObject).Row - 1;
      MessageBox('Error', ECompileError(ExceptObject).Message, mtError);
    end;
    on ERuntimeError do begin
      MessageBox('Run-time Error', ERuntimeError(ExceptObject).Message, mtError);
      if Prog.Code.FindPosition(ERuntimeError(ExceptObject).Addr, CPos) then begin
        CurrentMethod:=FindSource(TSource(CPos.Source));
        Writeln(HexStr(CurrentMethod));
        TSource(CPos.Source).GetRowAndColumn(CPos.Position, Y, X);
        CodeEditor.CursorX:=X - 1;
        CodeEditor.CursorY:=Y - 1;
      end;
    end;
  end;
  for I:=0 to High(Sources) do Sources[I].Free;
  VM.Free;
end;

end.
