unit RuntimeBASIC;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, LResources, Forms, Controls, Graphics, Dialogs, RBProg,
  RBCpl, RBVM, RBCode, RBSrc, RBNative, RBScan, RBVMStor;

type

  TRuntimeBASICErrorHandler = procedure(Sender: TObject; ErrorMessage: string; Column, Row: Integer) of object;
  TRuntimeBASICOutputString = procedure(Sender: TObject; AString: string) of object;

  { TRBNativeLibrary }

  TRBNativeLibrary = class(TComponent)
  protected
    function GetNativeInterface: TNativeInterface; virtual; abstract;
  public
    property NativeInterface: TNativeInterface read GetNativeInterface;
  end;

  { TRBStandardRuntimeLibrary }

  TRBStandardRuntimeLibrary = class(TRBNativeLibrary)
  protected
    function GetNativeInterface: TNativeInterface; override;
  end;

  { TRuntimeBASICInputOutput }

  TRuntimeBASICInputOutput = class(TComponent)
  protected
    procedure OutputString(AString: string); virtual; abstract;
    procedure OutputTab; virtual; abstract;
    procedure OutputNewLine; virtual; abstract;
  end;

  { TRuntimeBASICStandardInputOutput }

  TRuntimeBASICStandardInputOutput = class(TRuntimeBASICInputOutput)
  protected
    procedure OutputString(AString: string); override;
    procedure OutputTab; override;
    procedure OutputNewLine; override;
  end;

  { TRuntimeBASICCustomInputOutput }

  TRuntimeBASICCustomInputOutput = class(TRuntimeBASICInputOutput)
  private
    FOnOutputNewLine: TNotifyEvent;
    FOnOutputString: TRuntimeBASICOutputString;
    FOnOutputTab: TNotifyEvent;
  protected
    procedure OutputString(AString: string); override;
    procedure OutputTab; override;
    procedure OutputNewLine; override;
  published
    property OnOutputString: TRuntimeBASICOutputString read FOnOutputString write FOnOutputString;
    property OnOutputTab: TNotifyEvent read FOnOutputTab write FOnOutputTab;
    property OnOutputNewLine: TNotifyEvent read FOnOutputNewLine write FOnOutputNewLine;
  end;

  { TRuntimeBASIC }

  TRuntimeBASIC = class(TComponent)
  private
    FInputOutput: TRuntimeBASICInputOutput;
    FNativeLibrary: TRBNativeLibrary;
    FOnError: TRuntimeBASICErrorHandler;
    FProg: TProgram;
    FVirtualMachine: TVirtualMachine;
    FLastSource: string;
    VMIO: TVirtualMachineIO;
    procedure SetInputOutput(AValue: TRuntimeBASICInputOutput);
    procedure SetNativeLibrary(AValue: TRBNativeLibrary);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CompileSource(ASourceCode: string): Boolean;
    function CompileSource(ASource: TSource): Boolean;
    procedure ReassignVirtualMachineInputOutputProxy;
    procedure Reset;
    procedure Run;
    property Prog: TProgram read FProg;
    property VirtualMachine: TVirtualMachine read FVirtualMachine;
  published
    property NativeLibrary: TRBNativeLibrary read FNativeLibrary write SetNativeLibrary;
    property InputOutput: TRuntimeBASICInputOutput read FInputOutput write SetInputOutput;
    property OnError: TRuntimeBASICErrorHandler read FOnError write FOnError;
  end;

procedure Register;

implementation

uses
  RBRTL;

type

  { TVirtualMachineIOProxy }

  TVirtualMachineIOProxy = class(TVirtualMachineIO)
  public
    RB: TRuntimeBASIC;
    procedure OutputString(AString: string); override;
    procedure OutputTab; override;
    procedure OutputNewLine; override;
  end;

procedure Register;
begin
  {$I runtimebasic_icon.lrs}
  RegisterComponents('Runtime BASIC',[TRuntimeBASIC, TRBStandardRuntimeLibrary, TRuntimeBASICStandardInputOutput, TRuntimeBASICCustomInputOutput]);
end;

{ TVirtualMachineIOProxy }

procedure TVirtualMachineIOProxy.OutputString(AString: string);
begin
  if Assigned(RB.FInputOutput) then
    RB.FInputOutput.OutputString(AString)
  else
    VirtualMachineStandardIO.OutputString(AString);
end;

procedure TVirtualMachineIOProxy.OutputTab;
begin
  if Assigned(RB.FInputOutput) then
    RB.FInputOutput.OutputTab
  else
    VirtualMachineStandardIO.OutputTab;
end;

procedure TVirtualMachineIOProxy.OutputNewLine;
begin
  if Assigned(RB.FInputOutput) then
    RB.FInputOutput.OutputNewLine
  else
    VirtualMachineStandardIO.OutputNewLine;
end;

{ TRuntimeBASICCustomInputOutput }

procedure TRuntimeBASICCustomInputOutput.OutputString(AString: string);
begin
  if Assigned(FOnOutputString) then FOnOutputString(Self, AString);
end;

procedure TRuntimeBASICCustomInputOutput.OutputTab;
begin
  if Assigned(FOnOutputTab) then FOnOutputTab(Self);
end;

procedure TRuntimeBASICCustomInputOutput.OutputNewLine;
begin
  if Assigned(FOnOutputNewLine) then FOnOutputNewLine(Self);
end;

{ TRuntimeBASICStandardInputOutput }

procedure TRuntimeBASICStandardInputOutput.OutputString(AString: string);
begin
  VirtualMachineStandardIO.OutputString(AString);
end;

procedure TRuntimeBASICStandardInputOutput.OutputTab;
begin
  VirtualMachineStandardIO.OutputTab;
end;

procedure TRuntimeBASICStandardInputOutput.OutputNewLine;
begin
  VirtualMachineStandardIO.OutputNewLine;
end;

{ TRBStandardRuntimeLibrary }

function TRBStandardRuntimeLibrary.GetNativeInterface: TNativeInterface;
begin
  Result:=StandardRuntimeLibrary;
end;

{ TRuntimeBASIC }

procedure TRuntimeBASIC.SetNativeLibrary(AValue: TRBNativeLibrary);
begin
  if FNativeLibrary=AValue then Exit;
  FNativeLibrary:=AValue;
  FProg.NativeInterface:=FNativeLibrary.NativeInterface;
end;

procedure TRuntimeBASIC.SetInputOutput(AValue: TRuntimeBASICInputOutput);
begin
  if FInputOutput=AValue then Exit;
  FInputOutput:=AValue;
  FVirtualMachine.IO:=VMIO;
end;

constructor TRuntimeBASIC.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  VMIO:=TVirtualMachineIOProxy.Create;
  TVirtualMachineIOProxy(VMIO).RB:=Self;
  Reset;
end;

destructor TRuntimeBASIC.Destroy;
begin
  FreeAndNil(FVirtualMachine);
  FreeAndNil(FProg);
  FreeAndNil(VMIO);
  inherited Destroy;
end;

function TRuntimeBASIC.CompileSource(ASourceCode: string): Boolean;
var
  Source: TSource;
begin
  Result:=False;
  Source:=TSource.Create(Prog);
  Source.SourceCode:=ASourceCode;
  try
    Result:=CompileSource(Source);
  finally
    FreeAndNil(Source);
  end;
end;

function TRuntimeBASIC.CompileSource(ASource: TSource): Boolean;
var
  Compiler: TCompiler;
begin
  Result:=False;
  Compiler:=TCompiler.Create(Prog);
  Compiler.Prog:=Prog;
  Compiler.AddSource(ASource);
  FLastSource:=ASource.SourceCode;
  try
    Compiler.Compile;
    Result:=True;
  except
    on ScanError: EScanError do begin
      if Assigned(FOnError) then begin
        FOnError(Self, ScanError.Message, ScanError.Col, ScanError.Row);
      end else begin
        FreeAndNil(Compiler);
        raise;
      end;
    end;
    on CompileError: ECompileError do begin
      if Assigned(FOnError) then begin
        FOnError(Self, CompileError.Message, CompileError.Col, CompileError.Row);
      end else begin
        FreeAndNil(Compiler);
        raise;
      end;
    end;
  end;
end;

procedure TRuntimeBASIC.ReassignVirtualMachineInputOutputProxy;
begin
  VirtualMachine.IO:=VMIO;
end;

procedure TRuntimeBASIC.Reset;
begin
  FreeAndNil(FVirtualMachine);
  FreeAndNil(FProg);
  FProg:=TProgram.Create(nil);
  FVirtualMachine:=TVirtualMachine.Create(nil);
  if Assigned(FNativeLibrary) then FProg.NativeInterface:=FNativeLibrary.NativeInterface;
  ReassignVirtualMachineInputOutputProxy;
  FLastSource:='';
end;

procedure TRuntimeBASIC.Run;
var
  TempSource: TSource;
  CodePosition: TCodePosition;
  Row: Integer;
  Column: Integer;
begin
  try
    VirtualMachine.Run(Prog);
  except
    on RuntimeError: ERuntimeError  do begin
      if Assigned(FOnError) then begin
        if RuntimeError.Code.FindPosition(RuntimeError.Addr, CodePosition) then begin
          TempSource:=TSource.Create(Prog);
          TempSource.SourceCode:=FLastSource;
          TempSource.GetRowAndColumn(CodePosition.Position, Row, Column);
        end else begin
          TempSource:=nil;
          Column:=1;
          Row:=1;
        end;
        FOnError(Self, RuntimeError.Message, Column, Row);
        FreeAndNil(TempSource);
      end else raise;
    end;
  end;
end;

end.

