unit RBRTL;
{$MODE OBJFPC}{$H+}
interface

uses
  SysUtils, RBVMStor, RBNative;

type
  TStandardRuntimeLibrary = class(TNativeInterface)
  protected
    procedure RuntimeError(AMessage: string); inline;
    procedure ArgumentCheck(const Args: TNativeArguments; Min, Max: Integer; MetName: string); inline;
    function ToInt(const V: TVMValue): Int64; inline;
    function ToReal(const V: TVMValue): Double; inline;
    function ToStr(const V: TVMValue): string; inline;
    procedure NatAbs(Args: TNativeArguments; var Result: TVMValue);
    procedure NatAsc(Args: TNativeArguments; var Result: TVMValue);
    procedure NatAtn(Args: TNativeArguments; var Result: TVMValue);
    procedure NatBeep(Args: TNativeArguments; var Result: TVMValue);
    procedure NatChr(Args: TNativeArguments; var Result: TVMValue);
    procedure NatCos(Args: TNativeArguments; var Result: TVMValue);
    procedure NatDate(Args: TNativeArguments; var Result: TVMValue);
    procedure NatExp(Args: TNativeArguments; var Result: TVMValue);
    procedure NatHex(Args: TNativeArguments; var Result: TVMValue);
    procedure NatLBound(Args: TNativeArguments; var Result: TVMValue);
    procedure NatLCase(Args: TNativeArguments; var Result: TVMValue);
    procedure NatLeft(Args: TNativeArguments; var Result: TVMValue);
    procedure NatLen(Args: TNativeArguments; var Result: TVMValue);
    procedure NatLog(Args: TNativeArguments; var Result: TVMValue);
    procedure NatLTrim(Args: TNativeArguments; var Result: TVMValue);
    procedure NatMid(Args: TNativeArguments; var Result: TVMValue);
    procedure NatOct(Args: TNativeArguments; var Result: TVMValue);
    procedure NatRight(Args: TNativeArguments; var Result: TVMValue);
    procedure NatRnd(Args: TNativeArguments; var Result: TVMValue);
    procedure NatRTrim(Args: TNativeArguments; var Result: TVMValue);
    procedure NatSgn(Args: TNativeArguments; var Result: TVMValue);
    procedure NatShell(Args: TNativeArguments; var Result: TVMValue);
    procedure NatSin(Args: TNativeArguments; var Result: TVMValue);
    procedure NatSpace(Args: TNativeArguments; var Result: TVMValue);
    procedure NatSqr(Args: TNativeArguments; var Result: TVMValue);
    procedure NatStr(Args: TNativeArguments; var Result: TVMValue);
    procedure NatString(Args: TNativeArguments; var Result: TVMValue);
    procedure NatTan(Args: TNativeArguments; var Result: TVMValue);
    procedure NatTrim(Args: TNativeArguments; var Result: TVMValue);
    procedure NatUCase(Args: TNativeArguments; var Result: TVMValue);
    procedure NatUBound(Args: TNativeArguments; var Result: TVMValue);
    procedure NatVal(Args: TNativeArguments; var Result: TVMValue);
  protected
    procedure RegisterRTL;
    procedure RegisterFunctions; virtual;
  public
    procedure AfterConstruction; override;
  end;

var
  StandardRuntimeLibrary: TStandardRuntimeLibrary;

implementation

uses
  RBUtil, RBError, Math, Process;

{$PUSH}{$WARN 5024 OFF}

{ TStandardRuntimeLibrary }

// Helpers ////////////////////////////////////////////////////////////////////

procedure TStandardRuntimeLibrary.RuntimeError(AMessage: string);
begin
  raise ENativeError.Create(AMessage);
end;

procedure TStandardRuntimeLibrary.ArgumentCheck(const Args: TNativeArguments; Min, Max: Integer; MetName: string); inline;
begin
  if (Length(Args) < Min) or (Length(Args) > Max) then RuntimeError(emFmt(emInvalidArgumentCountToX, MetName));
end;

function TStandardRuntimeLibrary.ToInt(const V: TVMValue): Int64;
begin
  case V.ValueType of
    vtInteger: Result:=V.IntValue;
    vtReal: Result:=Round(V.RealValue);
    else RuntimeError(emIntegerValueExpected);
  end;
end;

function TStandardRuntimeLibrary.ToReal(const V: TVMValue): Double;
begin
  case V.ValueType of
    vtInteger: Result:=V.IntValue;
    vtReal: Result:=V.RealValue;
    else RuntimeError(emArithmeticValueExpected);
  end;
end;

function TStandardRuntimeLibrary.ToStr(const V: TVMValue): string;
begin
  if V.ValueType=vtString then Result:=V.StrValue else RuntimeError(emInvalidType);
end;

// Standard functions /////////////////////////////////////////////////////////

procedure TStandardRuntimeLibrary.NatAbs(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Abs');
  case Args[0].ValueType of
    vtInteger: SetInt(Result, Abs(Args[0].IntValue));
    vtReal: SetReal(Result, Abs(Args[0].RealValue));
    else RuntimeError(emArithmeticValueExpected);
  end;
end;

procedure TStandardRuntimeLibrary.NatAsc(Args: TNativeArguments; var Result: TVMValue);
var
  S: string;
begin
  ArgumentCheck(Args, 1, 1, 'Asc');
  S:=ToStr(Args[0]);
  if S='' then RuntimeError(emValueCannotBeEmpty);
  SetInt(Result, Ord(S[1]));
end;

procedure TStandardRuntimeLibrary.NatAtn(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Atn');
  SetReal(Result, ArcTan(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatBeep(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 0, 0, 'Beep');
  Beep;
end;

procedure TStandardRuntimeLibrary.NatChr(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Chr');
  SetString(Result, Chr(ToInt(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatCos(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Cos');
  SetReal(Result, Cos(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatDate(Args: TNativeArguments; var Result: TVMValue);
var
  S: TVMArraySize;
  ST: TSystemTime;
begin
  ArgumentCheck(Args, 0, 0, 'Date');
  S[0]:=8;
  CreateArray(VMP, S, 1, Result);
  DecRef(Result.ArrValue);
  DateTimeToSystemTime(Now, ST);
  with ST do begin
    SetInt(Result.ArrValue^.Data[0], Year);
    SetInt(Result.ArrValue^.Data[1], Month);
    SetInt(Result.ArrValue^.Data[2], Day);
    SetInt(Result.ArrValue^.Data[3], Hour);
    SetInt(Result.ArrValue^.Data[4], Minute);
    SetInt(Result.ArrValue^.Data[5], Second);
    SetInt(Result.ArrValue^.Data[6], MilliSecond);
    SetString(Result.ArrValue^.Data[7], DateTimeToStr(Now));
  end;
end;

procedure TStandardRuntimeLibrary.NatExp(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Exp');
  SetReal(Result, Exp(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatHex(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Hex');
  SetString(Result, HexStr(ToInt(Args[0]), 16));
  while (Result.StrValue <> '') and (Result.StrValue[1]='0') do
    Result.StrValue:=Copy(Result.StrValue, 2, Length(Result.StrValue));
end;

procedure TStandardRuntimeLibrary.NatLBound(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'LBound');
  if Args[0].ValueType <> vtArray then RuntimeError(emValueIsNotAnArray);
  if Args[0].ArrValue^.Dims <> 1 then RuntimeError(emValueIsAMultidimArray);
  SetInt(Result, 0);
end;

procedure TStandardRuntimeLibrary.NatLCase(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'LCase');
  SetString(Result, LowerCase(ToStr(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatLeft(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 2, 2, 'Left');
  SetString(Result, Copy(ToStr(Args[0]), 1, ToInt(Args[1])));
end;

procedure TStandardRuntimeLibrary.NatLen(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Len');
  case Args[0].ValueType of
    vtString: SetInt(Result, Length(Args[0].StrValue));
    vtArray: SetInt(Result, Args[0].ArrValue^.DLen);
    else RuntimeError(emFmt(emInvalidType));
  end;
end;

procedure TStandardRuntimeLibrary.NatLog(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Log');
  SetReal(Result, Ln(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatLTrim(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'LTrim');
  SetString(Result, TrimLeft(ToStr(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatMid(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 3, 3, 'Mid');
  SetString(Result, Copy(ToStr(Args[0]), ToInt(Args[1]), ToInt(Args[2])));
end;

procedure TStandardRuntimeLibrary.NatOct(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Oct');
  SetString(Result, OctStr(ToInt(Args[0]), 23));
  while (Result.StrValue <> '') and (Result.StrValue[1]='0') do
    Result.StrValue:=Copy(Result.StrValue, 2, Length(Result.StrValue));
end;

procedure TStandardRuntimeLibrary.NatRight(Args: TNativeArguments; var Result: TVMValue);
var
  S: string;
begin
  ArgumentCheck(Args, 2, 2, 'Right');
  S:=ToStr(Args[0]);
  SetString(Result, Copy(S, Length(S) - ToInt(Args[1]) + 1, Length(S)));
end;

procedure TStandardRuntimeLibrary.NatRnd(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 0, 0, 'Rnd');
  SetReal(Result, Random);
end;

procedure TStandardRuntimeLibrary.NatRTrim(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'RTrim');
  SetString(Result, TrimRight(ToStr(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatSgn(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Sgn');
  SetReal(Result, Sign(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatShell(Args: TNativeArguments; var Result: TVMValue);
var
  Output: string;
  I: Integer;
  CmdArgs: array of string;
begin
  if Length(Args) > 0 then begin
    SetLength(CmdArgs, Length(Args) - 1);
    for I:=1 to High(Args) do begin
      case Args[I].ValueType of
        vtInteger: CmdArgs[I - 1]:=IntToStr(Args[I].IntValue);
        vtReal: CmdArgs[I - 1]:=FloatStr(Args[I].RealValue);
        vtString: CmdArgs[I - 1]:=Args[I].StrValue;
        else RuntimeError(emInvalidType);
      end;
    end;
    Output:='';
    if RunCommand(ToStr(Args[0]), CmdArgs, Output) then SetString(Result, Output) else SetString(Result, '');
  end else SetString(Result, '');
end;

procedure TStandardRuntimeLibrary.NatSin(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Sin');
  SetReal(Result, Sin(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatSpace(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Space');
  SetString(Result, Space(ToInt(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatSqr(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Sqr');
  SetReal(Result, Sqrt(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatStr(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Str');
  SetString(Result, ValueToString(Args[0]));
end;

procedure TStandardRuntimeLibrary.NatString(Args: TNativeArguments; var Result: TVMValue);
var
  I, Cnt: Integer;
  S, FS: string;
begin
  ArgumentCheck(Args, 2, 2, 'String');
  case Args[1].ValueType of
    vtInteger, vtReal: S:=Chr(ToInt(Args[1]));
    vtString: S:=Args[1].StrValue;
    else RuntimeError(emInvalidType);
  end;
  Cnt:=ToInt(Args[0]);
  FS:='';
  for I:=1 to Cnt do FS:=FS + S;
  SetString(Result, FS);
end;

procedure TStandardRuntimeLibrary.NatTan(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Tan');
  SetReal(Result, Tan(ToReal(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatTrim(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Trim');
  SetString(Result, Trim(ToStr(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatUCase(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'UCase');
  SetString(Result, UpperCase(ToStr(Args[0])));
end;

procedure TStandardRuntimeLibrary.NatUBound(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'UBound');
  if Args[0].ValueType <> vtArray then RuntimeError(emValueIsNotAnArray);
  if Args[0].ArrValue^.Dims <> 1 then RuntimeError(emValueIsAMultidimArray);
  SetInt(Result, Args[0].ArrValue^.DLen);
end;

procedure TStandardRuntimeLibrary.NatVal(Args: TNativeArguments; var Result: TVMValue);
begin
  ArgumentCheck(Args, 1, 1, 'Val');
  try
    SetReal(Result, StrToFloat(ToStr(Args[0])));
    if Frac(Result.RealValue)=0 then begin
      Result.IntValue:=Round(Result.RealValue);
      Result.ValueType:=vtInteger;
    end;
  except
    RuntimeError(emArithmeticValueExpected);
  end;
end;

// Registration and Stuff /////////////////////////////////////////////////////

procedure TStandardRuntimeLibrary.RegisterRTL;
begin
  Register('Abs', mtFunction, @NatAbs);
  Register('Asc', mtFunction, @NatAsc);
  Register('Atn', mtFunction, @NatAtn);
  Register('Beep', mtSub, @NatBeep);
  Register('Chr', mtFunction, @NatChr);
  Register('Cos', mtFunction, @NatCos);
  Register('Date', mtFunction, @NatDate);
  Register('Exp', mtFunction, @NatExp);
  Register('Hex', mtFunction, @NatHex);
  Register('LBound', mtFunction, @NatLBound);
  Register('LCase', mtFunction, @NatLCase);
  Register('Left', mtFunction, @NatLeft);
  Register('Len', mtFunction, @NatLen);
  Register('Log', mtFunction, @NatLog);
  Register('LTrim', mtFunction, @NatLTrim);
  Register('Mid', mtFunction, @NatMid);
  Register('Oct', mtFunction, @NatOct);
  Register('Right', mtFunction, @NatRight);
  Register('Rnd', mtFunction, @NatRnd);
  Register('RTrim', mtFunction, @NatRTrim);
  Register('Sgn', mtFunction, @NatSgn);
  Register('Shell', mtFunction, @NatShell);
  Register('Sin', mtFunction, @NatSin);
  Register('Space', mtFunction, @NatSpace);
  Register('Sqr', mtFunction, @NatSqr);
  Register('Str', mtFunction, @NatStr);
  Register('String', mtFunction, @NatString);
  Register('Tan', mtFunction, @NatTan);
  Register('Trim', mtFunction, @NatTrim);
  Register('UCase', mtFunction, @NatUCase);
  Register('UBound', mtFunction, @NatUBound);
  Register('Val', mtFunction, @NatVal);
end;

procedure TStandardRuntimeLibrary.RegisterFunctions;
begin
  RegisterRTL;
end;

procedure TStandardRuntimeLibrary.AfterConstruction;
begin
  inherited AfterConstruction;
  RegisterFunctions;
end;

{$POP}

initialization
  StandardRuntimeLibrary:=TStandardRuntimeLibrary.Create(nil);
  Randomize;
finalization
  FreeAndNil(StandardRuntimeLibrary);
end.
