unit RBProg;
{$MODE OBJFPC}{$H+}
interface
uses
  FGL, RBOwn, RBStrDB, RBCode, RBVMStor, RBNative;

type

  { TVarList }

  PVarList = ^TVarList;
  TVarList = object
    Vars: array of string;
    procedure Reset;
    function Add(AName: string): Integer;
    function Find(AName: string): Integer;
    procedure Rename(OldName, NewName: string);
  end;

  { TMethod }

  TMethod = class
    Name: string;
    Code: TCode;
    Func: Boolean;
    Vars: TVarList;
    ArgCount: Integer;
    ID: Cardinal;
    Local: TVMStorage;
    constructor Create;
    destructor Destroy; override;
  end;

  TMethodList = specialize TFPGObjectList<TMethod>;

  { TProgram }

  TProgram = class(TOwnable)
  private
    FNativeInterface: TNativeInterface;
    FStringDB: TStringDB;
    FMethods: TMethodList;
    FCode: TCode;
    FCompiled: Boolean;
    FVars: TVarList;
    function GetVars: PVarList;
    function GetMethods(AIndex: Integer): TMethod; inline;
    function GetMethodCount: Integer; inline;
  public
    constructor Create(AOwner: TOwnable); override;
    destructor Destroy; override;
    procedure RemoveMethods;
    procedure ResetMethods;
    procedure AddMethod(AMethod: TMethod);
    procedure RemoveMethod(AMethod: TMethod);
    function FindMethod(AName: string): TMethod;
    property NativeInterface: TNativeInterface read FNativeInterface write FNativeInterface;
    property StringDB: TStringDB read FStringDB;
    property Code: TCode read FCode;
    property Compiled: Boolean read FCompiled;
    property Vars: PVarList read GetVars;
    property Methods[AIndex: Integer]: TMethod read GetMethods;
    property MethodCount: Integer read GetMethodCount;
  end;

implementation

uses
  RBRTL;

{ TVarList }
procedure TVarList.Reset;
begin
  SetLength(Vars, 0);
end;

function TVarList.Add(AName: string): Integer;
begin
  SetLength(Vars, Length(Vars) + 1);
  Vars[High(Vars)]:=AName;
  Result:=High(Vars);
end;

function TVarList.Find(AName: string): Integer;
var
  I: Integer;
begin
  for I:=0 to High(Vars) do if Vars[I]=AName then Exit(I);
  Result:=-1;
end;

procedure TVarList.Rename(OldName, NewName: string);
var
  I: Integer;
begin
  if Find(NewName) <> -1 then Exit;
  I:=Find(OldName);
  Vars[I]:=NewName;
end;

{ TMethod }
constructor TMethod.Create;
begin
  Code:=TCode.Create(nil);
end;

destructor TMethod.Destroy;
begin
  Local.Free;
  Code.Free;
  inherited Destroy;
end;

{ TProgram }
function TProgram.GetVars: PVarList;
begin
  Result:=@FVars;
end;

function TProgram.GetMethods(AIndex: Integer): TMethod;
begin
  Result:=FMethods[AIndex];
end;

function TProgram.GetMethodCount: Integer;
begin
  Result:=FMethods.Count;
end;

constructor TProgram.Create(AOwner: TOwnable);
begin
  inherited Create(AOwner);
  FNativeInterface:=StandardRuntimeLibrary;
  FStringDB:=TStringDB.Create(Self);
  FMethods:=TMethodList.Create(True);
  FCode:=TCode.Create(Self);
  FCode.StringDB:=StringDB;
end;

destructor TProgram.Destroy;
begin
  FCode.Free;
  FMethods.Free;
  FStringDB.Free;
  inherited Destroy;
end;

procedure TProgram.RemoveMethods;
begin
  FMethods.Clear;
end;

procedure TProgram.ResetMethods;
var
  I: Integer;
begin
  for I:=0 to FMethods.Count - 1 do begin
    SetLength(FMethods[I].Vars.Vars, 0);
    FMethods[I].Code.Reset;
    FMethods[I].ID:=I;
  end;
end;

procedure TProgram.AddMethod(AMethod: TMethod);
begin
  FMethods.Add(AMethod);
  AMethod.Code.StringDB:=StringDB;
end;

procedure TProgram.RemoveMethod(AMethod: TMethod);
begin
  FMethods.Remove(AMethod);
end;

function TProgram.FindMethod(AName: string): TMethod;
var
  I: Integer;
begin
  for I:=0 to FMethods.Count - 1 do
    if FMethods[I].Name=AName then Exit(FMethods[I]);
  Exit(nil);
end;

end.
