////////////////////////////////////////////////////////////////////////////////
// Delphi 2 compatible OpenGL Viewport control                      ..---.    //
// Copyright (C) 2018 Kostas Michalopoulos                         (( OpenGL  //
//                                                                  ``---'    //
// This software is provided 'as-is', without any express or implied          //
// warranty. In no event will the authors be held liable for any damages      //
// arising from the use of this software.                                     //
//                                                                            //
// Permission is granted to anyone to use this software for any purpose,      //
// including commercial applications, and to alter it and redistribute it     //
// freely, subject to the following restrictions:                             //
//                                                                            //
// 1. The origin of this software must not be misrepresented; you must not    //
//    claim that you wrote the original software. If you use this software    //
//    in a product, an acknowledgment in the product documentation would be   //
//    appreciated but is not required.                                        //
// 2. Altered source versions must be plainly marked as such, and must not be //
//    misrepresented as being the original software.                          //
// 3. This notice may not be removed or altered from any source distribution. //
////////////////////////////////////////////////////////////////////////////////
unit DGLVP;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  OpenGL;

type
  { TOpenGLViewport - provides a viewport to render using OpenGL }
  TOpenGLViewport = class(TWinControl)
  private
    FOnRender: TNotifyEvent;
    RC: THandle;
    DC: THandle;
    ShareVP: TOpenGLViewport;
    procedure CreateContext;
    procedure DestroyContext;
  protected
    procedure WMDestroy(var Msg: TWMDestroy); message WM_DESTROY;
    procedure WMEraseBkgnd(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure WMPaint(var Msg: TWMPaint); message WM_PAINT;
  public
    constructor Create(AOwner: TComponent); override;

    // Share resources with the given OpenGL viewport - this must be called
    // before any painting is done!
    procedure ShareResourcesWith(AViewport: TOpenGLViewport);

    // Attempts to make the OpenGL context the current one (not needed in OnRender)
    function MakeCurrent: Boolean;

    // Swaps the OpenGL buffers (not needed in OnRender)
    procedure SwapBuffers;
  published
    property OnRender: TNotifyEvent read FOnRender write FOnRender;
    // Standard properties and events
    property Align;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

procedure Register;

implementation

var
  GLProcsAlreadyLoaded: Boolean;

{ TOpenGLViewport }

constructor TOpenGLViewport.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width:=192;
  Height:=128;
end;

procedure TOpenGLViewport.CreateContext;
label Fail;
var
  PFD: TPixelFormatDescriptor;
  PF: Integer;
begin
  FillChar(PFD, SizeOf(PFD), 0);

  // Try to acquire a device context from the underlying window
  DC:=GetDC(Handle);
  if DC=0 then goto Fail;

  // Find a suitable PFD
  PFD.nSize:=SizeOf(PFD);
  PFD.nVersion:=1;
  PFD.dwFlags:=PFD_DRAW_TO_WINDOW or PFD_SUPPORT_OPENGL or PFD_DOUBLEBUFFER;
  PFD.iPixelType:=PFD_TYPE_RGBA;
  PFD.cColorBits:=24;
  PFD.cAlphaBits:=8;
  PFD.cDepthBits:=24;
  PFD.cStencilBits:=8;
  PF:=ChoosePixelFormat(DC, @PFD);
  if PF=0 then goto Fail;

  // Set pixel format
  if not SetPixelFormat(DC, PF, @PFD) then goto Fail;

  // Create GL rendering context
  RC:=wglCreateContext(DC);
  if RC=0 then goto Fail;

  // Try to load OpenGL procs
  if not GLProcsAlreadyLoaded then begin
    if not LoadGLProcs then goto Fail;
    GLProcsAlreadyLoaded:=True;
  end;

  // Share resources, if set
  if Assigned(ShareVP) then begin
    wglShareLists(RC, ShareVP.RC);
    ShareVP:=nil;
  end;

  // Success!
  Exit;
  
Fail:
  DestroyContext;
end;

procedure TOpenGLViewport.DestroyContext;
begin
  if RC <> 0 then wglDeleteContext(RC);
  if DC <> 0 then ReleaseDC(Handle, DC);
  RC:=0;
  DC:=0;
end;

procedure TOpenGLViewport.WMDestroy(var Msg: TWMDestroy);
begin
  DestroyContext;
end;

procedure TOpenGLViewport.WMEraseBkgnd(var Msg: TWMEraseBkgnd);
begin
  // Do not erase the background to avoid flickering
  Msg.Result:=0;
end;

procedure TOpenGLViewport.WMPaint(var Msg: TWMPaint);
var
  DC: THandle;
  Previous: THandle;
begin
  // For design mode, just draw some generic background
  if csDesigning in ComponentState then begin
    // Draw rectangle
    DC:=GetDC(Handle);
    Previous:=SelectObject(DC, GetStockObject(GRAY_BRUSH));
    Rectangle(DC, 0, 0, Width, Height);
    SelectObject(DC, Previous);
    ReleaseDC(Handle, DC);

    // Validate window rect
    ValidateRect(Handle, nil);

    // Message handled
    Msg.Result:=0;
    Exit;
  end;

  // Create context if necessary
  if RC=0 then begin
    CreateContext;
    if RC=0 then Exit;
  end;

  // Activate context
  if MakeCurrent then begin
    // Update viewport
    glViewport(0, 0, Width, Height);

    // Call render code
    if Assigned(FOnRender) then
      FOnRender(Self)
    else begin // no render code, just clear the background
      glClearColor(0, 0, 0, 0);
      glClear(GL_COLOR_BUFFER_BIT);
    end;

    // Swap buffers
    SwapBuffers();
  end;

  // Validate window rect
  ValidateRect(Handle, nil);

  // Message handled
  Msg.Result:=0;
end;

procedure TOpenGLViewport.ShareResourcesWith(AViewport: TOpenGLViewport);
begin
  if RC <> 0 then raise Exception.Create('Cannot share resources after the context has been created');
  ShareVP:=AViewport;
end;

function TOpenGLViewport.MakeCurrent: Boolean;
begin
  Result:=wglMakeCurrent(DC, RC);
end;

procedure TOpenGLViewport.SwapBuffers;
begin
  Windows.SwapBuffers(DC);
end;

procedure Register;
begin
  RegisterComponents('Additional', [TOpenGLViewport]);
end;

end.
