//---------------------------------------------------------------------------
#if defined(__BORLANDC__) || defined(WIN32) || defined(__WIN32)
#define UTILS_USE_WINDOWS_API
#ifdef __BORLANDC__
#include <vcl\vcl.h>
#pragma hdrstop
#include <io.h>
#else
#include <Windows.h>
#endif
#include <shlobj.h>
#endif
#include <stdio.h>
#include <time.h>
#include "UtilsUnit.h"
//---------------------------------------------------------------------------
// NUtilities functions
AString NUtilities::GetCurrentDateTime()
{
    time_t now;
    time(&now);
    struct tm* t = localtime(&now);
    char buff[256];
    sprintf(buff, "%i-%02i-%02i %02i:%02i:%02i",
        t->tm_year + 1900, t->tm_mon + 1, t->tm_mday,
        t->tm_hour, t->tm_min, t->tm_sec);
    return AString(buff);
}
//---------------------------------------------------------------------------
AString NUtilities::GetHumanFriendlySize(unsigned size)
{
    if (size < 1024) return AString::FromUnsigned(size);
    size /= 1024;
    if (size < 1024) return AString::FromUnsigned(size) + " KB";
    size /= 1024;
    if (size < 1024) return AString::FromUnsigned(size) + " MB";
    size /= 1024;
    return AString::FromUnsigned(size) + "G";
}
//---------------------------------------------------------------------------
AString NUtilities::ExtractFileNameOnly(const AString& path)
{
    // Find path parts
    size_t lastPeriod = 0, lastForwardSlash = 0, lastReverseSlash = 0, lastSlash = 0;
    bool hasLastPeriod = path.FindLastCharacter('.', lastPeriod);
    bool hasForwardSlash = path.FindLastCharacter('/', lastForwardSlash);
    bool hasReverseSlash = path.FindLastCharacter('\\', lastReverseSlash);
    bool hasLastSlash = hasForwardSlash || hasReverseSlash;
    if (hasForwardSlash && hasReverseSlash)
        lastSlash = lastForwardSlash > lastReverseSlash ? lastForwardSlash : lastReverseSlash;
    else if (hasForwardSlash)
        lastSlash = lastForwardSlash;
    else if (hasReverseSlash)
        lastSlash = lastReverseSlash;

    // No such parts, the path doesn't contain a directory or extension
    if (!hasLastSlash && !hasLastPeriod)
    {
        return path;
    }

    // If the last period is before the last slash, this is a period in the path
    if ((hasLastSlash && hasLastPeriod && lastPeriod < lastSlash) || !hasLastPeriod)
    {
        lastPeriod = path.GetLength();
    }

    // No path part
    if (!hasLastSlash)
    {
        return path.GetLeftSubstring(lastPeriod);
    }
    // Both path and period
    else
    {
        return path.GetSubstring(lastSlash + 1, lastPeriod);
    }
}
//---------------------------------------------------------------------------
AString NUtilities::ExtractFileName(const AString& path)
{
    if (path.IsEmpty()) return "";
    for (size_t i=path.GetLength() - 1;;--i)
    {
        if (path[i] == '/' || path[i] == '\\')
            return path.GetSubstring(i + 1, path.GetLength());
        if (i == 0) return path;
    }
}
//---------------------------------------------------------------------------
AString NUtilities::ExtractDirectoryPath(const AString& path)
{
    if (path.IsEmpty()) return "";
    for (size_t i=path.GetLength() - 1;;--i)
    {
        if (path[i] == '/' || path[i] == '\\')
            return path.GetLeftSubstring(i);
        if (i == 0) return "";
    }
}
//---------------------------------------------------------------------------
AString NUtilities::NormalizePath(const AString& path)
{
    TDynArray<AString> parts = path.GetReplaced('/', '\\').Split('\\', true);
    AString result;
    for (unsigned i=0; i < parts.GetSize(); ++i)
    {
        if (i > 0) result += '\\';
        result += parts[i];
    }
    return result;
}
//---------------------------------------------------------------------------
bool NUtilities::GetFileSize(const AString& path, unsigned& size)
{
    FILE* f = fopen(path.GetText(), "rb");
    if (!f) return false;
    if (fseek(f, 0, SEEK_END))
    {
        fclose(f);
        return false;
    }
    size = ftell(f);
    fclose(f);
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::FileExistsAt(const AString& path)
{
    FILE* f = fopen(path.GetText(), "rb");
    if (!f) return false;
    fclose(f);
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::IsFileReadOnly(const AString& path, bool& flag)
{
    #ifdef UTILS_USE_WINDOWS_API
    DWORD attr = GetFileAttributes(path.GetText());
    if (attr == 0xFFFFFFFF) return false;
    flag = (attr & FILE_ATTRIBUTE_READONLY) == FILE_ATTRIBUTE_READONLY;
    #else
    #error not supported
    #endif
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::SetFileReadOnly(const AString& path, bool flag)
{
    #ifdef UTILS_USE_WINDOWS_API
    DWORD attr = GetFileAttributes(path.GetText());
    if (attr == 0xFFFFFFFF) return false;
    if (flag) attr |= FILE_ATTRIBUTE_READONLY;
    else attr &= ~FILE_ATTRIBUTE_READONLY;
    return SetFileAttributes(path.GetText(), attr);
    #else
    #error not supported
    return false;
    #endif
}
//---------------------------------------------------------------------------
bool NUtilities::ForceDirectoryPath(const AString& path)
{
    // Tokenize individual path elements
    TDynArray<AString> dirs = path.GetReplaced('\\', '/').Split('/');
    AString pathSoFar;
    // Attempt to create each directory
    for (unsigned i=0; i < dirs.GetSize(); ++i)
    {
        // Append to the full path for this directory
        if (i > 0)
        {
            #ifdef UTILS_USE_WINDOWS_API
            pathSoFar += '\\';
            #else
            pathSoFar += '/';
            #endif
        }
        pathSoFar += dirs[i];
        // Ignore drives
        if (i == 0 && dirs[i].GetLength() == 2 && dirs[i][1] == ':')
            continue;

        #ifdef UTILS_USE_WINDOWS_API
        // Try to find a directory or file at the given path
        WIN32_FIND_DATA data;
        HANDLE ff = FindFirstFile(pathSoFar.GetText(), &data);
        if (ff != INVALID_HANDLE_VALUE)  // file exists
        {
            DWORD attribs = data.dwFileAttributes;
            FindClose(ff);
            // If the file is not a directory, abort
            if (!(attribs & FILE_ATTRIBUTE_DIRECTORY))
                return false;
        }
        else // file doesn't exist
        {
            // Try to create the directory
            if (!CreateDirectory(pathSoFar.GetText(), NULL)) return false;
        }
        #else
        #error not supported
        #endif
    }
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::DirectCopyFile(const AString& srcPath, const AString& dstPath)
{
    FILE* inf = fopen(srcPath.GetText(), "rb");
    if (!inf) return false;
    FILE* outf = fopen(dstPath.GetText(), "wb");
    if (!outf)
    {
        fclose(inf);
        return false;
    }
    char buffer[4096];
    while (true)
    {
        size_t bytes = fread(buffer, 1, 4096, inf);
        if (!bytes) break;
        if (!fwrite(buffer, bytes, 1, outf))
        {
            fclose(outf);
            fclose(inf);
            remove(dstPath.GetText());
            return false;
        }
    }
    fclose(outf);
    fclose(inf);
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::WriteBufferToFile(const AString& path, const void* buffer, size_t size)
{
    DEBUG_ASSERT(buffer, "null buffer was given");
    FILE* f = fopen(path.GetText(), "wb");
    if (!f) return false;
    if (fwrite(buffer, 1, size, f) != size)
    {
        fclose(f);
        return false;
    }
    fclose(f);
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::ReadFileToBuffer(const AString& path, void*& buffer, size_t& size, size_t extra)
{
    // Clear data just in case
    buffer = NULL;
    size = 0;
    // Try to read the data
    FILE* f = fopen(path.GetText(), "rb");
    if (!f) return false;
    if (fseek(f, 0, SEEK_END))
    {
        fclose(f);
        return false;
    }
    size = ftell(f);
    if (fseek(f, 0, SEEK_SET))
    {
        fclose(f);
        return false;
    }
    buffer = (void*)malloc(size + extra);
    if (!buffer)
    {
        size = 0;
        fclose(f);
        return false;
    }
    if (fread(buffer, 1, size, f) != size)
    {
        buffer = NULL;
        size = 0;
        fclose(f);
        return false;
    }
    fclose(f);
    return true;
}
//---------------------------------------------------------------------------
bool NUtilities::WriteStringToFile(const AString& path, const AString& text)
{
    return WriteBufferToFile(path, text.GetText(), text.GetLength());
}
//---------------------------------------------------------------------------
bool NUtilities::ReadStringFromFile(const AString& path, AString& text)
{
    char* buffer;
    size_t size;
    if (!ReadFileToBuffer(path, (void*)buffer, size, 1))
    {
        text.Clear();
        return false;
    }
    buffer[size] = '\0';
    text.Clear();
    AString::WrapTo(text, buffer, size);
    return true;
}
//---------------------------------------------------------------------------
void NUtilities::SplitTextToLines(const AString& text, AStringArray& lines)
{
    // First split at LF
    lines = text.Split('\n');
    // Then remove any trailing CRs
    for (unsigned i=0; i < lines.GetSize(); ++i)
    {
        if (lines[i].EndsWith('\r'))
        {
            lines[i].TrimFromEnd(1);            
        }
    }
} 
//---------------------------------------------------------------------------
bool NUtilities::ScanDirectoryContents(const AString& path, TDynArray<AString>& files, TDynArray<AString>& dirs)
{
#ifdef UTILS_USE_WINDOWS_API
    WIN32_FIND_DATA data;
    memset(&data, 0, sizeof(data));
    HANDLE ff = FindFirstFile((path + "\\*.*").GetText(), &data);
    if (ff != INVALID_HANDLE_VALUE)
    {
        while (true)
        {
            if (!strcmp(data.cFileName, ".") || !strcmp(data.cFileName, ".."))
            {
                memset(&data, 0, sizeof(data));
                if (!FindNextFile(ff, &data)) break;
                continue;
            }
            if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
                dirs.Add(data.cFileName);
            else
                files.Add(data.cFileName);
            memset(&data, 0, sizeof(data));
            if (!FindNextFile(ff, &data)) break;
        }
        FindClose(ff);
        return true;
    }
    else return false;
#else
#error not supported
#endif
}
//---------------------------------------------------------------------------
bool NUtilities::CompareFileContents(const AString& path1, const AString& path2, bool& identical)
{
    // Try to open the files
    FILE* f1 = fopen(path1.GetText(), "rb");
    if (!f1) return false;
    FILE* f2 = fopen(path2.GetText(), "rb");
    if (!f2)
    {
        fclose(f1);
        return false;
    }
    // Check their file sizes
    if (fseek(f1, 0, SEEK_END) || fseek(f2, 0, SEEK_END))
    {
        fclose(f2);
        fclose(f1);
        return false;
    }
    unsigned size1 = ftell(f1);
    unsigned size2 = ftell(f2);
    if (size1 != size2)
    {
        fclose(f2);
        fclose(f1);
        identical = false;
        return true;
    }
    // Special case for zero bytes
    if (size1 == 0)
    {
        fclose(f2);
        fclose(f1);
        identical = true;
        return true;
    }
    // Compare the contents
    char buff1[4096];
    char buff2[4096];
    identical = true;
    while (true)
    {
        unsigned bytes1 = fread(buff1, 1, sizeof(buff1), f1);
        unsigned bytes2 = fread(buff2, 1, sizeof(buff2), f2);
        if (bytes1 != bytes2)
        {
            fclose(f2);
            fclose(f1);
            identical = false;
            return true;
        }
        if (!bytes1) break;
        if (memcmp(buff1, buff2, bytes1))     
        {
            identical = false;
            break;
        }
    }
    // Close the files
    fclose(f2);
    fclose(f1);
    return true;
}
//---------------------------------------------------------------------------
#ifdef __BORLANDC__
//---------------------------------------------------------------------------
// Borland C++ Builder specific functions
bool NUtilities::BrowseForFolder(AString& path)
{
    BROWSEINFO bi;
    memset(&bi, 0, sizeof(bi));
    bi.lpszTitle = "Select folder...";
    bi.ulFlags = BIF_RETURNONLYFSDIRS;
    LPITEMIDLIST pidl = SHBrowseForFolder(&bi);
    if (pidl)
    {
        char target[MAX_PATH];
        SHGetPathFromIDList(pidl, target);
        IMalloc* mem;
        if (SUCCEEDED(SHGetMalloc(&mem)))
        {
            mem->Free(pidl);
            mem->Release();
        }
        path = target;
        return true;
    }
    return false;
}
//---------------------------------------------------------------------------
int NUtilities::ShowCenteredModal(TForm* parent, TForm* box)
{
    box->Left = parent->Left + (parent->Width - box->Width)/2;
    box->Top = parent->Top + (parent->Height - box->Height)/2;
    return box->ShowModal();
}
//---------------------------------------------------------------------------
void NUtilities::EnableFullRowReportMode(TListView* view)
{
   DWORD style = SendMessage(view->Handle, 4151, 0, 0);
   style |= 0x20;
   SendMessage(view->Handle, 4150, 0, style);
}
#endif
//---------------------------------------------------------------------------

