//---------------------------------------------------------------------------
#ifndef StringsH
#define StringsH
//---------------------------------------------------------------------------
#include <string.h>
#include "DebugAssert.h"
#include "DynArray.h"
//---------------------------------------------------------------------------
// String utilities
namespace NStringUtilities
{
    // Overloaded length function that maps to both char and wchar_t
    inline unsigned GetZStringLength(const char* str) { return ::strlen(str); }
    // Overloaded length function that maps to both char and wchar_t
    inline unsigned GetZStringLength(const wchar_t* str) { return ::wcslen(str); }

    // Overloaded function that returns an empty string
    inline const char* GetEmptyString(const char*) { return ""; }
    // Overloaded function that returns an empty string
    inline const wchar_t* GetEmptyString(const wchar_t*) { return L""; }

    // Duplicates the given string
    template <typename T> inline T* Duplicate(const T* str)
    {
        DEBUG_ASSERT(str, "null string was given");
        unsigned size = GetZStringLength(str) + 1;
        T* r = static_cast<T*>(::malloc(size*sizeof(T)));
        NMemory::TCopy<T>(r, str, size);
        return r;
    }
}
//---------------------------------------------------------------------------
// String template
template <typename T> class TString
{
public:
    // Default constructor for an empty string
    inline TString()
        : m_text(NULL)
        , m_length(0)
    {}

    // Copy constructor
    inline TString(const TString<T>& src)
    {
        if (src.m_length > 0)
        {
            m_text = new T[src.m_length + 1];
            m_length = src.m_length;
            ::memcpy(m_text, src.m_text, (src.m_length + 1)*sizeof(T));
        }
        else
        {
            m_text = NULL;
            m_length = 0;
        }
    }

    // Constructor from a single character
    explicit inline TString(T ch)
    {
        m_text = new T[2];
        m_text[0] = ch;
        m_text[1] = '\0';
        m_length = 1;
    }

    // Constructor from a string literal
    inline TString(const T* src)
    {
        DEBUG_ASSERT(src, "null string was given");
        if (src[0])
        {
            m_length = NStringUtilities::GetZStringLength(src);
            m_text = new T[m_length + 1];
            ::memcpy(m_text, src, (m_length + 1)*sizeof(T));
        }
        else
        {
            m_length = 0;
            m_text = NULL;
        }
    }

    // Construct from buffer and length
    inline TString(const T* src, unsigned length)
        : m_length(length)
    {
        DEBUG_ASSERT(src, "null string was given");
        m_text = new T[length + 1];
        ::memcpy(m_text, src, length*sizeof(T));
        m_text[length] = '\0';
    }

    // Destructor
    inline ~TString()
    {
        delete[] m_text;
    }

    // Returns the string's length in characters
    inline unsigned GetLength() const { return m_length; }

    // Returns the string's size in bytes
    inline unsigned GetDataSize() const { return m_length*sizeof(T); }

    // Returns true if this is an empty string
    inline bool IsEmpty() const { return m_length == 0; }

    // Returns the string's text buffer
    inline const T* GetText() const
    {
        return m_text ? m_text : NStringUtilities::GetEmptyString(m_text);
    }

    // Returns the string's text buffer as a generic data buffer
    inline void* GetData()
    {
        return m_text;
    }

    // Returns the string's text buffer as a generic data buffer
    inline const void* GetData() const
    {
        return m_text;
    }

    // Clear the string (equivalent, but faster, to assigning to an empty string)
    inline void Clear()
    {
        delete m_text;
        m_length = 0;
        m_text = NULL;
    }

    // Returns a substring starting from the left side of this string up to
    // the given number of characters
    inline TString<T> GetLeftSubstring(unsigned count) const
    {
        if (count >= m_length)
        {
            return *this;
        }
        else
        {
            TString<T> result = *this;
            result.m_text[count] = '\0';
            result.m_length = count;
            return result;
        }
    }

    // Returns the substring in the given character range with the second being
    // exclusive (ie. not included)
    inline TString<T> GetSubstring(unsigned first, unsigned second) const
    {
        // Clamp and empty string cases
        if (first > m_length) return TString<T>();
        if (second > m_length) second = m_length;
        if (first >= second) return TString<T>();
        if (first == 0 && second == m_length) return *this;
        if (first == 0) return GetLeftSubstring(second);
        // Create substring
        TString<T> result;
        result.m_length = second - first;
        result.m_text = new T[result.m_length + 1];
        for (unsigned i = first; i < second; ++i)
        {
            result.m_text[i - first] = m_text[i];
        }
        result.m_text[result.m_length] = '\0';
        return result;
    }

    // Splits the string to tokens using the given separator.  If skipMultiple
    // is true, multiple instances of a single separator will be treated as one
    inline TDynArray< TString<T> > Split(T separator, bool skipMultiple=false) const
    {
        TDynArray< TString<T> > tokens;
        tokens.Grow();
        for (unsigned i = 0; i < m_length; ++i)
        {
            if (m_text[i] == separator)
            {
                if (skipMultiple)
                {
                    while (i + 1 < m_length && m_text[i + 1] == separator) ++i;
                }
                tokens.Grow();
            }
            else
            {
                tokens.Last() += m_text[i];
            }
        }
        return tokens;
    }

    // Returns a copy of this string with the given character removed from the
    // beginning (if beginning is true) and end (if end is true)
    inline TString<T> GetTrimmed(T ch=' ', bool beginning=true, bool end=true) const
    {
        // Special case where both are false
        if (!beginning && !end)
        {
            return *this;
        }
        // Find the beginning
        unsigned beginIndex = 0;
        if (beginning)
        {
            for (;beginIndex < m_length && m_text[beginIndex] == ch;
                  ++beginIndex);
        }
        // Find the end
        unsigned endIndex=m_length;
        if (end)
        {
            for (;endIndex > beginIndex && m_text[endIndex - 1] == ch;
                  --endIndex);
        }
        // If indices are the same, just return an empty string
        if (beginIndex == endIndex)
        {
            return NStringUtilities::GetEmptyString(m_text);
        }
        // Return the substring
        return GetSubstring(beginIndex, endIndex);
    }
    
    // Trim the string at the given position - if the position is longer than
    // the string itself, ignore the call
    inline void TrimAt(unsigned index)
    {
        if (index < m_length)
        {
            m_length = index;
            m_text[index] = '\0';
        }
    }

    // Returns the string trimmed at the given position similar to TrimAt
    inline TString<T> GetTrimmedAt(unsigned index) const
    {
        TString<T> result = *this;
        result.TrimAt(index);
        return result;
    }

    // Trim the string by count characters from the end
    inline void TrimFromEnd(unsigned count)
    {
        if (count >= m_length)
        {
            delete m_text;
            m_text = NULL;
            m_length = 0;
        }
        else if (count > 0)
        {
            TrimAt(m_length - count);
        }
    }

    // Returns the string trimmed by count characters from the end
    inline TString<T> GetTrimmedFromEnd(unsigned count) const
    {
        TString<T> result = *this;
        result.TrimFromEnd(count);
        return result;
    }

    // Returns a copy of this string with all occurences of the first character
    // replace with the second character
    inline TString<T> GetReplaced(T first, T second) const
    {
        TString<T> result = *this;
        for (unsigned i=0; i < m_length; ++i)
        {
            if (result.m_text[i] == first) result.m_text[i] = second;
        }
        return result;
    }

    // Returns true if this string begins with the given string
    inline bool BeginsWith(const TString<T>& str) const
    {
        return str.m_length <= m_length && ::memcmp(m_text, str.m_text, str.m_length*sizeof(T)) == 0;
    }
    // Returns true if this string begins with the given string
    inline bool BeginsWith(const T* str) const
    {
        DEBUG_ASSERT(str != NULL, "Null string pointer passed in BeginsWith");
        unsigned length = NStringUtilities::GetZStringLength(str);
        return length <= m_length && ::memcmp(m_text, str, length*sizeof(T)) == 0;
    }
    // Returns true if this string begins with the given character
    inline bool BeginsWith(T ch) const
    {
        return m_length >= 1 && m_text[0] == ch;
    }

    // Returns true if this string ends with the given string
    inline bool EndsWith(const TString<T>& str) const
    {
        return str.m_length <= m_length && ::memcmp(m_text + (m_length - str.m_length), str.m_text, str.m_length*sizeof(T)) == 0;
    }
    // Returns true if this string ends with the given string
    inline bool EndsWith(const T* str) const
    {
        DEBUG_ASSERT(str != NULL, "Null string pointer passed in EndsWith");
        unsigned length = NStringUtilities::GetZStringLength(str);
        return length <= m_length && ::memcmp(m_text + (m_length - length), str, length*sizeof(T)) == 0;
    }
    // Returns true if this string ends with the given character
    inline bool EndsWith(T ch) const
    {
        return m_length >= 1 && m_text[m_length - 1] == ch;
    }

    // Tries to find the first index of the given character, returns false if there isn't such character in the string
    inline bool FindFirstCharacter(T ch, unsigned& index, unsigned start=0) const
    {
        for (unsigned i = start; i < GetLength(); ++i)
        {
            if (m_text[i] == ch)
            {
                index = i;
                return true;
            }
        }
        return false;
    }

    // Tries to find the last index of the given character, returns false if there isn't such character in the string
    inline bool FindLastCharacter(T ch, unsigned& index) const
    {
        for (int i = static_cast<int>(GetLength()) - 1; i >= 0; --i)
        {
            if (m_text[i] == ch)
            {
                index = static_cast<unsigned>(i);
                return true;
            }
        }
        return false;
    }

    // Returns true if the string contains the given characer
    inline bool Contains(T ch) const
    {
        for (unsigned i = 0; i < GetLength(); ++i)
        {
            if (m_text[i] == ch) return true;
        }
        return false;
    }

    // Returns the string converted to lower case (latin only)
    inline TString<T> ToLower() const
    {
        TString<T> result;
        result.m_text = new T[m_length + 1];
        result.m_length = m_length;
        for (unsigned i = 0; i < GetLength(); ++i)
        {
            if (m_text[i] >= 'A' && m_text[i] >= 'Z')
            {
                result.m_text[i] = (T)('a' + m_text[i] - 'A');
            }
            else
            {
                result.m_text[i] = m_text[i];
            }
        }
        result.m_text[m_length] = '\0';
        return result;
    }

    // Returns the string converted to upper case (latin only)
    inline TString<T> ToUpper() const
    {
        TString<T> result;
        result.m_text = new T[m_length + 1];
        result.m_length = m_length;
        for (unsigned i = 0; i < GetLength(); ++i)
        {
            if (m_text[i] >= 'a' && m_text[i] >= 'z')
            {
                result.m_text[i] = (T)('A' + m_text[i] - 'a');
            }
            else
            {
                result.m_text[i] = m_text[i];
            }
        }
        result.m_text[m_length] = '\0';
        return result;
    }

    // Compare the given string ignoring case (for latin only)
    inline bool CompareNCWith(const TString<T>& rhs) const
    {
        if (m_length == rhs.m_length)
        {
            for (unsigned i = 0; i < GetLength(); ++i)
            {
                if (m_text[i] >= 'a' && m_text[i] <= 'z')
                {
                    if (rhs.m_text[i] >= 'a' && rhs.m_text[i] <= 'z')
                    {
                        if (m_text[i] != rhs.m_text[i])
                            return false;
                    }
                    else if (rhs.m_text[i] >= 'A' && rhs.m_text[i] <= 'Z')
                    {
                        if (m_text[i] != rhs.m_text[i] - 'A' + 'a')
                            return false;
                    }
                    else return false;
                }
                else if (m_text[i] >= 'A' && m_text[i] <= 'Z')
                {
                    if (rhs.m_text[i] >= 'A' && rhs.m_text[i] <= 'Z')
                    {
                        if (m_text[i] != rhs.m_text[i])
                            return false;
                    }
                    else if (rhs.m_text[i] >= 'a' && rhs.m_text[i] <= 'z')
                    {
                        if (m_text[i] != rhs.m_text[i] - 'a' + 'A')
                            return false;
                    }
                    else return false;
                }
                else if (m_text[i] != rhs.m_text[i]) return false;
            }
            return true;
        }
        return false;
    }

    // Assignment operator
    inline TString<T>& operator=(const TString<T>& src)
    {
        if (&src != this)
        {
            // Kill existing buffer
            delete[] m_text;

            // Allocate new buffer
            if (src.m_length > 0)
            {
                m_text = new T[src.m_length + 1];
                m_length = src.m_length;
                ::memcpy(m_text, src.m_text, (src.m_length + 1)*sizeof(T));
            }
            else
            {
                m_text = NULL;
                m_length = 0;
            }
        }
        return *this;
    }

    // Append operator
    inline TString<T>& operator+=(const TString<T>& src)
    {
        if (src.m_length > 0)
        {
            T* newText = new T[m_length + src.m_length + 1];
            ::memcpy(newText, m_text, m_length*sizeof(T));
            ::memcpy(newText + m_length, src.m_text, (src.m_length + 1)*sizeof(T));
            delete[] m_text;
            m_text = newText;
            m_length += src.m_length;
        }
        return *this;
    }

    // Append operator
    inline TString<T>& operator+=(T ch)
    {
        T* newText = new T[m_length + 2];
        ::memcpy(newText, m_text, m_length*sizeof(T));
        newText[m_length++] = ch;
        newText[m_length] = '\0';
        delete[] m_text;
        m_text = newText;
        return *this;
    }

    // Add operator
    inline TString<T> operator+(const TString<T>& src) const
    {
        TString<T> newStr = *this;
        newStr += src;
        return newStr;
    }

    // Element operator
    inline T& operator[](const unsigned index)
    {
        DEBUG_ASSERT(index < GetLength(), "Invalid index");
        return m_text[index];
    }

    // Element operator
    inline T operator[](const unsigned index) const
    {
        DEBUG_ASSERT(index < GetLength(), "Invalid index");
        return m_text[index];
    }

    // Equality operator
    inline bool operator==(const TString<T>& other) const
    {
        return m_length == other.m_length && ::memcmp(m_text, other.m_text, m_length*sizeof(T)) == 0;
    }
    inline bool operator==(const T* other) const
    {
        if (other)
        {
            unsigned i;
            for (i=0; i < m_length; ++i)
            {
                if (other[i] != m_text[i]) return false;
            }
            return other[i] == '\0';
        }
        return false;
    }

    // Inequality operator
    inline bool operator!=(const TString<T>& other) const
    {
        return m_length != other.m_length || ::memcmp(m_text, other.m_text, m_length*sizeof(T)) != 0;
    }
    inline bool operator!=(const T* other) const
    {
        return !(operator==(other));
    }

    // Wrap the given buffer and length to the given string
    static void WrapTo(TString<T>& target, T* text, unsigned length)
    {
        DEBUG_ASSERT(target.m_text == NULL, "tried to wrap to an already used string");
        DEBUG_ASSERT(target.m_length == 0, "tried to wrap to an already used string");
        target.m_text = text;
        target.m_length = length;
    }

    // Convert the given value to a string
    static inline TString<T> FromInt(int i)
    {
        bool neg = i < 0;
        if (neg) i = -i;
        T buff[64];
        unsigned offset = 63;
        buff[63] = '\0';
        while (true)
        {
            buff[--offset] = (char)('0' + (i%10));
            i /= 10;
            if (offset == 0 || i == 0) break;
        }
        if (neg)
        {
            TString<T> result('-');
            result += TString<T>(buff + offset);
            return result;
        }
        return buff + offset;
    }

    // Convert the given value to a string
    static inline TString<T> FromUnsigned(unsigned i)
    {
        T buff[64];
        unsigned offset = 63;
        buff[63] = '\0';
        while (true)
        {
            buff[--offset] = (char)('0' + (i%10));
            i /= 10;
            if (offset == 0 || i == 0) break;
        }
        return TString<T>(buff + offset);
    }
    
    // Empty string constant
    static const TString<T> EMPTY;

private:
    T*          m_text;     // Text buffer
    unsigned    m_length;   // String length
};
//---------------------------------------------------------------------------
// Narrow character string
typedef TString<char>       AString;
// Wide character string
typedef TString<wchar_t>    AWString;
// Narrow character string array
typedef TDynArray<AString>  AStringArray;
// Wide character string array
typedef TDynArray<AWString> AWStringArray;
//---------------------------------------------------------------------------
// Add operator for native string
inline AString operator+(const char* as, const AString& s)
{
    return AString(as) + s;
}
//---------------------------------------------------------------------------
// Add operator for native wide string
inline AWString operator+(const wchar_t* as, const AWString& s)
{
    return AWString(as) + s;
}
//---------------------------------------------------------------------------
// Buffered string builder
#define STRING_BUILDER_CAPACITY_INCREASE    65536
template <typename T> class TStringBuilder
{
public:
    inline TStringBuilder()
        : m_text(NULL)
        , m_length(0)
        , m_capacity(0)
    {}
    inline TStringBuilder(const TString<T>& base)
    {
        m_length = base.GetLength();
        m_capacity = m_length + 1;
        m_text = new T[m_length];
        ::memcpy(m_text, base.GetText(), m_length + 1);
    }
    inline ~TStringBuilder()
    {
        delete m_text;
    }

    // Get the text in the builder
    inline const T* GetText() const { return m_text ? m_text : NStringUtilities::GetEmptyString(m_text); }
    // Get the text's length
    inline unsigned GetLength() const { return m_length; }
    // Move the text into a new string (string must be empty)
    inline void MoveTo(TString<T>& target)
    {
        TString<T>::WrapTo(target, m_text, m_length);
        m_text = NULL;
        m_length = m_capacity = 0;
    }

    // Add a single character
    inline TStringBuilder& operator+=(T ch)
    {
        Expand(2);
        m_text[m_length++] = ch;
        m_text[m_length] = '\0';
        return *this;
    }
    // Add a C string
    inline TStringBuilder& operator+=(const T* src)
    {
        unsigned length = NStringUtilities::GetZStringLength(src);
        if (length > 0)
        {
            Expand(length + 1);
            ::memcpy(m_text + m_length, src, (length + 1)*sizeof(T));
            m_length += length;
        }
        return *this;
    }
    // Add a string
    inline TStringBuilder& operator+=(const TString<T>& src)
    {
        if (src.GetLength() > 0)
        {
            Expand(src.GetLength() + 1);
            ::memcpy(m_text + m_length, src.GetText(), (src.GetLength() + 1)*sizeof(T));
            m_length += src.GetLength();
        }
        return *this;
    }

private:
    T*      m_text;     // Text buffer
    unsigned  m_length;   // String length
    unsigned  m_capacity; // Buffer capacity
    // Expand the buffer by the given size
    inline void Expand(unsigned chars)
    {
        // Check if we need to reallocate
        if (m_length + chars > m_capacity)
        {
            m_capacity += chars + STRING_BUILDER_CAPACITY_INCREASE;
            T* newText = new T[m_capacity];
            ::memcpy(newText, m_text, m_length*sizeof(T));
            delete[] m_text;
            m_text = newText;            
        }
    }
};
//---------------------------------------------------------------------------
// Narrow character string builder
typedef TStringBuilder<char>     AStringBuilder;
// Wide character string builder
typedef TStringBuilder<wchar_t>  AWStringBuilder;
//---------------------------------------------------------------------------
// Helper string functions
namespace NString
{
    template <typename T> TString<T> TRepeat(T ch, unsigned count)
    {
        T* buff = new T[count + 1];
        for (unsigned i=0; i < count; ++i) buff[i] = ch;
        buff[count] = '\0';
        TString<T> result;
        TString<T>::WrapTo(result, buff, count);
        return result;
    }
}
//---------------------------------------------------------------------------
// String hash function
namespace NHash
{
    template <typename T> inline unsigned Func(const TString<T>& str)
    {
        return NHash::Raw((const char*)str.GetText(), sizeof(T)*str.GetLength());
    }
}
//---------------------------------------------------------------------------
#endif

