//---------------------------------------------------------------------------
#ifndef RunnablesH
#define RunnablesH
//---------------------------------------------------------------------------
#include <DynArray.h>
//---------------------------------------------------------------------------
// Interface for a "runnable"
class IRunnable
{
public:
    virtual ~IRunnable(){}
    // Called to run
    virtual void Run(){}
    // Function call operator just calls Run
    inline void operator()() { Run(); }
};
//---------------------------------------------------------------------------
// A runnable that calls zero or more other runnables
class CRunGroup : public IRunnable
{
public:
    // Clear the group
    inline void Clear()
    {
        m_runs.Clear();
    }
    // Delete all runnables in the group
    inline void DeleteAll()
    {
        for (unsigned i=0; i < m_runs.GetSize(); ++i) delete m_runs[i];
        m_runs.Clear();
    }
    // Add a new runnable
    inline void Add(IRunnable* run)
    {
        m_runs.Add(run);
    }
    // Add a new runnable only if it isn't already part of the group
    inline void AddUnique(IRunnable* run)
    {
        if (!Contains(run)) Add(run);
    }
    // Remove the given runnable
    inline void Remove(IRunnable* run)
    {
        m_runs.Remove(run);
    }
    // Returns true if this group contains the given runnable
    inline bool Contains(IRunnable* run) const
    {
        return m_runs.Contains(run);
    }

    // Returns all runnables in this group
    inline const TDynArray<IRunnable*>& GetRunnables() const { return m_runs; }
protected:
    TDynArray<IRunnable*>   m_runs;
public:
    // IRunnable
    virtual void Run()
    {
        for (unsigned i=0; i < m_runs.GetSize(); ++i) m_runs[i]->Run();
    }
};
//---------------------------------------------------------------------------
// A runnable group that runs its runnables only once and then deletes them.
// If a runnable is added while Run is running, they'll be available for the
// next call to Run.  This group takes ownership of the runnables until the
// Run method is called (when they are deleted) or the runnable is removed
class CRunOnce : public CRunGroup
{
public:
    virtual ~CRunOnce()
    {
        DeleteAll();
    }
public:
    // IRunnable
    virtual void Run()
    {
        TDynArray<IRunnable*> runs;
        runs.Exchange(m_runs);
        for (unsigned i=0; i < runs.GetSize(); ++i)
        {
            runs[i]->Run();
            delete runs[i];
        }
    }
};
//---------------------------------------------------------------------------
// Contains functions to register runnables globally that will be run at some
// later point in time (actually when the RunNow function is called, which is
// really meant to be called at the main idle loop of a program)
namespace NRunLater
{
    extern CRunOnce GRunOnceLater;      // The group to run later

    // Add a runnable to run later
    void Add(IRunnable* run);
    // Remove the given runnable from the queue
    void Remove(IRunnable* run);
    // Run all runnables and clear the queue
    void RunNow();
}
//---------------------------------------------------------------------------
#endif
