//---------------------------------------------------------------------------
#ifndef RefObjectH
#define RefObjectH
//---------------------------------------------------------------------------
#include <stdlib.h>
#include "DynArray.h"
//---------------------------------------------------------------------------
// Base for object references
class CObjectReferenceBaseHeader
{
    friend class CRefObjBase;
public:
    virtual ~CObjectReferenceBaseHeader(){}
    // Clear this reference
    virtual void Clear()=0;
protected:
    // Invalidate this reference
    virtual void Invalidate()=0;
};
//---------------------------------------------------------------------------
// Referencable object base class (use TRefObject instead)
class CRefObjBase
{
    friend class CObjectReferenceBase;
public:
    // Return reference count
    inline int GetReferenceCount() const
    {
        return m_refs.GetSize();
    }
protected:
    TDynArray<CObjectReferenceBaseHeader*>  m_refs; // References
    void*                                   m_obj;  // Target object

    CRefObjBase()
        : m_obj(NULL)
    {
    }
    CRefObjBase(void* obj)
        : m_obj(obj)
    {
    }
    // Destroys the object
    virtual void Destroy()=0;
    // Clear any remainding references
    inline void InvalidateRefs()
    {
        for (size_t i=0; i < m_refs.GetSize(); ++i)
            m_refs[i]->Invalidate();
    }
};
//---------------------------------------------------------------------------
// Base for templated object references
class CObjectReferenceBase : public CObjectReferenceBaseHeader
{
public:
    CObjectReferenceBase()
        : m_obj(NULL)
    {
    }
    virtual void Clear()
    {
        Release();
    }
protected:
    CRefObjBase*    m_obj;  // Referenceable object

    inline void* GetTarget() const
    {
        DEBUG_ASSERT(m_obj, "null object");
        return m_obj->m_obj;
    }

    // Invalidate this reference
    virtual void Invalidate()
    {
        m_obj = NULL;
    }
    // Assign this reference from the given object
    inline void SetObject(CRefObjBase* obj)
    {
        if (m_obj != obj)
        {
            if (obj) obj->m_refs.Add(this);
            Release();
            m_obj = obj;
        }
    }
    // Assign this reference from the given reference
    inline void Assign(const CObjectReferenceBase* ref)
    {
        if (ref != this && ref->m_obj != m_obj)
        {
            if (ref->m_obj) ref->m_obj->m_refs.Add(this);
            Release();
            m_obj = ref->m_obj;
        }
    }
    // Release the reference
    virtual void Release()
    {
        if (m_obj)
        {
            m_obj->m_refs.Remove(this);
            m_obj = NULL;
        }
    }
};
//---------------------------------------------------------------------------
// Referencable object, all references to this object will become references
// to the target objects stored inside
template <typename T> class TRefObject : public CRefObjBase
{
public:
    TRefObject() : CRefObjBase()
    {
    }
    TRefObject(const TRefObject<T>& ref)
    {
        DEBUG_FAIL("Cannot assign a referenceable object");
    }
    TRefObject(const CRefObjBase& ref)
        : CRefObjBase()
    {
        DEBUG_FAIL("Cannot assign a referenceable object");
    }
    virtual ~TRefObject()
    {
        InvalidateRefs();
        Destroy();
    }
    // Set the object
    inline void Set(T* obj)
    {
        m_obj = obj;
    }
    // Returns the object
    inline const T* Get() const { return static_cast<T*>(m_obj); }
    inline T* Get() { return static_cast<T*>(m_obj); }
    // Block assignment
    inline TRefObject<T>& operator=(const TRefObject<T>& ref)
    {
        DEBUG_FAIL("Cannot assign a referenceable object");
        return *this;
    }
protected:
    virtual void Destroy()
    {
        delete (T*)m_obj;
        m_obj = NULL;
    }
};
//---------------------------------------------------------------------------
// Strong object reference to TRefObject
template <typename T> class TRef : public CObjectReferenceBase
{
public:
    TRef() : CObjectReferenceBase() {}
    TRef(TRefObject<T>& obj)
        : CObjectReferenceBase()
    {
        SetObject(&obj);
    }
    TRef(const CObjectReferenceBase& ref)
        : CObjectReferenceBase()
    {
        Assign(&ref);
    }
    virtual ~TRef()
    {
        Release();
    }
    // Returns true if the target object is null
    inline bool IsNull() const { return !(m_obj && GetTarget()); }
    // Returns true if the target is a valid (non-null) object
    inline bool IsValid() const { return m_obj && GetTarget(); }
    // Returns the target object
    inline const T* Get() const { return static_cast<T*>(GetTarget()); }
    inline T* Get() { return static_cast<T*>(GetTarget()); }
    inline const T& operator*() const { DEBUG_ASSERT(IsValid(), "invalid object access"); return *static_cast<T*>(GetTarget()); }
    inline T& operator*() { DEBUG_ASSERT(IsValid(), "invalid object access"); return *static_cast<T*>(GetTarget()); }
    inline const T* operator->() const { DEBUG_ASSERT(IsValid(), "invalid object access"); return static_cast<T*>(GetTarget()); }
    inline T* operator->() { DEBUG_ASSERT(IsValid(), "invalid object access"); return static_cast<T*>(GetTarget()); }
    // Assign another object
    inline TRef<T>& operator=(TRefObject<T>& obj)
    {
        SetObject(&obj);
        return *this;
    }
    // Assign an object from another referenceable
    inline TRef<T>& operator=(const CObjectReferenceBase& ref)
    {
        Assign(&ref);
        return *this;
    }
};
#endif
