#ifndef PolygonH
#define PolygonH

#include "Plane.h"

// Represents a flat polygon
template <typename T> class TPolygon
{
public:
    // Empty polygon
    inline TPolygon() {}

    // Copy constructor
    inline TPolygon(const TPolygon<T>& rhs)
    {
        operator=(rhs);
    }

    // Polygon from triangle
    inline TPolygon(const TVector<T>& a, const TVector<T>& b, const TVector<T>& c)
    {
        Add(a);
        Add(b);
        Add(c);
    }

    // Polygon from quad
    inline TPolygon(const TVector<T>& a, const TVector<T>& b, const TVector<T>& c, const TVector<T>& d)
    {
        Add(a);
        Add(b);
        Add(c);
        Add(d);
    }

    // Clear the polygon
    inline void Clear()
    {
        m_vertices.Clear();
    }

    // Add a new point to the polygon
    inline void Add(const TVector<T>& point)
    {
        m_vertices.Add(point);
    }

    // Calculate the plane of this polygon, returns false if the plane cannot be calculated
    inline bool CalcPlane(TPlane<T>& plane) const
    {
        if (m_vertices.GetSize() > 2)
        {
            plane = TPlane<T>(m_vertices[0], m_vertices[1], m_vertices[2]);
            return true;
        }
        return false;
    }

    // Returns this polygon clipped with the given plane
    TPolygon<T> Clipped(const TPlane<T>& plane) const
    {
        // Find the first vertex that is in front of the plane
        // and check if there are vertices behind the plane
        int ff = -1;
        bool hasBack = false;
        for (int i = 0; i < m_vertices.GetSignedSize(); ++i)
        {
            const T sd = NPlane::TSignedDistance(plane, m_vertices[i]);

            // Front?
            if (sd >= 0.0f)
            {
                // Set the first vertex in front of the plane
                if (ff == -1) ff = i;
            }
            // Back?
            else
            {
                hasBack = true;
            }

            // Stop the loop if we found vertices at the back and our first vertex
            if (ff != -1 && hasBack) break;
        }

        // No vertices at the front, return an empty polygon
        if (ff == -1) return TPolygon<T>();

        // No vertices at the back, return a copy of this polygon
        if (!hasBack) return *this;

        // Construct a new polygon clipped with the plane
        TPolygon<T> clipped;
        int prev = ff, idx = (ff + 1) % m_vertices.GetSize();
        clipped.Add(m_vertices[ff]);
        while (true)
        {
            TVector<T> ip;
            if (NPlane::TSegmentIntersection(plane, TSegment<T>(m_vertices[prev], m_vertices[idx]), ip))
                clipped.Add(ip);
            if (idx == ff) break;
            if (NPlane::TSignedDistance(plane, m_vertices[idx]) >= 0.0f)
                clipped.Add(m_vertices[idx]);
            prev = idx;
            idx = (idx + 1) % m_vertices.GetSize();
        }
        return clipped;
    }

    // Clip the polygon using the given plane
    void Clip(const TPlane<T>& plane)
    {
        operator=(Clipped(plane));
    }

    // Merges zero-length edges by removing adjacent overlapping points
    inline void MergeZeroLengthEdges()
    {
        for (int i = m_vertices.GetSignedSize() - 1; i > 0; --i)
        {
            if (NVector::TDistanceSquared3D(m_vertices[i], m_vertices[i - 1]) < 0.00001f)
            {
                m_vertices.RemoveAt(i);
            }
        }
    }

    // Returns the vertices of this polygon
    inline const TVector<T>::TAVectorArray& GetVertices() const { return m_vertices; }

    // Returns the vertex count of this polygon
    inline unsigned GetCount() const { return m_vertices.GetSize(); }

    // Vertex access operator
    inline const TVector<T>& operator[](unsigned idx) const { return m_vertices[idx]; }
    inline TVector<T>& operator[](unsigned idx) { return m_vertices[idx]; }

    // Assignment operator
    inline TPolygon<T>& operator=(const TPolygon<T>& rhs)
    {
        if (this != &rhs)
        {
            m_vertices = rhs.m_vertices;
        }
        return *this;
    }

    // Equality operator
    inline bool operator==(const TPolygon<T>& rhs) const
    {
        return m_vertices == rhs.m_vertices;
    }

private:
    TVector<T>::TAVectorArray   m_vertices;     // Polygon vertices
};

// Float polygon
typedef TPolygon<float> CPolygon;

#endif

