//---------------------------------------------------------------------------
#ifndef PlaneH
#define PlaneH
//---------------------------------------------------------------------------
#include "Segment.h"
//---------------------------------------------------------------------------
// Represents a place - X, Y, Z represent the normal and W the distance from origin
template <typename T> class TPlane : public TVector<T>
{
public:
    // Undefined plane
    inline TPlane() {}

    // Plane from vector
    inline TPlane(const TVector<T>& v)
        : TVector<T>(v)
    {
    }

    // Plane with the given values
    inline TPlane(T normalX, T normalY, T normalZ, T distance)
        : TVector<T>(normalX, normalY, normalZ, distance)
    {
    }

    // Plane from point and normal
    inline TPlane(const TVector<T>& point, const TVector<T>& normal)
    {
        CopyXYZ(normal);
        W = point.GetLength3D()*NVector::TDot3D(normal, point.GetNormalized3D());
    }

    // Plane from three points
    inline TPlane(const TVector<T>& a, const TVector<T>& b, const TVector<T>& c)
    {
        CopyXYZ(NVector::TCross3D(b - a, c - a));
        Normalize3D();
        W = NVector::TDot3D(*this, a);
    }
};
//---------------------------------------------------------------------------
// Float plane
typedef TPlane<float>   APlane;
//---------------------------------------------------------------------------
// Plane functions
namespace NPlane
{
    // Convert plane types
    template <typename T> inline TPlane<float> ToSingle(const TPlane<T>& src)
    {
        return TPlane<float>(src.X, src.Y, src.Z, src.W);
    }
    template <typename T> inline TPlane<double> ToDouble(const TPlane<T>& src)
    {
        return TPlane<double>(src.X, src.Y, src.Z, src.W);
    }

    // Returns the signed distance of the point to the plane (positive: front, negative: back)
    template <typename T> inline T TSignedDistance(const TPlane<T>& plane, const TVector<T>& point)
    {
        return NVector::TDot3D(point - plane.XYZ()*plane.W, plane);
    }

    // Returns the point projected to the plane
    template <typename T> inline TVector<T> TProjected(const TPlane<T>& plane, const TVector<T>& point)
    {
        return point - plane.XYZ()*TSignedDistance<T>(plane, point);
    }

    // Calculates the intersection point of a line segment and the plane
    // returns false if there is no intersection
    template <typename T> inline bool TSegmentIntersection(const TPlane<T>& plane, const TSegment<T>& seg, TVector<T>& ip, const T epsilon=0.00001f)
    {
        const TVector<T> u = seg.B - seg.A;
        const TVector<T> w = seg.A - plane.XYZ()*plane.W;
        const T vd = NVector::TDot3D(plane, u);
        const T vn = -NVector::TDot3D(plane, w);
        if (fabs(vd) > epsilon)
        {
            const T si = vn / vd;
            if (si >= 0.0f && si <= 1.0f)
            {
                ip = seg.A + u*si;
                return true;
            }
        }
        return false;
    }
    template <typename T> inline bool TSegmentIntersection(const TPlane<T>& plane, const TSegment<T>& seg, const T epsilon = 0.00001f)
    {
        const TVector<T> u = seg.B - seg.A;
        const TVector<T> w = seg.a - plane.XYZ()*plane.W;
        const T vd = NVector::TDot3D<T>(plane, u);
        const T vn = -NVector::TDot3D<T>(plane, w);
        if (fabs(vd) > epsilon)
        {
            const T si = vn / vd;
            if (si >= 0.0f && si <= 1.0f)
            {
                return true;
            }
        }
        return false;
    }
}
//---------------------------------------------------------------------------
// Plane hash function
namespace NHash
{
    template <typename T> inline unsigned Func(const TPlane<T>& plane)
    {
        return NHash::Raw((const char*)&plane, sizeof(plane));
    }
}
//---------------------------------------------------------------------------
#endif

