//---------------------------------------------------------------------------
#include "Nodes.h"
//---------------------------------------------------------------------------
CNode::CNode()
    : m_parent(NULL)
{
}
//---------------------------------------------------------------------------
CNode::~CNode()
{
    DeleteChildren();
}
//---------------------------------------------------------------------------
bool CNode::AddChild(CNode* child)
{
    DEBUG_ASSERT(child, "null child was given");
    DEBUG_ASSERT(child != this, "cannot add a node to itself as a child");
    DEBUG_ASSERT(child->m_parent != this, "the given node is already a child of this node");
    if (!CanAddChild(child)) return false;
    if (child->m_parent) child->m_parent->RemoveChild(child);
    child->m_parent = this;
    m_children.Add(child);
    OnChildAdded(child);
    return true;
}
//---------------------------------------------------------------------------
void CNode::RemoveChild(CNode* child)
{
    DEBUG_ASSERT(child, "null child was given");
    DEBUG_ASSERT(child != this, "cannot remove a node from itself");
    DEBUG_ASSERT(child->m_parent == this, "the given node is not a child of this node");
    DEBUG_ASSERT(m_children.Contains(child), "the given node has this node as its parent but it is not in the children array");
    child->m_parent = this;
    m_children.Remove(child);
    OnChildRemoved(child);
}
//---------------------------------------------------------------------------
void CNode::DeleteChild(CNode* child)
{
    RemoveChild(child);
    delete child;
}
//---------------------------------------------------------------------------
void CNode::RemoveChildren()
{
    while (m_children.HasItems()) RemoveChild(m_children.Last());
}
//---------------------------------------------------------------------------
void CNode::DeleteChildren()
{
    while (m_children.HasItems()) DeleteChild(m_children.Last());
}
//---------------------------------------------------------------------------
bool CNode::IsAncestor(CNode* node, bool includeThis) const
{
    DEBUG_ASSERT(node, "null node was given");
    if (includeThis && node == this) return true;
    return m_parent ? m_parent->IsAncestor(node, true) : false;
}
//---------------------------------------------------------------------------
bool CNode::IsDescendant(CNode* node, bool includeThis) const
{
    DEBUG_ASSERT(node, "null node was given");
    if (includeThis && node == this) return true;
    for (unsigned i=0; i < m_children.GetSize(); ++i)
    {
        if (m_children[i]->IsDescendant(node, true)) return true;
    }
    return false;
}
//---------------------------------------------------------------------------
CNode* CNode::FindChild(const AString& name) const
{
    DEBUG_ASSERT(!name.IsEmpty(), "empty name was given");
    for (unsigned i=0; i < m_children.GetSize(); ++i)
    {
        if (m_children[i]->GetName() == name) return m_children[i];
    }
    return NULL;
}
//---------------------------------------------------------------------------
void CNode::FindChildren(TDynArray<CNode*>& result, const AString& name, int max) const
{
    DEBUG_ASSERT(!name.IsEmpty(), "empty name was given");
    for (unsigned i=0; i < m_children.GetSize() && max != 0; ++i)
    {
        if (m_children[i]->GetName() == name)
        {
            result.Add(m_children[i]);
            max--;
        }
    }
}
//---------------------------------------------------------------------------
CNode* CNode::GetChildByPath(const AString& path) const
{
    unsigned dotpos;
    DEBUG_ASSERT(!path.IsEmpty(), "empty path was given");
    if (path.FindFirstCharacter('.', dotpos))
    {
        if (dotpos == 0 || dotpos + 1 == path.GetLength()) return NULL;
        CNode* child = FindChild(path.GetSubstring(0, dotpos));
        if (!child) return NULL;
        return child->GetChildByPath(path.GetSubstring(dotpos + 1, path.GetLength()));
    }
    else
    {
        return FindChild(path);
    }
}
//---------------------------------------------------------------------------
bool CNode::GetPath(AString& path, CNode* ancestor) const
{
    const CNode* node = this;
    path.Clear();
    for (;;)
    {
        if (!node->HasName()) return false;
        if (node == ancestor || !node) return !path.IsEmpty();
        if (path.IsEmpty())
        {
            path = node->GetName();
        }
        else
        {
            path = node->GetName() + "." + path;
        }
        node = node->GetParent();
    }
}
//---------------------------------------------------------------------------
void CNode::GetAllDescendants(TDynArray<CNode*>& result, bool includeThis, int max) const
{
    if (includeThis)
    {
        result.Add((CNode*)this);
        max--;
    }
    while (max != 0)
    {
        for (unsigned i=0; i < m_children.GetSize(); ++i)
        {
            unsigned prevSize = result.GetSize();
            GetAllDescendants(result, true, max);
            if (max > 0)
            {
                int diff = (int)(prevSize - result.GetSize());
                max -= diff;
                if (max <= 0) return;
            }
        }
    }
}
//---------------------------------------------------------------------------

