//---------------------------------------------------------------------------
#ifndef KeyValueTreeH
#define KeyValueTreeH
//---------------------------------------------------------------------------
#include "Strings.h"
//---------------------------------------------------------------------------
// Forward class declarations
class CKVNode;
//---------------------------------------------------------------------------
// An array of key-value nodes
typedef TDynArray<CKVNode*>     AKVNodeArray;
//---------------------------------------------------------------------------
// A key-value pair node, can have subnodes, etc
class CKVNode
{
public:
    CKVNode(const AString& name="", const AString& value="");
    ~CKVNode();
    // Set the node's name
    inline void SetName(const AString& name) { m_name = name; }
    // Set the node's value
    inline void SetValue(const AString& value) { m_value = value; }
    // Find a direct descendant with the given name
    CKVNode* DirectChildByName(const AString& name, bool createMissing=true);
    // Find a descendant with the given dot-separated path
    CKVNode* ChildByPath(const AString& path, bool createMissing=true);
    // Find a descendant with the given name and value
    CKVNode* ChildByNameAndValue(const AString& name, const AString& value);
    inline const CKVNode* ChildByNameAndValue(const AString& name, const AString& value) const
    {
        return const_cast<CKVNode*>(this)->ChildByNameAndValue(name, value);
    }
    // Find a descendant with the given child
    CKVNode* ChildByGrandchild(const AString& grandchildName, bool recursive=true);
    inline const CKVNode* ChildByGrandchild(const AString& grandchildName, bool recursive=true) const
    {
        return const_cast<CKVNode*>(this)->ChildByGrandchild(grandchildName, recursive);
    }
    // Find a descendant with the given child and value
    CKVNode* ChildByGrandchildAndValue(const AString& grandchildName, const AString& value, bool recursive=true);
    inline const CKVNode* ChildByGrandchildAndValue(const AString& grandchildName, const AString& value, bool recursive=true) const
    {
        return const_cast<CKVNode*>(this)->ChildByGrandchildAndValue(grandchildName, value, recursive);
    }
    // Create a new child with the given name and value
    CKVNode* CreateChild(const AString& name="", const AString& value="");
    // Set/change a child value (may create a new child node)
    void SetChildValue(const AString& path, const AString& value);
    // Retrieve a child value
    bool GetChildValue(const AString& path, AString& value) const;
    // Delete a child value
    bool DeleteChildValue(const AString& path);
    // Delete all children
    void DeleteAll();
    // Export this subnode to a string
    void Export(AString& target) const;
    // Import the nodes from the given string into this string
    bool Import(const AString& source, bool discardExisting=true);
    // Save the node's contents to a file
    bool WriteToFile(const AString& path) const;
    // Load the node's contents from a file
    bool LoadFromFile(const AString& path, bool discardExisting=true);
    // Return the node's name
    inline const AString& GetName() const { return m_name; }
    // Return the node's value
    inline const AString& GetValue() const { return m_value; }
    // Return the node's parent
    inline CKVNode* GetParent() const { return m_parent; }
    // Return the node's children
    inline const AKVNodeArray& GetChildren() const { return m_children; }
    // Return true if the node has children
    inline bool HasChildren() const { return m_children.HasItems(); }
private:
    AKVNodeArray    m_children; // Node children
    AString         m_name;     // Name of the node
    AString         m_value;    // Value of the node
    CKVNode*        m_parent;   // Parent node
    // Export node
    static void ExportNode(AStringBuilder& target, const CKVNode* node, unsigned depth);
    // Import node
    static bool ImportNode(const AString& source, CKVNode* node, unsigned& head);
    // Encode the given string for export
    static void EncodeString(const AString& str, AStringBuilder& builder);
    // Decode the given string for import
    static AString DecodeString(const AString& str, size_t& head);
};
//---------------------------------------------------------------------------
#endif
