//---------------------------------------------------------------------------
#ifndef HashMapH
#define HashMapH
//---------------------------------------------------------------------------
#include "HashFunc.h"
#include "Pair.h"
//---------------------------------------------------------------------------
// A hash map cell
template <typename K, typename V> struct THashMapCell
{
    TDynArray< TPair<K,V> > Entries;
};
//---------------------------------------------------------------------------
// A hash map
template <typename K, typename V, unsigned CellMask=0xFF, unsigned CellCount=256> struct THashMap
{
    THashMapCell<K,V>   Cells[CellCount];
    // Construct empty hash map
    inline THashMap(){}
    // Construct a copy of the given hashmap
    inline THashMap(const THashMap<K,V>& rhs)
    {
        if (rhs.CellCount == CellCount) // Fast case - the cell counts match
        {
            for (unsigned i=0; i < CellCount; ++i) Cells[i].Entries = rhs.Cells[i].Entries;
        }
        else    // Slow case - cell counts differ
        {
            for (unsigned i=0; i < rhs.CellCount; ++i)
            {
                for (unsigned j=0; j < rhs.Cells[i].Entries.GetSize(); ++j)
                {
                    Set(rhs.Cells[i].Entries[j].First, rhs.Cells[i].Entries[j].Second);
                }
            }
        }
    }
    // Clear the map
    inline void Clear()
    {
        for (unsigned i=0; i < CellCount; ++i) Cells[i].Entries.Clear();
    }
    // Reset the map (faster than clear but doesn't deallocate)
    inline void Reset()
    {
        for (unsigned i=0; i < CellCount; ++i) Cells[i].Entries.Reset();
    }
    // Put a value to the map
    inline void Set(const K& key, const V& value)
    {
        THashMapCell<K,V>*  Cell = &Cells[NHash::Func(key)&CellMask];
        // Find the entry for the key
        for (unsigned i=0; i < Cell->Entries.GetSize(); ++i)
        {
            if (Cell->Entries[i].First == key)
            {
                Cell->Entries[i].Second = value;
                return;
            }
        }
        // Add new entry
        Cell->Entries.Grow().First = key;
        Cell->Entries.Last().Second = value;
    }
    // Find the given key in the map, returns false if the key does not exist
    inline bool Find(const K& key, V& value) const
    {
        const THashMapCell<K,V>*    Cell = &Cells[NHash::Func(key)&CellMask];
        // Find the entry for the key
        for (unsigned i=0; i < Cell->Entries.GetSize(); ++i)
        {
            if (Cell->Entries[i].First == key)
            {
                value = Cell->Entries[i].Second;
                return true;
            }
        }
        // Entry not found
        return false;
    }
    // Check if the given key exists in the map
    inline bool Has(const K& key) const
    {
        const THashMapCell<K,V>*    Cell = &Cells[NHash::Func(key)&CellMask];
        // Find the entry for the key
        for (unsigned i=0; i < Cell->Entries.GetSize(); ++i)
        {
            if (Cell->Entries[i].First == key)
            {
                return true;
            }
        }
        // Entry not found
        return false;
    }
    // Copy all keys to the given array
    inline void GetKeys(TDynArray<K>& keys) const
    {
        for (unsigned i=0; i < CellCount; ++i)
        {
            for (unsigned j=0; j < Cells[i].Entries.GetSize(); ++j)
            {
                keys.Add(Cells[i].Entries[j].First);
            }
        }
    }
    // Copy all values to the given array
    inline void GetValues(TDynArray<V>& values) const
    {
        for (unsigned i=0; i < CellCount; ++i)
        {
            for (unsigned j=0; j < Cells[i].Entries.GetSize(); ++j)
            {
                values.Add(Cells[i].Entries[j].Second);
            }
        }
    }
    // Copy all key/value pairs to the given array
    inline void GetEntries(TDynArray< TPair<K,V> >& entries) const
    {
        for (unsigned i=0; i < CellCount; ++i)
        {
            entries.Add(Cells[i].Entries);
        }
    }
    // Operator based access, will add a new default value if it doesn't exist
    inline V& operator[](const K& key)
    {
        THashMapCell<K,V>*  Cell = &Cells[NHash::Func(key)&CellMask];
        // Find the entry for the key
        for (unsigned i=0; i < Cell->Entries.GetSize(); ++i)
        {
            if (Cell->Entries[i].First == key)
            {
                return Cell->Entries[i].Second;
            }
        }
        // Add new entry
        Cell->Entries.Grow().First = key;
        return Cell->Entries.Last().Second;
    }
    // Assign the given hashmap to this
    inline THashMap<K,V,CellMask,CellCount>& operator=(const THashMap<K,V>& rhs)
    {
        if ((void*)this == (void*)&rhs) return *this;
        if (rhs.CellCount == CellCount) // Fast case - the cell counts match
        {
            for (unsigned i=0; i < CellCount; ++i) Cells[i].Entries = rhs.Cells[i].Entries;
        }
        else    // Slow case - cell counts differ
        {
            Clear();
            for (unsigned i=0; i < rhs.CellCount; ++i)
            {
                for (unsigned j=0; j < rhs.Cells[i].Entries.GetSize(); ++j)
                {
                    Set(rhs.Cells[i].Entries[j].First, rhs.Cells[i].Entries[j].Second);
                }
            }
        }
        return *this;
    }
};
//---------------------------------------------------------------------------
#endif

