//---------------------------------------------------------------------------
#ifndef DocumentsH
#define DocumentsH
//---------------------------------------------------------------------------
#include "Strings.h"
//---------------------------------------------------------------------------
// Forward class declarations
class CDocView;
class CDocument;
class CDocumentManager;
//---------------------------------------------------------------------------
// Interface for document listeners
class IDocumentListener
{
public:
    virtual ~IDocumentListener(){}
    // Called after the document has been modified or marked as not modified
    virtual void OnDocumentModified(CDocument* doc, bool modified)=0;
    // Called after a view has been created
    virtual void OnViewCreated(CDocument* doc, CDocView* view)=0;
    // Called after a view has been destroyed
    virtual void OnViewDestroyed(CDocument* doc, CDocView* view)=0;
    // Called after a new document has been registered
    virtual void OnDocumentRegistered(CDocument* doc)=0;
    // Called before a document is unregistered
    virtual void OnDocumentUnregistered(CDocument* doc)=0;
};
//---------------------------------------------------------------------------
// Base class for document views
class CDocView
{
    friend class CDocument;
public:
    // Refresh the view
    virtual void Refresh()=0;
    // Returns true if this is the document's last view
    bool IsLastView() const; 
    // Returns the document this view is for
    inline CDocument* GetDocument() const { return m_document; }
protected:
    CDocView(CDocument* document);
    virtual ~CDocView();
    // Called when the document's modified flag changes
    virtual void OnModified(bool modified);
    // Called when the view has been invalidated (by default this calls Refresh
    // but a subclass can do more lazy update scheduling).  The "details"
    // parameter is a document-specific object that describes what part of the
    // document was invalidated and it can be null if the entire view was
    // invalidated (or the document doesn't support partial invalidation).
    virtual void OnInvalidated(const void* details);
private:
    CDocument*  m_document;     // The document this view is for
};
//---------------------------------------------------------------------------
// Base class for documents
class CDocument
{
    friend class CDocView;
    friend class CDocumentManager;
public:
    CDocument();
    virtual ~CDocument();
    // Returns true if the document can be modified
    virtual bool CanModify() const;
    // Mark the document as modified, returns false if the document is readonly
    bool MarkModified();
    // Remove the modification flag from the document
    void ClearModified();
    // Create a new view.  If the document supports multiple types of views,
    // the "type" parameter can be used to specify which type of view to create,
    // otherwise it is ignored.
    CDocView* NewView(int type);
    // Destroy all document views
    void DestroyAllViews();
    // Refresh all views regardless of their validity status
    void RefreshViews();
    // Invalidate all views using the given detail.  If detail is null or the
    // document and/or view subclasses do not support partial invalidation,
    // this will invalidate the entire view
    void InvalidateViews(const void* detail=0);
    // Returns the document's views
    inline const TDynArray<CDocView*>& GetViews() const { return m_views; }
    // Returns the document's filename
    inline const AString& GetFileName() const { return m_filename; }
    // Returns the document manager this document is registered to
    inline CDocumentManager* GetManager() const { return m_manager; }
    // Returns true if the document has been modified
    inline bool IsModified() const { return m_modified; }
protected:
    // Change the document's filename
    void SetFileName(const AString& filename) { m_filename = filename; }
    // Called to create a new view of the given type
    virtual CDocView* CreateView(int type)=0;
    // Called by CDocView if it gets destroyed
    void OnDocViewDestroyed(CDocView* view);
    // Called after a view has been added
    virtual OnViewAdded(CDocView* view) { (void)view; }
    // Called after a view has been removed
    virtual OnViewRemoved(CDocView* view) { (void)view; }
    // Called when the document's modified flag changes
    virtual void OnModified(bool modified) { (void) modified; }
private:
    TDynArray<CDocView*>    m_views;        // Active views for the document
    AString                 m_filename;     // The filename of the document
    CDocumentManager*       m_manager;      // Associated document manager
    bool                    m_modified;     // Modification flag
};
//---------------------------------------------------------------------------
// Document manager
class CDocumentManager
{
    friend class CDocument;
public:
    // Registers a new document with the manager
    void Register(CDocument* doc);
    // Unregisters the given document
    void Unregister(CDocument* doc);
    // Add a document listener
    void AddListener(IDocumentListener* listener);
    // Remove the given document listener
    void RemoveListener(IDocumentListener* listener);
    // Marks the given document as modified
    // Returns the registered documents in this manager
    inline const TDynArray<CDocument*>& GetDocuments() const { return m_docs; }
private:
    TDynArray<CDocument*>           m_docs;         // Registered documents
    TDynArray<IDocumentListener*>   m_listeners;    // Registered listeners
    // Call OnDocumentModified in all listeners
    void DocumentModified(CDocument* doc, bool modified);
    // Call OnViewCreated in all listeners
    void ViewCreated(CDocument* doc, CDocView* view);
    // Call OnViewDestroyed in all listeners
    void ViewDestroyed(CDocument* doc, CDocView* view);
};
//---------------------------------------------------------------------------
#endif
