//---------------------------------------------------------------------------
#ifndef ColorsH
#define ColorsH
//---------------------------------------------------------------------------
#include <math.h>
#include "DynArray.h"
#include "MathUtils.h"
//---------------------------------------------------------------------------
// A 32bit color
struct SColor
{
    unsigned char   Blue;
    unsigned char   Green;
    unsigned char   Red;
    unsigned char   Alpha;

    // Construct undefined color
    inline SColor(){}
    // Construct a color using the given values
    inline SColor(unsigned char r, unsigned char g, unsigned char b, unsigned char a=255)
        : Alpha(a), Blue(b), Green(g), Red(r)
    {}
    // Construct a copy of the given color
    inline SColor(const SColor& rhs)
        : Alpha(rhs.Alpha), Blue(rhs.Blue), Green(rhs.Green), Red(rhs.Red)
    {}
    // Construct a copy of the given color with overridden alpha
    inline SColor(const SColor& rhs, unsigned char a)
        : Alpha(a), Blue(rhs.Blue), Green(rhs.Green), Red(rhs.Red)
    {}
    // Construct a color using the given 32bit value
    inline explicit SColor(unsigned color)
        : Alpha(static_cast<unsigned char>((color&0xFF000000UL)>>24UL))
        , Blue(static_cast<unsigned char>((color&0xFF0000UL)>>16UL))
        , Green(static_cast<unsigned char>((color&0xFF000UL)>>8UL))
        , Red(static_cast<unsigned char>(color&0xFFUL))
    {}
    // Convert the color to a VCL TColor
    #if defined(GraphicsHPP) && defined(__BORLANDC__)
    inline TColor ToTColor() const
    {
        return (TColor)RGB(static_cast<unsigned>(Red),
                           static_cast<unsigned>(Green),
                           static_cast<unsigned>(Blue));
    }
    #endif
    // Convert the color to a 32bit value
    inline operator unsigned() const
    {
        return static_cast<unsigned>(Red)|
               static_cast<unsigned>(Green<<8UL)|
               static_cast<unsigned>(Blue<<16UL)|
               static_cast<unsigned>(Alpha<<24UL);
    }
    // Comparison
    inline bool operator==(const SColor& rhs) const
    {
        return Alpha == rhs.Alpha &&
               Blue == rhs.Blue &&
               Green == rhs.Green &&
               Red == rhs.Red;
    }
    inline bool operator!=(const SColor& rhs) const
    {
        return !(Alpha == rhs.Alpha &&                   
                 Blue == rhs.Blue &&
                 Green == rhs.Green &&
                 Red == rhs.Red);
    }
};
//---------------------------------------------------------------------------
// Blending modes
enum EBlendMode
{
    BM_REPLACE,     // Replace blending mode (ie. no blend)
    BM_MIX,         // Mix colors
    BM_ADD,         // Add colors
    BM_SUBTRACT,    // Subtract colors
    BM_MULTIPLY,    // Multiply colors
    BM_COLORIZE,    // Colorize (use the source's hue and saturation in dest)
    BM_HUE,         // Use the source's hue in the destination
    BM_SATURATION,  // Use the source's saturation in the destination
    BM_VALUE        // Use the source's value in the destination
};
//---------------------------------------------------------------------------
// Color functions
namespace NColor
{
    // Convert the color to HSV
    inline void ToHSV(const SColor& color, float& h, float& s, float& v)
    {
        float delta, minc, maxc, r, g, b;
        r = static_cast<float>(color.Red)/255.0f;
        g = static_cast<float>(color.Green)/255.0f;
        b = static_cast<float>(color.Blue)/255.0f;
        minc = TMin(TMin(r, g), b);
        maxc = TMax(TMax(r, g), b);
        v = maxc;
        delta = maxc - minc;
        if (maxc != 0.0f)
        {
            s = delta/maxc;
        }
        else
        {
            s = 0.0f;
            h = 0.0f;
            return;
        }
        if (delta == 0.0f)
        {
            h = 0.0f;
            return;
        }
        else if (r == maxc)
        {
            h = (g - b)/delta;
        }
        else if (g == maxc)
        {
            h = 2.0f + (b - r)/delta;
        }
        else
        {
            h = 4.0f + (r - g)/delta;
        }
        h = h*60.0f;
        if (h < 0) h += 360.0f;
    }
    // Create a color with the given HSV values
    inline SColor FromHSV(float h, float s, float v)
    {
        int i;
        float f, p, q, t;
        if (s == 0.0f)
        {
            unsigned char vb = static_cast<unsigned char>(v*255);
            return SColor(vb, vb, vb);
        }
        h /= 60.0f;
        i = (int)floor(h);
        f = h - i;
        p = v*(1.0f - s);
        q = v*(1.0f - s*f);
        t = v*(1.0f - s*(1.0f - f));
        switch (i)
        {
        case 0:
            return SColor(static_cast<unsigned char>(v*255.0f),
                          static_cast<unsigned char>(t*255.0f),
                          static_cast<unsigned char>(p*255.0f));
        case 1:
            return SColor(static_cast<unsigned char>(q*255.0f),
                          static_cast<unsigned char>(v*255.0f),
                          static_cast<unsigned char>(p*255.0f));
        case 2:
            return SColor(static_cast<unsigned char>(p*255.0f),
                          static_cast<unsigned char>(v*255.0f),
                          static_cast<unsigned char>(t*255.0f));
        case 3:
            return SColor(static_cast<unsigned char>(p*255.0f),
                          static_cast<unsigned char>(q*255.0f),
                          static_cast<unsigned char>(v*255.0f));
        case 4:
            return SColor(static_cast<unsigned char>(t*255.0f),
                          static_cast<unsigned char>(p*255.0f),
                          static_cast<unsigned char>(v*255.0f));
        default:
            return SColor(static_cast<unsigned char>(v*255.0f),
                          static_cast<unsigned char>(p*255.0f),
                          static_cast<unsigned char>(q*255.0f));
        }
    }
    // Calculate the squared distance between two colors
    inline unsigned Distance(const SColor& a, const SColor& b)
    {
        int dr = a.Red - b.Red;
        int dg = a.Green - b.Green;
        int db = a.Blue - b.Blue;
        return dr*dr + dg*dg + db*db;
    }
    // Mix the given colors
    SColor Mix(const SColor& target, const SColor& source, unsigned alpha);
}
//---------------------------------------------------------------------------
// A color palette
class CPalette
{
public:
    // Create an empty palette
    inline CPalette(){}
    // Create a palette with the given number of colors
    inline CPalette(unsigned size)
    {
        m_colors.SetSize(size);
    }
    // Create a copy of the given palette
    inline CPalette(const CPalette& rhs)
        : m_colors(rhs.m_colors)
    {}
    // Resize the palette
    inline void SetSize(unsigned size)
    {
        m_colors.SetSize(size);
    }
    // Find the closest index for the given color
    inline unsigned FindClosest(const SColor& ref) const
    {
        DEBUG_ASSERT(HasColors(), "The palette is empty");
        unsigned best = 0;
        unsigned bestd = 0xFFFFFFFF;
        for (unsigned i=0; i < GetSize(); ++i)
        {
            unsigned d = NColor::Distance(ref, m_colors[i]);
            if (d < bestd)
            {
                bestd = d;
                best = i;
            }
        }
        return best;
    }
    // Direct color access
    inline const SColor& operator[](size_t index) const { return m_colors[index]; }
    inline SColor& operator[](size_t index) { return m_colors[index]; }
    // Returns the colors in this palette
    inline const TDynArray<SColor>& GetColors() const { return m_colors; }
    // Returns the number of colors in the palette
    inline unsigned GetSize() const { return m_colors.GetSize(); }
    // Returns true if the palette is empty
    inline bool IsEmpty() const { return m_colors.IsEmpty(); }
    // Returns true if the palette has colors (not empty)
    inline bool HasColors() const { return m_colors.HasItems(); }
private:
    TDynArray<SColor>   m_colors;    
};
//---------------------------------------------------------------------------
// Blending function, blends the source into target
typedef void (*FBlender)(SColor* target, const SColor* source, size_t count, unsigned opacity);
//---------------------------------------------------------------------------
// Blending functions
namespace NBlender
{
    // Returns the blender for the given blending mode
    FBlender ByMode(EBlendMode mode);
    // BM_REPLACE function
    void Replace(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_MIX function
    void Mix(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_ADD function
    void Add(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_SUBTRACT function
    void Subtract(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_MULTIPLY function
    void Multiply(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_COLORIZE function
    void Colorize(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_HUE function
    void Hue(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_SATURATION function
    void Saturation(SColor* target, const SColor* source, size_t count, unsigned opacity);
    // BM_VALUE function
    void Value(SColor* target, const SColor* source, size_t count, unsigned opacity);
}
#endif

