//---------------------------------------------------------------------------
#ifndef CamerasH
#define CamerasH
//---------------------------------------------------------------------------
#include "Matrix.h"
#include "Ray.h"
//---------------------------------------------------------------------------
// Camera modes
enum ECameraMode
{
    CM_ORTHOGRAPHIC,
    CM_PERSPECTIVE
};
//---------------------------------------------------------------------------
// Represents a 3D camera
class CCamera
{
public:
    CCamera();
    // Move towards the given direction
    void MoveTowards(const AVector& dir);
    // Move towards the given direction in view space
    void MoveTowardsInView(const AVector& dir);
    // Move forward
    void MoveForward(float speed);
    // Rotate horizontally
    void RotateHorizontally(float angle);
    // Rotate vertically
    void RotateVertically(float angle);
    // Rotate around a point in world space
    void RotateAroundPoint(const AVector& point, float hangle, float vangle);
    // Set the camera's position
    void SetPosition(const AVector& pos);
    // Set the camera's direction
    void SetDirection(const AVector& dir);
    // Set the camera's position and direction
    void SetPositionAndDirection(const AVector& pos, const AVector& dir);
    // Set the camera's aspect ratio
    void SetAspect(float aspect);
    // Set the camera's near and far clip planes
    void SetClipPlanes(float znear, float zfar);
    // Set the camera's field of view
    void SetFOV(float fov);
    // Set the camera's orthographic scale
    void SetOrthoScale(float orthoScale);
    // Set the camera's mode
    void SetMode(ECameraMode mode);
    // Setup the camera's parameters 
    void Setup(ECameraMode mode, float fov, float aspect, float znear, float zfar, float orthoScale);
    // Convert the given vector from world space to screen space using
    // the given screen dimensions
    AVector WorldToScreen(const AVector& v, int width, int height) const;
    // Convert the given vector from screen space to world space using
    // the given screen dimensions
    AVector ScreenToWorld(const AVector& v, int width, int height) const;
    // Create a ray using the given screen coordinates for the given screen
    // dimensions
    ARay RayAtScreenCoords(int x, int y, int width, int height);
    // Returns the projection matrix (view-to-screen)
    inline const AMatrix& GetProjectionMatrix() const { return m_projection; }
    // Returns the view matrix (world-to-view)
    inline const AMatrix& GetViewMatrix() const { return m_view; }
    // Returns the unprojection matrix (screen-to-world)
    inline const AMatrix& GetUnprojectionMatrix() const { return m_unprojection; }
    // Returns the camera position
    inline const AVector& GetPosition() const { return m_position; }
    // Returns the camera's forward direction vector
    inline const AVector& GetDirection() const { return m_direction; }
    // Returns the camera's right direction vector
    inline const AVector& GetRight() const { return m_right; }
    // Returns the camera's up direction vector
    inline const AVector& GetUp() const { return m_up; }
    // Returns the camera's aspect ratio
    inline float GetAspect() const { return m_aspect; }
    // Returns the camera's near clip plane
    inline float GetZNear() const { return m_zNear; }
    // Returns the camera's far clip plane
    inline float GetZFar() const { return m_zFar; }
    // Returns the camera's vertical field of view
    inline float GetFOV() const { return m_FOV; }
    // Returns the camera's orthographic scale
    inline float GetOrthoScale() const { return m_orthoScale; }
    // Returns the camera's mode
    inline ECameraMode GetMode() const { return m_mode; }
private:
    AMatrix     m_projection;       // Projection matrix (view-to-screen)
    AMatrix     m_view;             // View matrix (world-to-view)
    AMatrix     m_unprojection;     // Unprojection matrix (screen-to-world)
    AVector     m_position;         // Camera position
    AVector     m_direction;        // Forward direction vector
    AVector     m_right;            // Right direction vector
    AVector     m_up;               // Up direction vector
    float       m_aspect;           // Aspect ratio
    float       m_zNear;            // Near clip plane
    float       m_zFar;             // Far clip plane
    float       m_FOV;              // Vertical field of view
    float       m_orthoScale;       // Scale for orthographic mode
    ECameraMode m_mode;             // Camera mode (ortho or perspective)
    // Recalculate the derived fields
    void RecalcDeriveds();
};
//---------------------------------------------------------------------------
#endif
